Встроенные валидаторы
===============

Yii предоставляет встроенный набор часто используемых валидаторов, расположенных, в первую очередь,
в пространстве имен `yii\validators`. Вместо того, чтобы использовать длинные имена классов валидаторов,
вы можете использовать *псевдонимы*, чтобы указать на использование этих валидаторов.
Например, вы можете использовать псевдоним `required`, чтобы сослаться на класс [[yii\validators\RequiredValidator]]:

```php
public function rules()
{
    return [
        [['email', 'password'], 'required'],
    ];
}
```

Все поддерживаемые псевдонимы валидаторов можно увидеть в свойстве [[yii\validators\Validator::builtInValidators]].

Ниже мы опишем основные способы использования и свойства всех встроенных валидаторов.

## [[yii\validators\BooleanValidator|boolean]] <span id="boolean"></span>

```php
[
    // Проверяет 'selected' на равенство 0 или 1, без учета типа данных
    ['selected', 'boolean'],

    // Проверяет, что "deleted" - это тип данных boolean и содержит true или false
    ['deleted', 'boolean', 'trueValue' => true, 'falseValue' => false, 'strict' => true],
]
```

Этот валидатор проверяет, что второе значение является *boolean*.

- `trueValue`: значение, соответствующее `true`. По умолчанию - `'1'`.
- `falseValue`: значение, соответствующее `false`. По умолчанию - `'0'`.
- `strict`: должна ли проверка учитывать соответствие типов данных `trueValue` или `falseValue`. По умолчанию - `false`.

> Note: Из-за того, что как правило данные, полученные из HTML-форм, представляются в виде строки, обычно вам стоит
оставить свойство [[yii\validators\BooleanValidator::strict|strict]] равным `false`.


## [[yii\captcha\CaptchaValidator|captcha]] <span id="captcha"></span>

```php
[
    ['verificationCode', 'captcha'],
]
```
Этот валидатор обычно используется вместе с [[yii\captcha\CaptchaAction]] и [[yii\captcha\Captcha]], чтобы
убедиться, что данные в инпуте соответствуют верификационному коду, отображенному с помощью виджета
[[yii\captcha\Captcha|CAPTCHA]].

- `caseSensitive`: необходимо ли учитывать чувствительность к регистру при сравнении. По умолчанию - `false`.
- `captchaAction`: [маршрут](structure-controllers.md#routes), соответствующий
  [[yii\captcha\CaptchaAction|CAPTCHA action]], который рендерит изображение с *CAPTCHA*. По умолчанию - `'site/captcha'`.
- `skipOnEmpty`: может ли валидация быть пропущена, если *input* пустой. По умолчанию - `false`,
  что означает, что *input* обязателен.


## [[yii\validators\CompareValidator|compare]] <span id="compare"></span>

```php
[
    // проверяет, является ли значение атрибута "password" таким же, как "password_repeat"
    ['password', 'compare'],
    
    // то же, что и выше, но атрибут для сравнения указан явно
    ['password', 'compare', 'compareAttribute' => 'password_repeat'],

    // проверяет, что возраст больше или равен 30
    ['age', 'compare', 'compareValue' => 30, 'operator' => '>=', 'type' => 'number'],
]
```

Этот валидатор сравнивает значение указанного атрибута с другим, чтобы удостовериться, что их отношение
соответствует описанному в свойстве `operator`.

- `compareAttribute`: имя атрибута, с которым нужно сравнить значение. Когда валидатор используется
  для проверки атрибута, значением по умолчанию для этого свойства будет имя этого же атрибута
  с суффиксом `_repeat`. Например, если проверяющийся атрибут - `password`,
  то значение свойства по умолчанию будет `password_repeat`.
- `compareValue`: постоянное значение, с которым будут сравниваться входящие данные. Когда одновременно указаны
   это свойство и `compareAttribute`, это свойство получит приоритет.
- `operator`: оператор сравнения. По умолчанию `==`, что означает проверку на эквивалентность входящих данных и в
  `compareAttribute`, и в `compareValue`. Поддерживаются следующие операторы:
     * `==`: проверяет два значения на эквивалентность. Сравнение не учитывает тип данных.
     * `===`: проверяет два значения на эквивалентность. Сравнение строгое и учитывает тип данных.
     * `!=`: проверяет, что два значения не эквивалентны. Сравнение не учитывает тип данных.
     * `!==`: проверяет, что два значения не эквивалентны. Сравнение строгое и учитывает тип данных.
     * `>`: проверяет, что валидируемое значение больше, чем то, с которым происходит сравнение.
     * `>=`: проверяет, что валидируемое значение больше или равно тому, с которым происходит сравнение.
     * `<`: проверяет, что валидируемое значение меньше, чем то, с которым происходит сравнение.
     * `<=`: проверяет, что валидируемое значение меньше или равно тому, с которым происходит сравнение.
- `type`: По умолчанию при сравнении используется тип '[[yii\validators\CompareValidator::TYPE_STRING|string]]'. То есть
   значения сравниваются побайтово. При сравнении чисел необходимо задать [[yii\validators\CompareValidator::$type|$type]]
   как '[[yii\validators\CompareValidator::TYPE_NUMBER|number]]'.     


## [[yii\validators\DateValidator|date]] <span id="date"></span>

Валидатор [[yii\validators\DateValidator|date]] можно использовать тремя способами:

```php
[
    [['from', 'to'], 'date'],
    [['from_datetime', 'to_datetime'], 'datetime'],
    [['some_time'], 'time'],
]
```

Этот валидатор проверяет соответствие входящих данных форматам *date*, *time* или *datetime*.
Опционально, он может конвертировать входящее значение в UNIX timestamp и сохранить в атрибуте,
описанном здесь: [[yii\validators\DateValidator::timestampAttribute|timestampAttribute]].

- `format`: формат даты/времени, согласно которому должна быть сделана проверка.
   Значение может быть паттерном, описанным в [руководстве ICU](https://unicode-org.github.io/icu/userguide/format_parse/datetime/#datetime-format-syntax)
   или форматом PHP префиксированным `php:`. Формат должен приниматься классом `Datetime`. Подробнее о нём можно
   прочитать в [руководстве PHP по date_create_from_format()](https://www.php.net/manual/ru/datetime.createfromformat.php).
   Если значение не задано, используется `Yii::$app->formatter->dateFormat`.
   Подробнее читайте в [[yii\validators\DateValidator::$format|документации по API]].
- `timestampAttribute`: имя атрибута, которому данный валидатор может присваивать значение UNIX timestamp,
   получаемое из входных даты и времени. Это может быть как тот же атрибут, что валидируется в данный момент. Если это так,
   после валидации оригинальное значение будет перезаписано значеним timestamp.
   Дополнительные примеры вы можете найти в разделе ["Handling date input with the DatePicker"](https://github.com/yiisoft/yii2-jui/blob/master/docs/guide/topics-date-picker.md).

  Начиная с версии 2.0.4, для атрибута могут быть заданы формат и часовой пояс через
  [[yii\validators\DateValidator::$timestampAttributeFormat|$timestampAttributeFormat]] и
  [[yii\validators\DateValidator::$timestampAttributeTimeZone|$timestampAttributeTimeZone]] соответственно.
  
  При использовании `timestampAttribute`, входное значение будет конвертировано в unix timestamp, который по определению
  содержит время в UTC. То есть будет произведено преобразование из [[yii\validators\DateValidator::timeZone|входного часового пояса]]
  в UTC.

- Начиная с версии 2.0.4 также можно задать [[yii\validators\DateValidator::$min|минимальное]] и
  [[yii\validators\DateValidator::$max|максимальное]] значение timestamp.

В том случае, если ввод не обязателен, вам может понадобится добавить [фильтр значения по умолчанию](#default) в
дополнение к валидатору даты для того, чтобы пустой ввод сохранялся как `null`. В противном случае вы можете получить
даты вроде `0000-00-00` в базе данных или `1970-01-01` в полях ввода виджета date picker.

```php
[
    [['from_date', 'to_date'], 'default', 'value' => null],
    [['from_date', 'to_date'], 'date'],
],
```


## [[yii\validators\DefaultValueValidator|default]] <span id="default"></span>

```php
[
    // установить null для "age" в качестве значения по умолчанию
    ['age', 'default', 'value' => null],

    // установить "USA" в качестве значения по умолчанию для "country"
    ['country', 'default', 'value' => 'USA'],

    // установить в "from" и "to" дату 3 дня и 6 дней от сегодняшней, если они пустые
    [['from', 'to'], 'default', 'value' => function ($model, $attribute) {
        return date('Y-m-d', strtotime($attribute === 'to' ? '+3 days' : '+6 days'));
    }],
]
```

Этот валидатор не проверяет данные. Вместо этого он присваивает значения по умолчанию проходящим проверку
атрибутам, если они пусты.

- `value`: значение по умолчанию или функция обратного вызова, которая возвращает значение по умолчанию,
  которое будет присвоено проверяемому атрибуту, если он пустой. Функция обратного вызова должна выглядеть так:

```php
function foo($model, $attribute) {
    // ... вычисление $value ...
    return $value;
}
```

> Info: Как определить, является значение пустым или нет, более подробно описано в отдельной статье
  в секции [Пустые значения](input-validation.md#handling-empty-inputs).


## [[yii\validators\NumberValidator|double]] <span id="double"></span>

```php
[
    // проверяет, является ли "salary" числом типа double
    ['salary', 'double'],
]
```

Этот валидатор проверяет, что входящее значение является корректным *double* числом. Он идентичен
валидатору [number](#number). (Прим. пер.: корректным float числом).

- `max`: верхний лимит (включительно) для значений. Если не установлен, значит, валидатор не будет проверять верхний лимит.
- `min`: Нижний лимит (включительно) для значений. Если не установлен, валидатор не будет проверять нижний лимит.


## [[yii\validators\EmailValidator|email]] <span id="email"></span>

```php
[
    // проверяет, что "email" - это корректный email-адрес
    ['email', 'email'],
]
```
Валидатор проверяет, что значение входящих данных является корректным email-адресом.

- `allowName`: можно ли передавать в атрибут имя (пример: `John Smith <john.smith@example.com>`). По умолчанию - `false`.
- `checkDNS`, проверяет, существует ли доменное имя для введенного адреса (и A, и MX запись).
  Учтите, что проверка может закончится неудачей, что может быть вызвано временными проблемами с DNS, даже если
  email-адрес корректен. По умолчанию - `false`.
- `enableIDN`, нужно ли учитывать IDN (многоязычные доменные имена). По умолчанию - `false`. Учтите, что для использования
  IDN-валидации вам нужно установить и включить PHP расширение `intl`, иначе будет выброшено исключение.


## [[yii\validators\ExistValidator|exist]] <span id="exist"></span>

```php
[
    // a1 должно существовать в столбце, который представляется атрибутом "a1"
    ['a1', 'exist'],

    // a1 должно существовать, но его значение будет использовать a2 для проверки существования
    ['a1', 'exist', 'targetAttribute' => 'a2'],

    // и a1, и a2 должны существовать, в противном случае оба атрибута будут возвращать ошибку
    [['a1', 'a2'], 'exist', 'targetAttribute' => ['a1', 'a2']],

    // и a1, и a2 должны существовать, но только атрибут a1 будет возвращать ошибку
    ['a1', 'exist', 'targetAttribute' => ['a1', 'a2']],

    // a1 требует проверки существования a2 и a3 (используя значение a1)
    ['a1', 'exist', 'targetAttribute' => ['a2', 'a1' => 'a3']],

    // a1 должен существовать. Если a1 - массив, то каждый его элемент должен существовать
    ['a1', 'exist', 'allowArray' => true],
]
```

Этот валидатор ищет входящие данные в столбце таблицы. Он работает только с атрибутами
модели [Active Record](db-active-record.md). Он поддерживает проверку и одного столбца, и нескольких.

- `targetClass`: имя класса [Active Record](db-active-record.md), который должен быть использован для проверки
  входящего значения. Если не установлен, будет использован класс текущей модели.
- `targetAttribute`: имя атрибута в `targetClass` который должен быть использован для проверки существования
  входящего значения. Если не установлен, будет использовано имя атрибута, который проверяется в данный момент.
  Вы можете использовать массив для валидации нескольких столбцов одновременно. Значения массива являются атрибутами,
  которые будут использованы для проверки существования, тогда как ключи массива будут являться атрибутами, чьи значения
  будут проверены. Если ключ и значения одинаковы, вы можете указать только значение.
- `filter`: дополнительный фильтр, который будет добавлен к запросу в базу данных для проверки на существование значения.
  Это может быть строка или массив, представляющие дополнительные условия в запросе (подробнее о формате
  значений запроса: [[yii\db\Query::where()]]), или анонимная функция с сигнатурой `function ($query)`,
  где `$query` - это [[yii\db\Query|Query]] объект, который вы можете модифицировать в функции.
- `allowArray`: разрешать ли значению быть массивом. По умолчанию - `false`. Если свойство установлено в `true`
  и тип входящих данных - массив, тогда каждый его элемент должен существовать в соответствующем столбце таблицы.
  Помните, что это свойство не может быть установлено в `true`, если вы валидируете несколько столбцов, передавая
  их в `targetAttribute` как массив.


## [[yii\validators\FileValidator|file]] <span id="file"></span>

```php
[
    // проверяет, что "primaryImage" - это загруженное изображение в формате PNG, JPG или GIF
    // размер файла должен быть меньше 1MB
    ['primaryImage', 'file', 'extensions' => ['png', 'jpg', 'gif'], 'maxSize' => 1024*1024],
]
```

Этот валидатор проверяет, что input является корректным загруженным файлом.

- `extensions`: список имен расширений, которые допустимы для загрузки. Это также может быть или массив, или
  строка, содержащая имена файловых расширений, разделенных пробелом или запятой (пр.: "gif, jpg").
  Имя расширения не чувствительно к регистру. По умолчанию - `null`, что значит, что все имена файловых расширений
  допустимы.
- `mimeTypes`: список MIME-типов, которые допустимы для загрузки. Это может быть или массив, или строка,
  содержащая MIME-типы файлов, разделенные пробелом или запятой (пример: "image/jpeg, image/png").
  В именах MIME-типов допустимо использовать символ `*` для выбора группы mime-типов.
  Например, `image/*` разрешит все типы, которы начинаются с `image/` (пример: `image/jpeg`, `image/png`).
  Имена MIME-типов не чувствительны к регистру. По умолчанию - `null`, что значит, что допустимы все MIME-типы.
  Более детальную информацию можно найти в [списке MIME-типов](https://ru.wikipedia.org/wiki/Список_MIME-типов).
- `minSize`: минимальный размер файла в байтах, разрешенный для загрузки. По умолчанию - `null`, что значит, что нет
  минимального лимита.
- `maxSize`: максимальный размер файла в байтах, разрешенный для загрузки. По умолчанию - `null`, что значит, что нет
  максимального лимита.
- `maxFiles`: максимальное количество файлов, которое может быть передано в атрибут. По умолчанию 1, что значит, что
  input должен быть файлом в единственном экземпляре. Если больше, чем 1, то атрибут должен быть массивом,
  состоящим из не более, чем `maxFiles` загруженных файлов.
- `checkExtensionByMimeType`: нужно ли проверять расширение файла исходя из его MIME-типа. Если они не соответствуют
  друг другу, то файл будет считаться некорректным. По умолчанию - `true`, то есть проверка будет произведена.

`FileValidator` используется вместе с [[yii\web\UploadedFile]]. Пожалуйста, посмотрите раздел
[Загрузка файлов](input-file-upload.md) для более полного понимания загрузки и проверки файлов.


## [[yii\validators\FilterValidator|filter]] <span id="filter"></span>

```php
[
    // обрезает пробелы вокруг "username" и "email"
    [['username', 'email'], 'filter', 'filter' => 'trim', 'skipOnArray' => true],

    // нормализует значение "phone"
    ['phone', 'filter', 'filter' => function ($value) {
        // нормализация значения происходит тут
        return $value;
    }],
    
    // нормализует значение "phone" используя функцию "normalizePhone"
    ['phone', 'filter', 'filter' => [$this, 'normalizePhone']],
        
    public function normalizePhone($value) {
        return $value;
    }
]
```

Этот валидатор не проверяет данные. Вместо этого он применяет указанный фильтр к входящему значению и
присваивает результат применения фильтра атрибуту.

- `filter`: PHP-callback, осуществляющий фильтрацию. Это может быть глобальная php функция, анонимная функция
  и т.д. Функция должна выглядеть как `function ($value) { return $newValue; }`. Это свойство обязательно должно
  быть установлено.
- `skipOnArray`: нужно ли пропускать валидацию, если входящим значением является массив. По умолчанию - `false`.
  Помните, что если фильтр не может принимать массив, вы должны установить это значение в `true`. Иначе могут
  произойти различные ошибки PHP.

> Трюк: Если вы хотите удалить пробелы вокруг значений атрибута, вы можете использовать валидатор [trim](#trim).

## [[yii\validators\ImageValidator|image]] <span id="image"></span>

```php
[
    // проверяет, что "primaryImage" - это валидное изображение с указанными размерами
    ['primaryImage', 'image', 'extensions' => 'png, jpg',
        'minWidth' => 100, 'maxWidth' => 1000,
        'minHeight' => 100, 'maxHeight' => 1000,
    ],
]
```

Этот валидатор проверяет, что входящие данные являются корректным файлом изображения. Он расширяет [file](#file)
валидатор и наследует все его свойства. Кроме того, он поддерживает следующие дополнительные свойства, специфичные
для валидации изображений:

- `minWidth`: минимальная ширина изображения. По умолчанию `null`, что значит, что нет нижнего лимита.
- `maxWidth`: максимальная ширина изображения. По умолчанию `null`, что значит, что нет верхнего лимита.
- `minHeight`: минимальная высота изображения. По умолчанию `null`, что значит, что нет нижнего лимита.
- `maxHeight`: максимальная высота изображения. По умолчанию `null`, что значит, что нет верхнего лимита.


## [[yii\validators\IpValidator|ip]] <span id="ip"></span>
```php
[
    // проверяет, что "ip_address" - это валидный IPv4 или IPv6 адрес
    ['ip_address', 'ip'],


    // проверяет, что "ip_address" - это валидный IPv6 адрес или подсеть,
    // значение будет развернуто в формат полной записи IPv6 адреса
    ['ip_address', 'ip', 'ipv4' => false, 'subnet' => null, 'expandIPv6' => true],

    // проверяет, что "ip_address" - это валидный IPv4 или IPv6 адрес,
    // разрешает использования символа отрацания `!`
    ['ip_address', 'ip', 'negation' => true],
]
```

Этот валидатор проверяет, является ли входящее значение валидным IPv4/IPv6 адресом или подсетью.
Он также может изменять значение атрибута, если включена нормализация или развертывание IPv6 адресов.

Валидатор имеет такие свойства:

- `ipv4`: может ли проверяемое значение быть IPv4 адрессом. По умолчанию `true`.
- `ipv6`: может ли проверяемое значение быть IPv6 адрессом. По умолчанию `true`.
- `subnet`: может ли проверяемое значение быть IP адресом с CIDR (подсетью), например `192.168.10.0/24`
     * `true` - указание подсети обязательно;
     * `false` - указание подсети запрещено;
     * `null` - указание подсети возможно, но не обязательно.

    По умолчанию `false`.
- `normalize`: нормализировать ли проверяемый IP адрес без CIDR к IP адресу с наименьшим CIDR
(32 для IPv4 или 128 для IPv6). Свойство действует только если `subnet` не установлен в `false`. Например:
    * `10.0.1.5` будет приведен к `10.0.1.5/32`
    * `2008:db0::1` будет приведен к `2008:db0::1/128`
- `negation`: может ли проверяемое значение иметь символ отрицания `!` в начале, например `!192.168.0.1`.
По умолчанию `false`.
- `expandIPv6`: разворачивать ли IPv6 адрес в формат полной записи.
Например, IPv6 адрес `2008:db0::1` будет развернут в `2008:0db0:0000:0000:0000:0000:0000:0001`. По умолчанию `false`.
- `ranges`: массив IPv4 или IPv6 диапазонов, которые разрешены или запрещены.

    Если свойство не установлено, все IP адреса разрешены. Иначе, правила будут проверяться последовательно до первого
вхождения. IP адрес будет запрещен, если не подпадет ни под одно правило. Например:
    ```php
    [
         'client_ip', 'ip', 'ranges' => [
             '192.168.10.128'
             '!192.168.10.0/24',
             'any' // разрешает все остальные IP адреса
         ]
    ]
    ```
В этом примере, доступ разрешен для всех IPv4 и IPv6 адресов кроме подсети `192.168.10.0/24`.
IPv4 адрес `192.168.10.128` также разрешен, так как находится перед запрещающим правилом.
- `networks`: массив псевдониимов, которые могут быть использованы в `ranges`. Формат массива:
    * ключ - имя псевдонима
    * значение - массив строк. Строка может быть как диапазоном адресов, так и другим псведонимом. Строка может иметь
    символ отрицания `!` в начале (не зависит от свойства `negation`).

    Следующие псевдонимы определены по умолчанию:
    
    * `*`: `any`
    * `any`: `0.0.0.0/0, ::/0`
    * `private`: `10.0.0.0/8, 172.16.0.0/12, 192.168.0.0/16, fd00::/8`
    * `multicast`: `224.0.0.0/4, ff00::/8`
    * `linklocal`: `169.254.0.0/16, fe80::/10`
    * `localhost`: `127.0.0.0/8', ::1`
    * `documentation`: `192.0.2.0/24, 198.51.100.0/24, 203.0.113.0/24, 2001:db8::/32`
    * `system`: `multicast, linklocal, localhost, documentation`

> Info: Этот валидатор стал доступным начиная с версии 2.0.7.


## [[yii\validators\RangeValidator|in]] <span id="in"></span>

```php
[
    // проверяет, что значение "level" равно 1, 2 или 3
    ['level', 'in', 'range' => [1, 2, 3]],
]
```

Этот валидатор проверяет, что входящее значение соответствует одному из значений, указанных в `range`.

- `range`: список значений, с которыми будет сравниваться входящее значение.
- `strict`: должно ли сравнение входящего значения со списком значений быть строгим (учитывать тип данных).
  По умолчанию `false`.
- `not`: должен ли результат проверки быть инвертирован. По умолчанию - `false`. Если свойство установлено в `true`,
  валидатор проверяет, что входящее значение НЕ соответствует ни одному из значений, указанных в `range`.
- `allowArray`: устанавливает, допустимо ли использовать массив в качестве входных данных. Если установлено в `true`
  и входящие данные являются массивом, для каждого элемента входящего массива должно быть найдено соответствие в
  `range`.


## [[yii\validators\NumberValidator|integer]] <span id="integer"></span>

```php
[
    // проверяет "age" на то, что это integer значение
    ['age', 'integer'],
]
```

Проверяет, что входящее значение является integer значением.

- `max`: верхний лимит (включительно) для числа. Если не установлено, валидатор не будет проверять верхний лимит.
- `min`: нижний лимит (включительно) для числа. Если не установлено, валидатор не будет проверять нижний лимит.


## [[yii\validators\RegularExpressionValidator|match]] <span id="match"></span>

```php
[
    // проверяет, что "username" начинается с буквы и содержит только буквенные символы,
    // числовые символы и знак подчеркивания
    ['username', 'match', 'pattern' => '/^[a-z]\w*$/i']
]
```

Этот валидатор проверяет, что входящее значение совпадает с указанным регулярным выражением.

- `pattern`: регулярное выражение, с которым должно совпадать входящее значение. Это свойство должно быть установлено,
  иначе будет выброшено исключение.
- `not`: инвертирует регулярное выражение. По умолчанию `false`, что значит, что валидация будет успешна,
   только если входящее значение совпадают с шаблоном. Если установлено в `true`, валидация будет успешна,
   только если входящее значение НЕ совпадает с шаблоном.


## [[yii\validators\NumberValidator|number]] <span id="number"></span>

```php
[
    // проверяет, является ли "salary" числом
    ['salary', 'number'],
]
```

Этот валидатор проверяет, являются ли входящие значения числовыми. Он эквивалентен валидатору [double](#double).

- `max`: верхний лимит (включительно) для числа. Если не установлено, валидатор не будет проверять верхний лимит.
- `min`: нижний лимит (включительно) для числа. Если не установлено, валидатор не будет проверять нижний лимит.


## [[yii\validators\RequiredValidator|required]] <span id="required"></span>

```php
[
    // проверяет, являются ли "username" и "password" не пустыми
    [['username', 'password'], 'required'],
]
```

Этот валидатор проверяет, являются ли входящие значения не пустыми.

- `requiredValue`: желаемое значение, которому должны соответствовать проверяемые данные. Если не установлено,
  это значит, что данные должны быть не пусты.
- `strict`: учитывать или нет соответствие типу данных при валидации (строгое сравнение).
  Если `requiredValue` не установлено, а это свойство установлено в `true`, валидатор проверит, что входящее значение
  строго не соответствует `null`; если свойство установлено в `false`, валидатор будет проверять значение на пустоту с
  приведением типов.
  Если `requiredValue` установлено, сравнение между входящими данными и `requiredValue` будет также учитывать тип
  данных, если это свойство установлено в `true`.

> Info: как определить, является ли значение пустым или нет, подробнее рассказывается
  в секции [Пустые значения](input-validation.md#handling-empty-inputs).


## [[yii\validators\SafeValidator|safe]] <span id="safe"></span>

```php
[
    // обозначает "description" как safe атрибут
    ['description', 'safe'],
]
```

Этот валидатор не проверяет данные. Вместо этого он указывает, что атрибут является
[безопасным атрибутом](structure-models.md#safe-attributes).


## [[yii\validators\StringValidator|string]] <span id="string"></span>

```php
[
    // проверяет, что "username" это строка с длиной от 4 до 24 символов
    ['username', 'string', 'length' => [4, 24]],
]
```

Этот валидатор проверяет, что входящее значение - это корректная строка с указанной длиной.

- `length`: описывает длину для строки, проходящей валидацию. Может быть определен следующими
   способами:
     * числом: точная длина, которой должна соответствовать строка;
     * массив с одним элементом: минимальная длина входящей строки (напр.: `[8]`). Это перезапишет `min`.
     * массив с двумя элементами: минимальная и максимальная длина входящей строки (напр.: `[8, 128]`).
     Это перезапишет и `min`, и `max`.
- `min`: минимальная длина входящей строки. Если не установлено, то не будет ограничения на минимальную длину.
- `max`: максимальная длина входящей строки. Если не установлено, то не будет ограничения на максимальную длину.
- `encoding`: кодировка входящей строки. Если не установлено, будет использовано значение из
  [[yii\base\Application::charset|charset]], которое по умолчанию установлено в `UTF-8`.


## [[yii\validators\FilterValidator|trim]] <span id="trim"></span>

```php
[
    // обрезает пробелы вокруг "username" и "email"
    [['username', 'email'], 'trim'],
]
```

Этот валидатор не производит проверки данных. Вместо этого он будет обрезать пробелы вокруг входящих данных.
Помните, что если входящие данные являются массивом, то они будут проигнорированы этим валидатором.


## [[yii\validators\UniqueValidator|unique]] <span id="unique"></span>

```php
[
    // a1 должен быть уникальным в столбце, который представляет "a1" атрибут
    ['a1', 'unique'],

    // a1 должен быть уникальным, но для проверки на уникальность
    // будет использован столбец a2
    ['a1', 'unique', 'targetAttribute' => 'a2'],

    // a1 и a2 вместе должны быть уникальны, и каждый из них
    // будет получать сообщения об ошибке
    [['a1', 'a2'], 'unique', 'targetAttribute' => ['a1', 'a2']],

    // a1 и a2 вместе должны быть уникальны, но только a1 будет получать сообщение об ошибке
    ['a1', 'unique', 'targetAttribute' => ['a1', 'a2']],

    // a1 должен быть уникальным, что устанавливается проверкой уникальности a2 и a3
    // (используя значение a1)
    ['a1', 'unique', 'targetAttribute' => ['a2', 'a1' => 'a3']],
]
```

Этот валидатор проверяет входящие данные на уникальность в столбце таблицы. Он работает только с
атрибутами модели [Active Record](db-active-record.md). Он поддерживает проверку либо одного столбца,
либо нескольких.

- `targetClass`: имя класса [Active Record](db-active-record.md), который должен быть использован
  для проверки значения во входящих данных. Если не установлен, будет использован класс модели, которая
  в данный момент проходит проверку.
- `targetAttribute`: имя атрибута в `targetClass`, который должен быть использован для проверки на
  уникальность входящего значения. Если не установлено, будет использован атрибут, проверяемый
  в данный момент.
  Вы можете использовать массив для проверки нескольких столбцов таблицы на уникальность. Значения массива -
  это атрибуты, которые будут использованы для валидации, а ключи массива - это атрибуты, которые предоставляют
  данные для валидации. Если ключ и значение одинаковые, вы можете указать только значение.
- `filter`: дополнительный фильтр, который можно присоединить к запросу в БД, чтобы использовать его при
  проверке значения на уникальность. Это может быть строка или массив, представляющие дополнительные условия для запроса
  (см. [[yii\db\Query::where()]] о формате условий в запросе), или анонимная функция вида `function ($query)`,
  где `$query` это объект [[yii\db\Query|Query]], который вы можете изменить в функции.


## [[yii\validators\UrlValidator|url]] <span id="url"></span>

```php
[
    // Проверяет, что "website" является корректным URL. Добавляет http:// к атрибуту "website".
    // если у него нет URI схемы
    ['website', 'url', 'defaultScheme' => 'http'],
]
```

Этот валидатор проверяет, что входящее значение является корректным URL.

- `validSchemes`: массив с указанием на URI-схему, которая должна считаться корректной. По умолчанию
  `['http', 'https']`, что означает, что и `http`, и `https` URI будут считаться корректными.
- `defaultScheme`: схема URI, которая будет присоединена к входящим данным, если в них отсутствует URI-схема.
  По умолчанию `null`, что значит, что входящие данные не будут изменены.
- `enableIDN`: должна ли валидация учитывать IDN (интернационализованные доменные имена).
  По умолчанию - `false`. Учтите, что для того, чтобы IDN валидация работала корректно, вы должны установить `intl`
  PHP расширение, иначе будет выброшено исключение.

> Note: Валидатор проверяет, что протокол и хост в URL являются корректными. Валидатор НЕ проверяет другие части URL
и НЕ предназначен для защиты от XSS или любых других видов атак. Обратитесь к секции
[Лучшие практики безопасности](security-best-practices.md) чтобы узнать больше о том, как предтвращать известные угрозы
при разработке приложений.
