<?php

/**
 * Email Template Task Class.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class VTEmailTemplateTask extends VTTask
{
	/** @var bool Sending email takes more time, this should be handled via queue all the time. */
	public $executeImmediately = true;

	/**
	 * Get field names.
	 *
	 * @return string[]
	 */
	public function getFieldNames()
	{
		return [
			'template', 'email', 'relations_email', 'emailoptout',
			'smtp', 'copy_email', 'reply_to', 'address_emails', 'attachments', 'filesFromFields'
		];
	}

	/**
	 * Execute task.
	 *
	 * @param Vtiger_Record_Model $recordModel
	 */
	public function doTask($recordModel)
	{
		if (!empty($this->template)) {
			$mailerContent['template'] = $this->template;
			if (empty($this->smtp)) {
				$mailerContent['smtp_id'] = \App\Mail::getDefaultSmtp();
			} elseif (-1 === (int) $this->smtp) {
				$templateMail = \App\Mail::getTemplate($this->template);
				$mailerContent['smtp_id'] = $templateMail['smtp_id'];
			} else {
				$mailerContent['smtp_id'] = $this->smtp;
			}
			$emailParser = \App\EmailParser::getInstanceByModel($recordModel);
			$emailParser->emailoptout = $this->emailoptout ? true : false;
			$mailerContent['to'] = [];
			if ($this->email) {
				$email = \is_array($this->email) ? implode(',', $this->email) : $this->email;
				$mailerContent['to'] = $emailParser->setContent($email)->parse()->getContent(true);
			}
			if ($this->address_emails) {
				$emails = $emailParser->setContent($this->address_emails)->getContent(true);
				foreach ($emails as $email) {
					$mailerContent['to'][] = $email;
				}
			}
			if ($this->relations_email && '-' !== $this->relations_email) {
				[$relatedModule,$relatedFieldName,$onlyFirst] = array_pad(explode('::', $this->relations_email), 3, false);
				$relationListView = Vtiger_RelationListView_Model::getInstance($recordModel, $relatedModule);
				$relationListView->setFields(['id', $relatedFieldName]);
				$relationListView->set('search_key', $relatedFieldName);
				$relationListView->set('operator', 'ny');
				if ($onlyFirst) {
					$relationListView->getQueryGenerator()->setLimit(1);
				}
				foreach ($relationListView->getAllEntries() as $relatedRecordModel) {
					$mailerContent['to'][] = $relatedRecordModel->get($relatedFieldName);
				}
			}
			if (!empty($this->reply_to)) {
				$emailsReply = \is_array($this->reply_to) ? implode(',', $this->reply_to) : $this->reply_to;
				$mailerContent['reply_to'] = $emailParser->setContent($emailsReply)->parse()->getContent(true);
			}
			unset($emailParser);
			if (empty($mailerContent['to'])) {
				return false;
			}
			if ('Contacts' === $recordModel->getModuleName() && !$recordModel->isEmpty('notifilanguage')) {
				$mailerContent['language'] = $recordModel->get('notifilanguage');
			}
			$mailerContent['recordModel'] = $recordModel;
			if (!empty($this->copy_email)) {
				$mailerContent['bcc'] = $this->copy_email;
			}
			$mailerContent['attachments'] = [];
			if ($attachments = $this->getAttachments($recordModel)) {
				$mailerContent['attachments']['ids'] = $attachments;
			}
			if (!empty($this->filesFromFields) && ($files = $this->getFilesFromFields($recordModel))) {
				$mailerContent['attachments'] = array_merge($mailerContent['attachments'], $files);
			}
			\App\Mailer::sendFromTemplate($mailerContent);
		}
	}

	/**
	 * Get attachments from document relation or multireference field.
	 *
	 * @param Vtiger_Record_Model $recordModel
	 *
	 * @return array
	 */
	public function getAttachments(Vtiger_Record_Model $recordModel): array
	{
		$documentIds = [];
		if (!empty($this->attachments)) {
			[$moduleOrFieldName, $fieldNameOrAction] = array_pad(explode('::', $this->attachments), 2, null);
			$relationListView = null;
			if ($moduleOrFieldName && $fieldNameOrAction) {
				$fieldModelList = array_intersect_key($recordModel->getModule()->getFields(), array_flip([$moduleOrFieldName]));
				$isFieldActive = false;
				foreach ($fieldModelList as $fieldModel) {
					$isFieldActive = $fieldModel->isActiveField();
				}
				if (!$recordModel->isEmpty($fieldNameOrAction) && App\Record::isExists($recordModel->get($fieldNameOrAction), $moduleOrFieldName)) {
					$relationListView = Vtiger_RelationListView_Model::getInstance(
						Vtiger_Record_Model::getInstanceById($recordModel->get($fieldNameOrAction), $moduleOrFieldName),
						'Documents'
					);
				} elseif ($isFieldActive && 'allAttachments' === $fieldNameOrAction) {
					$currentValue = $recordModel->get($moduleOrFieldName);
					$documentIds = $currentValue ? explode(',', $currentValue) : [];
				} elseif ($isFieldActive && 'latestAttachments' === $fieldNameOrAction
					&& false !== $recordModel->getPreviousValue($moduleOrFieldName)
				) {
					$previousValue = $recordModel->getPreviousValue($moduleOrFieldName);
					$previousAttachments = $previousValue ? explode(',', $previousValue) : [];
					$currentValue = $recordModel->get($moduleOrFieldName);
					$currentAttachments = $currentValue ? explode(',', $currentValue) : [];
					$documentIds = array_diff($currentAttachments, $previousAttachments);
				}
			} else {
				$relationListView = Vtiger_RelationListView_Model::getInstance($recordModel, 'Documents');
			}
			if ($relationListView) {
				$queryGenerator = $relationListView->getRelationQuery(true);
				$queryGenerator->setFields(['id']);
				$documentIds = $queryGenerator->createQuery()->column();
			}
		}
		return $documentIds;
	}

	/**
	 * Get files from fields.
	 *
	 * @param Vtiger_Record_Model $recordModel
	 *
	 * @return array
	 */
	protected function getFilesFromFields(Vtiger_Record_Model $recordModel): array
	{
		$files = [];
		foreach ($this->filesFromFields as $value) {
			$value = explode('::', $value);
			if (!$recordModel->isEmpty($value[1])) {
				$fieldValue = $recordModel->get($value[1]);
				switch ($value[0]) {
					case 'record':
						if ($attachments = $this->getFilesMultiAttachment($fieldValue)) {
							$files = array_merge($files, $attachments);
						}
						break;
					case 'relatedRecord':
						if (!empty($fieldValue)) {
							if ('Users' === $value[3]) {
							} elseif (\App\Record::isExists($fieldValue, $value[3])) {
								$relatedRecordModel = \Vtiger_Record_Model::getInstanceById($fieldValue, $value[3]);
								if (
									$relatedRecordModel->isViewable()
									&& ($relValue = $relatedRecordModel->get($value[2]))
									&& ($attachments = $this->getFilesMultiAttachment($relValue))
								) {
									$files = array_merge($files, $attachments);
								}
							}
						}
						break;
					default:
						break;
				}
			}
		}
		return $files;
	}

	/**
	 * Get files from multi attachment field type.
	 *
	 * @param string $json
	 *
	 * @return string[]
	 */
	protected function getFilesMultiAttachment(string $json): array
	{
		$files = [];
		if (\App\Json::isJson($json) && ($fieldFiles = \App\Json::decode($json))) {
			foreach ($fieldFiles as $fieldFile) {
				if (file_exists($fieldFile['path'])) {
					$files[$fieldFile['path']] = $fieldFile['name'];
				}
			}
		}
		return $files;
	}
}
