<?php
/**
 * Mail account file.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Mail;

use App\Integrations\OAuth\AbstractProvider;
use App\Log;
use App\Mail\Account\Entity\AccountPrivateEntity;
use App\Mail\Account\Entity\EntityInterface;
use App\Mail\Account\Entity\Enum\Status;
use App\Mail\Account\Repository\AccountPrivateRepository;
use App\Mail\Account\Repository\AccountRepositoryInterface;

/**
 * Mail account class.
 */
final class Account
{
	/** @var string Base module name */
	public const MODULE_NAME = 'MailAccount';
	/**
	 * List of scopes that will be used for authentication.
	 *
	 * @var array
	 */
	private array $scopes = [];
	private Server $server;
	/** @var AbstractProvider OAuth2 provider */
	private ?AbstractProvider $provider = null;
	private string $password;
	private string $userName;
	/** @var bool Debug */
	private bool $debug = false;
	private ?string $refreshToken;
	/** @var string Date time */
	private ?string $expireTime = null;
	private ?string $redirectUri;

	/** @var int Number of failed login attempts */
	private int $attempt = 0;

	public function __construct(
		protected ?EntityInterface $accountEntity = null,
		protected readonly ?AccountRepositoryInterface $repository = null
	) {
		if ($accountEntity) {
			$this
				->setLogin($accountEntity?->getLogin())
				->setPassword($accountEntity?->getPassword())
				->setRefreshToken($accountEntity?->getRefreshToken())
				->setServer($accountEntity?->getServer())
				->setRedirectUri($this->getServer()?->getRedirectUri());
		}
	}

	/**
	 * Get instance by ID.
	 *
	 * @param int $id
	 */
	public static function getInstanceById(int $id): ?self
	{
		$repository = new AccountPrivateRepository();
		$accountEntity = $repository->findById($id);

		return $accountEntity ? (new self($accountEntity, $repository)) : null;
	}

	public function setLogin(string $login)
	{
		$this->userName = $login;

		return $this;
	}

	public function setPassword(#[\SensitiveParameter] string $pwd)
	{
		$this->password = $pwd;

		return $this;
	}

	public function setRefreshToken(#[\SensitiveParameter] string $token)
	{
		$this->refreshToken = $token;

		return $this;
	}

	public function setServer(Server $server)
	{
		$this->server = $server;

		return $this;
	}

	public function setRedirectUri(string $uri)
	{
		$this->redirectUri = $uri;

		return $this;
	}

	public function isPrivate(): bool
	{
		return (bool) $this->accountEntity?->isPrivate();
	}

	public function setError(string $message)
	{
		$this->accountEntity?->setLogs($message);
	}

	public function getError(): string
	{
		return $this->accountEntity?->getLogs();
	}

	public function getSource(): ?AccountPrivateEntity
	{
		return $this->accountEntity ?? null;
	}

	public function setSource(AccountPrivateEntity $entity)
	{
		$this->accountEntity = $entity;

		return $this;
	}

	public function getPassword(): string
	{
		return $this->password;
	}

	public function getLogin(): string
	{
		return $this->userName;
	}

	/**
	 * Mail server instance.
	 *
	 * @return Server
	 */
	public function getServer(): Server
	{
		return $this->server;
	}

	/**
	 * Returns the refresh token, if defined.
	 *
	 * @return string|null
	 */
	public function getRefreshToken(): ?string
	{
		return $this->refreshToken;
	}

	/**
	 * Get expire time.
	 *
	 * @param array $fields
	 *
	 * @return string|null DateTime: date('Y-m-d H:i:s')
	 */
	public function getExpireTime(): ?string
	{
		return $this->expireTime;
	}

	/**
	 * Update require data to databese.
	 *
	 * @param string[] $fields
	 *
	 * @return void
	 */
	public function update(array $fields = [])
	{
		if (!$fields) {
			$fields = ['password', 'refresh_token', 'expire_time'];
		}
		if (!isset($this->accountEntity)) {
			return;
		}
		foreach ($fields as $fieldName) {
			switch ($fieldName) {
				case 'password':
					$this->accountEntity->setPassword($this->{$fieldName});
					break;
				case 'refresh_token':
					if (!$this->refreshToken) {
						break;
					}
					$this->accountEntity->setRefreshToken($this->refreshToken);
					break;
				case 'expire_time':
					if (!$this->expireTime) {
						break;
					}
					$this->accountEntity->setExpireTime($this->expireTime);
					break;
				case 'last_login':
					$this->accountEntity->setLastLogin(date('Y-m-d H:i:s'));
					break;
				default:
					break;
			}
		}
		if ($this->accountEntity->getChanges()) {
			$this->repository?->update($this->accountEntity);
		}
	}

	/**
	 * Requests an access token using a specified option set.
	 *
	 * @param array $options
	 *
	 * @return string
	 */
	public function getAccessToken(array $options = []): string
	{
		$provider = $this->getOAuthProvider();
		if (isset($options['code'])) {
			$provider->getAccessToken('authorization_code', $options);
		} elseif ($this->refreshToken) {
			$provider->setData(['refreshToken' => $this->refreshToken]);
			$provider->refreshToken();
		}

		if ($provider->getToken()) {
			$this->password = $provider->getToken();
			if ($provider->getRefreshToken()) {
				$this->refreshToken = $provider->getRefreshToken();
			}
			$this->expireTime = date('Y-m-d H:i:s', $provider->getExpires());
		}

		return $provider->getToken();
	}

	/**
	 * Get OAuth provider.
	 *
	 * @return \App\Integrations\OAuth\AbstractProvider
	 */
	public function getOAuthProvider(): AbstractProvider
	{
		if (!$this->provider) {
			$this->provider = \App\Integrations\OAuth::getProviderByName($this->getServer()->get('oauth_provider'));
			$this->provider->setData([
				'clientId' => $this->getServer()->get('client_id'),
				'clientSecret' => $this->getServer()->getClientSecret(),
				'redirectUri' => $this->redirectUri ?? $this->getServer()->getRedirectUri(),
				'scopes' => $this->scopes ?: $this->provider->getScopesByAction(self::MODULE_NAME)
			]);
		}

		return $this->provider;
	}

	/**
	 * Check if mail account is active.
	 *
	 * @return bool
	 */
	public function isActive(): bool
	{
		return Status::STATUS_ACTIVE === $this->getSource()->getStatus();
	}

	/**
	 * Open imap connection.
	 *
	 * @return Connections\Imap
	 */
	public function openImap(): Connections\Imap
	{
		$imap = new Connections\Imap([
			'host' => $this->getServer()->get('imap_host'),
			'port' => $this->getServer()->get('imap_port'),
			'encryption' => $this->getServer()->get('imap_encrypt'), // 'ssl',
			'validateCert' => (bool) $this->getServer()->get('validate_cert'),
			'authentication' => 'oauth2' === $this->getServer()->get('auth_method') ? 'oauth' : null,
			'username' => $this->userName,
			'password' => $this->password,
			'debug' => $this->debug,
		]);
		try {
			++$this->attempt;
			$imap->connect();
			$this->update(['last_login']);
		} catch (\Throwable $th) {
			// try only once if token has expired
			if (1 === $this->attempt && 'oauth2' === $this->getServer()->get('auth_method')) {
				$this->getAccessToken();
				$this->update();
				return $this->openImap();
			}

			Log::error("IMAP connect - Account: {$this->accountEntity?->getId()}, message: " . $th->getMessage());
			throw $th;
		}

		return $imap;
	}

	/**
	 * Lock mail account.
	 *
	 * @param string $messages
	 *
	 * @return $this
	 */
	public function lock(string $messages): self
	{
		$messages = \App\Purifier::decodeHtml(\App\Purifier::encodeHtml($messages));
		$messages = \App\TextUtils::textTruncate($messages, 65500, true, true);

		$this->accountEntity?->setStatus(Status::STATUS_LOCKED);
		$this->accountEntity?->setLogs($messages);
		$this->repository?->update($this->accountEntity);

		return $this;
	}

	/**
	 * Unlock mail account.
	 *
	 * @return $this
	 */
	public function unlock(): self
	{
		$this->accountEntity?->setStatus(Status::STATUS_ACTIVE);
		$this->accountEntity?->setLogs('');
		$this->repository?->update($this->accountEntity);

		return $this;
	}

	/**
	 * Deactiveate mail account.
	 *
	 * @param string|null $messages
	 *
	 * @return $this
	 */
	public function deactivate(?string $messages = null): self
	{
		if (null !== $messages) {
			$messages = \App\Purifier::decodeHtml(\App\Purifier::encodeHtml($messages));
			$messages = \App\TextUtils::textTruncate($messages, 65500, true, true);
			$this->accountEntity?->setLogs($messages);
		}

		$this->accountEntity?->setStatus(Status::STATUS_INACTIVE);
		$this->repository?->update($this->accountEntity);

		return $this;
	}

	/**
	 * Get actions.
	 *
	 * @return array
	 */
	public function getActions(): array
	{
		$actions = $this->accountEntity?->get('scanner_actions');
		return $actions ? explode(',', $actions) : [];
	}

	/**
	 * Get folders.
	 *
	 * @return array
	 */
	public function getFolders(): array
	{
		$folders = $this->accountEntity?->getFolders();
		return $folders ?: [];
	}

	/**
	 * Sets the debug mode for the account.
	 *
	 * @param bool $debug True to enable debug mode, false to disable it.
	 *
	 * @return $this
	 */
	public function setDebug(bool $debug): self
	{
		$this->debug = $debug;
		return $this;
	}
}
