<?php
/**
 * System cli file.
 *
 * @package Cli
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Cli;

use App\Colors;
use App\Company;
use App\Config;
use App\ConfigFile;
use App\Db;
use App\Db\Fixer;
use App\Exceptions\DbException;
use App\Exceptions\NoPermitted;
use App\Language;
use App\Module;
use App\Session;
use App\UserPrivilegesFile;
use App\YetiForce\Register;
use App\YetiForce\Updater;
use vtlib\Package;
use yii\db\Exception;

/**
 * System cli class.
 */
class System extends Base
{
	/** {@inheritdoc} */
	public string $moduleName = 'System';

	/** {@inheritdoc} */
	protected array $methods = [
		'history' => 'History of uploaded updates',
		'update' => 'Update',
		'checkRegStatus' => 'Check registration status',
		'deleteRegistration' => 'Delete registration data',
		'showProducts' => 'Show active products',
		'reloadModule' => 'Reload modules',
		'reloadUserPrivileges' => 'Reload users privileges',
		'reloadMenus' => 'Reload menus',
	];

	/**
	 * History of uploaded updates.
	 *
	 * @return void
	 */
	public function history(): void
	{
		$table = array_map(function ($item) {
			$item['result'] = $item['result'] ? 'OK' : 'Error';
			unset($item['id']);
			return $item;
		}, \Settings_Updates_Module_Model::getUpdates());
		if ($table) {
			$this->climate->table($table);
		} else {
			$this->climate->lightGreen('No updates');
		}
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	/**
	 * Update CRM.
	 *
	 * @return void
	 */
	public function update(): void
	{
		$maxExecutionTime = \ini_get('max_execution_time');
		if ($maxExecutionTime < 1 || $maxExecutionTime > 600) {
			$this->climate->lightGreen('Max execution time = ' . $maxExecutionTime);
		} else {
			$this->climate->lightRed('Max execution time = ' . $maxExecutionTime);
		}
		$this->climate->arguments->add(
			[
				'type' => [
					'prefix' => 't',
					'description' => 'Update type. Values: patches, version',
				],
				'version' => [
					'prefix' => 'v',
					'description' => 'Chose version',
				]
			]
		);
		if ($this->isHelpMode()) {
			return;
		}
		$this->climate->arguments->parse();

		if ($this->climate->arguments->defined('version')) {
			$version = trim((string) $this->climate->arguments->get('version'));
			$version ? $this->updateByHash($version) : $this->updateTable();
		} elseif ($this->climate->arguments->defined('type')) {
			$this->updateByType($this->climate->arguments->get('type'));
		} else {
			$this->updateBySelect();
		}
	}

	/**
	 * Check registration status.
	 *
	 * @throws DbException
	 *
	 * @return void
	 */
	public function checkRegStatus(): void
	{
		$registration = new Register();
		$this->climate->bold('Status: ' . Language::translate(
			Register::STATUS_MESSAGES[$registration->getStatus(true)],
			'Settings::Companies'
		));
		if ($error = $registration->getError()) {
			$this->climate->lightRed('Status error: ' . Language::translateSingleMod($error, 'Other.Exceptions'));
		}
		$this->climate->border('─', 200);
		$this->climate->bold('APP ID: ' . Register::getInstanceKey());
		$this->climate->border('─', 200);
		$this->climate->bold('CRM ID: ' . Register::getCrmKey());
		$this->climate->border('─', 200);
		$this->climate->bold('Provider: ' . Register::getProvider());
		$this->climate->border('─', 200);
		$table = [];
		foreach (Company::getCompany() as $key => $value) {
			$table[$key] = $value;
		}
		$this->climate->table([$table]);
		$this->climate->border('─', 200);
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	/**
	 * Show active products.
	 *
	 * @return void
	 */
	public function showProducts(): void
	{
		$table = Register::getProducts();

		$table ? $this->climate->table($table) : $this->climate->bold('None');
		$this->climate->border('─', 200);
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	/**
	 * Reload modules.
	 *
	 * @throws NoPermitted
	 *
	 * @return void
	 */
	public function reloadModule(): void
	{
		$this->climate->bold('Tools: ' . Fixer::baseModuleTools());
		$this->climate->bold('Actions: ' . Fixer::baseModuleActions());
		$this->climate->bold('Profile field: ' . Fixer::profileField());
		$this->climate->bold('Share: ' . Fixer::share());
		Module::createModuleMetaFile();
		$this->climate->bold('Create module meta file');
		Colors::generate();
		$this->climate->bold('Colors');
		$this->climate->lightYellow()->border('─', 200);
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	/**
	 * Reload users privileges.
	 *
	 * @return void
	 */
	public function reloadUserPrivileges(): void
	{
		$this->climate->bold('Users: ' . UserPrivilegesFile::recalculateAll());
		$this->climate->lightYellow()->border('─', 200);
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	/**
	 * Delete registration data.
	 *
	 * @throws Exception
	 *
	 * @return void
	 */
	public function deleteRegistration(): void
	{
		$confirmation = $this->climate->confirm(Language::translate('DELETE_REGISTRATION_QUESTION', 'CLI'));
		if (!$confirmation->confirmed()) {
			$this->climate->lightYellow(Language::translate('OPERATION_CANCELED', 'CLI'));
			return;
		}

		$this->deleteDatabaseTable();
		$this->deleteRegistrationFile();
		$this->updateApplicationUniqueKey();
		Session::cleanAll();

		$this->climate->lightGreen(Language::translate('REGISTRATION_DELETED', 'CLI'));
	}

	/**
	 * Reload menus.
	 *
	 * @return void
	 */
	public function reloadMenus(): void
	{
		$menuRecordModel = new \Settings_Menu_Record_Model();
		$menuRecordModel->refreshMenuFiles();
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	private function deleteDatabaseTable(): void
	{
		$db = Db::getInstance('admin');
		if (null !== $db->getSchema()->getTableSchema('s_#__reg_data')) {
			$db->createCommand()->truncateTable('s_#__reg_data')->execute();
		}
	}

	private function deleteRegistrationFile(): void
	{
		$registerFile = Register::REGISTRATION_FILE;
		if (file_exists($registerFile)) {
			unlink($registerFile);
		}
	}

	private function updateApplicationUniqueKey(): void
	{
		$configFile = new ConfigFile('main');
		Config::set('main', 'application_unique_key', sha1(microtime() . '-' . random_int(1, 9999999)));
		$configFile->create();
	}

	/**
	 * Update by package type.
	 *
	 * @param string $type Package type. Values: patches, version
	 *
	 * @return void
	 */
	private function updateByType(string $type): void
	{
		$types = ['patches', 'version'];
		if (!\in_array($this->climate->arguments->get('type'), $types)) {
			$this->climate->white('Type not found. Allowed types:')->columns($types);
			return;
		}
		$versionUpdate = 'version' === $type;
		foreach (Updater::getToInstall() as $package) {
			$versionCompare = $package['fromVersion'] !== $package['toVersion'];
			if (($versionCompare && !$versionUpdate) || (!$versionCompare && $versionUpdate)) {
				continue;
			}
			if (!Updater::isDownloaded($package)) {
				$this->climate->inline($package['label'] . ' - Downloading a package ...');
				Updater::download($package);
				$this->climate->out('- downloaded');
			}
			$this->updateByPackage($package);
		}
	}

	/**
	 * Update by selecting a package.
	 *
	 * @return void
	 */
	private function updateBySelect(): void
	{
		$options = [];
		$toInstall = Updater::getToInstall();
		foreach ($toInstall as $package) {
			$option = "{$package['label']}";
			if ($package['fromVersion'] !== $package['toVersion']) {
				$option .= " ({$package['fromVersion']} >> {$package['toVersion']})";
			}
			if (Updater::isDownloaded($package)) {
				$option .= ' - Downloaded, ready to install';
			} else {
				$option .= ' - To download';
			}
			$options[$package['hash']] = $option;
		}
		if (!$options) {
			$this->climate->lightBlue('No updates available');
			return;
		}
		$input = $this->climate->radio('Updates available:', $options);
		$hash = $input->prompt();
		foreach ($toInstall as $package) {
			if ($package['hash'] === $hash) {
				if (Updater::isDownloaded($package)) {
					$this->updateByPackage($package);
				} else {
					Updater::download($package);
					$this->update();
				}
				return;
			}
		}
	}

	/**
	 * Update package.
	 *
	 * @param array $package
	 *
	 * @return void
	 */
	private function updateByPackage(array $package): void
	{
		$startTime = microtime(true);
		try {
			$packageInstance = new Package();
			$this->climate->white($package['label'] . ' - Installing the package');
			$path = ROOT_DIRECTORY . \DIRECTORY_SEPARATOR . \Settings_ModuleManager_Module_Model::getUploadDirectory() . \DIRECTORY_SEPARATOR . $package['hash'] . '.zip';
			$response = $packageInstance->import($path, true);
			if ($packageInstance->_errorText) {
				$this->climate->lightRed($packageInstance->_errorText);
			} else {
				echo $response . PHP_EOL;
				unlink($path);
			}
		} catch (\Throwable $th) {
			$this->climate->lightRed($th->__toString());
		}
		$this->climate->lightBlue('Total time: ' . round(microtime(true) - $startTime, 2));
	}

	/**
	 * Show table of available updates.
	 */
	private function updateTable()
	{
		$toInstall = Updater::getToInstall();
		$table = [];

		foreach ($toInstall as $package) {
			if (Updater::isDownloaded($package)) {
				$desc = 'Downloaded, ready to install';
			} else {
				$desc = 'To download';
			}

			$table[] = [
				'Parameter' => $package['label'],
				'fromVersion' => $package['fromVersion'] ?? '-',
				'toVersion' => $package['toVersion'] ?? '-',
				'Value' => $package['hash'],
				'description' => $desc
			];
		}

		$table ? $this->climate->table($table) : $this->climate->bold('No updates available');
	}

	/**
	 * Install the update package by hash.
	 *
	 * @param string $hash
	 */
	private function updateByHash(string $hash)
	{
		$toInstall = Updater::getToInstall();
		if (false === array_search($hash, array_column($toInstall, 'hash'))) {
			$this->climate->lightBlue('No update');
		}

		foreach ($toInstall as $package) {
			if ($package['hash'] === $hash) {
				if (!Updater::isDownloaded($package)) {
					Updater::download($package);
				}
				$this->updateByPackage($package);
				break;
			}
		}
	}
}
