<?php

/* +***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.0
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * All Rights Reserved.
 * Contributor(s): YetiForce S.A.
 * *********************************************************************************** */
use App\Colors;
use App\Config;
use App\Exceptions\Security;
use App\Fields\Picklist;
use App\Language;
use App\Layout;
use App\Layout\Media;
use App\Purifier;
use App\TextUtils;
use App\Utils;
use App\Utils\Completions;

class Vtiger_Multipicklist_UIType extends Vtiger_Base_UIType
{
	/** @var string Value separator in the database */
	const SEPARATOR = ' |##| ';

	/** {@inheritdoc} */
	public function getDbConditionBuilderValue($value, string $operator)
	{
		$values = [];
		if (!\is_array($value)) {
			$value = $value ? explode('##', $value) : [];
		}
		foreach ($value as $val) {
			$values[] = parent::getDbConditionBuilderValue($val, $operator);
		}
		return implode('##', $values);
	}

	/** {@inheritdoc} */
	public function getDBValue($value, $recordModel = false)
	{
		if (\is_array($value)) {
			$value = implode(self::SEPARATOR, $value);
		}
		return Purifier::decodeHtml($value);
	}

	/** {@inheritdoc} */
	public function validate($value, $isUserFormat = false)
	{
		$hashValue = '';
		if (\is_string($value)) {
			$hashValue = $value;
			$value = explode(self::SEPARATOR, $value);
		} elseif (\is_array($value)) {
			$hashValue = implode(self::SEPARATOR, $value);
		}

		if (empty($value) || isset($this->validate[$hashValue])) {
			return;
		}
		if (!\is_array($value)) {
			throw new Security('ERR_ILLEGAL_FIELD_VALUE||' . $this->getFieldModel()->getName() . '||' . $this->getFieldModel()->getModuleName() . '||' . $value, 406);
		}
		foreach ($value as $item) {
			if (!\is_string($item)) {
				throw new Security('ERR_ILLEGAL_FIELD_VALUE||' . $this->getFieldModel()->getName() . '||' . $this->getFieldModel()->getModuleName() . '||' . Utils::varExport($value), 406);
			}
			if ($item != strip_tags($item)) {
				throw new Security('ERR_ILLEGAL_FIELD_VALUE||' . $this->getFieldModel()->getName() . '||' . $this->getFieldModel()->getModuleName() . '||' . Utils::varExport($value), 406);
			}
		}

		if ($hashValue) {
			$this->validateFieldLength($hashValue);
		}

		$this->validate[$hashValue] = true;
	}

	/** {@inheritdoc} */
	public function getDisplayValue($value, $record = false, $recordModel = false, $rawText = false, $length = false)
	{
		if (empty($value)) {
			return null;
		}
		$valueRaw = $valueHtml = '';
		$values = explode(self::SEPARATOR, $value);
		$trValueRaw = $trValue = [];
		$moduleName = $this->getFieldModel()->getModuleName();
		$fieldName = Colors::sanitizeValue($this->getFieldModel()->getName());
		foreach ($values as $value) {
			$popoverClass = '';
			$popoverContent = '';
			$displayValue = Language::translate($value, $moduleName);
			if (!$rawText) {
				if ($icon = Picklist::getIcon($this->getFieldModel()->getName(), $value) ?: '') {
					['type' => $type, 'name' => $name] = $icon;
					$icon = '';
					if ('icon' === $type) {
						$icon = "<span class=\"{$name} mr-1\"></span>";
					} elseif ('image' === $type && ($src = Media::getImageUrl($name))) {
						$icon = '<img class="icon-img--picklist mr-1" src="' . $src . '">';
					}
				}
				if ($description = Picklist::getDescription($this->getFieldModel()->getName(), $value) ?: '') {
					$popoverClass = ' js-popover-tooltip';
					$popoverContent = \sprintf(' data-placement="top" data-content="%s"', Purifier::encodeHtml(Completions::decode($description)));
				}
			}
			$value = Colors::sanitizeValue($value);
			$trValueRaw[] = $displayValue;
			$trValue[] = "<span class=\"picklistValue picklistLb_{$moduleName}_{$fieldName}_{$value}$popoverClass\"$popoverContent>{$icon}{$displayValue}</span>";
		}
		if ($rawText) {
			$valueRaw = str_ireplace(self::SEPARATOR, ', ', implode(self::SEPARATOR, $trValueRaw));
			if (\is_int($length)) {
				$valueRaw = TextUtils::textTruncate($valueRaw, $length);
			}
		} else {
			$valueHtml = str_ireplace(self::SEPARATOR, ' ', implode(self::SEPARATOR, $trValue));
			if (\is_int($length)) {
				$valueHtml = Layout::truncateHtml($valueHtml, 'miniHtml', $length ?: Config::main('listview_max_textlength'), false);
			}
		}
		return $rawText ? $valueRaw : $valueHtml;
	}

	/** {@inheritdoc} */
	public function getEditViewDisplayValue($value, $recordModel = false)
	{
		if (\is_array($value)) {
			return $value;
		}

		return $value ? explode(self::SEPARATOR, Purifier::encodeHtml($value)) : [];
	}

	/** {@inheritdoc} */
	public function getValueFromImport($value, $defaultValue = null)
	{
		$trimmedValue = trim($value);
		if ('' === $trimmedValue) {
			return $defaultValue ?? '';
		}
		$explodedValue = explode(self::SEPARATOR, $trimmedValue);
		foreach ($explodedValue as $key => $value) {
			$explodedValue[$key] = trim($value);
		}
		return implode(self::SEPARATOR, $explodedValue);
	}

	/** {@inheritdoc} */
	public function getTemplateName()
	{
		return 'Edit/Field/MultiPicklist.tpl';
	}

	/** {@inheritdoc} */
	public function getListSearchTemplateName()
	{
		return 'List/Field/MultiPicklist.tpl';
	}

	/** {@inheritdoc} */
	public function getAllowedColumnTypes()
	{
		return ['text'];
	}

	/** {@inheritdoc} */
	public function getDefaultOperator()
	{
		return 'c';
	}

	/** {@inheritdoc} */
	public function getQueryOperators()
	{
		return ['e', 'n', 'c', 'k', 'y', 'ny', 'ef', 'nf'];
	}

	/**
	 * Returns template for operator.
	 *
	 * @param string $operator
	 *
	 * @return string
	 */
	public function getOperatorTemplateName(string $operator = '')
	{
		return 'ConditionBuilder/Picklist.tpl';
	}
}
