<?php
/**
 * Aggregate fields.
 *
 * @package   Fields
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Adrian Kon <a.kon@yetiforce.com>
 */

 /** Aggregate fields class */
class ReportTemplate_Aggregate_Field extends ReportTemplate_BaseReportField_Model
{
	public function __construct(
		private string $sourceModuleName,
		private array $relatedItems = [],
	) {
	}

	/** Get fields */
	public function getFields(): array
	{
		$moduleFields = $this->getBaseModuleFields($this->sourceModuleName);
		$moduleInventoryFields = $this->getInventoryFields($this->sourceModuleName);
		$reportFields = array_merge($moduleFields, $moduleInventoryFields);
		foreach ($this->relatedItems as $item) {
			if (isset($item['isRelatedAdvModule']) && $item['isRelatedAdvModule']) {
				$advRelatedModuleFields = $this->getAdvRelatedModuleFields($this->sourceModuleName, $item['relatedItem']);
				$reportFields = array_merge($reportFields, $advRelatedModuleFields);
			} elseif (App\Module::getModuleId($item['relatedItem'])) {
				$moduleReferenceFields = $this->getReferenceReportFields($item['sourceModule'], $item['relatedItem']);
				$moduleReferenceInventoryFields = $this->getInventoryFields($item['relatedItem']);
				$reportFields = array_merge($reportFields, $moduleReferenceFields, $moduleReferenceInventoryFields);
			} else {
				$relationFields = $this->getRelationFields($item['relatedItem']);
				$reportFields = array_merge($reportFields, $relationFields);
			}
		}
		return $reportFields;
	}

	/** Get base module fields */
	private function getBaseModuleFields(string $moduleName): array
	{
		$moduleFields = [];
		$moduleModel = Vtiger_Module_Model::getInstance($moduleName);
		if (App\Privilege::isPermitted($moduleName)) {
			$translatedModuleName = App\Language::translate($moduleName, $moduleName);
			foreach ($moduleModel->getFields() as $fieldModel) {
				if ($fieldModel->isActiveField() && $fieldModel->getPermissions() && ($aggregateFunctions = $this->getAggregations($fieldModel->getFieldDataType()))) {
					$fieldLabel = App\Language::translate($fieldModel->getLabel(), $moduleName);
					if (in_array($fieldModel->getFieldDataType(), self::AGGREGATE_FIELD_TYPES)){
						$reportField = $this->createReportField($fieldModel);
						$reportField->setAggregateLabel($fieldLabel);
						$moduleFields[$translatedModuleName][] = $reportField;
					}
					foreach ($aggregateFunctions as $functionName) {
						$reportField = $this->createReportField($fieldModel);
						$reportField->setAggregateFunction($functionName);
						$reportField->setAggregateLabel(App\Language::translate($functionName, 'ReportTemplate') . ' ' . $fieldLabel);
						$moduleFields[$translatedModuleName][] = $reportField;
					}
				}
			}
			foreach ($this->getAggregations('entityidfield') as $functionName) {
				$baseEntityField = $this->getBaseIdModuleField($moduleName);
				$baseEntityField->setAggregateFunction($functionName);
				$baseEntityField->setAggregateLabel($translatedModuleName . ' ' . App\Language::translate($functionName, 'ReportTemplate') . ' ' . $baseEntityField->getLabel());
				$moduleFields = $this->unshiftField($moduleFields, $baseEntityField);
			}
		}
		return $moduleFields;
	}

	/** Get inventory fields */
	private function getInventoryFields(string $moduleName): array
	{
		$inventoryFields = [];
		$advancedBlockLabel = App\Language::translate('LBL_INVENTORY_BLOCK', 'ReportTemplate');
		$moduleModel = Vtiger_Module_Model::getInstance($moduleName);
		if ($moduleModel->isInventory() && $moduleModel->isActive() && App\Privilege::isPermitted($moduleName)) {
			$moduleName = $moduleModel->getName();
			$inventoryModel = \Vtiger_Inventory_Model::getInstance($moduleName);
			$translatedModuleName = $this->getTranslatedModuleName($moduleName);
			$inventoryGroupLabel = sprintf(
				'[%s] - %s',
				$translatedModuleName,
				$advancedBlockLabel
			);
			foreach ($inventoryModel->getFields() as $fieldName => $fieldModel) {
				$label = App\Language::translate($fieldModel->getLabel(), 'ReportTemplate');
				if (in_array($fieldModel->getType(), self::AGGREGATE_FIELD_TYPES)){
					$reportField = ReportTemplate_FieldReport_Model::fromInventory($fieldModel);
					$reportField->setExpressionField(sprintf(
						'%s:%s:%s',
						$fieldName,
						$moduleName,
						'INVENTORY'
					));
					$reportField->setAggregateLabel(sprintf(
						'[%s] [%s] %s',
						$translatedModuleName,
						$advancedBlockLabel,
						$label,

					));
					$inventoryFields[$inventoryGroupLabel][] = $reportField;
				}
				if ($aggregateFunctions = $this->getAggregations($fieldModel->getType())) {
					foreach ($aggregateFunctions as $functionName) {
						$reportField = ReportTemplate_FieldReport_Model::fromInventory($fieldModel);
						$reportField->setAggregateFunction($functionName);
						$reportField->setExpressionField("{$fieldName}:{$moduleName}:INVENTORY");
						$reportField->setAggregateLabel(sprintf(
							'[%s] [%s] %s %s',
							$translatedModuleName,
							$advancedBlockLabel,
							App\Language::translate($functionName, 'ReportTemplate'),
							$label,

						));
						$inventoryFields[$inventoryGroupLabel][] = $reportField;
					}
				}
			}
		}
		return $inventoryFields;
	}

	/** Get field related with advanced block */
	private function getAdvRelatedModuleFields(string $inventoryModule, string $relatedInvModule): array
	{
		$inventoryFields = [];
		$moduleModel = Vtiger_Module_Model::getInstance($inventoryModule);
		if ($moduleModel->isInventory() && $moduleModel->isActive()) {
			$inventoryModel = \Vtiger_Inventory_Model::getInstance($inventoryModule);
			if ($inventoryModel->isField('name')) {
				$inventoryFieldName = $inventoryModel->getField('name');
				foreach ($inventoryFieldName->getModules() as $inventoryModuleName) {
					if ($relatedInvModule === $inventoryModuleName && \App\Module::isModuleActive($inventoryModuleName)) {
						$advancedModuleModel = Vtiger_Module_Model::getInstance($inventoryModuleName);
						$translatedModuleName = $this->getTranslatedModuleName($relatedInvModule);
						foreach ($advancedModuleModel->getFields() as $fieldName => $fieldModel) {
							$fieldModel = Vtiger_Field_Model::getInstance($fieldName, $advancedModuleModel);
							if ($fieldModel->isActiveField() && $fieldModel->getPermissions() && ($aggregateFunctions = $this->getAggregations($fieldModel->getFieldDataType()))) {
								$blockLabel = $this->getTranslatedBlockName($relatedInvModule, $fieldModel->getBlockName());
								$labelGroup = "[{$translatedModuleName}] {$blockLabel}";
								if (in_array($fieldModel->getFieldDataType(), self::AGGREGATE_FIELD_TYPES)){
									$reportField = $this->createReportField($fieldModel);
									$reportField->setExpressionField(sprintf(
										'%s:%s:%s:%s',
										$fieldModel->getName(),
										$relatedInvModule,
										'name',
										'INVENTORY'
									));
									$reportField->setAggregateLabel(sprintf(
										'[%s] %s',
										$translatedModuleName,
										App\Language::translate($fieldModel->getLabel(), $relatedInvModule),

									));
									$inventoryFields[$labelGroup][] = $reportField;
								}
								foreach ($aggregateFunctions as $functionName) {
									$reportField = $this->createReportField($fieldModel);
									$reportField->setAggregateFunction($functionName);
									$reportField->setAggregateLabel(sprintf(
										'[%s] %s %s',
										$translatedModuleName,
										App\Language::translate($functionName, 'ReportTemplate'),
										App\Language::translate($fieldModel->getLabel(), $relatedInvModule),

									));
									$reportField->setExpressionField("{$fieldModel->getName()}:{$relatedInvModule}:name:INVENTORY");
									$inventoryFields[$labelGroup][] = $reportField;
								}
							}
						}
						foreach ($this->getAggregations('entityidfield') as $functionName) {
							$baseEntityField = $this->getBaseIdModuleField($inventoryModuleName);
							$baseEntityField->setAggregateFunction($functionName);
							$baseEntityField->setExpressionField("{$baseEntityField->getFieldName()}:{$inventoryModuleName}:name:INVENTORY");
							$baseEntityField->setAggregateLabel(App\Language::translate($functionName, 'ReportTemplate') . ' ' . $baseEntityField->getLabel());
							$inventoryFields = $this->unshiftField($inventoryFields, $baseEntityField);
						}
					}
				}
			}
		}
		return $inventoryFields;
	}

	/** Get field from reference modules */
	public function getReferenceReportFields(string $sourceModule, string $referenceModule): array
	{
		$referenceModuleFields = [];
		$moduleModel = Vtiger_Module_Model::getInstance($sourceModule);
		$fieldTypes = 'Users' === $referenceModule ? 'owner' : \Vtiger_Field_Model::$referenceTypes;
		foreach ($moduleModel->getFieldsByType($fieldTypes) as $parentFieldName => $field) {
			if ('owner' === $field->getFieldDataType()) {
				$relatedModule = 'Users';
			} else {
				$referenceModuleList = $field->getReferenceList();
				$relatedModule = \in_array($referenceModule, $referenceModuleList) ? $referenceModule : null;
			}
			if ($referenceModule === $relatedModule && App\Privilege::isPermitted($relatedModule)) {
				$relatedModuleModel = \Vtiger_Module_Model::getInstance($relatedModule);
				$translatedModuleName = App\Language::translate($relatedModule, $relatedModule);
				foreach ($relatedModuleModel->getFields() as $fieldModel) {
					$fieldModelDataType = $fieldModel->getFieldDataType();
					if ($fieldModel->isActiveField() && $fieldModel->getPermissions() && ($aggregateFunctions = $this->getAggregations($fieldModelDataType))) {
						if(in_array($fieldModelDataType, self::AGGREGATE_FIELD_TYPES)){
							$reportField = $this->createReportField($fieldModel);
							$reportField->setExpressionField(sprintf(
								'%s:%s:%s',
								$fieldModel->getName(),
								$relatedModule, $parentFieldName,
							));
							$reportField->setAggregateLabel(\App\Language::translate($fieldModel->getLabel(), $relatedModule));
							$referenceModuleFields[$translatedModuleName][] = $reportField;
						}
						foreach ($aggregateFunctions as $functionName) {
							$reportField = $this->createReportField($fieldModel);
							$reportField->setExpressionField("{$fieldModel->getName()}:{$relatedModule}:{$parentFieldName}");
							$reportField->setAggregateFunction($functionName);
							$reportField->setAggregateLabel(sprintf(
								'[%s] %s - %s',
								$translatedModuleName,
								App\Language::translate($functionName, 'ReportTemplate'),
								\App\Language::translate($fieldModel->getLabel(), $relatedModule),
							));
							$referenceModuleFields[$translatedModuleName][] = $reportField;
						}
					}
				}
				foreach ($this->getAggregations('entityidfield') as $functionName) {
					$baseEntityField = $this->getBaseIdModuleField($relatedModule);
					$baseEntityField->setAggregateFunction($functionName);
					$baseEntityField->setExpressionField("{$baseEntityField->getFieldName()}:{$relatedModule}:{$parentFieldName}");
					$baseEntityField->setAggregateLabel(App\Language::translate($functionName, 'ReportTemplate') . ' ' . $baseEntityField->getLabel());
					$referenceModuleFields = $this->unshiftField($referenceModuleFields, $baseEntityField);
				}
			}
		}
		return $referenceModuleFields;
	}

	/** Get field from related module */
	private function getRelationFields(int $relationId): array
	{
		$relationFields = [];
		$relation = Vtiger_Relation_Model::getInstanceById($relationId);
		$relationModuleModel = $relation->getRelationModuleModel();
		$relatedModuleName = $relationModuleModel->getName();
		if (App\Privilege::isPermitted($relatedModuleName)) {
			$translatedModuleName = App\Language::translate($relatedModuleName, $relatedModuleName);
			foreach ($relationModuleModel->getFields() as $fieldModel) {
				$fieldModelDataType = $fieldModel->getFieldDataType();
				if ($fieldModel->isActiveField() && $fieldModel->getPermissions() && ($aggregateFunctions = $this->getAggregations($fieldModelDataType))) {
					if(in_array($fieldModelDataType, self::AGGREGATE_FIELD_TYPES)){
						$reportField = $this->createReportField($fieldModel);
						$reportField->setExpressionField(sprintf(
							'%s:%s:%s:M2M',
							$fieldModel->getName(),
							$relatedModuleName, $relationId,
						));
						$reportField->setAggregateLabel(sprintf(
							'[%s] %s',
							$translatedModuleName,
							\App\Language::translate($fieldModel->getLabel(), $relatedModuleName),
						));
						$relationFields[$translatedModuleName][] = $reportField;
					}
					foreach ($aggregateFunctions as $functionName) {
						$reportField = $this->createReportField($fieldModel);
						$reportField->setExpressionField(sprintf(
							'%s:%s:%s:M2M',
							$fieldModel->getName(),
							$relatedModuleName, $relationId,
						));
						$reportField->setAggregateFunction($functionName);
						$reportField->setAggregateLabel(sprintf(
							'[%s] %s - %s',
							$translatedModuleName,
							App\Language::translate($functionName, 'ReportTemplate'),
							\App\Language::translate($fieldModel->getLabel(), $relatedModuleName),
						));
						$relationFields[$translatedModuleName][] = $reportField;
					}
				}
			}
			foreach ($this->getAggregations('entityidfield') as $functionName) {
				$baseEntityField = $this->getBaseIdModuleField($relatedModuleName);
				$baseEntityField->setAggregateFunction($functionName);
				$baseEntityField->setExpressionField(sprintf(
					'%s:%s:%s:M2M',
					$baseEntityField->getFieldName(),
					$relatedModuleName, $relationId,
				));
				$baseEntityField->setAggregateLabel(App\Language::translate($functionName, 'ReportTemplate') . ' ' . $baseEntityField->getLabel());
				$relationFields = $this->unshiftField($relationFields, $baseEntityField);
			}
		}
		return $relationFields;
	}
}
