<?php

class googleMap extends waMapAdapter
{

    public function getName()
    {
        return 'Google Maps';
    }

    protected function getByAddress($address, $options = array())
    {
        $address = preg_replace('@российская\s+федерация@ui', 'Россия', $address);

        $locale = wa()->getLocale();
        $zoom = ifset($options['zoom'], 10);
        if (!empty($options['static'])) {
            return $this->getStaticImg($address, $options);
        } else {

            if ($key = $this->getSettings('key')) {
                $key = "&key={$key}";
            } else {
                $key = '';
            }

            $id = uniqid();
            $width = ifset($options['width'], '100%');
            $height = ifset($options['height'], '400px');
            $address = json_encode($address);
            $style = sprintf('width: %s; height: %s', $width, $height);

            $html = <<<HTML
<div id="google-map-{$id}" class="map" style="{$style}"></div>
<script type="text/javascript">
$(function () {
        if (!window.init_google_maps) {
            window.init_google_maps = function () {
                $(window).trigger('init_google_maps');
            }
        }
        $(window).one('init_google_maps', function () {
            var geocoder = new google.maps.Geocoder();
            geocoder.geocode( { 'address': {$address}}, function(results, status) {
                if (status === google.maps.GeocoderStatus.OK) {
                    var map =  new google.maps.Map(document.getElementById('google-map-{$id}'), {
                        center: new google.maps.LatLng(55.753994, 37.622093),
                        zoom: {$zoom}
                    });
                    map.setCenter(results[0].geometry.location);
                    var marker = new google.maps.Marker({
                        map: map,
                        title: {$address},
                        position: results[0].geometry.location
                    });
                }
            });
        });
        if (!(window.google)) {
            $.getScript('https://maps.googleapis.com/maps/api/js?sensor=false{$key}&lang={$locale}&callback=init_google_maps')
        } else {
            init_google_maps();
        }
});
</script>
HTML;
            return $html;
        }
    }

    /**
     * https://developers.google.com/maps/documentation/urls/get-started?hl=ru
     * @param $address
     * @param $longitude
     * @param $latitude
     * @param int $zoom
     * @return string
     */
    public function getUrlToMap($address, $longitude, $latitude, $zoom)
    {
        if (isset($longitude, $latitude)) {
            $query = "$latitude,$longitude";
        } else {
            $query = $address;
        }
        $query = urlencode($query);

        return "https://maps.google.com/maps?q=$query&z=$zoom";
    }

    private function getStaticImg($address, $options = array())
    {
        if ($key = $this->getSettings('key')) {
            $key = "&key={$key}";
        } else {
            $key = '';
        }
        $zoom = ifset($options['zoom'], 10);
        $width = ifset($options['width'], '600');
        $height = ifset($options['height'], '400');
        $size = (int)$width.'x'.(int)$height;
        $url = '//maps.googleapis.com/maps/api/staticmap?center='.urlencode($address).'.&zoom='.$zoom.'&size='.$size.'&markers=color:red%7Clabel:A%7C'.urlencode($address).'&sensor=false';
        $url_with_key = $url . $key;
        $url_to_map = $this->getUrlToMap($address, null, null, $zoom);

        $image_id = 'google-static-map-id'.uniqid();

        $link_text = _ws('Link to map');
        $remove_bad_map = <<<JS
function() {
    var badMap = document.getElementById('{$image_id}'),
        span = document.createElement('span');
    span.innerHTML = '<i class=\'icon16 marker\'></i> {$link_text}';
    badMap.parentNode.replaceChild(span, badMap);
}
JS;

        return '<a target="_blank" href="'.$url_to_map.'"><img id="'. $image_id .'" src="'. $url_with_key .'" onerror="this.onerror='. $remove_bad_map .';this.src=\''. $url .'\';" /></a>';
    }

    protected function getByLatLng($lat, $lng, $options = array())
    {
        $lat_str = str_replace(',', '.',(string)$lat);
        $lng_str = str_replace(',', '.',(string)$lng);
        return $this->getByAddress(join(',', array($lat_str, $lng_str)), $options);
    }

    protected function initControls()
    {
        parent::initControls(); // TODO: Change the autogenerated stub
        $env = $this->getEnvironment();
        if ($env == waMapAdapter::FRONTEND_ENVIRONMENT) {
            $description = _ws('<a href="https://developers.google.com/maps/documentation/javascript/get-api-key">Obtain a Google Maps API key</a> to get a map displayed.')
                . '<br>'
                . _ws('To be able to use an API key, <a href="https://console.developers.google.com/apis/api/static_maps_backend/">enable Static Maps API</a> in your Google account.');
        } else {
            $description = _ws('<a href="https://developers.google.com/maps/documentation/javascript/get-api-key">Obtain a Google Maps API key</a> to display a map within the Webasyst backend. Without a key, only a link to the map will be displayed.')
                . '<br>'
                . _ws('To be able to use an API key, <a href="https://console.developers.google.com/apis/api/static_maps_backend/">enable Static Maps API</a> in your Google account.');

        }
        $this->controls['key'] = array(
            'title'        => _ws('Google Maps API key'),
            'description'  => $description,
            'control_type' => waHtmlControl::INPUT,
        );
    }

    public function getJs($html = true)
    {
        $url = 'https://maps.googleapis.com/maps/api/js';
        $params = array(
            'sensor' => 'false',
            'key'    => $this->getSettings('key'),
            'lang'   => wa()->getLocale(),

        );

        $url .= '?' . http_build_query($params);
        if ($html) {
            return <<<HTML
<script type="text/javascript" src="{$url}"></script>
HTML;

        } else {
            return $url;
        }
    }

    public function geocode($address)
    {
        $data = array();
        if ($this->geocodingAllowed()) {
            if ($response = $this->sendGeoCodingRequest($address)) {
                switch ($response['status']) {
                    case 'ZERO_RESULTS':
                        //XXX
                        $this->geocodingAllowed(true);
                        break;
                    case 'OK':
                        $this->geocodingAllowed(true);
                        foreach ($response['results'] as $result) {
                            if (empty($result['partial_match']) || (count($response['results']) === 1)) {
                                // address correct, geocoding without errors
                                $data['lat'] = ifset($result['geometry']['location']['lat'], '');
                                $data['lng'] = ifset($result['geometry']['location']['lng'], '');
                                break;
                            }
                        }
                        break;
                    case 'REQUEST_DENIED':
                    case 'OVER_QUERY_LIMIT':
                        $this->geocodingAllowed(false);
                        break;

                }
            } else {
                $this->geocodingAllowed(false);
            }
        }
        return $data;
    }

    protected function sendGeoCodingRequest($address)
    {
        /**
         * @link https://developers.google.com/maps/documentation/geocoding/intro?hl=ru
         */

        $response = null;
        if ($address) {
            $url = 'https://maps.googleapis.com/maps/api/geocode/json';
            $address = preg_replace('@российская\s+федерация@ui', 'Россия', $address);
            $params = array(
                'address' => $address,
                'sensor'  => 'false',
            );

            $options = array(
                'format'  => waNet::FORMAT_JSON,
                'timeout' => 9,
            );
            $net = new waNet($options);
            try {
                $response = $net->query($url, $params);
            } catch (waException $ex) {
                if ( ( $response = $net->getResponse())) {
                    $log_template = "%s: %s\n\$request=%s;\n";
                    $response += array(
                        'status'        => 'UNKNOWN',
                        'error_message' => $ex->getMessage(),
                    );
                    $message = sprintf($log_template, $response['status'], $response['error_message'], var_export($params, true));
                } else {
                    $message = $ex->getMessage()."\n".$ex->getFullTraceAsString();
                    $response = array();
                }

                waLog::log(get_class($this).":\n".$message, 'geocode.log');
            }
        }
        return $response;
    }
}
