<?php

class siteCustomPartnerlogosBlockType extends siteBlockType {
    /** @var array Элементы основного блока */
    public array $elements = [
        'main'    => 'site-block-columns',
        'wrapper' => 'site-block-columns-wrapper',
    ];

    /** @var array Элементы колонок */
    public array $column_elements = [
        'main'    => 'site-block-column',
        'wrapper' => 'site-block-column-wrapper',
    ];

    public function __construct(array $options = []) {
        if (!isset($options['columns']) || !wa_is_int($options['columns'])) {
            $options['columns'] = 1;
        }
        $options['type'] = 'site.CustomPartnerLogos';
        parent::__construct($options);
    }

    public function getExampleBlockData(): siteBlockData {
        // Создаём основной блок
        $result = $this->getEmptyBlockData();

        // Создаём горизонтальную последовательность
        $hseq = $this->createSequence(true, 'only_columns', true);

        // Добавляем последовательности в основной блок
        $hseq->addChild($this->getHeadingColumn());
        foreach ($this->getExampleData() as $item) {
            $hseq->addChild($this->getLogoColumn($item));
        }

        $result->addChild($hseq, '');

        // Настраиваем свойства основного блока
        $result->data = [
            'block_props'   => [
                $this->elements['main']    => [
                    'padding-top'    => "p-t-18",
                    'padding-bottom' => "p-b-18",
                    'padding-left' => 'p-l-blc',
                    'padding-right' => 'p-r-blc',
                ],
                $this->elements['wrapper'] => [
                    'padding-top'    => "p-t-12",
                    'padding-bottom' => "p-b-12",
                    'flex-align'     => "y-c",
                    'max-width'      => "cnt",
                    'padding-left'   => "p-l-19 p-l-0-mb",
                    'padding-right'  => "p-r-19 p-r-0-mb",
                ],
            ],
            'wrapper_props' => [
                'justify-align' => 'y-j-cnt',
            ],
            'elements'      => $this->elements,
        ];

        return $result;
    }

    public function render(siteBlockData $data, bool $is_backend, array $tmpl_vars = []) {
        return parent::render($data, $is_backend, $tmpl_vars + [
                'children' => array_reduce($data->getRenderedChildren($is_backend), 'array_merge', []),
            ]);
    }

    public function getRawBlockSettingsFormConfig() {
        return [
                'type_name'    => _w('Block'),
                'sections'     => [
                    [
                        'type' => 'ColumnsGroup',
                        'name' => _w('Columns'),
                    ],
                    [
                        'type' => 'RowsAlignGroup',
                        'name' => _w('Columns alignment'),
                    ],
                    [
                        'type' => 'RowsWrapGroup',
                        'name' => _w('Wrap line'),
                    ],
                    [
                        'type' => 'TabsWrapperGroup',
                        'name' => _w('Tabs'),
                    ],                    
                    [   'type' => 'CommonLinkGroup',
                        'name' => _w('Link or action'),
                        'is_hidden' => true,
                    ],
                    [
                        'type' => 'MaxWidthToggleGroup',
                        'name' => _w('Max width'),
                    ],
                    [
                        'type' => 'BackgroundColorGroup',
                        'name' => _w('Background'),
                    ],
                    [   'type' => 'HeightGroup',
                        'name' => _w('Height'),
                    ],
                    [
                        'type' => 'PaddingGroup',
                        'name' => _w('Padding'),
                    ],
                    [
                        'type' => 'MarginGroup',
                        'name' => _w('Margin'),
                    ],
                    [
                        'type' => 'BorderGroup',
                        'name' => _w('Border'),
                    ],
                    [
                        'type' => 'BorderRadiusGroup',
                        'name' => _w('Angle'),
                    ],
                    [
                        'type' => 'ShadowsGroup',
                        'name' => _w('Shadows'),
                    ],
                    [
                        'type' => 'IdGroup',
                        'name' => _w('Identifier (ID)'),
                    ],
                ],
                'elements'     => $this->elements,
                'semi_headers' => [
                    'main'    => _w('Whole block'),
                    'wrapper' => _w('Container'),
                ],
            ] + parent::getRawBlockSettingsFormConfig();
    }

    private function getExampleData(): array {
        return [
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"/>
<path d="M64.771 10.1317C65.4548 10.0783 66.1394 10.041 66.825 10.0197C67.5106 9.99837 68.1964 9.99436 68.8822 10.0077C69.5681 10.021 70.2532 10.0517 70.9374 10.0997C71.6217 10.145 72.3045 10.2077 72.9858 10.2877C73.6669 10.3677 74.3457 10.465 75.0222 10.5797C75.6988 10.6917 76.3721 10.821 77.0422 10.9677C77.7126 11.1144 78.3789 11.277 79.041 11.4557C79.7032 11.6344 80.3606 11.829 81.0134 12.0397C81.6662 12.2503 82.3134 12.477 82.955 12.7197C83.5966 12.9624 84.232 13.2197 84.861 13.4917C85.4901 13.7663 86.112 14.0557 86.7266 14.3597C87.3416 14.6637 87.9488 14.9824 88.5482 15.3157C89.1474 15.649 89.7382 15.997 90.3206 16.3597C90.9028 16.7224 91.4758 17.0997 92.0398 17.4917C92.6036 17.881 93.1576 18.285 93.7018 18.7037C94.2458 19.1197 94.7794 19.5503 95.3026 19.9957C95.8256 20.4383 96.3374 20.8944 96.8382 21.3637C97.339 21.833 97.8281 22.3144 98.3054 22.8077C98.7825 23.301 99.2473 23.805 99.6998 24.3197C100.153 24.8344 100.592 25.361 101.019 25.8997C101.446 26.4357 101.86 26.9823 102.26 27.5397C102.66 28.097 103.046 28.6637 103.419 29.2397C103.791 29.8157 104.149 30.401 104.493 30.9957C104.837 31.5903 105.166 32.1917 105.48 32.7997C105.795 33.4103 106.094 34.0277 106.378 34.6517C106.663 35.2757 106.931 35.9063 107.185 36.5437C107.438 37.181 107.676 37.825 107.897 38.4757C108.119 39.1237 108.325 39.7783 108.515 40.4397C108.705 41.0983 108.879 41.761 109.037 42.4277C109.194 43.097 109.335 43.769 109.46 44.4437C109.585 45.1183 109.693 45.7957 109.784 46.4757C109.876 47.1557 109.951 47.837 110.009 48.5197C110.067 49.2024 110.109 49.8877 110.133 50.5757C110.158 51.261 110.166 51.9463 110.157 52.6317C110.149 53.317 110.123 54.0023 110.081 54.6877C110.039 55.373 109.98 56.057 109.904 56.7397C109.828 57.4197 109.736 58.0984 109.627 58.7757C109.518 59.453 109.393 60.1277 109.251 60.7997C109.109 61.4717 108.951 62.1397 108.776 62.8037C108.602 63.4677 108.411 64.1263 108.204 64.7797C107.998 65.433 107.775 66.081 107.537 66.7237C107.298 67.369 107.044 68.0063 106.775 68.6357C106.505 69.2677 106.22 69.8917 105.92 70.5077C105.62 71.1264 105.305 71.7357 104.975 72.3357C104.645 72.9383 104.301 73.5317 103.942 74.1157C103.583 74.6997 103.21 75.2757 102.823 75.8437C102.436 76.409 102.036 76.965 101.621 77.5117C101.207 78.061 100.78 78.5984 100.339 79.1237C99.8989 79.649 99.4458 80.1637 98.9802 80.6677C98.5146 81.1717 98.0369 81.6637 97.547 82.1437C97.0574 82.6237 96.5564 83.0917 96.0438 83.5477C95.5313 84.0037 95.0078 84.4477 94.4734 84.8797C93.9393 85.309 93.3949 85.725 92.8402 86.1277C92.2856 86.533 91.7214 86.9237 91.1478 87.2997C90.5742 87.6757 89.9917 88.037 89.4002 88.3837C88.8088 88.733 88.2093 89.0663 87.6018 89.3837C86.9941 89.701 86.379 90.0037 85.7566 90.2917C85.134 90.5797 84.5048 90.853 83.869 91.1117C83.2333 91.3704 82.5916 91.6117 81.9438 91.8357C81.2961 92.0623 80.6433 92.273 79.9854 92.4677C79.3276 92.6624 78.6653 92.8397 77.9986 92.9997C77.332 93.1624 76.6618 93.3077 75.9882 93.4357C75.3146 93.5663 74.6382 93.6797 73.959 93.7757C73.2798 93.8717 72.5986 93.9504 71.9154 94.0117C71.2269 94.0757 70.5372 94.1224 69.8462 94.1517C69.1556 94.1784 68.4645 94.1903 67.773 94.1877C67.0816 94.1823 66.3908 94.1597 65.7006 94.1197C65.0102 94.0797 64.3213 94.0237 63.6338 93.9517C62.9464 93.877 62.261 93.7863 61.5778 93.6797C60.8946 93.573 60.2145 93.4504 59.5374 93.3117C58.8606 93.1704 58.1876 93.013 57.5182 92.8397C56.8489 92.6663 56.1842 92.4757 55.5242 92.2677C54.8645 92.0624 54.2101 91.841 53.561 91.6037C52.912 91.3637 52.2693 91.109 51.633 90.8397C50.9965 90.5677 50.367 90.281 49.7446 89.9797C49.1225 89.6783 48.508 89.3623 47.901 89.0317C47.2944 88.701 46.6962 88.3543 46.1066 87.9917C45.517 87.6317 44.9366 87.257 44.3654 86.8677C43.7942 86.4757 43.233 86.0717 42.6818 85.6557C42.1304 85.237 41.5897 84.8064 41.0598 84.3637C40.5297 83.9184 40.0109 83.461 39.5034 82.9917C38.996 82.5223 38.5003 82.041 38.0164 81.5477C37.5326 81.0517 37.0613 80.545 36.6024 80.0277C36.1436 79.5104 35.6977 78.9823 35.2649 78.4437C34.8321 77.905 34.4128 77.3557 34.0071 76.7957C33.6013 76.2357 33.2096 75.6663 32.832 75.0877C32.4543 74.5063 32.0911 73.917 31.7424 73.3197C31.3938 72.7224 31.06 72.117 30.7412 71.5037C30.4223 70.8904 30.1188 70.269 29.8306 69.6397C29.5423 69.013 29.2697 68.3783 29.0128 67.7357C28.7559 67.093 28.5149 66.445 28.29 65.7917C28.065 65.1357 27.8562 64.4757 27.6637 63.8117C27.4712 63.1477 27.2952 62.4797 27.1356 61.8077C26.9761 61.1357 26.8332 60.4597 26.707 59.7797C26.5807 59.0997 26.4714 58.4157 26.3788 57.7277C26.2862 57.0424 26.2106 56.3557 26.1518 55.6677C26.0931 54.9797 26.0515 54.289 26.0268 53.5957C26.0021 52.905 25.9945 52.2144 26.0039 51.5237C26.0133 50.833 26.0397 50.1423 26.0831 49.4517C26.1266 48.761 26.187 48.0717 26.2644 47.3837C26.3417 46.6957 26.436 46.0104 26.5471 45.3277C26.6583 44.645 26.7861 43.9663 26.9308 43.2917C27.0754 42.6143 27.2366 41.941 27.4143 41.2717C27.5921 40.605 27.7862 39.9423 27.9966 39.2837C28.2071 38.625 28.4336 37.9717 28.6762 37.3237C28.9189 36.6757 29.1774 36.0343 29.4516 35.3997C29.7258 34.765 30.0155 34.137 30.3207 33.5157C30.6259 32.8944 30.9462 32.281 31.2815 31.6757C31.6169 31.073 31.967 30.477 32.3317 29.8877C32.6965 29.301 33.0755 28.7237 33.4687 28.1557C33.8619 27.585 34.2689 27.0263 34.6897 26.4797C35.1105 25.9303 35.5446 25.3917 35.9918 24.8637C36.4391 24.3357 36.8992 23.8197 37.3719 23.3157C37.8447 22.8117 38.3296 22.3183 38.8266 21.8357C39.3234 21.3557 39.832 20.8877 40.3522 20.4317C40.8722 19.9757 41.4032 19.533 41.945 19.1037C42.4869 18.6744 43.039 18.2584 43.6014 17.8557C44.1638 17.453 44.7358 17.065 45.3174 16.6917C45.899 16.3183 46.4894 15.9584 47.0886 15.6117C47.6878 15.2677 48.2952 14.9384 48.9106 14.6237C49.5261 14.309 50.149 14.009 50.7794 13.7237C51.4098 13.4384 52.0468 13.169 52.6902 12.9157C53.3337 12.665 53.983 12.429 54.6382 12.2077C55.2934 11.9864 55.9537 11.781 56.619 11.5917C57.2841 11.405 57.9536 11.233 58.6274 11.0757C59.3013 10.921 59.9785 10.7824 60.659 10.6597C61.3396 10.5397 62.0228 10.4357 62.7086 10.3477C63.3942 10.257 64.0817 10.185 64.771 10.1317Z" fill="black"/>
<path d="M92.3028 42.4629C94.3984 43.6429 97.586 42.5829 99.9184 43.2469C101.244 43.6229 102.183 44.4909 102.84 45.6789C103.674 47.1829 103.956 49.1349 103.455 50.7949C102.971 52.3949 101.892 53.1109 100.506 53.8509C101.324 55.7029 102.56 57.3909 103.302 59.2669C103.647 60.1389 103.748 60.5149 103.384 61.3989C102.258 61.6789 101.377 61.3189 100.317 60.9789L99.6384 60.9469C97.92 58.3869 96.9916 55.1909 96.0944 52.2669C95.9116 54.6509 96.7748 59.5669 95.1724 61.3549C94.08 61.5309 93.3096 61.5069 92.2284 61.3189C91.3992 58.6429 91.496 45.0149 92.3028 42.4629Z" fill="white"/>
<path d="M96.0126 46.3838C96.637 46.4838 97.3654 46.5638 97.9494 46.8118C98.9622 47.2398 99.3162 47.8158 99.6638 48.7998C99.4466 49.4478 99.247 49.7118 98.7638 50.1918C98.167 50.9118 97.7134 51.2078 96.8394 51.5318L96.2242 51.1478C95.6718 49.6238 95.9098 47.9678 96.0126 46.3838Z" fill="black"/>
<path d="M41.9416 51.5757C41.99 49.0397 41.5224 46.1357 41.8484 43.6757C43.282 42.9677 44.5872 43.0517 46.0356 42.5157C46.1092 42.4877 46.182 42.4596 46.2552 42.4316C46.2688 46.2316 47.0036 57.9237 45.9972 60.8157L45.1524 61.2396C42.7716 60.4236 38.7676 53.8717 37.6396 51.6037C37.5576 54.1157 38.0174 56.9157 37.633 59.3757C37.4702 60.4197 37.2574 60.9116 36.4315 61.5396C35.49 61.5276 34.3448 61.7716 33.6403 61.1316C33.2672 56.7156 32.7469 46.6236 33.8003 42.4796C37.2577 43.2116 38.9876 49.5237 41.9416 51.5757Z" fill="white"/>
<path d="M80.7844 42.4834C81.6608 42.9714 82.6332 42.6914 83.5252 43.0514C84.986 43.6434 89.0668 58.1194 90.1848 60.6594L90.3452 61.0194C89.7664 61.3754 88.646 61.6314 87.97 61.6714C85.6928 61.8114 85.6408 59.1994 84.1648 58.0274C84.0968 57.9754 84.0264 57.9234 83.9572 57.8714C82.7876 58.4634 81.3632 58.4674 80.0704 58.5914C79.0792 59.1754 78.6896 60.4074 78.2092 61.4114C76.8084 61.9594 75.682 61.5034 74.3132 60.9594C76.6868 56.5634 79.0156 47.6154 80.7844 42.4834Z" fill="white"/>
<path d="M82.2711 48.8916C82.7995 50.3796 84.1203 53.1636 83.5215 54.6316L82.3963 54.3036L81.3467 54.7756L80.8687 54.5556C80.6439 52.8356 81.7375 50.5356 82.2711 48.8916Z" fill="black"/>
<path d="M54.3001 42.4912C55.0609 42.8872 56.1865 42.9352 57.0357 43.0712C59.9802 45.8112 61.2285 56.7512 63.7837 60.8672C63.0497 61.2832 61.7406 61.6712 60.8966 61.6472C59.1374 61.5952 58.6758 59.2592 57.625 58.1352C57.569 58.0752 57.5113 58.0152 57.4545 57.9552C56.5061 58.3712 53.3873 58.2632 52.9493 58.9912C52.4561 59.8112 52.8238 60.5912 51.9858 61.4112C50.6618 61.5192 49.675 61.5632 48.4006 61.1392C47.8794 59.5352 53.2833 45.3912 54.3001 42.4912Z" fill="white"/>
<path d="M55.5996 49.2197C55.716 49.4037 55.7284 49.4157 55.8396 49.6437C56.4368 50.8757 57.5004 53.3157 56.9796 54.6397L55.8956 54.3477L54.668 54.7797L54.3208 54.1877C54.5408 52.4797 55.0888 50.8597 55.5996 49.2197Z" fill="black"/>
<path d="M72.6012 42.4355C73.3856 42.9515 74.1608 42.6716 75.0068 42.8716C75.9404 43.0916 76.134 43.4796 76.6584 44.2196C76.7004 45.5756 75.4316 47.5596 74.6716 48.7276C74.0944 51.8796 72.7632 56.5276 71.2444 59.3396C70.6448 60.4516 70.1252 60.9076 68.9304 61.2596C67.8704 60.9876 67.3624 60.3676 66.8216 59.4676C65.4436 57.1796 61.2508 44.7156 61.8204 42.4956C62.1276 42.6156 62.3868 42.6796 62.7156 42.7196C63.268 42.7836 63.9056 42.8076 64.4344 42.9756C67.5648 43.9516 67.5608 50.3276 69.0092 53.0996C69.0424 53.1636 69.0784 53.2276 69.1128 53.2916C70.922 51.0516 71.8196 45.3595 72.6012 42.4355Z" fill="white"/>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="#000000" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"></rect>
<path d="M56.1503 10.1031C56.8973 10.5163 57.1286 10.8994 57.5435 11.6355C59.0878 14.3743 64.0671 14.2682 66.5643 16.349C67.8832 17.4479 69.1609 19.4783 69.3199 21.1973L68.5248 21.9476C68.9075 23.1136 71.775 25.1441 72.7138 26.206C73.5667 27.171 74.2386 28.3632 74.0636 29.6924C73.8632 31.2145 72.8161 31.8509 71.6562 32.6652C72.3393 34.808 73.08 37.2178 73.1481 39.4762C73.184 40.6626 73.0418 42.0435 72.1962 42.9443C71.2827 43.4992 70.3863 43.2568 69.385 43.1071L69.0474 43.3564C69.0931 43.737 69.1863 44.0874 69.3442 44.4386C72.6078 51.7026 80.5809 49.2345 86.6007 51.5627C88.6991 52.3744 89.6941 53.4775 90.606 55.487C92.6715 53.8428 95.3115 52.3577 98.0536 52.5051C105.76 52.9194 102.938 66.3059 104.042 70.0341C104.889 72.8919 107.037 74.9512 109 77.0884C105.826 78.0439 102.211 78.3953 99.1187 76.9437C91.9107 73.5598 96.6524 67.4972 94.7956 62.7104C94.405 61.7032 93.8087 60.7782 92.7995 60.317L92.2542 60.525C91.5089 61.851 91.835 64.0278 91.68 65.5486C91.3 69.28 89.3467 71.398 86.5721 73.6492C87.8103 74.9391 89.6563 75.4871 90.7958 76.8614C91.3802 77.5663 91.6389 78.1618 91.5569 79.0825C91.2707 82.3108 84.2473 90.7825 81.5444 93C80.2137 92.9941 78.7072 92.9401 77.3999 92.6951C76.6792 92.5598 76.0978 92.3 75.6651 91.6905C75.6289 90.979 75.6575 90.7101 76.0221 90.0681C76.8017 88.6967 78.2094 87.4406 79.2268 86.2195C80.5407 84.6423 81.6312 82.9532 82.6595 81.1821C81.4486 80.5857 80.0643 79.81 78.7095 79.6839C77.8678 80.3973 78.2396 81.9512 77.5184 82.4602C75.5838 83.8251 73.3597 83.2666 71.5779 85.1821C69.7815 87.5814 67.6608 90.0603 65.515 92.1543C63.3409 92.1471 59.1947 92.6171 57.4817 91.2588C57.8311 89.6358 60.7913 88.1477 61.9947 87.0154C64.5031 84.6548 67.0694 81.51 68.8444 78.5712L69.0092 78.2948C67.874 76.0933 66.6792 73.9361 66.026 71.5365C65.4998 69.6037 65.5508 67.7245 65.5722 65.7396C59.686 64.755 47.011 61.8306 41.7422 62.9731C41.0992 63.1123 40.7036 63.1769 40.344 63.7471C41.7046 69.0478 46.5474 69.9188 48.7794 74.2329C49.3865 75.4065 49.3339 76.4478 48.9387 77.6695C48.0667 78.078 47.5053 78.0465 46.5714 77.7835C41.9732 76.4893 37.5526 73.4071 35.5789 68.9977C34.9132 67.511 34.5934 65.8323 33.7776 64.4258C32.6931 64.5519 31.7622 64.549 30.822 63.9217C28.7812 62.5601 27.9171 59.5915 27.4798 57.3246C26.6812 53.1857 26.6572 47.3161 29.158 43.7655C30.0313 42.525 31.2238 41.6086 32.7569 41.3773C35.5848 40.9505 38.0899 42.6767 40.2512 44.2322C41.8771 40.2587 44.5582 36.979 47.118 33.5935C47.7656 32.7373 48.7086 31.9617 49.147 30.9846C46.2095 31.4618 43.4707 33.2872 40.501 33.978C39.7751 33.6924 39.0455 33.3943 38.3634 33.0141C37.5579 32.5647 36.9261 31.9531 36.7132 31.0173C35.6134 26.19 46.4388 19.6747 48.4378 15.5911C49.1733 14.0884 47.9962 11.4916 47.5211 10C49.27 10.858 52.019 12.8424 53.994 12.3413C55.169 12.0433 55.6199 11.0788 56.1503 10.1031ZM36.1209 57.3567L37.8718 54.8634C36.6549 53.1611 34.7605 49.7491 33.0244 48.7363C32.6964 51.2014 32.8533 55.9227 34.5312 58.0114C34.6793 58.1955 34.7362 58.2283 34.9274 58.3953L35.518 58.3478L36.1209 57.3567Z" fill="black"></path>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"/>
<path d="M55.6157 21.0769C68.4608 21.0201 81.3057 21.0394 94.1505 21.1349C94.9709 21.6969 95.7897 22.2693 96.5677 22.8897C101.863 27.1121 104.848 35.4177 105.586 41.9445C106.791 52.6109 102.917 63.6649 96.2485 71.9489C92.4777 76.6329 86.6473 82.1653 80.3529 82.7985C77.2413 83.1113 73.8781 82.4481 70.7201 82.4597L50.1717 82.3985C48.2173 82.4029 46.2305 82.6977 44.2861 82.6297C43.5845 82.6053 42.9709 82.4677 42.4365 82.0053C41.4821 81.1797 41.2521 79.9993 41.1361 78.8005C40.9497 76.8753 40.8889 74.2673 42.2961 72.7481C43.1089 71.8709 44.3529 71.4601 45.5153 71.3237C48.5717 70.9653 52.0649 71.3993 55.1765 71.3969C64.0981 71.3897 73.0373 71.5593 81.9561 71.3373C83.7281 69.9197 85.2693 68.3317 86.3089 66.3013C80.1917 65.1169 73.9625 65.4897 67.7621 65.5361C61.1789 65.6235 54.5962 65.5921 48.0141 65.4417C44.0593 65.3389 37.3977 66.5665 33.8703 65.3585C32.7807 64.9853 31.9397 64.2425 31.4875 63.1733C30.7837 61.5089 30.8861 59.0005 31.5123 57.3353C31.8475 56.4441 32.1742 55.9633 33.1112 55.5985C36.5785 54.2485 42.3053 54.9505 46.0129 54.9661C52.5949 54.9937 86.1097 55.9633 89.6913 54.1797C91.1445 53.4561 91.7021 51.1781 92.2153 49.7505L59.6325 49.6905C53.7737 49.7009 47.6717 50.0653 41.8373 49.6461C40.8901 49.5781 40.1389 49.4845 39.3501 48.9049C38.0781 47.9701 37.8789 45.2089 37.7421 43.7345C37.5953 42.1557 38.0945 40.2613 39.2101 39.0921C39.7973 38.4765 40.5405 38.2141 41.3625 38.1061C46.6313 37.4133 52.4089 37.8749 57.7253 37.8861L89.7437 37.9109C89.1961 36.4261 88.6873 34.3185 87.2489 33.4581C82.9241 30.8709 50.9629 35.2201 47.7257 31.8077C46.5465 30.5649 45.8933 28.4253 45.9185 26.7409C45.9409 25.2361 46.4401 23.8001 47.5501 22.7529C49.8517 20.5805 52.7073 21.0137 55.6157 21.0769Z" fill="black"/>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"/>
<path d="M102.649 46.7809C102.731 47.4407 102.798 48.1024 102.851 48.7661C102.904 49.43 102.943 50.0949 102.968 50.7607C102.993 51.4265 103.003 52.0927 102.999 52.7591C102.996 53.4259 102.978 54.092 102.946 54.7575C102.914 55.423 102.867 56.0871 102.807 56.7498C102.746 57.4128 102.672 58.0736 102.583 58.7321C102.495 59.3907 102.392 60.0463 102.275 60.6989C102.159 61.3518 102.028 62.0009 101.884 62.6462C101.74 63.2915 101.582 63.9323 101.41 64.5685C101.238 65.2046 101.053 65.8356 100.855 66.4614C100.656 67.0871 100.444 67.7068 100.219 68.3202C99.9943 68.9337 99.7563 69.5405 99.5053 70.1407C99.2546 70.7406 98.9911 71.333 98.7148 71.9178C98.4385 72.5028 98.1501 73.0796 97.8495 73.6479C97.5487 74.2163 97.2359 74.7758 96.9111 75.3265C96.5865 75.8772 96.2504 76.4182 95.9027 76.9493C95.555 77.4808 95.1961 78.002 94.8261 78.5129C94.456 79.0239 94.0754 79.5238 93.6841 80.0127C93.2928 80.5019 92.8913 80.9796 92.4795 81.4457C92.0678 81.9119 91.6465 82.3661 91.2157 82.8083C90.7846 83.2505 90.3443 83.6801 89.8947 84.0971C89.4453 84.514 88.9874 84.9179 88.5209 85.3087C88.0541 85.6995 87.5794 86.0767 87.0968 86.4404C86.614 86.8041 86.1238 87.1537 85.6262 87.4893C85.1286 87.8249 84.6241 88.1461 84.1126 88.453C83.6014 88.7599 83.0838 89.052 82.5598 89.3292C82.0361 89.6065 81.5068 89.8686 80.9719 90.1155C80.4367 90.3628 79.8968 90.5945 79.3521 90.8105C78.8071 91.0266 78.2579 91.227 77.7044 91.4118C77.151 91.5967 76.594 91.7656 76.0335 91.9185C75.4727 92.0715 74.9091 92.2084 74.3426 92.3292C73.7761 92.45 73.2074 92.5546 72.6365 92.6429C72.0656 92.7312 71.4931 92.8031 70.919 92.8586C70.3451 92.9144 69.7703 92.9537 69.1945 92.9764C68.6187 92.9991 68.0428 93.0054 67.4668 92.9953C66.891 92.9849 66.3156 92.9583 65.7406 92.9154C65.166 92.8722 64.5925 92.8127 64.0203 92.737C63.4478 92.6613 62.8773 92.5692 62.3089 92.4607C61.7407 92.3525 61.175 92.2281 60.6118 92.0874C60.0488 91.9464 59.4891 91.7897 58.9327 91.6172C58.3765 91.4447 57.8242 91.2562 57.2757 91.0518C56.7274 90.8474 56.1838 90.6277 55.6448 90.3928C55.1058 90.1575 54.5723 89.9069 54.0442 89.641C53.5158 89.3751 52.9936 89.0944 52.4775 88.7988C51.9611 88.5033 51.4514 88.1931 50.9484 87.8682C50.4454 87.5437 49.9496 87.2048 49.461 86.8515C48.9724 86.4986 48.4916 86.1318 48.0186 85.7511C47.5458 85.3707 47.0813 84.9769 46.6252 84.5697C46.169 84.1628 45.7218 83.743 45.2837 83.3103C44.8452 82.8776 44.4165 82.4326 43.9974 81.9756C43.578 81.5186 43.1687 81.0498 42.7695 80.5691C42.3703 80.0887 41.9816 79.5972 41.6034 79.0944C41.2252 78.5916 40.8579 78.0784 40.5014 77.5549C40.145 77.0313 39.8 76.4978 39.4665 75.9543C39.133 75.4109 38.8112 74.8583 38.5011 74.2966C38.1912 73.7348 37.8934 73.1646 37.6076 72.5858C37.322 72.007 37.0488 71.4204 36.788 70.8258C36.5275 70.2316 36.2797 69.6301 36.0446 69.0214C35.8097 68.4129 35.5878 67.7981 35.3791 67.1767C35.1704 66.5554 34.975 65.9285 34.793 65.2961C34.611 64.6637 34.4426 64.0266 34.2878 63.3847C34.133 62.7425 33.992 62.0964 33.8648 61.4464C33.7375 60.7963 33.6241 60.1431 33.5247 59.4867C33.4255 58.8301 33.3402 58.171 33.2688 57.5096C33.1985 56.8554 33.1418 56.1999 33.0988 55.5429C33.056 54.8856 33.0271 54.2275 33.0121 53.5686C32.9971 52.9097 32.996 52.2508 33.0089 51.5919C33.0217 50.933 33.0482 50.275 33.0885 49.6176C33.1291 48.96 33.1836 48.3041 33.252 47.65C33.3202 46.9955 33.4021 46.3434 33.4977 45.6937C33.5933 45.0439 33.7027 44.3973 33.8259 43.7539C33.9488 43.1105 34.0852 42.4708 34.2351 41.835C34.3852 41.1991 34.5486 40.5678 34.7251 39.9411C34.9017 39.3147 35.0915 38.6935 35.2945 38.0775C35.4973 37.4618 35.7129 36.8521 35.9416 36.2485C36.1702 35.6448 36.4113 35.0478 36.665 34.4577C36.9187 33.8676 37.1847 33.285 37.4629 32.7101C37.7414 32.1351 38.0317 31.5681 38.3339 31.0092C38.6361 30.4506 38.9499 29.9009 39.2752 29.36C39.6006 28.819 39.9373 28.2877 40.2852 27.766C40.6332 27.244 40.9918 26.7323 41.361 26.2308C41.7305 25.729 42.1104 25.2381 42.5006 24.758C42.8908 24.2776 43.2908 23.8088 43.7006 23.3514C44.1107 22.8938 44.5301 22.4479 44.9587 22.0139C45.3874 21.5799 45.825 21.1584 46.2716 20.7493C46.7182 20.3402 47.1732 19.9439 47.6364 19.5604C48.0996 19.1769 48.5706 18.8067 49.0494 18.45C49.5282 18.0933 50.0143 17.7501 50.5078 17.4205C51.001 17.0912 51.501 16.776 52.0079 16.4748C52.5144 16.1736 53.0271 15.887 53.5459 15.6151C54.0645 15.3426 54.5886 15.0851 55.1183 14.8425C55.6478 14.6 56.1822 14.3726 56.7214 14.1603C57.2607 13.948 57.804 13.751 58.3515 13.5694C58.8989 13.3874 59.4499 13.2212 60.0044 13.0707C60.5589 12.9203 61.1163 12.7854 61.6766 12.6662C62.2368 12.5473 62.7992 12.4442 63.3638 12.3568C63.9284 12.2694 64.4945 12.1982 65.0621 12.143C65.6297 12.0874 66.1982 12.0482 66.7674 12.0251C67.3366 12.0018 67.906 11.9947 68.4755 12.0039C69.045 12.0127 69.614 12.0376 70.1824 12.0786C70.7508 12.1199 71.318 12.177 71.884 12.2499C72.4502 12.323 73.0144 12.4121 73.5766 12.5172C74.139 12.6222 74.6988 12.743 75.2561 12.8796C75.8133 13.0165 76.3673 13.1688 76.918 13.3366C77.4687 13.5044 78.0156 13.6878 78.5587 13.8868C79.1018 14.0855 79.6404 14.2995 80.1744 14.5288C80.7083 14.7581 81.237 15.0023 81.7608 15.2612C82.2845 15.5205 82.8025 15.7943 83.3148 16.0825C83.8268 16.3711 84.3325 16.674 84.832 16.9909C85.3315 17.3079 85.8239 17.6388 86.3092 17.9835C86.7948 18.3283 87.2727 18.6866 87.7431 19.0584C88.2134 19.43 88.6755 19.8148 89.1295 20.2128C89.5835 20.6106 90.0289 21.021 90.4657 21.4443C90.9025 21.8673 91.33 22.3024 91.7483 22.7497C92.1668 23.1969 92.5756 23.6557 92.9745 24.1259C93.3737 24.5962 93.7627 25.0773 94.1415 25.5694C94.5202 26.0617 94.8884 26.5645 95.2459 27.0777C95.6034 27.5905 95.9498 28.1133 96.2853 28.646C96.6207 29.1787 96.9447 29.7206 97.2573 30.2716C97.5698 30.8223 97.8706 31.3817 98.1598 31.9498C98.4489 32.5178 98.7257 33.0936 98.9903 33.6771C99.2552 34.2606 99.5075 34.8514 99.7473 35.4494C99.9871 36.0471 100.214 36.6512 100.428 37.2619C100.642 37.8725 100.844 38.4888 101.032 39.1108C101.22 39.7328 101.395 40.3598 101.557 40.9919C101.719 41.6236 101.868 42.2598 102.003 42.9004C102.138 43.5407 102.259 44.1843 102.367 44.8312C102.475 45.4784 102.569 46.1283 102.649 46.7809ZM52.643 55.4682C52.6918 55.7489 52.7464 56.028 52.8069 56.3055C52.8677 56.5834 52.9342 56.8593 53.0064 57.133C53.0786 57.4068 53.1565 57.6785 53.2402 57.9482C53.3238 58.2175 53.4132 58.4846 53.5083 58.7492C53.6031 59.0135 53.7036 59.2749 53.8096 59.5336C53.9158 59.7925 54.0273 60.0482 54.1439 60.3005C54.2608 60.5528 54.383 60.8018 54.5106 61.0475C54.6378 61.2929 54.7702 61.5347 54.9078 61.7728C55.0454 62.0107 55.188 62.2447 55.3354 62.4749C55.4831 62.7049 55.6354 62.9305 55.7924 63.1519C55.9493 63.373 56.1109 63.5897 56.2771 63.802C56.4434 64.0143 56.6139 64.2217 56.7889 64.4242C56.9638 64.6266 57.143 64.8241 57.3263 65.0165C57.5095 65.2089 57.6967 65.3959 57.888 65.5776C58.0793 65.7596 58.2742 65.9359 58.4729 66.1065C58.6715 66.2772 58.8737 66.4423 59.0794 66.6019C59.2849 66.7612 59.4938 66.9144 59.706 67.0617C59.918 67.2093 60.133 67.3508 60.351 67.4861C60.5693 67.6214 60.7901 67.7504 61.0136 67.8731C61.2373 67.9961 61.4633 68.1127 61.6917 68.2228C61.9203 68.3328 62.151 68.4363 62.3837 68.5331C62.6166 68.6303 62.8514 68.7206 63.0879 68.8042C63.3247 68.8878 63.5632 68.9648 63.8032 69.0351C64.043 69.1051 64.2842 69.1685 64.5267 69.2253C64.7695 69.2817 65.0132 69.3313 65.2579 69.3738C65.5026 69.4167 65.748 69.4527 65.9941 69.4817C66.2404 69.5104 66.4871 69.5322 66.7343 69.547C66.9814 69.5618 67.2287 69.5697 67.4762 69.5707C67.7236 69.5716 67.9709 69.5656 68.218 69.5527C68.4654 69.5394 68.7123 69.5192 68.9586 69.4921C69.205 69.465 69.4506 69.4311 69.6956 69.3904C69.9406 69.3494 70.1844 69.3015 70.4272 69.2466C70.67 69.1917 70.9116 69.13 71.1519 69.0616C71.3923 68.9932 71.631 68.9179 71.8681 68.8359C72.1051 68.7542 72.3404 68.6656 72.5739 68.57C72.8072 68.4748 73.0384 68.3731 73.2676 68.2649C73.4967 68.1564 73.7235 68.0415 73.9477 67.9204C74.1717 67.7993 74.3931 67.6719 74.6119 67.5382C74.8307 67.4044 75.0467 67.2646 75.2598 67.1185C75.4726 66.9725 75.6823 66.8206 75.8888 66.6629C76.0954 66.5049 76.2984 66.3414 76.4978 66.1723C76.6976 66.0029 76.8935 65.828 77.0856 65.6476C77.2777 65.4672 77.466 65.2814 77.6505 65.0903C77.8347 64.8991 78.0148 64.703 78.1908 64.5017C78.3669 64.3005 78.5387 64.0944 78.7062 63.8834C78.8736 63.6724 79.0362 63.4568 79.1943 63.2366C79.3526 63.0165 79.5062 62.7919 79.6549 62.5629C79.8037 62.3343 79.9475 62.1013 80.0862 61.8641C80.2251 61.6273 80.3589 61.3864 80.4875 61.1417C80.6164 60.8972 80.74 60.6492 80.8583 60.3975C80.9763 60.1461 81.0891 59.8914 81.1967 59.6334C81.3041 59.3754 81.406 59.1146 81.5024 58.8509C81.5989 58.5872 81.6896 58.3209 81.7747 58.0518C81.8599 57.7831 81.9394 57.512 82.0129 57.2385C82.0868 56.9651 82.1548 56.6897 82.2169 56.4125C82.2788 56.1355 82.3349 55.8569 82.3853 55.5765C82.4354 55.2961 82.4797 55.0144 82.5182 54.7315C82.5566 54.4486 82.589 54.1647 82.6154 53.8799C82.6419 53.5951 82.6622 53.3098 82.6763 53.0241C82.6908 52.7383 82.6991 52.4522 82.7013 52.1658C82.7034 51.8794 82.6996 51.5932 82.6898 51.3071C82.68 51.0207 82.6641 50.735 82.642 50.4498C82.6202 50.1647 82.5923 49.8802 82.5582 49.5964C82.5244 49.3128 82.4845 49.0304 82.4384 48.749C82.3921 48.4655 82.3398 48.1835 82.2815 47.9031C82.2234 47.6224 82.1593 47.3437 82.089 47.0671C82.0189 46.7902 81.9429 46.5155 81.8609 46.2429C81.7792 45.9704 81.6917 45.7004 81.5985 45.433C81.505 45.1652 81.406 44.9004 81.3013 44.6386C81.1967 44.3768 81.0866 44.1181 80.9711 43.8623C80.8555 43.6068 80.7346 43.3546 80.6081 43.1058C80.4817 42.8572 80.3501 42.6123 80.2133 42.371C80.0765 42.13 79.9346 41.8928 79.7878 41.6594C79.6409 41.4264 79.4891 41.1977 79.3324 40.9734C79.1758 40.7489 79.0144 40.5288 78.8485 40.3134C78.6825 40.0983 78.512 39.8879 78.3367 39.6823C78.1618 39.477 77.9826 39.2765 77.7993 39.081C77.6156 38.8857 77.428 38.6959 77.2364 38.5114C77.0449 38.3268 76.8495 38.1477 76.6503 37.9739C76.4511 37.8004 76.2484 37.6328 76.0421 37.471C75.8355 37.3092 75.6259 37.1532 75.413 37.0031C75.1999 36.8532 74.9839 36.7094 74.7648 36.5716C74.5457 36.4341 74.3238 36.3029 74.099 36.178C73.8741 36.0531 73.6469 35.9345 73.4172 35.8222C73.1875 35.7102 72.9554 35.6049 72.7211 35.5061C72.487 35.4074 72.2509 35.3153 72.0128 35.2298C71.7746 35.1447 71.5347 35.0662 71.293 34.9942C71.0515 34.9226 70.8086 34.858 70.5642 34.8003C70.32 34.7425 70.0746 34.6918 69.828 34.6479C69.5817 34.6041 69.3346 34.5673 69.0866 34.5377C68.8386 34.508 68.5901 34.4853 68.341 34.4696C68.092 34.4541 67.8428 34.4458 67.5935 34.4445C67.3444 34.4432 67.0952 34.4491 66.8459 34.462C66.5968 34.4746 66.3482 34.4945 66.0999 34.5216C65.8517 34.5487 65.6043 34.583 65.3576 34.6243C65.1108 34.6653 64.865 34.7134 64.6203 34.7686C64.3756 34.8238 64.1323 34.8859 63.8903 34.955C63.648 35.024 63.4076 35.0999 63.1689 35.1825C62.9299 35.2655 62.6929 35.3551 62.4581 35.4513C62.2229 35.5478 61.9901 35.6508 61.7595 35.7602C61.5287 35.8697 61.3005 35.9857 61.0749 36.1084C60.8493 36.2311 60.6262 36.3601 60.4058 36.4954C60.1856 36.6307 59.9684 36.7722 59.7543 36.9198C59.5401 37.0674 59.3292 37.221 59.1215 37.3806C58.9139 37.5405 58.7098 37.7061 58.5093 37.8774C58.3087 38.0486 58.1118 38.2256 57.9186 38.4082C57.7257 38.5908 57.5367 38.7788 57.3517 38.9722C57.1667 39.1655 56.9859 39.3641 56.8093 39.5678C56.6327 39.7712 56.4605 39.9797 56.2926 40.1933C56.1251 40.4068 55.962 40.6249 55.8034 40.8476C55.6451 41.0706 55.4915 41.2978 55.3427 41.5293C55.194 41.7608 55.0502 41.9965 54.9115 42.2362C54.7728 42.4759 54.6393 42.7194 54.511 42.9667C54.3826 43.2142 54.2596 43.4652 54.1419 43.7194C54.0241 43.9736 53.912 44.2311 53.8055 44.492C53.6986 44.7528 53.5976 45.0165 53.5022 45.283C53.4066 45.5495 53.3167 45.8186 53.2328 46.0901C53.1489 46.3617 53.0707 46.6356 52.9982 46.9119C52.926 47.1879 52.8597 47.4659 52.7992 47.746C52.7387 48.0258 52.6842 48.3071 52.6357 48.59C52.5871 48.873 52.5447 49.1572 52.5081 49.4426C52.4719 49.728 52.4416 50.0144 52.4174 50.3018C52.3931 50.5888 52.375 50.8764 52.363 51.1647C52.351 51.4527 52.3452 51.741 52.3455 52.0296C52.3457 52.3182 52.3521 52.6065 52.3647 52.8944C52.3775 53.1827 52.3963 53.4704 52.4211 53.7574C52.4459 54.0444 52.4768 54.3306 52.5139 54.6161C52.5509 54.9012 52.594 55.1852 52.643 55.4682Z" fill="black"/>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"/>
<path d="M89.8752 5.23873C91.1327 5.01769 92.0806 4.81811 93.3235 5.28033C95.8475 7.49185 97.0624 11.0796 97.9395 14.2439C103.051 32.6807 102.075 52.4965 92.61 69.3677C86.2974 80.6188 71.4915 95.4851 58.9456 99C55.9479 97.2952 53.178 95.3535 50.3795 93.338C42.7282 87.2646 36.8933 79.6715 34.8796 69.9081C32.3867 57.821 35.3282 45.6797 41.988 35.4598C52.3646 19.536 71.7248 8.99983 89.8752 5.23873ZM51.9476 79.7945C54.7988 66.0617 58.06 55.0616 66.3742 43.5372C68.5906 42.5878 72.3139 42.8586 74.5549 43.7133C75.0457 43.9 75.4657 44.1424 75.8842 44.4561C76.2214 47.3036 71.0158 54.6826 69.5691 57.6742C66.05 64.95 64.3821 73.6077 63.8436 81.644C74.9649 75.9441 82.5074 67.007 86.4163 55.0282C89.6103 45.241 90.3293 26.964 85.5623 17.7461C85.3771 17.3878 85.346 17.3676 85.1348 17.0938C70.1803 23.1511 55.7 31.7268 49.115 47.3719C45.7033 55.4765 44.7153 64.7855 48.0918 73.0774C49.0653 75.4679 50.4578 77.7005 51.9476 79.7945Z" fill="black"/>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<g clip-path="url(#clip0_2269_17687)">
<rect width="136" height="104" fill="white"/>
<path d="M94.1821 67.724C94.1358 67.751 94.0911 67.7819 94.0428 67.805C93.4063 68.1098 92.5016 68.1407 91.8083 68.3374C87.3801 69.6183 83.247 70.7024 78.7247 71.6514C75.6148 72.2996 72.5909 73.3412 69.4644 73.9161C66.7059 74.4253 63.863 74.5565 61.0794 74.8999C57.5486 75.3397 54.0525 76.1151 50.5059 76.4237C39.3721 77.3882 22.6481 76.9561 12.0456 73.3798C8.4484 72.1645 4.47699 69.8922 2.77871 66.3197C1.87787 64.4215 1.88866 62.3035 2.61937 60.3514C5.10507 53.7195 18.2897 46.1154 24.77 43.1679C28.3649 41.5321 32.1943 40.4866 35.9277 39.2173C59.9506 31.0384 101.485 21.9181 125.932 30.8957C128.167 31.7174 129.548 34.0862 131.676 34.7498C131.731 34.7652 131.787 34.7768 131.842 34.7922C133.524 36.8601 134.403 38.8007 134.093 41.5206C133.531 46.4511 129.305 50.2859 125.59 53.1639C117.967 59.0628 109.489 63.0712 100.404 66.1808C98.6196 66.7903 96.0741 67.8667 94.1821 67.724Z" fill="black"/>
<ellipse cx="68.1123" cy="52.136" rx="68.0829" ry="23.1443" transform="rotate(-10.2371 68.1123 52.136)" fill="black"/>
<path d="M117.673 57.2392L114.421 59.5C112.716 57.0888 111.768 54.257 110.87 51.4754C110.708 53.5703 110.959 55.4684 110.963 57.5325C110.965 58.5703 110.732 59.473 109.982 60.2253C108.959 60.6188 107.987 60.3141 106.959 60.0903C105.83 57.5595 106.594 45.6113 106.622 42.251C108.481 43.4894 111.94 42.386 114.321 43.3003C115.862 43.8945 117.29 44.967 117.928 46.5372C118.361 47.602 118.384 48.8636 117.897 49.9168C117.388 51.0163 116.185 52.0309 115.09 52.4939C115.843 54.1528 116.533 55.8041 117.673 57.2392Z" fill="white"/>
<path d="M111.078 45.46C112.412 46.0001 113.364 46.5479 113.972 47.902C113.844 48.7662 113.851 48.8164 113.213 49.4067C113.123 49.4877 113.039 49.5726 112.943 49.6459C112.254 50.1744 111.979 50.3249 111.14 50.1705C110.346 48.8627 110.832 46.8874 111.078 45.46Z" fill="black"/>
<path d="M58.1307 42.4658C58.2333 42.5314 58.3332 42.597 58.4382 42.6549C60.0273 43.5345 63.0863 43.0831 64.734 42.5353L64.7371 60.3167C64.6009 60.3862 64.4157 60.4942 64.2606 60.5405C63.256 60.8298 61.5504 60.6755 60.6735 60.1238L60.6098 50.5445L60.5917 48.4689C58.6017 51.1116 59.2395 57.6354 56.8012 60.6678C56.0207 61.1038 56.0211 61.2311 55.1168 60.9919C52.5655 59.1825 51.3591 51.2196 50.652 48.2258L50.5941 60.1084C50.402 60.2241 50.2048 60.336 49.9911 60.4132C49.0169 60.7643 47.6624 60.718 46.7326 60.2897C46.7234 54.441 46.6034 48.5653 46.8615 42.7243C48.7195 43.3455 51.508 43.249 53.302 42.516C53.7777 45.9149 54.4432 49.9735 55.6153 53.1872C57.1615 51.1965 57.6342 45.1047 58.1307 42.4658Z" fill="white"/>
<path d="M94.3023 42.8017C96.1163 43.1412 98.08 43.2763 99.7914 42.4893C100.91 48.2338 103.507 54.2214 103.88 60.0083C103.608 60.2012 103.32 60.4057 103.006 60.5253C102.291 60.8031 101.257 60.5793 100.615 60.2051C99.6232 59.6264 99.6765 58.523 99.4739 57.5006C99.2158 57.3077 99.0515 57.1766 98.7367 57.084C94.729 55.9073 94.9574 59.7383 92.2533 60.969C91.7201 61.2082 91.4204 60.942 90.9022 60.7375C89.9613 59.1557 93.6642 45.537 94.3023 42.8017Z" fill="white"/>
<path d="M97.0294 47.5049C97.6023 49.2371 98.3547 51.2818 98.2717 53.1144L97.4272 53.1877L96.4287 53.6005L95.8917 53.3497C95.5892 51.5557 96.5507 49.2448 97.0294 47.5049Z" fill="black"/>
<path d="M26.9385 52.9599C26.4084 52.8287 25.9053 52.5355 25.4454 52.2461C24.3652 51.5632 23.4674 50.564 23.2221 49.2793C23.0099 48.1682 23.4261 46.9761 24.043 46.0541C25.1028 44.4723 26.8516 43.2454 28.7417 42.9137C29.9924 42.6938 31.2825 42.8751 32.3138 43.6659C33.5916 44.6497 34.0329 45.9962 34.2301 47.5278C33.3825 48.2801 33.0908 48.5347 31.9484 48.7894C30.4454 48.2338 29.7718 46.6906 28.2209 46.4514C27.69 47.0494 27.5168 47.1613 27.3841 47.9715C28.3073 49.5725 31.8794 51.2932 33.6922 51.7832C34.4357 53.8395 35.0167 56.2661 34.0503 58.3726C33.5341 59.4953 32.6822 60.2592 31.5163 60.6604C29.6441 61.3047 27.3933 60.9575 25.6495 60.0933C24.278 59.4143 23.452 58.6311 22.9165 57.1843C22.9771 56.4089 22.962 56.2353 23.4277 55.5755C24.5307 55.2283 25.8169 54.9467 26.7988 54.3371C26.9875 55.6682 26.9489 56.0964 27.83 57.1496C28.7972 57.4351 29.0036 57.4891 29.9716 57.2461C30.3123 56.6249 30.4878 56.4784 30.2752 55.7569C29.8366 54.2677 28.1985 53.6273 26.9385 52.9599Z" fill="white"/>
<path d="M78.454 42.7314C79.6481 43.1404 81.0778 43.0285 82.3541 43.3911C83.001 47.4497 82.5184 52.3263 82.4879 56.4697C83.9401 57.2838 85.4416 56.9211 87.0392 57.1989C87.7174 58.1094 87.5442 58.5337 87.4887 59.6255L86.2992 60.4164C84.0311 60.6904 80.341 61.0684 78.3302 59.8416C78.3225 54.1434 78.2457 48.4297 78.454 42.7314Z" fill="white"/>
<path d="M69.2981 42.833C70.6414 43.0529 71.9755 43.0374 73.3304 43.3229C73.9693 47.586 73.7922 55.4717 73.369 59.812C72.7668 60.2093 72.2271 60.4524 71.4991 60.5334C70.5357 60.6376 70.0666 60.5874 69.327 59.9779C68.5384 54.7503 68.9551 48.1262 69.2981 42.833Z" fill="white"/>
<path d="M38.2432 42.7363C39.4017 43.1453 40.811 43.0141 42.0541 43.3729C42.7454 48.5811 42.3091 54.5032 42.1702 59.7847C41.8781 60.1011 41.6254 60.3287 41.2177 60.4869C40.235 60.865 39.0861 60.7338 38.1606 60.2786C38.1456 54.4337 38.0649 48.5812 38.2432 42.7363Z" fill="white"/>
<path d="M31.7417 77.2656C34.6333 78.45 39.7887 77.6784 42.9075 77.6089L43.1991 77.933C41.318 78.913 34.1298 78.3767 31.9161 77.9215L31.7417 77.2656Z" fill="black" fill-opacity="0.054902"/>
</g>
<defs>
<clipPath id="clip0_2269_17687">
<rect width="136" height="104" fill="white"/>
</clipPath>
</defs>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"/>
<path d="M61.6153 13.3636C68.4123 11.9558 93.2895 14.9368 99.447 17.1711C102.61 18.3191 104.994 20.3528 106.356 23.4398C110.113 31.9547 106.71 47.9414 103.336 56.3094C98.1065 69.2833 86.5012 84.2205 73.2571 89.8704C71.6944 90.4431 70.1611 90.9957 68.4756 91C60.2254 91.019 51.8975 84.3736 46.2866 78.9406C34.8296 67.8471 27.142 52.6293 27.0017 36.5507C26.9522 30.8746 27.9558 24.3015 32.1745 20.1392C38.5737 13.8258 53.1449 13.3764 61.6153 13.3636ZM66.5743 49.3744C63.2826 46.1618 54.0464 36.585 49.609 36.6349C48.863 36.643 48.6629 36.9992 48.1303 37.4695C46.193 42.9121 50.3147 52.0657 51.8471 57.428C54.7493 62.1714 58.7904 67.4319 64.4767 68.8734C67.8644 69.7318 71.1739 68.8777 74.1058 67.1056C81.4338 62.6765 86.6965 53.3674 88.5945 45.2401C89.2666 42.3628 89.839 38.2443 88.0384 35.6361C87.5231 34.889 86.8668 34.4966 86.067 34.1029L83.6753 34.0415C77.1188 37.9447 72.6257 44.7265 66.5743 49.3744Z" fill="black"/>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"/>
<path d="M44.7978 25.8642C45.0875 25.757 45.3548 25.7294 45.5263 25.4691C46.4772 24.0267 53.3853 19.7639 55.0126 19.1759C57.2558 18.3644 60.1673 17.8193 62.5357 17.3906C70.372 15.9665 79.5696 18.5665 86.03 23.0774C89.4252 25.4477 92.4536 28.4519 95.5601 31.1896C95.1479 29.5727 93.7662 28.0905 93.0845 26.5256C93.1368 26.5685 93.1874 26.6144 93.2413 26.6543C94.4469 27.5699 96.3003 27.8271 97.7482 28.2038C99.1183 28.559 100.592 28.8653 101.827 29.5758C105.654 31.7715 107.711 37.3144 108.395 41.4945C109.129 45.9748 109.087 55.4192 108.169 59.829C107.762 61.7889 107.003 63.7397 105.836 65.375C105.738 65.5138 105.637 65.6506 105.533 65.7854C105.429 65.9201 105.322 66.0528 105.212 66.1835C105.103 66.3121 104.99 66.4386 104.874 66.5632C104.758 66.6877 104.639 66.8092 104.518 66.9276C104.396 67.0481 104.272 67.1645 104.145 67.2767C104.019 67.3911 103.89 67.5013 103.758 67.6075C103.626 67.7157 103.492 67.8208 103.355 67.9229C103.219 68.0229 103.08 68.1209 102.939 68.2169C102.798 68.3108 102.654 68.4016 102.509 68.4894C102.364 68.5793 102.217 68.665 102.067 68.7467C101.918 68.8283 101.767 68.9059 101.614 68.9794C101.462 69.0549 101.307 69.1264 101.151 69.1938C100.995 69.2612 100.838 69.3245 100.679 69.3837C100.52 69.4449 100.359 69.5021 100.198 69.5551C100.036 69.6082 99.8736 69.6572 99.7096 69.7021C99.5459 69.747 99.3811 69.7889 99.2154 69.8277C96.0936 76.1821 88.5993 81.9394 81.9849 84.2178C72.9603 87.3261 62.5602 86.1502 53.7865 82.8857C52.7192 82.4876 51.5558 82.2456 50.5364 81.7342C49.6474 81.2902 48.9461 80.7389 48.37 79.9305C47.1772 80.0132 46.8483 79.3701 45.733 79.1465C45.7897 79.5661 45.8307 79.9764 45.8577 80.399C45.4602 80.2674 45.2011 80.0315 44.8734 79.7774C43.1842 79.7345 41.3204 79.3946 39.671 79.024C35.9772 78.1941 32.4748 76.3506 30.4184 73.0707C26.6222 67.0195 26.8975 59.7555 27.1024 52.9111C27.2016 49.5915 27.2485 46.2994 27.9694 43.0441C29.111 37.8871 31.1509 34.105 36.3539 32.2125C37.7209 31.7164 39.2836 31.4622 40.5046 30.6568C40.557 30.6231 40.6069 30.5833 40.6581 30.5496C42.1378 29.1654 43.4509 27.3923 44.7978 25.8642Z" fill="black"/>
<path d="M61.7904 57.9827C60.4295 59.6394 58.9687 60.9807 56.7387 61.1859C55.6485 61.287 54.5757 61.0144 53.7272 60.307C52.4422 59.2352 51.8132 57.4069 51.6781 55.7838C51.3302 51.5976 53.1306 46.171 55.8714 43.0382C57.0223 41.7214 58.6303 40.5087 60.4353 40.3709C62.3554 40.227 63.8808 42.4717 64.9924 42.3492C65.1544 41.9358 65.2818 41.2896 65.6873 41.112C67.0647 40.5026 68.7411 41.5101 70.1939 40.5883C69.797 43.2556 66.4639 55.6613 67.8653 57.2997C68.1243 57.6029 68.3399 57.5171 68.7102 57.5355C70.4092 56.5862 71.1794 55.2541 71.7049 53.4197C72.8719 49.3498 73.6933 44.649 74.2029 40.4383C75.4085 41.1794 78.1086 41.2835 79.4653 40.9068C79.758 40.8272 80.0042 40.7077 80.2547 40.5393C79.8085 43.3353 76.5839 55.0826 77.5758 56.8434C78.247 57.3395 78.3607 57.4865 79.1786 57.511C80.3779 56.9414 81.0307 55.5909 81.4613 54.3935C83.0366 50.0266 83.3459 44.992 84.3433 40.4413C85.5787 41.0997 88.3722 41.0385 89.7175 40.6128C89.9288 40.5454 89.9953 40.5026 90.209 40.4015C88.9776 46.56 87.0771 58.8922 84.6419 64.135C83.6411 66.2878 82.1451 68.2661 80.1414 69.5738C77.9252 71.0223 74.8916 71.5459 72.3079 70.9641C71.8013 70.8507 71.4966 70.716 71.1071 70.3608C70.1577 70.1066 69.5569 69.54 69.0642 68.704C68.9956 68.5876 68.9325 68.4682 68.8694 68.3488C68.8005 68.3549 68.7313 68.3641 68.6624 68.3702C65.8637 68.6122 62.4871 67.6751 59.6657 67.3597C55.3704 66.8819 41.4525 65.9387 38.4404 69.3778C38.1807 70.2567 38.1204 70.7711 38.5855 71.6133C40.0659 74.2898 45.4333 79.0763 48.3701 79.9307C47.1773 80.0134 46.8484 79.3703 45.7331 79.1468C45.7898 79.5663 45.8308 79.9767 45.8578 80.3993C45.4603 80.2676 45.2012 80.0318 44.8735 79.7776C44.706 77.784 34.0342 72.434 37.3156 68.8173C39.6629 66.2296 47.4383 66.0979 50.7175 66.0826C56.7871 66.0551 62.4755 67.1943 68.4643 67.6812C68.8682 66.3215 69.5937 64.6862 70.8924 63.9727C71.8868 63.4245 77.7644 62.2608 78.9468 62.518C79.0396 62.5364 79.1832 62.6099 79.2595 62.6436C77.4692 64.1717 73.3889 64.3065 73.1267 67.13C73.0967 67.4515 73.3488 67.7516 73.5031 68.0334C74.1921 68.4223 74.7719 68.4989 75.5374 68.2661C77.2092 67.7547 78.3708 66.1286 79.1033 64.6341C80.0517 62.6987 80.6155 60.2764 80.9318 58.1511C79.8591 59.5046 78.0039 60.9409 76.2501 61.2104C74.536 61.4768 73.0634 60.3958 71.805 59.3913C71.6233 59.2464 71.4429 59.0994 71.2639 58.9503C68.901 59.2106 66.7757 63.55 64.1772 61.5166C62.903 60.5213 62.3095 59.4832 61.7904 57.9827Z" fill="white"/>
<path d="M61.4895 42.7227C61.8383 42.8023 62.2045 42.888 62.5043 43.0902C63.0332 43.4515 63.2412 43.9813 63.3719 44.5815C64.0257 47.5949 62.5111 53.0643 60.8059 55.6061C60.2936 56.3717 59.901 56.6871 59.0114 56.8647C58.9618 56.8831 58.9131 56.9045 58.8623 56.9198C58.0308 57.171 57.9178 57.1036 57.1577 56.733C56.7872 56.1665 56.771 55.6092 56.7789 54.9508C56.8295 50.8043 58.5003 45.6135 61.4895 42.7227Z" fill="black"/>
<path d="M40.658 30.5497C42.1377 29.1655 43.4509 27.3924 44.7977 25.8643C44.3726 27.2362 43.2604 28.4581 42.1463 29.3339C44.1083 30.0965 50.7558 29.0889 53.7037 29.4258C67.3968 30.9937 79.4158 36.9072 94.7081 34.5522C86.7624 38.812 62.4645 32.0625 52.6322 30.96C49.9979 30.666 42.9266 31.2418 41.2279 30.7579C41.0319 30.6998 40.8442 30.6324 40.658 30.5497Z" fill="white"/>
<path d="M36.1837 34.623C36.2728 34.6537 36.4075 34.7027 36.5049 34.7272C38.2725 35.1528 42.7843 35.6796 44.4105 34.7608C43.9382 38.157 42.988 42.9895 41.6111 46.1315C45.6128 44.4839 46.6941 36.6595 51.3088 35.3611C52.3613 35.064 55.4399 34.7945 56.3902 35.3488C56.6851 36.1695 56.6976 36.2032 56.4107 37.0239C55.1548 38.3928 53.1955 38.5796 51.8331 39.7587C49.6689 41.6359 46.9403 46.5173 44.555 47.5034C45.2002 49.8216 48.4675 56.9447 50.0688 58.6597C50.1368 58.9292 50.2026 59.165 50.3074 59.4222C50.8653 60.8003 51.9923 62.996 53.4197 63.6054C53.258 63.7095 53.0905 63.8045 52.9104 63.8718C51.6337 64.3465 47.3764 64.6007 46.1572 63.9882C44.533 62.359 41.2516 53.6618 40.5761 51.1721C39.8245 55.1042 39.1961 59.3181 37.8217 63.0756C37.5642 63.2226 37.3654 63.3267 37.0785 63.4064C35.6588 63.8014 33.0698 63.7922 31.7959 63.0205C33.3041 57.9094 37.1232 42.5025 37.0173 37.6762C36.9937 36.6044 36.579 35.603 36.1837 34.623Z" fill="white"/>
<path d="M96.7506 40.2422C98.7951 41.1242 101.21 40.1472 103.266 41.5621C103.579 42.196 103.596 42.5512 103.477 43.2433C101.916 44.7745 99.3977 43.1576 97.4617 43.6016C96.6774 43.7823 96.5984 44.0211 96.2003 44.6642C96.0729 47.4663 100.602 49.7172 101.7 52.2008C102.327 53.6156 101.99 55.5847 101.428 56.9689C100.829 58.445 99.7827 59.8077 98.2683 60.4172C97.4066 60.7663 96.2995 61.0694 95.4124 60.6928C95.0072 60.8888 94.554 61.0358 94.105 61.0817C92.5937 61.2409 90.9318 60.9071 89.7362 59.9425C88.9976 59.3453 88.6913 58.7604 88.6025 57.8295C90.6341 55.2417 94.1369 58.8247 96.539 57.5538C97.2302 57.1894 97.2709 56.8005 97.4528 56.1053C96.825 53.0858 93.6579 51.6986 92.0321 49.2456C91.4215 48.3238 91.1302 47.3561 91.3954 46.2536C92.0578 43.5036 94.4554 41.6049 96.7506 40.2422Z" fill="white"/>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"/>
<path d="M65.8801 7.38124C66.7524 7.11219 67.6899 7.10271 68.5958 7.05487C75.0533 6.71287 81.2447 7.90559 86.161 12.4467C94.7447 20.3756 95.2979 34.2651 95.7884 45.1587C96.1193 52.5191 96.7131 75.3309 91.7203 80.5797C90.6101 81.7473 89.0514 82.7652 87.3921 82.7363C83.4312 82.6676 81.9221 78.0289 79.4042 75.8093C78.6178 75.1158 78.614 75.0741 77.5684 75.1594C76.3708 78.6347 77.5089 83.8712 76.9543 87.743C76.385 91.7172 75.4135 94.0898 72.2026 96.5169C71.4242 96.6841 70.6378 96.839 69.8467 96.9266C67.7055 97.164 64.7075 96.8708 63.0157 95.3744C58.7947 91.639 60.3836 83.0579 60.0253 78.0455C59.9517 77.01 59.7086 76.3104 58.9514 75.608C55.6023 77.3051 54.5288 82.1703 50.0165 82.6198C48.3691 82.7841 46.9624 82.0438 45.7705 80.9752C42.2051 77.7788 41.8379 68.3276 41.5575 63.8068C40.6587 49.2901 39.5173 25.9953 49.8726 14.5086C54.1913 9.71835 59.6042 7.74266 65.8801 7.38124ZM71.2312 52.9166C72.3801 51.9863 73.1751 51.253 73.9223 49.949C73.8765 49.3422 73.9648 49.0045 73.4965 48.5611C71.6768 46.8398 67.5039 47.484 65.1791 47.5991C64.0042 47.9775 63.8493 47.9037 63.2447 48.9581C63.3877 50.0821 63.8777 50.9049 64.7713 51.6045C66.6892 53.1075 68.491 53.359 70.8507 52.9843C70.9777 52.9639 71.1042 52.9388 71.2312 52.9166ZM58.4954 40.1278C59.4754 37.947 60.4157 35.879 59.6453 33.4841C58.8386 32.5443 58.5115 32.1719 57.2053 32.0824C56.941 32.0644 56.6738 32.0668 56.409 32.0692C55.6287 33.7806 54.7073 35.8387 55.3643 37.7495C55.6287 38.5183 56.2721 39.3486 57.0245 39.6755C57.4947 39.8796 57.9998 40.0066 58.4954 40.1278ZM81.0733 40.2045C82.1123 38.3847 83.0757 36.3119 82.4017 34.1718C82.1911 33.503 81.7257 32.9739 81.2612 32.468C80.6395 32.234 80.0282 31.9797 79.3749 31.8499C78.4374 33.6513 77.5439 35.7695 78.2599 37.8276C78.7565 39.2548 79.5826 39.6432 80.8897 40.1453C80.9497 40.1686 81.012 40.1847 81.0733 40.2045Z" fill="black"/>
</svg>',
            ],
            [
                'svg_html' => '<svg width="136" height="104" viewBox="0 0 136 104" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="136" height="104" fill="white"/>
<path d="M59.9095 13.1439C61.5738 12.8153 63.6538 13.1565 65.3873 13.1259C79.0401 12.8883 93.0003 14.6018 106 18.9491C105.942 31.75 105.839 44.9474 101.773 57.2218C97.1732 71.1077 85.7017 82.0599 72.8492 88.552C70.5303 89.8231 68.258 91.2073 65.7215 92C58.9509 88.0706 52.6369 83.5461 47.1223 77.9686C29.4737 60.117 29.9214 42.0125 30.0223 18.5059C39.3546 15.6612 50.1268 13.2127 59.9095 13.1439ZM47.8763 54.8241C52.7654 55.3785 57.3462 54.742 62.1932 54.8547C62.1821 60.0196 62.2135 65.1891 62.1317 70.3528C65.1182 70.4155 69.8371 71.1502 72.6129 70.3123L72.853 69.8531C72.8507 64.8876 72.5976 59.782 72.8985 54.8337C76.8737 54.8363 84.6525 55.6184 88.0715 54.87C88.5314 51.0915 88.1177 47.0204 88.3338 43.1747C85.0372 42.4687 76.8531 43.3398 72.8694 43.3111C72.8316 38.5974 72.7459 33.8788 72.9012 29.167C69.4413 29.3309 66.0408 29.379 62.5844 29.1081C60.652 31.4203 63.3961 41.7726 61.6943 42.9474C61.4973 43.0834 60.5044 43.1411 60.2467 43.1606C57.5523 43.3673 49.7418 42.2406 47.692 43.1121C47.5509 47.0231 48.0786 50.8722 47.8763 54.8241Z" fill="black"/>
</svg>',
            ],
        ];
    }

    private function getHeadingColumn(): siteBlockData {
        $vseq = $this->createSequence();

        $heading = (new siteHeadingBlockType())->getEmptyBlockData();
        $heading->data = [
            'html'        => 'Condimentum&nbsp;pulvinar',
            'tag'         => 'h1',
            'block_props' => [
                'align'         => 't-c',
                'font-header'   => 't-hdn',
                'font-size'     => ['name' => 'Size #3', 'value' => 't-3', 'unit' => 'px', 'type' => 'library'],
                'margin-bottom' => 'm-b-16',
                'margin-top'    => 'm-t-0',
            ],
        ];

        $vseq->addChild($heading);

        $block_props = [
            $this->column_elements['main']    => [
                'padding-bottom' => 'p-b-0',
                'padding-left' => 'p-l-clm',
                'padding-right' => 'p-r-clm',
            ],
            $this->column_elements['wrapper'] => [
                'flex-align'   => 'y-c',
                'margin-left'  => 'm-l-a',
                'margin-right' => 'm-r-a',
            ],
        ];

        return $this->createColumn(
            [
                'column'        => 'st-12-mb st-12 st-12-lp st-12-tb',
                'block_props'   => $block_props,
                'wrapper_props' => ['flex-align' => 'y-c'],
            ],
            $vseq
        );
    }

    /**
     * Получает колонку с
     *
     * @return siteBlockData
     */
    private function getLogoColumn($item): siteBlockData {
        $vseq = $this->createSequence();

        $vseq->addChild($this->getLogo($item['svg_html']));


        $block_props = [
            $this->column_elements['main']    => [
                'flex-align-vertical' => 'a-c-c',
                'padding-bottom'      => 'p-b-12',
                'padding-top'         => 'p-t-12',
                'padding-left' => 'p-l-clm',
                'padding-right' => 'p-r-clm',
            ],
            $this->column_elements['wrapper'] => [
                'flex-align' => 'y-c',
            ],
        ];

        return $this->createColumn(
            [
                'column'        => 'st-2 st-2-lp st-3-tb st-4-mb',
                'block_props'   => $block_props,
                'wrapper_props' => ['flex-align' => 'y-c'],
            ],
            $vseq
        );
    }

    private function getLogo(string $svg_html) {
        $imageBlock = (new siteImageBlockType())->getEmptyBlockData();

        $imageBlock->data = [
            'image' => [
                'type'     => 'svg',
                'svg_html' => $svg_html,
            ],
        ];

        return $imageBlock;
    }

    /**
     * Создаёт последовательность блоков
     *
     * @param bool   $is_horizontal
     * @param string $complex_type
     * @param bool   $indestructible
     * @return siteBlockData
     */
    private function createSequence(bool $is_horizontal = false, string $complex_type = 'with_row', bool $indestructible = false): siteBlockData {
        $seq = (new siteVerticalSequenceBlockType())->getEmptyBlockData();
        $seq->data['is_horizontal'] = $is_horizontal;
        $seq->data['is_complex'] = $complex_type;

        if ($indestructible) {
            $seq->data['indestructible'] = true;
        }

        return $seq;
    }

    /**
     * Создаёт колонку с настройками
     *
     * @param array         $params
     * @param siteBlockData $content
     * @return siteBlockData
     */
    private function createColumn(array $params, siteBlockData $content): siteBlockData {
        $column = (new siteColumnBlockType())->getEmptyBlockData();

        $column->data = [
            'elements'      => $this->column_elements,
            'column'        => $params['column'] ?? 'st-12 st-12-lp st-12-tb st-12-mb',
            'block_props'   => $params['block_props'] ?? [],
            'wrapper_props' => $params['wrapper_props'] ?? [],
            'inline_props'  => $params['inline_props'] ?? [],
        ];

        $column->addChild($content, '');

        return $column;
    }
}
