<?php
/**
 * Adds settings to the permalinks admin settings page
 *
 * @class       KKART_Admin_Permalink_Settings
 * @author      WooThemes
 * @category    Admin
 * @package     Kkart\Admin
 * @version     2.3.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( class_exists( 'KKART_Admin_Permalink_Settings', false ) ) {
	return new KKART_Admin_Permalink_Settings();
}

/**
 * KKART_Admin_Permalink_Settings Class.
 */
class KKART_Admin_Permalink_Settings {

	/**
	 * Permalink settings.
	 *
	 * @var array
	 */
	private $permalinks = array();

	/**
	 * Hook in tabs.
	 */
	public function __construct() {
		$this->settings_init();
		$this->settings_save();
	}

	/**
	 * Init our settings.
	 */
	public function settings_init() {
		add_settings_section( 'kkart-permalink', __( 'Product permalinks', 'kkart' ), array( $this, 'settings' ), 'permalink' );

		add_settings_field(
			'kkart_product_category_slug',
			__( 'Product category base', 'kkart' ),
			array( $this, 'product_category_slug_input' ),
			'permalink',
			'optional'
		);
		add_settings_field(
			'kkart_product_tag_slug',
			__( 'Product tag base', 'kkart' ),
			array( $this, 'product_tag_slug_input' ),
			'permalink',
			'optional'
		);
		add_settings_field(
			'kkart_product_attribute_slug',
			__( 'Product attribute base', 'kkart' ),
			array( $this, 'product_attribute_slug_input' ),
			'permalink',
			'optional'
		);

		$this->permalinks = kkart_get_permalink_structure();
	}

	/**
	 * Show a slug input box.
	 */
	public function product_category_slug_input() {
		?>
		<input name="kkart_product_category_slug" type="text" class="regular-text code" value="<?php echo esc_attr( $this->permalinks['category_base'] ); ?>" placeholder="<?php echo esc_attr_x( 'product-category', 'slug', 'kkart' ); ?>" />
		<?php
	}

	/**
	 * Show a slug input box.
	 */
	public function product_tag_slug_input() {
		?>
		<input name="kkart_product_tag_slug" type="text" class="regular-text code" value="<?php echo esc_attr( $this->permalinks['tag_base'] ); ?>" placeholder="<?php echo esc_attr_x( 'product-tag', 'slug', 'kkart' ); ?>" />
		<?php
	}

	/**
	 * Show a slug input box.
	 */
	public function product_attribute_slug_input() {
		?>
		<input name="kkart_product_attribute_slug" type="text" class="regular-text code" value="<?php echo esc_attr( $this->permalinks['attribute_base'] ); ?>" /><code>/attribute-name/attribute/</code>
		<?php
	}

	/**
	 * Show the settings.
	 */
	public function settings() {
		/* translators: %s: Home URL */
		echo wp_kses_post( wpautop( sprintf( __( 'If you like, you may enter custom structures for your product URLs here. For example, using <code>shop</code> would make your product links like <code>%sshop/sample-product/</code>. This setting affects product URLs only, not things such as product categories.', 'kkart' ), esc_url( home_url( '/' ) ) ) ) );

		$shop_page_id = kkart_get_page_id( 'shop' );
		$base_slug    = urldecode( ( $shop_page_id > 0 && get_post( $shop_page_id ) ) ? get_page_uri( $shop_page_id ) : _x( 'shop', 'default-slug', 'kkart' ) );
		$product_base = _x( 'product', 'default-slug', 'kkart' );

		$structures = array(
			0 => '',
			1 => '/' . trailingslashit( $base_slug ),
			2 => '/' . trailingslashit( $base_slug ) . trailingslashit( '%product_cat%' ),
		);
		?>
		<table class="form-table kkart-permalink-structure">
			<tbody>
				<tr>
					<th><label><input name="product_permalink" type="radio" value="<?php echo esc_attr( $structures[0] ); ?>" class="wctog" <?php checked( $structures[0], $this->permalinks['product_base'] ); ?> /> <?php esc_html_e( 'Default', 'kkart' ); ?></label></th>
					<td><code class="default-example"><?php echo esc_html( home_url() ); ?>/?product=sample-product</code> <code class="non-default-example"><?php echo esc_html( home_url() ); ?>/<?php echo esc_html( $product_base ); ?>/sample-product/</code></td>
				</tr>
				<?php if ( $shop_page_id ) : ?>
					<tr>
						<th><label><input name="product_permalink" type="radio" value="<?php echo esc_attr( $structures[1] ); ?>" class="wctog" <?php checked( $structures[1], $this->permalinks['product_base'] ); ?> /> <?php esc_html_e( 'Shop base', 'kkart' ); ?></label></th>
						<td><code><?php echo esc_html( home_url() ); ?>/<?php echo esc_html( $base_slug ); ?>/sample-product/</code></td>
					</tr>
					<tr>
						<th><label><input name="product_permalink" type="radio" value="<?php echo esc_attr( $structures[2] ); ?>" class="wctog" <?php checked( $structures[2], $this->permalinks['product_base'] ); ?> /> <?php esc_html_e( 'Shop base with category', 'kkart' ); ?></label></th>
						<td><code><?php echo esc_html( home_url() ); ?>/<?php echo esc_html( $base_slug ); ?>/product-category/sample-product/</code></td>
					</tr>
				<?php endif; ?>
				<tr>
					<th><label><input name="product_permalink" id="kkart_custom_selection" type="radio" value="custom" class="tog" <?php checked( in_array( $this->permalinks['product_base'], $structures, true ), false ); ?> />
						<?php esc_html_e( 'Custom base', 'kkart' ); ?></label></th>
					<td>
						<input name="product_permalink_structure" id="kkart_permalink_structure" type="text" value="<?php echo esc_attr( $this->permalinks['product_base'] ? trailingslashit( $this->permalinks['product_base'] ) : '' ); ?>" class="regular-text code"> <span class="description"><?php esc_html_e( 'Enter a custom base to use. A base must be set or WordPress will use default instead.', 'kkart' ); ?></span>
					</td>
				</tr>
			</tbody>
		</table>
		<?php wp_nonce_field( 'kkart-permalinks', 'kkart-permalinks-nonce' ); ?>
		<script type="text/javascript">
			jQuery( function() {
				jQuery('input.wctog').change(function() {
					jQuery('#kkart_permalink_structure').val( jQuery( this ).val() );
				});
				jQuery('.permalink-structure input').change(function() {
					jQuery('.kkart-permalink-structure').find('code.non-default-example, code.default-example').hide();
					if ( jQuery(this).val() ) {
						jQuery('.kkart-permalink-structure code.non-default-example').show();
						jQuery('.kkart-permalink-structure input').removeAttr('disabled');
					} else {
						jQuery('.kkart-permalink-structure code.default-example').show();
						jQuery('.kkart-permalink-structure input:eq(0)').click();
						jQuery('.kkart-permalink-structure input').attr('disabled', 'disabled');
					}
				});
				jQuery('.permalink-structure input:checked').change();
				jQuery('#kkart_permalink_structure').focus( function(){
					jQuery('#kkart_custom_selection').click();
				} );
			} );
		</script>
		<?php
	}

	/**
	 * Save the settings.
	 */
	public function settings_save() {
		if ( ! is_admin() ) {
			return;
		}

		// We need to save the options ourselves; settings api does not trigger save for the permalinks page.
		if ( isset( $_POST['permalink_structure'], $_POST['kkart-permalinks-nonce'], $_POST['kkart_product_category_slug'], $_POST['kkart_product_tag_slug'], $_POST['kkart_product_attribute_slug'] ) && wp_verify_nonce( wp_unslash( $_POST['kkart-permalinks-nonce'] ), 'kkart-permalinks' ) ) { // WPCS: input var ok, sanitization ok.
			kkart_switch_to_site_locale();

			$permalinks                   = (array) get_option( 'kkart_permalinks', array() );
			$permalinks['category_base']  = kkart_sanitize_permalink( wp_unslash( $_POST['kkart_product_category_slug'] ) ); // WPCS: input var ok, sanitization ok.
			$permalinks['tag_base']       = kkart_sanitize_permalink( wp_unslash( $_POST['kkart_product_tag_slug'] ) ); // WPCS: input var ok, sanitization ok.
			$permalinks['attribute_base'] = kkart_sanitize_permalink( wp_unslash( $_POST['kkart_product_attribute_slug'] ) ); // WPCS: input var ok, sanitization ok.

			// Generate product base.
			$product_base = isset( $_POST['product_permalink'] ) ? kkart_clean( wp_unslash( $_POST['product_permalink'] ) ) : ''; // WPCS: input var ok, sanitization ok.

			if ( 'custom' === $product_base ) {
				if ( isset( $_POST['product_permalink_structure'] ) ) { // WPCS: input var ok.
					$product_base = preg_replace( '#/+#', '/', '/' . str_replace( '#', '', trim( wp_unslash( $_POST['product_permalink_structure'] ) ) ) ); // WPCS: input var ok, sanitization ok.
				} else {
					$product_base = '/';
				}

				// This is an invalid base structure and breaks pages.
				if ( '/%product_cat%/' === trailingslashit( $product_base ) ) {
					$product_base = '/' . _x( 'product', 'slug', 'kkart' ) . $product_base;
				}
			} elseif ( empty( $product_base ) ) {
				$product_base = _x( 'product', 'slug', 'kkart' );
			}

			$permalinks['product_base'] = kkart_sanitize_permalink( $product_base );

			// Shop base may require verbose page rules if nesting pages.
			$shop_page_id   = kkart_get_page_id( 'shop' );
			$shop_permalink = ( $shop_page_id > 0 && get_post( $shop_page_id ) ) ? get_page_uri( $shop_page_id ) : _x( 'shop', 'default-slug', 'kkart' );

			if ( $shop_page_id && stristr( trim( $permalinks['product_base'], '/' ), $shop_permalink ) ) {
				$permalinks['use_verbose_page_rules'] = true;
			}

			update_option( 'kkart_permalinks', $permalinks );
			kkart_restore_locale();
		}
	}
}

return new KKART_Admin_Permalink_Settings();
