<?php
/**
 * Kkart Email Settings
 *
 * @package Kkart\Admin
 * @version 2.1.0
 */

defined( 'ABSPATH' ) || exit;

if ( class_exists( 'KKART_Settings_Emails', false ) ) {
	return new KKART_Settings_Emails();
}

/**
 * KKART_Settings_Emails.
 */
class KKART_Settings_Emails extends KKART_Settings_Page {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id    = 'email';
		$this->label = __( 'Emails', 'kkart' );

		add_action( 'kkart_admin_field_email_notification', array( $this, 'email_notification_setting' ) );
		parent::__construct();
	}

	/**
	 * Get sections.
	 *
	 * @return array
	 */
	public function get_sections() {
		$sections = array(
			'' => __( 'Email options', 'kkart' ),
		);
		return apply_filters( 'kkart_get_sections_' . $this->id, $sections );
	}

	/**
	 * Get settings array.
	 *
	 * @return array
	 */
	public function get_settings() {
		$settings = apply_filters(
			'kkart_email_settings',
			array(
				array(
					'title' => __( 'Email notifications', 'kkart' ),
					'desc'  => __( 'Email notifications sent from Kkart are listed below. Click on an email to configure it.', 'kkart' ),
					'type'  => 'title',
					'id'    => 'email_notification_settings',
				),

				array( 'type' => 'email_notification' ),

				array(
					'type' => 'sectionend',
					'id'   => 'email_notification_settings',
				),

				array(
					'type' => 'sectionend',
					'id'   => 'email_recipient_options',
				),

				array(
					'title' => __( 'Email sender options', 'kkart' ),
					'type'  => 'title',
					'desc'  => '',
					'id'    => 'email_options',
				),

				array(
					'title'    => __( '"From" name', 'kkart' ),
					'desc'     => __( 'How the sender name appears in outgoing Kkart emails.', 'kkart' ),
					'id'       => 'kkart_email_from_name',
					'type'     => 'text',
					'css'      => 'min-width:400px;',
					'default'  => esc_attr( get_bloginfo( 'name', 'display' ) ),
					'autoload' => false,
					'desc_tip' => true,
				),

				array(
					'title'             => __( '"From" address', 'kkart' ),
					'desc'              => __( 'How the sender email appears in outgoing Kkart emails.', 'kkart' ),
					'id'                => 'kkart_email_from_address',
					'type'              => 'email',
					'custom_attributes' => array(
						'multiple' => 'multiple',
					),
					'css'               => 'min-width:400px;',
					'default'           => get_option( 'admin_email' ),
					'autoload'          => false,
					'desc_tip'          => true,
				),

				array(
					'type' => 'sectionend',
					'id'   => 'email_options',
				),

				array(
					'title' => __( 'Email template', 'kkart' ),
					'type'  => 'title',
					/* translators: %s: Nonced email preview link */
					'desc'  => sprintf( __( 'This section lets you customize the Kkart emails. <a href="%s" target="_blank">Click here to preview your email template</a>.', 'kkart' ), wp_nonce_url( admin_url( '?preview_kkart_mail=true' ), 'preview-mail' ) ),
					'id'    => 'email_template_options',
				),

				array(
					'title'       => __( 'Header image', 'kkart' ),
					'desc'        => __( 'URL to an image you want to show in the email header. Upload images using the media uploader (Admin > Media).', 'kkart' ),
					'id'          => 'kkart_email_header_image',
					'type'        => 'text',
					'css'         => 'min-width:400px;',
					'placeholder' => __( 'N/A', 'kkart' ),
					'default'     => '',
					'autoload'    => false,
					'desc_tip'    => true,
				),

				array(
					'title'       => __( 'Footer text', 'kkart' ),
					/* translators: %s: Available placeholders for use */
					'desc'        => __( 'The text to appear in the footer of all Kkart emails.', 'kkart' ) . ' ' . sprintf( __( 'Available placeholders: %s', 'kkart' ), '{site_title} {site_url}' ),
					'id'          => 'kkart_email_footer_text',
					'css'         => 'width:400px; height: 75px;',
					'placeholder' => __( 'N/A', 'kkart' ),
					'type'        => 'textarea',
					'default'     => '{site_title} &mdash; Built with {Kkart}',
					'autoload'    => false,
					'desc_tip'    => true,
				),

				array(
					'title'    => __( 'Base color', 'kkart' ),
					/* translators: %s: default color */
					'desc'     => sprintf( __( 'The base color for Kkart email templates. Default %s.', 'kkart' ), '<code>#96588a</code>' ),
					'id'       => 'kkart_email_base_color',
					'type'     => 'color',
					'css'      => 'width:6em;',
					'default'  => '#0f89db',
					'autoload' => false,
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'Background color', 'kkart' ),
					/* translators: %s: default color */
					'desc'     => sprintf( __( 'The background color for Kkart email templates. Default %s.', 'kkart' ), '<code>#f7f7f7</code>' ),
					'id'       => 'kkart_email_background_color',
					'type'     => 'color',
					'css'      => 'width:6em;',
					'default'  => '#f7f7f7',
					'autoload' => false,
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'Body background color', 'kkart' ),
					/* translators: %s: default color */
					'desc'     => sprintf( __( 'The main body background color. Default %s.', 'kkart' ), '<code>#ffffff</code>' ),
					'id'       => 'kkart_email_body_background_color',
					'type'     => 'color',
					'css'      => 'width:6em;',
					'default'  => '#ffffff',
					'autoload' => false,
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'Body text color', 'kkart' ),
					/* translators: %s: default color */
					'desc'     => sprintf( __( 'The main body text color. Default %s.', 'kkart' ), '<code>#3c3c3c</code>' ),
					'id'       => 'kkart_email_text_color',
					'type'     => 'color',
					'css'      => 'width:6em;',
					'default'  => '#3c3c3c',
					'autoload' => false,
					'desc_tip' => true,
				),

				array(
					'type' => 'sectionend',
					'id'   => 'email_template_options',
				),

			)
		);

		return apply_filters( 'kkart_get_settings_' . $this->id, $settings );
	}

	/**
	 * Output the settings.
	 */
	public function output() {
		global $current_section;

		// Define emails that can be customised here.
		$mailer          = KKART()->mailer();
		$email_templates = $mailer->get_emails();

		if ( $current_section ) {
			foreach ( $email_templates as $email_key => $email ) {
				if ( strtolower( $email_key ) === $current_section ) {
					$email->admin_options();
					break;
				}
			}
		} else {
			$settings = $this->get_settings();
			KKART_Admin_Settings::output_fields( $settings );
		}
	}

	/**
	 * Save settings.
	 */
	public function save() {
		global $current_section;

		if ( ! $current_section ) {
			KKART_Admin_Settings::save_fields( $this->get_settings() );

		} else {
			$kkart_emails = KKART_Emails::instance();

			if ( in_array( $current_section, array_map( 'sanitize_title', array_keys( $kkart_emails->get_emails() ) ), true ) ) {
				foreach ( $kkart_emails->get_emails() as $email_id => $email ) {
					if ( sanitize_title( $email_id ) === $current_section ) {
						do_action( 'kkart_update_options_' . $this->id . '_' . $email->id );
					}
				}
			} else {
				do_action( 'kkart_update_options_' . $this->id . '_' . $current_section );
			}
		}
	}

	/**
	 * Output email notification settings.
	 */
	public function email_notification_setting() {
		// Define emails that can be customised here.
		$mailer          = KKART()->mailer();
		$email_templates = $mailer->get_emails();

		?>
		<tr valign="top">
		<td class="kkart_emails_wrapper" colspan="2">
			<table class="kkart_emails widefat" cellspacing="0">
				<thead>
					<tr>
						<?php
						$columns = apply_filters(
							'kkart_email_setting_columns',
							array(
								'status'     => '',
								'name'       => __( 'Email', 'kkart' ),
								'email_type' => __( 'Content type', 'kkart' ),
								'recipient'  => __( 'Recipient(s)', 'kkart' ),
								'actions'    => '',
							)
						);
						foreach ( $columns as $key => $column ) {
							echo '<th class="kkart-email-settings-table-' . esc_attr( $key ) . '">' . esc_html( $column ) . '</th>';
						}
						?>
						</tr>
					</thead>
					<tbody>
						<?php
						foreach ( $email_templates as $email_key => $email ) {
							echo '<tr>';

							foreach ( $columns as $key => $column ) {

								switch ( $key ) {
									case 'name':
										echo '<td class="kkart-email-settings-table-' . esc_attr( $key ) . '">
										<a href="' . esc_url( admin_url( 'admin.php?page=kkart-settings&tab=email&section=' . strtolower( $email_key ) ) ) . '">' . esc_html( $email->get_title() ) . '</a>
										' . kkart_help_tip( $email->get_description() ) . '
									</td>';
										break;
									case 'recipient':
										echo '<td class="kkart-email-settings-table-' . esc_attr( $key ) . '">
										' . esc_html( $email->is_customer_email() ? __( 'Customer', 'kkart' ) : $email->get_recipient() ) . '
									</td>';
										break;
									case 'status':
										echo '<td class="kkart-email-settings-table-' . esc_attr( $key ) . '">';

										if ( $email->is_manual() ) {
											echo '<span class="status-manual tips" data-tip="' . esc_attr__( 'Manually sent', 'kkart' ) . '">' . esc_html__( 'Manual', 'kkart' ) . '</span>';
										} elseif ( $email->is_enabled() ) {
											echo '<span class="status-enabled tips" data-tip="' . esc_attr__( 'Enabled', 'kkart' ) . '">' . esc_html__( 'Yes', 'kkart' ) . '</span>';
										} else {
											echo '<span class="status-disabled tips" data-tip="' . esc_attr__( 'Disabled', 'kkart' ) . '">-</span>';
										}

										echo '</td>';
										break;
									case 'email_type':
										echo '<td class="kkart-email-settings-table-' . esc_attr( $key ) . '">
										' . esc_html( $email->get_content_type() ) . '
									</td>';
										break;
									case 'actions':
										echo '<td class="kkart-email-settings-table-' . esc_attr( $key ) . '">
										<a class="button alignright" href="' . esc_url( admin_url( 'admin.php?page=kkart-settings&tab=email&section=' . strtolower( $email_key ) ) ) . '">' . esc_html__( 'Manage', 'kkart' ) . '</a>
									</td>';
										break;
									default:
										do_action( 'kkart_email_setting_column_' . $key, $email );
										break;
								}
							}

							echo '</tr>';
						}
						?>
					</tbody>
				</table>
			</td>
		</tr>
		<?php
	}
}

return new KKART_Settings_Emails();
