/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { Fragment } from 'react';
import { InspectorControls } from '@wordpress/block-editor';
import { ServerSideRender } from '@wordpress/editor';
import PropTypes from 'prop-types';
import { PanelBody, ToggleControl, Placeholder } from '@wordpress/components';
import { Icon, list } from '@kkart/icons';
import ToggleButtonControl from '@kkart/editor-components/toggle-button-control';

const EmptyPlaceholder = () => (
	<Placeholder
		icon={ <Icon srcElement={ list } /> }
		label={ __(
			'Product Categories List',
			'kkart'
		) }
		className="kkart-block-product-categories"
	>
		{ __(
			"This block shows product categories for your store. To use it, you'll first need to create a product and assign it to a category.",
			'kkart'
		) }
	</Placeholder>
);

/**
 * Component displaying the categories as dropdown or list.
 *
 * @param {Object} props Incoming props for the component.
 * @param {Object} props.attributes Incoming block attributes.
 * @param {function(any):any} props.setAttributes Setter for block attributes.
 * @param {string} props.name Name for block.
 */
const ProductCategoriesBlock = ( { attributes, setAttributes, name } ) => {
	const getInspectorControls = () => {
		const {
			hasCount,
			hasImage,
			hasEmpty,
			isDropdown,
			isHierarchical,
		} = attributes;

		return (
			<InspectorControls key="inspector">
				<PanelBody
					title={ __(
						'List Settings',
						'kkart'
					) }
					initialOpen
				>
					<ToggleButtonControl
						label={ __(
							'Display style',
							'kkart'
						) }
						value={ isDropdown ? 'dropdown' : 'list' }
						options={ [
							{
								label: __(
									'List',
									'kkart'
								),
								value: 'list',
							},
							{
								label: __(
									'Dropdown',
									'kkart'
								),
								value: 'dropdown',
							},
						] }
						onChange={ ( value ) =>
							setAttributes( {
								isDropdown: value === 'dropdown',
							} )
						}
					/>
				</PanelBody>
				<PanelBody
					title={ __( 'Content', 'kkart' ) }
					initialOpen
				>
					<ToggleControl
						label={ __(
							'Show product count',
							'kkart'
						) }
						help={
							hasCount
								? __(
										'Product count is visible.',
										'kkart'
								  )
								: __(
										'Product count is hidden.',
										'kkart'
								  )
						}
						checked={ hasCount }
						onChange={ () =>
							setAttributes( { hasCount: ! hasCount } )
						}
					/>
					{ ! isDropdown && (
						<ToggleControl
							label={ __(
								'Show category images',
								'kkart'
							) }
							help={
								hasImage
									? __(
											'Category images are visible.',
											'kkart'
									  )
									: __(
											'Category images are hidden.',
											'kkart'
									  )
							}
							checked={ hasImage }
							onChange={ () =>
								setAttributes( { hasImage: ! hasImage } )
							}
						/>
					) }
					<ToggleControl
						label={ __(
							'Show hierarchy',
							'kkart'
						) }
						help={
							isHierarchical
								? __(
										'Hierarchy is visible.',
										'kkart'
								  )
								: __(
										'Hierarchy is hidden.',
										'kkart'
								  )
						}
						checked={ isHierarchical }
						onChange={ () =>
							setAttributes( {
								isHierarchical: ! isHierarchical,
							} )
						}
					/>
					<ToggleControl
						label={ __(
							'Show empty categories',
							'kkart'
						) }
						help={
							hasEmpty
								? __(
										'Empty categories are visible.',
										'kkart'
								  )
								: __(
										'Empty categories are hidden.',
										'kkart'
								  )
						}
						checked={ hasEmpty }
						onChange={ () =>
							setAttributes( { hasEmpty: ! hasEmpty } )
						}
					/>
				</PanelBody>
			</InspectorControls>
		);
	};

	return (
		<Fragment>
			{ getInspectorControls() }
			<ServerSideRender
				block={ name }
				attributes={ attributes }
				EmptyResponsePlaceholder={ EmptyPlaceholder }
			/>
		</Fragment>
	);
};

ProductCategoriesBlock.propTypes = {
	/**
	 * The attributes for this block
	 */
	attributes: PropTypes.object.isRequired,
	/**
	 * The register block name.
	 */
	name: PropTypes.string.isRequired,
	/**
	 * A callback to update attributes
	 */
	setAttributes: PropTypes.func.isRequired,
};

export default ProductCategoriesBlock;
