<?php
/**
 * Setup menus in WP admin.
 *
 * @package Kkart\Admin
 * @version 2.5.0
 */

defined( 'ABSPATH' ) || exit;

if ( class_exists( 'KKART_Admin_Menus', false ) ) {
	return new KKART_Admin_Menus();
}

/**
 * KKART_Admin_Menus Class.
 */
class KKART_Admin_Menus {

	/**
	 * Hook in tabs.
	 */
	public function __construct() {
		// Add menus.
		add_action( 'admin_menu', array( $this, 'admin_menu' ), 9 );
		add_action( 'admin_menu', array( $this, 'reports_menu' ), 20 );
		add_action( 'admin_menu', array( $this, 'settings_menu' ), 50 );
		add_action( 'admin_menu', array( $this, 'status_menu' ), 60 );

		if ( apply_filters( 'kkart_show_addons_page', true ) ) {
			add_action( 'admin_menu', array( $this, 'addons_menu' ), 70 );
		}

		add_action( 'admin_head', array( $this, 'menu_highlight' ) );
		add_action( 'admin_head', array( $this, 'menu_order_count' ) );
		add_filter( 'menu_order', array( $this, 'menu_order' ) );
		add_filter( 'custom_menu_order', array( $this, 'custom_menu_order' ) );
		add_filter( 'set-screen-option', array( $this, 'set_screen_option' ), 10, 3 );

		// Add endpoints custom URLs in Appearance > Menus > Pages.
		add_action( 'admin_head-nav-menus.php', array( $this, 'add_nav_menu_meta_boxes' ) );

		// Admin bar menus.
		if ( apply_filters( 'kkart_show_admin_bar_visit_store', true ) ) {
			add_action( 'admin_bar_menu', array( $this, 'admin_bar_menus' ), 31 );
		}

		// Handle saving settings earlier than load-{page} hook to avoid race conditions in conditional menus.
		add_action( 'wp_loaded', array( $this, 'save_settings' ) );
	}

	/**
	 * Add menu items.
	 */
	public function admin_menu() {
		global $menu;

		if ( current_user_can( 'edit_others_shop_orders' ) ) {
			$menu[] = array( '', 'read', 'separator-kkart', '', 'wp-menu-separator kkart' ); // WPCS: override ok.
		}

		add_menu_page( __( KKART_ORDERS_MENU, 'kkart' ), __( KKART_ORDERS_MENU, 'kkart' ), 'edit_others_shop_orders', 'kkart', null, 'dashicons-cart', '55.5' );

		add_submenu_page( 'edit.php?post_type=product', __( 'Attributes', 'kkart' ), __( 'Attributes', 'kkart' ), 'manage_product_terms', 'product_attributes', array( $this, 'attributes_page' ) );
	}

	/**
	 * Add menu item.
	 */
	public function reports_menu() {
		if ( current_user_can( 'edit_others_shop_orders' ) ) {
			add_submenu_page( 'kkart', __( 'Reports', 'kkart' ), __( 'Reports', 'kkart' ), 'view_kkart_reports', 'kkart-reports', array( $this, 'reports_page' ) );
		} else {
			add_menu_page( __( 'Sales reports', 'kkart' ), __( 'Sales reports', 'kkart' ), 'view_kkart_reports', 'kkart-reports', array( $this, 'reports_page' ), null, '55.6' );
		}
	}

	/**
	 * Add menu item.
	 */
	public function settings_menu() {
		$settings_page = add_submenu_page( 'options-general.php', __( 'Kkart settings', 'kkart' ), __( 'Kkart Settings', 'kkart' ), 'manage_kkart', 'kkart-settings', array( $this, 'settings_page' ), 1 );

		add_action( 'load-' . $settings_page, array( $this, 'settings_page_init' ) );
	}

	/**
	 * Loads gateways and shipping methods into memory for use within settings.
	 */
	public function settings_page_init() {
		KKART()->payment_gateways();
		KKART()->shipping();

		// Include settings pages.
		KKART_Admin_Settings::get_settings_pages();

		// Add any posted messages.
		if ( ! empty( $_GET['kkart_error'] ) ) { // WPCS: input var okay, CSRF ok.
			KKART_Admin_Settings::add_error( wp_kses_post( wp_unslash( $_GET['kkart_error'] ) ) ); // WPCS: input var okay, CSRF ok.
		}

		if ( ! empty( $_GET['kkart_message'] ) ) { // WPCS: input var okay, CSRF ok.
			KKART_Admin_Settings::add_message( wp_kses_post( wp_unslash( $_GET['kkart_message'] ) ) ); // WPCS: input var okay, CSRF ok.
		}

		do_action( 'kkart_settings_page_init' );
	}

	/**
	 * Handle saving of settings.
	 *
	 * @return void
	 */
	public function save_settings() {
		global $current_tab, $current_section;

		// We should only save on the settings page.
		if ( ! is_admin() || ! isset( $_GET['page'] ) || 'kkart-settings' !== $_GET['page'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return;
		}

		// Include settings pages.
		KKART_Admin_Settings::get_settings_pages();

		// Get current tab/section.
		$current_tab     = empty( $_GET['tab'] ) ? 'general' : sanitize_title( wp_unslash( $_GET['tab'] ) ); // WPCS: input var okay, CSRF ok.
		$current_section = empty( $_REQUEST['section'] ) ? '' : sanitize_title( wp_unslash( $_REQUEST['section'] ) ); // WPCS: input var okay, CSRF ok.

		// Save settings if data has been posted.
		if ( '' !== $current_section && apply_filters( "kkart_save_settings_{$current_tab}_{$current_section}", ! empty( $_POST['save'] ) ) ) { // WPCS: input var okay, CSRF ok.
			KKART_Admin_Settings::save();
		} elseif ( '' === $current_section && apply_filters( "kkart_save_settings_{$current_tab}", ! empty( $_POST['save'] ) ) ) { // WPCS: input var okay, CSRF ok.
			KKART_Admin_Settings::save();
		}
	}

	/**
	 * Add menu item.
	 */
	public function status_menu() {
		add_submenu_page( 'kkart', __( 'Kkart status', 'kkart' ), __( 'Status', 'kkart' ), 'manage_kkart', 'kkart-status', array( $this, 'status_page' ) );
	}

	/**
	 * Addons menu item.
	 */
	public function addons_menu() {
		return;
		$count_html = KKART_Helper_Updater::get_updates_count_html();
		/* translators: %s: extensions count */
		$menu_title = sprintf( __( 'Extensions %s', 'kkart' ), $count_html );
		add_submenu_page( 'kkart', __( 'Kkart extensions', 'kkart' ), $menu_title, 'manage_kkart', 'kkart-addons', array( $this, 'addons_page' ) );
	}

	/**
	 * Highlights the correct top level admin menu item for post type add screens.
	 */
	public function menu_highlight() {
		global $parent_file, $submenu_file, $post_type;

		switch ( $post_type ) {
			case 'shop_order':
			case 'shop_coupon':
				$parent_file = 'kkart'; // WPCS: override ok.
				break;
			case 'product':
				$screen = get_current_screen();
				if ( $screen && taxonomy_is_product_attribute( $screen->taxonomy ) ) {
					$submenu_file = 'product_attributes'; // WPCS: override ok.
					$parent_file  = 'edit.php?post_type=product'; // WPCS: override ok.
				}
				break;
		}
	}

	/**
	 * Adds the order processing count to the menu.
	 */
	public function menu_order_count() {
		global $submenu;

		if ( isset( $submenu['kkart'] ) ) {
			// Remove 'Kkart' sub menu item.
			unset( $submenu['kkart'][0] );

			// Add count if user has access.
			if ( apply_filters( 'kkart_include_processing_order_count_in_menu', true ) && current_user_can( 'edit_others_shop_orders' ) ) {
				$order_count = apply_filters( 'kkart_menu_order_count', kkart_processing_order_count() );

				if ( $order_count ) {
					foreach ( $submenu['kkart'] as $key => $menu_item ) {
						if ( 0 === strpos( $menu_item[0], _x( 'All Orders', 'Admin menu name', 'kkart' ) ) ) {
							$submenu['kkart'][ $key ][0] .= ' <span class="awaiting-mod update-plugins count-' . esc_attr( $order_count ) . '"><span class="processing-count">' . number_format_i18n( $order_count ) . '</span></span>'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
							break;
						}
					}
				}
			}
		}
	}

	/**
	 * Reorder the KKART menu items in admin.
	 *
	 * @param int $menu_order Menu order.
	 * @return array
	 */
	public function menu_order( $menu_order ) {
		// Initialize our custom order array.
		$kkart_menu_order = array();

		// Get the index of our custom separator.
		$kkart_separator = array_search( 'separator-kkart', $menu_order, true );

		// Get index of product menu.
		$kkart_product = array_search( 'edit.php?post_type=product', $menu_order, true );

		// Loop through menu order and do some rearranging.
		foreach ( $menu_order as $index => $item ) {

			if ( 'kkart' === $item ) {
				$kkart_menu_order[] = 'separator-kkart';
				$kkart_menu_order[] = $item;
				$kkart_menu_order[] = 'edit.php?post_type=product';
				unset( $menu_order[ $kkart_separator ] );
				unset( $menu_order[ $kkart_product ] );
			} elseif ( ! in_array( $item, array( 'separator-kkart' ), true ) ) {
				$kkart_menu_order[] = $item;
			}
		}

		// Return order.
		return $kkart_menu_order;
	}

	/**
	 * Custom menu order.
	 *
	 * @param bool $enabled Whether custom menu ordering is already enabled.
	 * @return bool
	 */
	public function custom_menu_order( $enabled ) {
		return $enabled || current_user_can( 'edit_others_shop_orders' );
	}

	/**
	 * Validate screen options on update.
	 *
	 * @param bool|int $status Screen option value. Default false to skip.
	 * @param string   $option The option name.
	 * @param int      $value  The number of rows to use.
	 */
	public function set_screen_option( $status, $option, $value ) {
		if ( in_array( $option, array( 'kkart_keys_per_page', 'kkart_webhooks_per_page' ), true ) ) {
			return $value;
		}

		return $status;
	}

	/**
	 * Init the reports page.
	 */
	public function reports_page() {
		KKART_Admin_Reports::output();
	}

	/**
	 * Init the settings page.
	 */
	public function settings_page() {
		KKART_Admin_Settings::output();
	}

	/**
	 * Init the attributes page.
	 */
	public function attributes_page() {
		KKART_Admin_Attributes::output();
	}

	/**
	 * Init the status page.
	 */
	public function status_page() {
		KKART_Admin_Status::output();
	}

	/**
	 * Init the addons page.
	 */
	public function addons_page() {
		KKART_Admin_Addons::output();
	}

	/**
	 * Add custom nav meta box.
	 *
	 * Adapted from http://www.johnmorrisonline.com/how-to-add-a-fully-functional-custom-meta-box-to-wordpress-navigation-menus/.
	 */
	public function add_nav_menu_meta_boxes() {
		add_meta_box( 'kkart_endpoints_nav_link', __( 'Kkart endpoints', 'kkart' ), array( $this, 'nav_menu_links' ), 'nav-menus', 'side', 'low' );
	}

	/**
	 * Output menu links.
	 */
	public function nav_menu_links() {
		// Get items from account menu.
		$endpoints = kkart_get_account_menu_items();

		// Remove dashboard item.
		if ( isset( $endpoints['dashboard'] ) ) {
			unset( $endpoints['dashboard'] );
		}

		// Include missing lost password.
		$endpoints['lost-password'] = __( 'Lost password', 'kkart' );

		$endpoints = apply_filters( 'kkart_custom_nav_menu_items', $endpoints );

		?>
		<div id="posttype-kkart-endpoints" class="posttypediv">
			<div id="tabs-panel-kkart-endpoints" class="tabs-panel tabs-panel-active">
				<ul id="kkart-endpoints-checklist" class="categorychecklist form-no-clear">
					<?php
					$i = -1;
					foreach ( $endpoints as $key => $value ) :
						?>
						<li>
							<label class="menu-item-title">
								<input type="checkbox" class="menu-item-checkbox" name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-object-id]" value="<?php echo esc_attr( $i ); ?>" /> <?php echo esc_html( $value ); ?>
							</label>
							<input type="hidden" class="menu-item-type" name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-type]" value="custom" />
							<input type="hidden" class="menu-item-title" name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-title]" value="<?php echo esc_attr( $value ); ?>" />
							<input type="hidden" class="menu-item-url" name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-url]" value="<?php echo esc_url( kkart_get_account_endpoint_url( $key ) ); ?>" />
							<input type="hidden" class="menu-item-classes" name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-classes]" />
						</li>
						<?php
						$i--;
					endforeach;
					?>
				</ul>
			</div>
			<p class="button-controls">
				<span class="list-controls">
					<a href="<?php echo esc_url( admin_url( 'nav-menus.php?page-tab=all&selectall=1#posttype-kkart-endpoints' ) ); ?>" class="select-all"><?php esc_html_e( 'Select all', 'kkart' ); ?></a>
				</span>
				<span class="add-to-menu">
					<button type="submit" class="button-secondary submit-add-to-menu right" value="<?php esc_attr_e( 'Add to menu', 'kkart' ); ?>" name="add-post-type-menu-item" id="submit-posttype-kkart-endpoints"><?php esc_html_e( 'Add to menu', 'kkart' ); ?></button>
					<span class="spinner"></span>
				</span>
			</p>
		</div>
		<?php
	}

	/**
	 * Add the "Visit Store" link in admin bar main menu.
	 *
	 * @since 2.4.0
	 * @param WP_Admin_Bar $wp_admin_bar Admin bar instance.
	 */
	public function admin_bar_menus( $wp_admin_bar ) {
		if ( ! is_admin() || ! is_admin_bar_showing() ) {
			return;
		}

		// Show only when the user is a member of this site, or they're a super admin.
		if ( ! is_user_member_of_blog() && ! is_super_admin() ) {
			return;
		}

		// Don't display when shop page is the same of the page on front.
		if ( intval( get_option( 'page_on_front' ) ) === kkart_get_page_id( 'shop' ) ) {
			return;
		}

		// Add an option to visit the store.
		$wp_admin_bar->add_node(
			array(
				'parent' => 'site-name',
				'id'     => 'view-store',
				'title'  => __( 'Visit Store', 'kkart' ),
				'href'   => kkart_get_page_permalink( 'shop' ),
			)
		);
	}
}

return new KKART_Admin_Menus();
