import path from 'path';
import type { Plugin } from 'vite';
import { findFilesRecursively } from '../utils';

type Options = {
    root: string;
    pluginEntryFile: string;
};

/**
 * @sw-package framework
 * @private - Only to be used by shopware directly
 *
 * Creates a Vite plugin that automatically imports all *.override.vue files
 * into the bundle entry file.
 */
export default function viteOverridePlugin(options: Options): Plugin {
    const { root, pluginEntryFile } = options;
    const pattern = /\.override\.vue$/;

    let overrideFiles = [] as string[];

    return {
        name: 'shopware-vite-plugin-override-component',

        configResolved() {
            // Find all override files
            overrideFiles = findFilesRecursively(root, pattern);
        },

        transform(code, id) {
            // Only transform the entry file
            if (id !== pluginEntryFile || overrideFiles.length === 0) {
                return null;
            }

            const componentNames = [] as string[];

            // Generate import statements for each override file
            const imports = overrideFiles
                .map((file) => {
                    // Convert absolute path to relative path from entry file
                    const relativePath = path.relative(path.dirname(id), file);

                    // Ensure proper path separators
                    const importPath = relativePath.split(path.sep).join('/');

                    // Get component name from file name and prefix with _ to avoid name conflicts
                    const componentName = `_${path.basename(file, '.override.vue').replace(/[-_/\\:*?"<>|]/g, '')}`;
                    componentNames.push(`${componentName}`);

                    return `import ${componentName} from './${importPath}';`;
                })
                .join('\n');

            const registers = componentNames
                .map((name) => {
                    return `Shopware.Component.registerOverrideComponent(${name});`;
                })
                .join('\n');

            // Add imports at the beginning of the file
            return {
                code: `
                /* Start: auto generated by Shopware */\n
                ${imports}\n${registers}\n
                /* End: auto generated by Shopware */\n\n
                ${code}`,
                map: null,
            };
        },
    };
}
