<?php

/*
 * This file is part of the PrestaSitemapBundle package.
 *
 * (c) PrestaConcept <https://prestaconcept.net>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Presta\SitemapBundle\Command;

use Presta\SitemapBundle\Service\DumperInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\RouterInterface;

/**
 * Command to dump the sitemaps to provided directory
 */
class DumpSitemapsCommand extends Command
{
    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var DumperInterface
     */
    private $dumper;

    /**
     * @var string
     */
    private $defaultTarget;

    public function __construct(RouterInterface $router, DumperInterface $dumper, string $defaultTarget)
    {
        $this->router = $router;
        $this->dumper = $dumper;
        $this->defaultTarget = $defaultTarget;

        parent::__construct();
    }

    public static function getDefaultName(): ?string
    {
        return 'presta:sitemaps:dump';
    }

    /**
     * @inheritdoc
     */
    protected function configure(): void
    {
        $this
            ->setDescription('Dumps sitemaps to given location')
            ->addOption(
                'section',
                null,
                InputOption::VALUE_REQUIRED,
                'Name of sitemap section to dump, all sections are dumped by default'
            )
            ->addOption(
                'base-url',
                null,
                InputOption::VALUE_REQUIRED,
                'Base url to use for absolute urls. Good example - http://acme.com/, bad example - acme.com.' .
                ' Defaults to Symfony\'s default routing context'
            )
            ->addOption(
                'gzip',
                null,
                InputOption::VALUE_NONE,
                'Gzip sitemap'
            )
            ->addArgument(
                'target',
                InputArgument::OPTIONAL,
                'Location where to dump sitemaps. Generated urls will not be related to this folder.',
                $this->defaultTarget
            );
    }

    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        /** @var string $targetDir */
        $targetDir = $input->getArgument('target');
        $targetDir = rtrim($targetDir, '/');

        /** @var string|null $baseUrl */
        $baseUrl = $input->getOption('base-url');
        if ($baseUrl) {
            $baseUrl = rtrim($baseUrl, '/') . '/';

            //sanity check
            if (!parse_url($baseUrl, PHP_URL_HOST)) {
                throw new \InvalidArgumentException(
                    'Invalid base url. Use fully qualified base url, e.g. http://acme.com/',
                    -1
                );
            }

            // Set Router's host used for generating URLs from configuration param
            // There is no other way to manage domain in CLI
            $request = Request::create($baseUrl);
            $this->router->getContext()->fromRequest($request);
        } else {
            $baseUrl = $this->getBaseUrl();
        }

        /** @var string|null $section */
        $section = $input->getOption('section');
        if ($section) {
            $output->writeln(
                sprintf(
                    "Dumping sitemaps section <comment>%s</comment> into <comment>%s</comment> directory",
                    $section,
                    $targetDir
                )
            );
        } else {
            $output->writeln(
                sprintf(
                    "Dumping <comment>all sections</comment> of sitemaps into <comment>%s</comment> directory",
                    $targetDir
                )
            );
        }

        $options = [
            'gzip' => (bool)$input->getOption('gzip'),
        ];
        $filenames = $this->dumper->dump($targetDir, $baseUrl, $section, $options);

        if (!is_array($filenames)) {
            $output->writeln(
                "<error>No URLs were added to sitemap by EventListeners</error>" .
                " - this may happen when provided section is invalid"
            );

            return 1;
        }

        $output->writeln("<info>Created/Updated the following sitemap files:</info>");
        foreach ($filenames as $filename) {
            $output->writeln("    <comment>$filename</comment>");
        }

        return 0;
    }

    private function getBaseUrl(): string
    {
        $context = $this->router->getContext();

        if ('' === $host = $context->getHost()) {
            throw new \RuntimeException(
                'Router host must be configured to be able to dump the sitemap, please see documentation.'
            );
        }

        $scheme = $context->getScheme();
        $port = '';

        if ('http' === $scheme && 80 != $context->getHttpPort()) {
            $port = ':' . $context->getHttpPort();
        } elseif ('https' === $scheme && 443 != $context->getHttpsPort()) {
            $port = ':' . $context->getHttpsPort();
        }

        return rtrim($scheme . '://' . $host . $port, '/') . '/';
    }
}
