<?php
/*
 * Copyright 2014 Osclass
 * Copyright 2025 Osclass by OsclassPoint.com
 *
 * Osclass maintained & developed by OsclassPoint.com
 * You may not use this file except in compliance with the License.
 * You may download copy of Osclass at
 *
 *     https://osclass-classifieds.com/download
 *
 * Do not edit or add to this file if you wish to upgrade Osclass to newer
 * versions in the future. Software is distributed on an "AS IS" basis, without
 * warranties or conditions of any kind, either express or implied. Do not remove
 * this NOTICE section as it contains license information and copyrights.
 */


/**
 * Gets category array from view and cache it if not exists
 *
 * @return array
 */
function osc_get_category_row($id, $cache = true) {
  $id = (int)$id;
  
  if($id <= 0) {
    return false;
  }

  if($cache === true && View::newInstance()->_exists('category_' . $id)) {
    return View::newInstance()->_get('category_' . $id);
  }
  
  // If there is more categories in DB, it's not effective way
  if(OPTIMIZE_CATEGORIES === true && osc_count_categories_all() < OPTIMIZE_CATEGORIES_LIMIT) {
    $categories = osc_get_categories_flat();
    
    // Search in session array with flat categories
    if(is_array($categories) && isset($categories[$id])) {
      View::newInstance()->_exportVariableToView('category_' . $id, $categories[$id]);
      return $categories[$id];
    }
  }

  // Search in database
  $category = Category::newInstance()->findByPrimaryKey($id);
  View::newInstance()->_exportVariableToView('category_' . $id, $category);
  
  return $category;
}


/**
 * Gets category array from view and cache it if not exists by slug
 *
 * @return array
 */
function osc_get_category_row_by_slug($slug) {
  if(View::newInstance()->_exists('category_' . $slug)) {
    return View::newInstance()->_get('category_' . $slug);
  }
  
  // If there is more categories in DB, it's not effective way
  if(OPTIMIZE_CATEGORIES === true && osc_count_categories_all() < OPTIMIZE_CATEGORIES_LIMIT) {
    $id = 0;
    $slug_map = osc_get_categories_slug();
      
    if(is_array($slug_map) && isset($slug_map[$id])) {
      $id = $slug_map[$id];
    }

    if($id > 0) {
      $category = osc_get_category_row($id);
      View::newInstance()->_exportVariableToView('category_' . $slug, $category);

      return $category;
    }
  }
  
  $category = Category::newInstance()->findBySlug($slug);
  View::newInstance()->_exportVariableToView('category_' . $slug, $category);

  return $category;
}

/**
 * Count all categories
 *
 * @return array
 */
function osc_count_categories_all() {
  if(View::newInstance()->_exists('count_categories')) {
    return View::newInstance()->_get('count_categories');
  }

  $count = Category::newInstance()->count();
  View::newInstance()->_exportVariableToView('count_categories', (int)$count);
  
  return (int)$count;
}

/**
* Helper Categories
* @package Osclass
* @subpackage Helpers
* @author Osclass
*/

/**
 * Gets current category
 *
 * @return array
 */
function osc_category() {
  $category = null;
  if(View::newInstance()->_exists('subcategories')) {
    $category = View::newInstance()->_current('subcategories');
  } elseif(View::newInstance()->_exists('categories')) {
    $category = View::newInstance()->_current('categories');
  } elseif(View::newInstance()->_exists('category')) {
    $category = View::newInstance()->_get('category');
  }

  return $category;
}


/**
 * Low level function: Gets the list of categories as a tree
 *
 * @return mixed|string <array>
 * @throws \Exception
 */
function osc_get_categories() {
  if(!View::newInstance()->_exists('categories')) {
    $tree = Category::newInstance()->toTree();
    osc_export_categories($tree);
    
    return $tree;
  }
  
  return View::newInstance()->_get('categories');
}


/**
 * Gets the list of categories as a flat structure
 *
 * @return array
 */
function osc_get_categories_flat($enabled = false) {
  if(!View::newInstance()->_exists('categories_flat' . (int)$enabled)) {
    $output = array();
    
    if($enabled === true) {
      $categories = Category::newInstance()->listEnabled();
    } else {
      $categories = Category::newInstance()->listAll();
    }
    
    // Create array in way that category ID is also key in array
    if(is_array($categories) && count($categories) > 0) {
      foreach($categories as $cat_row) {
        if(isset($cat_row['pk_i_id'])) {
          $output[$cat_row['pk_i_id']] = $cat_row;
        }
      }
    }

    View::newInstance()->_exportVariableToView('categories_flat' . (int)$enabled, $output);
    return $output;
  }
  
  return View::newInstance()->_get('categories_flat' . (int)$enabled);
}



/**
 * Gets the list of categories as a flat structure
 *
 * @return array
 */
function osc_get_categories_slug() {
  if(!View::newInstance()->_exists('categories_slug')) {
    $output = array();
    $categories = osc_get_categories_flat();
    
    // Create array in way that category ID is also key in array
    if(is_array($categories) && count($categories) > 0) {
      foreach($categories as $cat_row) {
        if(isset($cat_row['pk_i_id'])) {
          $output[$cat_row['s_slug']] = $cat_row['pk_i_id'];
        }
      }
    }

    View::newInstance()->_exportVariableToView('categories_slug', $output);
    return $output;
  }
  
  return View::newInstance()->_get('categories_slug');
}


/**
 * Low level function: Gets the value of the category attribute
 *
 * @param    $field
 * @param string $locale
 *
 * @return string <array>
 */
function osc_category_field($field, $locale = '') {
  return osc_field(osc_category(), $field, '');
}

/**
 * Gets the number of categories
 *
 * @return int
 */
function osc_priv_count_categories() {
  return View::newInstance()->_count('categories');
}

/**
 * Gets the number of subcategories
 *
 * @return int
 */
function osc_priv_count_subcategories() {
  return View::newInstance()->_count('subcategories');
}


/**
 * Gets the total of categories. If categories are not loaded, this function will load them.
 *
 * @return int
 * @throws \Exception
 */
function osc_count_categories() {
  if(!View::newInstance()->_exists('categories')) {
    View::newInstance()->_exportVariableToView('categories', Category::newInstance()->toTree());
  }
  return osc_priv_count_categories();
}


/**
 * Let you know if there are more categories in the list. If categories are not loaded, this function will load them.
 *
 * @return boolean
 * @throws \Exception
 */
function osc_has_categories() {
  if(!View::newInstance()->_exists('categories')) {
    View::newInstance()->_exportVariableToView('categories', Category::newInstance()->toTree());
  }
  
  return View::newInstance()->_next('categories');
}

/**
 * Gets the total of subcategories for the current category. If subcategories are not loaded, this function will load them and
 * it will prepare the the pointer to the first element
 *
 * @return int
 */
function osc_count_subcategories() {
  $category = View::newInstance()->_current('categories');
  
  if($category == '') {
    return - 1;
  }
  
  if(!isset($category['categories'])) {
    return 0;
  }
  
  if(!is_array($category['categories'])) {
    return 0;
  }
  
  if(count($category['categories']) == 0) {
    return 0;
  }
  
  if(!View::newInstance()->_exists('subcategories')) {
    View::newInstance()->_exportVariableToView('subcategories', $category['categories']);
  }
  
  return osc_priv_count_subcategories();
}

/**
 * Let you know if there are more subcategories for the current category in the list. If subcategories are not loaded, this
 * function will load them and it will prepare the pointer to the first element
 *
 * @return boolean
 */
function osc_has_subcategories() {
  $category = View::newInstance()->_current('categories');
  
  if($category == '') {
    return - 1;
  }
  
  if(!isset($category['categories'])) {
    return false;
  }

  if(!View::newInstance()->_exists('subcategories')) {
    View::newInstance()->_exportVariableToView('subcategories', $category['categories']);
  }
  
  $ret = View::newInstance()->_next('subcategories');
  
  //we have to delete for next iteration
  if(!$ret) {
    View::newInstance()->_erase('subcategories');
  }
  
  return $ret;
}

/**
 * Gets the name of the current category
 *
 * @param string $locale
 * @return string
 */
function osc_category_name($locale = '') {
  if($locale == '') {
    $locale = osc_current_user_locale();
  }
  
  return osc_category_field('s_name', $locale);
}

/**
 * Gets the description of the current category
 *
 * @param string $locale
 * @return string
 */
function osc_category_description($locale = '') {
  if($locale == '') {
    $locale = osc_current_user_locale();
  }
  
  return osc_category_field('s_description', $locale);
}

/**
 * Gets the id of the current category
 *
 * @param string $locale
 * @return string
 */
function osc_category_id($locale = '') {
  if($locale == '') {
    $locale = osc_current_user_locale();
  }
  
  return osc_category_field('pk_i_id', $locale);
}

/**
 * Gets the slug of the current category. WARNING: This slug could NOT be used as a valid W3C HTML tag attribute as it could have other characters besides [A-Za-z0-9-_] We only did a urlencode to the variable
 *
 * @param string $locale
 * @return string
 */
function osc_category_slug($locale = '') {
  if($locale == '') {
    $locale = osc_current_user_locale();
  }
  
  return osc_category_field('s_slug', $locale);
}

/**
 * Returns if the category has the prices enabled or not
 *
 * @return boolean
 */
function osc_category_price_enabled() {
  return (boolean)osc_category_field('b_price_enabled');
}

/**
 * Returns category's parent id
 *
 * @return int
 */
function osc_category_parent_id() {
  return osc_category_field('fk_i_parent_id');
}


function osc_category_color() {
  return osc_category_field('s_color');
}


function osc_category_icon() {
  return osc_category_field('s_icon');
}

/**
 * Gets the total items related with the current category
 *
 * @return int
 */
function osc_category_total_items() {
  return osc_category_field('i_num_items');
  //$category = osc_category();
  //return CategoryStats::newInstance()->getNumItems($category);
}

/**
 * Reset the pointer of the array to the first category
 *
 * @return void
 */
function osc_goto_first_category() {
  View::newInstance()->_reset('categories');
}


/**
 * Gets list of non-empty categories
 *
 * @return array
 * @throws \Exception
 */
function osc_get_non_empty_categories() {
  $aCategories = Category::newInstance()->toTree(false);
  View::newInstance()->_exportVariableToView('categories', $aCategories);
  return  View::newInstance()->_get('categories');
}


/**
 * Prints category select
 *
 * @param string $name
 * @param null   $category
 * @param null   $default_str
 *
 * @return void
 * @throws \Exception
 */
function osc_categories_select($name = 'sCategory', $category = null, $default_str = null) {
  if($default_str == null) {
    $default_str = __('Select a category');
  }
  CategoryForm::category_select(Category::newInstance()->toTree(), $category, $default_str, $name);
}


/**
 * Get th category by id or slug
 *
 * @since 3.0
 *
 * @param $by   //two possibilities: slug or id
 * @param $what //the id or slug category we're looking for
 *
 * @return array|bool
 * @throws \Exception
 */
function osc_get_category($by, $what) {
  if(!in_array($by, array('slug', 'id'))) {
    return false;
  }

  switch ($by) {
    case 'slug':
      return osc_get_category_row_by_slug($what);
    break;
    case 'id':
      return osc_get_category_row((int)$what);
    break;
  }
}


/**
 * @return bool|int
 */
function osc_category_move_to_children() {
  $category = View::newInstance()->_current('categories');
  if($category == '') {
    return - 1;
  }
  if(!isset($category['categories'])) {
    return false;
  }

  if(View::newInstance()->_exists('categoryTrail')) {
    $catTrail = View::newInstance()->_get('categoryTrail');
  } else {
    $catTrail = array();
  }
  $catTrail[] = View::newInstance()->_key('categories');
  View::newInstance()->_exportVariableToView('categoryTrail', $catTrail);
  View::newInstance()->_exportVariableToView('categories', $category['categories']);
  View::newInstance()->_reset('categories');
}


/**
 * @return bool|int
 * @throws \Exception
 */
function osc_category_move_to_parent() {
  $category = View::newInstance()->_get('categories');
  $category = end($category);

  if($category == '') {
    return - 1;
  }
  if(!isset($category['fk_i_parent_id'])) {
    return false;
  }

  $keys = View::newInstance()->_get('categoryTrail');
  $position = array_pop($keys);
  View::newInstance()->_exportVariableToView('categoryTrail', $keys);
  if(!View::newInstance()->_exists('categories_tree')) {
    View::newInstance()->_exportVariableToView('categories_tree', Category::newInstance()->toTree());
  }
  $scats['categories'] = Category::newInstance()->toTree();
  if(count($keys)>0) {
    foreach($keys as $k) {
      $scats = $scats['categories'][$k];
    }
  }

  $scats = $scats['categories'];

  View::newInstance()->_erase('categories');
  View::newInstance()->_erase('subcategories');
  View::newInstance()->_exportVariableToView('categories', $scats);
  View::newInstance()->_seek('categories', $position);
}

/**
 * Gets the total of subcategories for the current category. If subcategories are not loaded, this function will load them and
 * it will prepare the the pointer to the first element
 *
 * @return int
 */
function osc_count_subcategories2() {
  $category = View::newInstance()->_current('categories');
  if($category == '') {
    return - 1;
  }
  if(!isset($category['categories'])) {
    return 0;
  }
  if(!is_array($category['categories'])) {
    return 0;
  }
  return count($category['categories']);
}


/**
 * @param null $categories
 *
 * @throws \Exception
 */
function osc_export_categories($categories = null) {
  if($categories==null) {
    $categories = Category::newInstance()->toTree();
  }
  
  View::newInstance()->_exportVariableToView('categories', $categories);
  View::newInstance()->_exportVariableToView('categories_tree', $categories);
}



/*
  GET USER RELATED CATEGORY
*/

/**
 * Gets list of categories
 *
 * @return array
 */
function osc_get_user_item_categories($user_id = null) {
  $user_id = (int)($user_id === null ? osc_logged_user_id() : $user_id);
  
  if(!View::newInstance()->_exists('user_item_categories')) {
    $categories = Category::newInstance()->listUser($user_id);
    View::newInstance()->_exportVariableToView('user_item_categories', $categories);
    return $categories;
  }
  
  return View::newInstance()->_get('user_item_categories');
}