<?php

/**
 * @file controllers/grid/settings/category/CategoryCategoryGridHandler.php
 *
 * Copyright (c) 2014-2025 Simon Fraser University
 * Copyright (c) 2003-2025 John Willinsky
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class CategoryCategoryGridHandler
 *
 * @ingroup controllers_grid_settings_category
 *
 * @brief Handle operations for category management operations.
 */

namespace PKP\controllers\grid\settings\category;

use APP\core\Request;
use APP\facades\Repo;
use PKP\controllers\grid\CategoryGridHandler;
use PKP\controllers\grid\DataObjectGridCellProvider;
use PKP\controllers\grid\feature\OrderCategoryGridItemsFeature;
use PKP\controllers\grid\GridColumn;
use PKP\controllers\grid\settings\category\form\CategoryForm;
use PKP\core\JSONMessage;
use PKP\core\PKPRequest;
use PKP\facades\Locale;
use PKP\file\ContextFileManager;
use PKP\file\TemporaryFileManager;
use PKP\linkAction\LinkAction;
use PKP\linkAction\request\AjaxModal;
use PKP\security\authorization\CanAccessSettingsPolicy;
use PKP\security\authorization\ContextAccessPolicy;
use PKP\security\Role;

class CategoryCategoryGridHandler extends CategoryGridHandler
{
    public $_contextId;

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        $this->addRoleAssignment(
            [Role::ROLE_ID_MANAGER, Role::ROLE_ID_SITE_ADMIN],
            [
                'fetchGrid',
                'fetchCategory',
                'fetchRow',
                'addCategory',
                'editCategory',
                'updateCategory',
                'deleteCategory',
                'uploadImage',
                'deleteImage',
                'saveSequence',
            ]
        );
    }

    //
    // Overridden methods from PKPHandler.
    //
    /**
     * @copydoc PKPHandler::authorize()
     */
    public function authorize($request, &$args, $roleAssignments)
    {
        $this->addPolicy(new ContextAccessPolicy($request, $roleAssignments));
        $this->addPolicy(new CanAccessSettingsPolicy());
        return parent::authorize($request, $args, $roleAssignments);
    }


    /**
     * @copydoc CategoryGridHandler::initialize()
     *
     * @param null|mixed $args
     */
    public function initialize($request, $args = null)
    {
        parent::initialize($request, $args);

        $context = $request->getContext();
        $this->_contextId = $context->getId();

        // Set the grid title.
        $this->setTitle('grid.category.categories');

        // Add grid-level actions.
        $router = $request->getRouter();
        $this->addAction(
            new LinkAction(
                'addCategory',
                new AjaxModal(
                    $router->url($request, null, null, 'addCategory'),
                    __('grid.category.add'),
                ),
                __('grid.category.add'),
                'add_category'
            )
        );

        // Add grid columns.
        $cellProvider = new DataObjectGridCellProvider();
        $cellProvider->setLocale(Locale::getLocale());

        $this->addColumn(
            new GridColumn(
                'title',
                'grid.category.name',
                null,
                null,
                $cellProvider
            )
        );
    }

    /**
     * @copydoc GridHandler::loadData
     */
    public function loadData($request, $filter)
    {
        // For top-level rows, only list categories without parents.
        return Repo::category()->getCollector()
            ->filterByContextIds([$this->_getContextId()])
            ->filterByParentIds([null])
            ->getMany()
            ->toArray();
    }

    /**
     * @copydoc GridHandler::initFeatures()
     */
    public function initFeatures($request, $args)
    {
        return array_merge(
            parent::initFeatures($request, $args),
            [new OrderCategoryGridItemsFeature(OrderCategoryGridItemsFeature::ORDER_CATEGORY_GRID_CATEGORIES_AND_ROWS, true, $this)]
        );
    }

    /**
     * @copydoc CategoryGridHandler::getDataElementInCategorySequence()
     */
    public function getDataElementInCategorySequence($categoryId, &$category)
    {
        return $category->getSequence();
    }

    /**
     * @copydoc CategoryGridHandler::setDataElementInCategorySequence()
     */
    public function setDataElementInCategorySequence($parentCategoryId, &$category, $newSequence)
    {
        $category->setSequence($newSequence);
        Repo::category()->edit($category, []);
    }

    /**
     * @copydoc GridHandler::getDataElementSequence()
     */
    public function getDataElementSequence($gridDataElement)
    {
        return $gridDataElement->getSequence();
    }

    /**
     * @copydoc GridHandler::setDataElementSequence()
     */
    public function setDataElementSequence($request, $categoryId, $category, $newSequence)
    {
        $category->setSequence($newSequence);
        Repo::category()->edit($category, []);
    }

    /**
     * @copydoc CategoryGridHandler::getCategoryRowIdParameterName()
     */
    public function getCategoryRowIdParameterName()
    {
        return 'parentCategoryId';
    }

    /**
     * @copydoc GridHandler::getRowInstance()
     */
    public function getRowInstance()
    {
        return new \PKP\controllers\grid\settings\category\CategoryGridRow();
    }

    /**
     * @copydoc CategoryGridHandler::getCategoryRowInstance()
     */
    public function getCategoryRowInstance()
    {
        return new \PKP\controllers\grid\settings\category\CategoryGridCategoryRow();
    }

    /**
     * @copydoc CategoryGridHandler::loadCategoryData()
     *
     * @param null|mixed $filter
     */
    public function loadCategoryData($request, &$category, $filter = null)
    {
        $categoryId = $category->getId();
        return Repo::category()->getCollector()
            ->filterByContextIds([$this->_getContextId()])
            ->filterByParentIds([$categoryId])
            ->getMany()
            ->toArray();
    }

    /**
     * Handle the add category operation.
     *
     * @param array $args
     * @param PKPRequest $request
     */
    public function addCategory($args, $request)
    {
        return $this->editCategory($args, $request);
    }

    /**
     * Handle the edit category operation.
     *
     * @param array $args
     * @param PKPRequest $request
     *
     * @return JSONMessage JSON object
     */
    public function editCategory($args, $request)
    {
        $categoryForm = $this->_getCategoryForm($request);

        $categoryForm->initData();

        return new JSONMessage(true, $categoryForm->fetch($request));
    }

    /**
     * Update category data in database and grid.
     *
     * @param array $args
     * @param PKPRequest $request
     *
     * @return JSONMessage JSON object
     */
    public function updateCategory($args, $request)
    {
        $categoryForm = $this->_getCategoryForm($request);

        $categoryForm->readInputData();
        if ($categoryForm->validate()) {
            $categoryForm->execute();
            return \PKP\db\DAO::getDataChangedEvent();
        } else {
            return new JSONMessage(true, $categoryForm->fetch($request));
        }
    }

    /**
     * Delete a category
     *
     * @param array $args
     * @param PKPRequest $request
     *
     * @return JSONMessage JSON object
     */
    public function deleteCategory($args, $request)
    {
        if (!$request->checkCSRF()) {
            return new JSONMessage(false);
        }

        $context = $request->getContext();
        $category = Repo::category()->get((int) $request->getUserVar('categoryId'));
        if ($category && $category->getContextId() == $context->getId()) {
            Repo::category()->delete($category);
        }

        // FIXME delete dependent objects?

        return \PKP\db\DAO::getDataChangedEvent();
    }

    /**
     * Handle file uploads for cover images for Categories.
     *
     * @param PKPRequest $request
     * @param array $args
     *
     * @return JSONMessage JSON object
     */
    public function uploadImage($args, $request)
    {
        $user = $request->getUser();

        $temporaryFileManager = new TemporaryFileManager();
        $temporaryFile = $temporaryFileManager->handleUpload('uploadedFile', $user->getId());
        if ($temporaryFile) {
            $json = new JSONMessage(true);
            $json->setAdditionalAttributes([
                'temporaryFileId' => $temporaryFile->getId()
            ]);
            return $json;
        } else {
            return new JSONMessage(false, __('common.uploadFailed'));
        }
    }

    /**
     * Handle file deletion for cover images for Categories.
     *
     * @param array $args
     *    `name` string Filename of the cover image to be deleted.
     *    `thumbnailName` string Filename of the cover image thumbnail to be deleted.
     *    `categoryId` int ID of the category this cover image is attached to.
     *
     * @throws \Exception
     */
    public function deleteImage(array $args, PKPRequest $request): JSONMessage
    {
        if (empty($args['name']) || empty($args['thumbnailName']) || empty($args['categoryId'])) {
            throw new \Exception('File name, thumbnail name, or category ID not provided');
        }

        // Check that the category exists and is in the current context.
        $category = Repo::category()->get((int) $args['categoryId']);
        $context = $request->getContext();
        if (!$category || ($category->getContextId() != $context->getId())) {
            return new JSONMessage(false, __('manager.categories.form.removeCoverImageOnDifferentContextNowAllowed'));
        }

        // Remove the image from the category.
        $category->setImage(null);
        Repo::category()->edit($category, []);

        // Remove the image files.
        $contextFileManager = new ContextFileManager($category->getContextId());
        $basePath = $contextFileManager->getBasePath() . '/categories/';
        $name = $args['name'];
        $thumbnailName = $args['thumbnailName'];
        if (
            $contextFileManager->deleteByPath($basePath . $thumbnailName) &&
            $contextFileManager->deleteByPath($basePath . $name)
        ) {
            $json = new JSONMessage(true);
            $json->setEvent('fileDeleted');
            return $json;
        } else {
            return new JSONMessage(false, __('manager.categories.form.removeCoverImageFileNotFound'));
        }
    }

    //
    // Private helper methods.
    //
    /**
     * Get a CategoryForm instance.
     *
     * @param Request $request
     *
     * @return CategoryForm
     */
    public function _getCategoryForm($request)
    {
        // Get the category ID.
        $categoryId = (int) $request->getUserVar('categoryId');

        // Instantiate the files form.
        $contextId = $this->_getContextId();
        return new CategoryForm($contextId, $categoryId);
    }

    /**
     * Get context id.
     *
     * @return int
     */
    public function _getContextId()
    {
        return $this->_contextId;
    }
}
