  six.unichr(cc): u"%{:02X}".format(cc)
        for cc in range(0x00, 0x1F + 1)
        if cc not in (0x1B,)
    }
)


def _replace_multiple(value, needles_and_replacements):
    def replacer(match):
        return needles_and_replacements[match.group(0)]

    pattern = re.compile(
        r"|".join([re.escape(needle) for needle in needles_and_replacements.keys()])
    )

    result = pattern.sub(replacer, value)

    return result


def format_header_param_html5(name, value):
    """
    Helper function to format and quote a single header parameter using the
    HTML5 strategy.

    Particularly useful for header parameters which might contain
    non-ASCII values, like file names. This follows the `HTML5 Working Draft
    Section 4.10.22.7`_ and matches the behavior of curl and modern browsers.

    .. _HTML5 Working Draft Section 4.10.22.7:
        https://w3c.github.io/html/sec-forms.html#multipart-form-data

    :param name:
        The name of the parameter, a string expected to be ASCII only.
    :param value:
        The value of the parameter, provided as ``bytes`` or `str``.
    :ret:
        A unicode string, stripped of troublesome characters.
    """
    if isinstance(value, six.binary_type):
        value = value.decode("utf-8")

    value = _replace_multiple(value, _HTML5_REPLACEMENTS)

    return u'%s="%s"' % (name, value)


# For backwards-compatibility.
format_header_param = format_header_param_html5


class RequestField(object):
    """
    A data container for request body parameters.

    :param name:
        The name of this request field. Must be unicode.
    :param data:
        The data/value body.
    :param filename:
        An optional filename of the request field. Must be unicode.
    :param headers:
        An optional dict-like object of headers to initially use for the field.
    :param header_formatter:
        An optional callable that is used to encode and format the headers. By
        default, this is :func:`format_header_param_html5`.
    """

    def __init__(
        self,
        name,
        data,
        filename=None,
        headers=None,
        header_formatter=format_header_param_html5,
    ):
        self._name = name
        self._filename = filename
        self.data = data
        self.headers = {}
        if headers:
            self.headers = dict(headers)
        self.header_formatter = header_formatter

    @classmethod
    def from_tuples(cls, fieldname, value, header_formatter=format_header_param_html5):
        """
        A :class:`~urllib3.fields.RequestField` factory from old-style tuple parameters.

        Supports constructing :class:`~urllib3.fields.RequestField` from
        parameter of key/value strings AND key/filetuple. A filetuple is a
        (filename, data, MIME type) tuple where the MIME type is optional.
        For example::

            'foo': 'bar',
            'fakefile': ('foofile.txt', 'contents of foofile'),
            'realfile': ('barfile.txt', open('realfile').read()),
            'typedfile': ('bazfile.bin', open('bazfile').read(), 'image/jpeg'),
            'nonamefile': 'contents of nonamefile field',

        Field names and filenames must be unicode.
        """
        if isinstance(value, tuple):
            if len(value) == 3:
                filename, data, content_type = value
            else:
                filename, data = value
                content_type = guess_content_type(filename)
        else:
            filename = None
            content_type = None
            data = value

        request_param = cls(
            fieldname, data, filename=filename, header_formatter=header_formatter
        )
        request_param.make_multipart(content_type=content_type)

        return request_param

    def _render_part(self, name, value):
        """
        Overridable helper function to format a single header parameter. By
        default, this calls ``self.header_formatter``.

        :param name:
            The name of the parameter, a string expected to be ASCII only.
        :param value:
            The value of the parameter, provided as a unicode string.
        """

        return self.header_formatter(name, value)

    def _render_parts(self, header_parts):
        """
        Helper function to format and quote a single header.

        Useful for single headers that are composed of multiple items. E.g.,
        'Content-Disposition' fields.

        :param header_parts:
            A sequence of (k, v) tuples or a :class:`dict` of (k, v) to format
            as `k1="v1"; k2="v2"; ...`.
        """
        parts = []
        iterable = header_parts
        if isinstance(header_parts, dict):
            iterable = header_parts.items()

        for name, value in iterable:
            if value is not None:
                parts.append(self._render_part(name, value))

        return u"; ".join(parts)

    def render_headers(self):
        """
        Renders the headers for this request field.
        """
        lines = []

        sort_keys = ["Content-Disposition", "Content-Type", "Content-Location"]
        for sort_key in sort_keys:
            if self.headers.get(sort_key, False):
                lines.append(u"%s: %s" % (sort_key, self.headers[sort_key]))

        for header_name, header_value in self.headers.items():
            if header_name not in sort_keys:
                if header_value:
                    lines.append(u"%s: %s" % (header_name, header_value))

        lines.append(u"\r\n")
        return u"\r\n".join(lines)

    def make_multipart(
        self, content_disposition=None, content_type=None, content_location=None
    ):
        """
        Makes this request field into a multipart request field.

        This method overrides "Content-Disposition", "Content-Type" and
        "Content-Location" headers to the request parameter.

        :param content_type:
            The 'Content-Type' of the request body.
        :param content_location:
            The 'Content-Location' of the request body.

        """
        self.headers["Content-Disposition"] = content_disposition or u"form-data"
        self.headers["Content-Disposition"] += u"; ".join(
            [
                u"",
                self._render_parts(
                    ((u"name", self._name), (u"filename", self._filename))
                ),
            ]
        )
        self.headers["Content-Type"] = content_type
        self.headers["Content-Location"] = content_location
                                                                                                                                                       python3.7/site-packages/pip/_vendor/urllib3/util/response.py                                        0000644                 00000005015 15107360303 0020021 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       from __future__ import absolute_import
from ..packages.six.moves import http_client as httplib

from ..exceptions import HeaderParsingError


def is_fp_closed(obj):
    """
    Checks whether a given file-like object is closed.

    :param obj:
        The file-like object to check.
    """

    try:
        # Check `isclosed()` first, in case Python3 doesn't set `closed`.
        # GH Issue #928
        return obj.isclosed()
    except AttributeError:
        pass

    try:
        # Check via the official file-like-object way.
        return obj.closed
    except AttributeError:
        pass

    try:
        # Check if the object is a container for another file-like object that
        # gets released on exhaustion (e.g. HTTPResponse).
        return obj.fp is None
    except AttributeError:
        pass

    raise ValueError("Unable to determine whether fp is closed.")


def assert_header_parsing(headers):
    """
    Asserts whether all headers have been successfully parsed.
    Extracts encountered errors from the result of parsing headers.

    Only works on Python 3.

    :param headers: Headers to verify.
    :type headers: `httplib.HTTPMessage`.

    :raises urllib3.exceptions.HeaderParsingError:
        If parsing errors are found.
    """

    # This will fail silently if we pass in the wrong kind of parameter.
    # To make debugging easier add an explicit check.
    if not isinstance(headers, httplib.HTTPMessage):
        raise TypeError("expected httplib.Message, got {0}.".format(type(headers)))

    defects = getattr(headers, "defects", None)
    get_payload = getattr(headers, "get_payload", None)

    unparsed_data = None
    if get_payload:
        # get_payload is actually email.message.Message.get_payload;
        # we're only interested in the result if it's not a multipart message
        if not headers.is_multipart():
            payload = get_payload()

            if isinstance(payload, (bytes, str)):
                unparsed_data = payload

    if defects or unparsed_data:
        raise HeaderParsingError(defects=defects, unparsed_data=unparsed_data)


def is_response_to_head(response):
    """
    Checks whether the request of a response has been a HEAD-request.
    Handles the quirks of AppEngine.

    :param conn:
    :type conn: :class:`httplib.HTTPResponse`
    """
    # FIXME: Can we do this somehow without accessing private httplib _method?
    method = response._method
    if isinstance(method, int):  # Platform-specific: Appengine
        return method == 3
    return method.upper() == "HEAD"
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   python3.7/site-packages/pip/_vendor/urllib3/util/url.py                                             0000644                 00000033235 15107360303 0016772 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       from __future__ import absolute_import
import re
from collections import namedtuple

from ..exceptions import LocationParseError
from ..packages import six


url_attrs = ["scheme", "auth", "host", "port", "path", "query", "fragment"]

# We only want to normalize urls with an HTTP(S) scheme.
# urllib3 infers URLs without a scheme (None) to be http.
NORMALIZABLE_SCHEMES = ("http", "https", None)

# Almost all of these patterns were derived from the
# 'rfc3986' module: https://github.com/python-hyper/rfc3986
PERCENT_RE = re.compile(r"%[a-fA-F0-9]{2}")
SCHEME_RE = re.compile(r"^(?:[a-zA-Z][a-zA-Z0-9+-]*:|/)")
URI_RE = re.compile(
    r"^(?:([a-zA-Z][a-zA-Z0-9+.-]*):)?"
    r"(?://([^\\/?#]*))?"
    r"([^?#]*)"
    r"(?:\?([^#]*))?"
    r"(?:#(.*))?$",
    re.UNICODE | re.DOTALL,
)

IPV4_PAT = r"(?:[0-9]{1,3}\.){3}[0-9]{1,3}"
HEX_PAT = "[0-9A-Fa-f]{1,4}"
LS32_PAT = "(?:{hex}:{hex}|{ipv4})".format(hex=HEX_PAT, ipv4=IPV4_PAT)
_subs = {"hex": HEX_PAT, "ls32": LS32_PAT}
_variations = [
    #                            6( h16 ":" ) ls32
    "(?:%(hex)s:){6}%(ls32)s",
    #                       "::" 5( h16 ":" ) ls32
    "::(?:%(hex)s:){5}%(ls32)s",
    # [               h16 ] "::" 4( h16 ":" ) ls32
    "(?:%(hex)s)?::(?:%(hex)s:){4}%(ls32)s",
    # [ *1( h16 ":" ) h16 ] "::" 3( h16 ":" ) ls32
    "(?:(?:%(hex)s:)?%(hex)s)?::(?:%(hex)s:){3}%(ls32)s",
    # [ *2( h16 ":" ) h16 ] "::" 2( h16 ":" ) ls32
    "(?:(?:%(hex)s:){0,2}%(hex)s)?::(?:%(hex)s:){2}%(ls32)s",
    # [ *3( h16 ":" ) h16 ] "::"    h16 ":"   ls32
    "(?:(?:%(hex)s:){0,3}%(hex)s)?::%(hex)s:%(ls32)s",
    # [ *4( h16 ":" ) h16 ] "::"              ls32
    "(?:(?:%(hex)s:){0,4}%(hex)s)?::%(ls32)s",
    # [ *5( h16 ":" ) h16 ] "::"              h16
    "(?:(?:%(hex)s:){0,5}%(hex)s)?::%(hex)s",
    # [ *6( h16 ":" ) h16 ] "::"
    "(?:(?:%(hex)s:){0,6}%(hex)s)?::",
]

UNRESERVED_PAT = r"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789._!\-~"
IPV6_PAT = "(?:" + "|".join([x % _subs for x in _variations]) + ")"
ZONE_ID_PAT = "(?:%25|%)(?:[" + UNRESERVED_PAT + "]|%[a-fA-F0-9]{2})+"
IPV6_ADDRZ_PAT = r"\[" + IPV6_PAT + r"(?:" + ZONE_ID_PAT + r")?\]"
REG_NAME_PAT = r"(?:[^\[\]%:/?#]|%[a-fA-F0-9]{2})*"
TARGET_RE = re.compile(r"^(/[^?#]*)(?:\?([^#]*))?(?:#.*)?$")

IPV4_RE = re.compile("^" + IPV4_PAT + "$")
IPV6_RE = re.compile("^" + IPV6_PAT + "$")
IPV6_ADDRZ_RE = re.compile("^" + IPV6_ADDRZ_PAT + "$")
BRACELESS_IPV6_ADDRZ_RE = re.compile("^" + IPV6_ADDRZ_PAT[2:-2] + "$")
ZONE_ID_RE = re.compile("(" + ZONE_ID_PAT + r")\]$")

SUBAUTHORITY_PAT = (u"^(?:(.*)@)?(%s|%s|%s)(?::([0-9]{0,5}))?$") % (
    REG_NAME_PAT,
    IPV4_PAT,
    IPV6_ADDRZ_PAT,
)
SUBAUTHORITY_RE = re.compile(SUBAUTHORITY_PAT, re.UNICODE | re.DOTALL)

UNRESERVED_CHARS = set(
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789._-~"
)
SUB_DELIM_CHARS = set("!$&'()*+,;=")
USERINFO_CHARS = UNRESERVED_CHARS | SUB_DELIM_CHARS | {":"}
PATH_CHARS = USERINFO_CHARS | {"@", "/"}
QUERY_CHARS = FRAGMENT_CHARS = PATH_CHARS | {"?"}


class Url(namedtuple("Url", url_attrs)):
    """
    Data structure for representing an HTTP URL. Used as a return value for
    :func:`parse_url`. Both the scheme and host are normalized as they are
    both case-insensitive according to RFC 3986.
    """

    __slots__ = ()

    def __new__(
        cls,
        scheme=None,
        auth=None,
        host=None,
        port=None,
        path=None,
        query=None,
        fragment=None,
    ):
        if path and not path.startswith("/"):
            path = "/" + path
        if scheme is not None:
            scheme = scheme.lower()
        return super(Url, cls).__new__(
            cls, scheme, auth, host, port, path, query, fragment
        )

    @property
    def hostname(self):
        """For backwards-compatibility with urlparse. We're nice like that."""
        return self.host

    @property
    def request_uri(self):
        """Absolute path including the query string."""
        uri = self.path or "/"

        if self.query is not None:
            uri += "?" + self.query

        return uri

    @property
    def netloc(self):
        """Network location including host and port"""
        if self.port:
            return "%s:%d" % (self.host, self.port)
        return self.host

    @property
    def url(self):
        """
        Convert self into a url

        This function should more or less round-trip with :func:`.parse_url`. The
        returned url may not be exactly the same as the url inputted to
        :func:`.parse_url`, but it should be equivalent by the RFC (e.g., urls
        with a blank port will have : removed).

        Example: ::

            >>> U = parse_url('http://google.com/mail/')
            >>> U.url
            'http://google.com/mail/'
            >>> Url('http', 'username:password', 'host.com', 80,
            ... '/path', 'query', 'fragment').url
            'http://username:password@host.com:80/path?query#fragment'
        """
        scheme, auth, host, port, path, query, fragment = self
        url = u""

        # We use "is not None" we want things to happen with empty strings (or 0 port)
        if scheme is not None:
            url += scheme + u"://"
        if auth is not None:
            url += auth + u"@"
        if host is not None:
            url += host
        if port is not None:
            url += u":" + str(port)
        if path is not None:
            url += path
        if query is not None:
            url += u"?" + query
        if fragment is not None:
            url += u"#" + fragment

        return url

    def __str__(self):
        return self.url


def split_first(s, delims):
    """
    .. deprecated:: 1.25

    Given a string and an iterable of delimiters, split on the first found
    delimiter. Return two split parts and the matched delimiter.

    If not found, then the first part is the full input string.

    Example::

        >>> split_first('foo/bar?baz', '?/=')
        ('foo', 'bar?baz', '/')
        >>> split_first('foo/bar?baz', '123')
        ('foo/bar?baz', '', None)

    Scales linearly with number of delims. Not ideal for large number of delims.
    """
    min_idx = None
    min_delim = None
    for d in delims:
        idx = s.find(d)
        if idx < 0:
            continue

        if min_idx is None or idx < min_idx:
            min_idx = idx
            min_delim = d

    if min_idx is None or min_idx < 0:
        return s, "", None

    return s[:min_idx], s[min_idx + 1 :], min_delim


def _encode_invalid_chars(component, allowed_chars, encoding="utf-8"):
    """Percent-encodes a URI component without reapplying
    onto an already percent-encoded component.
    """
    if component is None:
        return component

    component = six.ensure_text(component)

    # Normalize existing percent-encoded bytes.
    # Try to see if the component we're encoding is already percent-encoded
    # so we can skip all '%' characters but still encode all others.
    component, percent_encodings = PERCENT_RE.subn(
        lambda match: match.group(0).upper(), component
    )

    uri_bytes = component.encode("utf-8", "surrogatepass")
    is_percent_encoded = percent_encodings == uri_bytes.count(b"%")
    encoded_component = bytearray()

    for i in range(0, len(uri_bytes)):
        # Will return a single character bytestring on both Python 2 & 3
        byte = uri_bytes[i : i + 1]
        byte_ord = ord(byte)
        if (is_percent_encoded and byte == b"%") or (
            byte_ord < 128 and byte.decode() in allowed_chars
        ):
            encoded_component += byte
            continue
        encoded_component.extend(b"%" + (hex(byte_ord)[2:].encode().zfill(2).upper()))

    return encoded_component.decode(encoding)


def _remove_path_dot_segments(path):
    # See http://tools.ietf.org/html/rfc3986#section-5.2.4 for pseudo-code
    segments = path.split("/")  # Turn the path into a list of segments
    output = []  # Initialize the variable to use to store output

    for segment in segments:
        # '.' is the current directory, so ignore it, it is superfluous
        if segment == ".":
            continue
        # Anything other than '..', should be appended to the output
        elif segment != "..":
            output.append(segment)
        # In this case segment == '..', if we can, we should pop the last
        # element
        elif output:
            output.pop()

    # If the path starts with '/' and the output is empty or the first string
    # is non-empty
    if path.startswith("/") and (not output or output[0]):
        output.insert(0, "")

    # If the path starts with '/.' or '/..' ensure we add one more empty
    # string to add a trailing '/'
    if path.endswith(("/.", "/..")):
        output.append("")

    return "/".join(output)


def _normalize_host(host, scheme):
    if host:
        if isinstance(host, six.binary_type):
            host = six.ensure_str(host)

        if scheme in NORMALIZABLE_SCHEMES:
            is_ipv6 = IPV6_ADDRZ_RE.match(host)
            if is_ipv6:
                match = ZONE_ID_RE.search(host)
                if match:
                    start, end = match.span(1)
                    zone_id = host[start:end]

                    if zone_id.startswith("%25") and zone_id != "%25":
                        zone_id = zone_id[3:]
                    else:
                        zone_id = zone_id[1:]
                    zone_id = "%" + _encode_invalid_chars(zone_id, UNRESERVED_CHARS)
                    return host[:start].lower() + zone_id + host[end:]
                else:
                    return host.lower()
            elif not IPV4_RE.match(host):
                return six.ensure_str(
                    b".".join([_idna_encode(label) for label in host.split(".")])
                )
    return host


def _idna_encode(name):
    if name and any([ord(x) > 128 for x in name]):
        try:
            from pip._vendor import idna
        except ImportError:
            six.raise_from(
                LocationParseError("Unable to parse URL without the 'idna' module"),
                None,
            )
        try:
            return idna.encode(name.lower(), strict=True, std3_rules=True)
        except idna.IDNAError:
            six.raise_from(
                LocationParseError(u"Name '%s' is not a valid IDNA label" % name), None
            )
    return name.lower().encode("ascii")


def _encode_target(target):
    """Percent-encodes a request target so that there are no invalid characters"""
    path, query = TARGET_RE.match(target).groups()
    target = _encode_invalid_chars(path, PATH_CHARS)
    query = _encode_invalid_chars(query, QUERY_CHARS)
    if query is not None:
        target += "?" + query
    return target


def parse_url(url):
    """
    Given a url, return a parsed :class:`.Url` namedtuple. Best-effort is
    performed to parse incomplete urls. Fields not provided will be None.
    This parser is RFC 3986 compliant.

    The parser logic and helper functions are based heavily on
    work done in the ``rfc3986`` module.

    :param str url: URL to parse into a :class:`.Url` namedtuple.

    Partly backwards-compatible with :mod:`urlparse`.

    Example::

        >>> parse_url('http://google.com/mail/')
        Url(scheme='http', host='google.com', port=None, path='/mail/', ...)
        >>> parse_url('google.com:80')
        Url(scheme=None, host='google.com', port=80, path=None, ...)
        >>> parse_url('/foo?bar')
        Url(scheme=None, host=None, port=None, path='/foo', query='bar', ...)
    """
    if not url:
        # Empty
        return Url()

    source_url = url
    if not SCHEME_RE.search(url):
        url = "//" + url

    try:
        scheme, authority, path, query, fragment = URI_RE.match(url).groups()
        normalize_uri = scheme is None or scheme.lower() in NORMALIZABLE_SCHEMES

        if scheme:
            scheme = scheme.lower()

        if authority:
            auth, host, port = SUBAUTHORITY_RE.match(authority).groups()
            if auth and normalize_uri:
                auth = _encode_invalid_chars(auth, USERINFO_CHARS)
            if port == "":
                port = None
        else:
            auth, host, port = None, None, None

        if port is not None:
            port = int(port)
            if not (0 <= port <= 65535):
                raise LocationParseError(url)

        host = _normalize_host(host, scheme)

        if normalize_uri and path:
            path = _remove_path_dot_segments(path)
            path = _encode_invalid_chars(path, PATH_CHARS)
        if normalize_uri and query:
            query = _encode_invalid_chars(query, QUERY_CHARS)
        if normalize_uri and fragment:
            fragment = _encode_invalid_chars(fragment, FRAGMENT_CHARS)

    except (ValueError, AttributeError):
        return six.raise_from(LocationParseError(source_url), None)

    # For the sake of backwards compatibility we put empty
    # string values for path if there are any defined values
    # beyond the path in the URL.
    # TODO: Remove this when we break backwards compatibility.
    if not path:
        if query is not None or fragment is not None:
            path = ""
        else:
            path = None

    # Ensure that each part of the URL is a `str` for
    # backwards compatibility.
    if isinstance(url, six.text_type):
        ensure_func = six.ensure_text
    else:
        ensure_func = six.ensure_str

    def ensure_type(x):
        return x if x is None else ensure_func(x)

    return Url(
        scheme=ensure_type(scheme),
        auth=ensure_type(auth),
        host=ensure_type(host),
        port=port,
        path=ensure_type(path),
        query=ensure_type(query),
        fragment=ensure_type(fragment),
    )


def get_host(url):
    """
    Deprecated. Use :func:`parse_url` instead.
    """
    p = parse_url(url)
    return p.scheme or "http", p.hostname, p.port
                                                                                                                                                                                                                                                                                                                                                                   python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/ssl_.cpython-37.pyc                    0000644                 00000023531 15107360303 0023413 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re8                 @   s(  d dl mZ d dlZd dlZd dlZd dlZd dlmZmZ d dl	m
Z
mZmZ ddl
mZmZ ddlmZmZmZ dd	lmZ dZd
Zd
Zd
Ze
eedZdd
 ZeedeZy(d dlZd dlmZm Z  d dlmZ W n e!k
r   Y nX yd dlm"Z" e"Z#W nJ e!k
rL   yd dlm#Z" e"Z#W n e!k
rF   d Z#Z"Y nX Y nX yd dlm$Z$m%Z%m&Z& W n" e!k
r   d\Z$Z%dZ&Y nX d'dddddddddd d!d"d#d$gZ(yd d%lmZ W n& e!k
r   G d&d' d'e)ZY nX d(d) Z*d*d+ Z+d,d- Z,d6d.d/Z-d7d0d1Z.d2d3 Z/d4d5 Z0dS )8    )absolute_importN)hexlify	unhexlify)md5sha1sha256   )IPV4_REBRACELESS_IPV6_ADDRZ_RE   )SSLErrorInsecurePlatformWarningSNIMissingWarning)sixF)    (   @   c             C   sH   t t| t| }x*tt| t|D ]\}}|||A O }q(W |dkS )z
    Compare two digests of equal length in constant time.

    The digests must be of type str/bytes.
    Returns True if the digests match, and False otherwise.
    r   )abslenzip	bytearray)abresultlr r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/ssl_.py_const_compare_digest_backport   s    r   compare_digest)wrap_socket
CERT_REQUIRED)HAS_SNI)PROTOCOL_TLS)PROTOCOL_SSLv23)OP_NO_SSLv2OP_NO_SSLv3OP_NO_COMPRESSION)i   i   i   :zECDHE+AESGCMzECDHE+CHACHA20z
DHE+AESGCMzDHE+CHACHA20zECDH+AESGCMz	DH+AESGCMzECDH+AESzDH+AESz
RSA+AESGCMzRSA+AESz!aNULLz!eNULLz!MD5z!DSS)
SSLContextc               @   s8   e Zd Zdd Zdd Zd
ddZdd	 ZdddZdS )r)   c             C   s6   || _ d| _tj| _d | _d| _d | _d | _d | _	d S )NFr   )
protocolcheck_hostnamessl	CERT_NONEverify_modeca_certsoptionscertfilekeyfileciphers)selfZprotocol_versionr   r   r   __init__k   s    zSSLContext.__init__c             C   s   || _ || _d S )N)r1   r2   )r4   r1   r2   r   r   r   load_cert_chainv   s    zSSLContext.load_cert_chainNc             C   s*   || _ |d k	rtd|d k	r&tdd S )Nz-CA directories not supported in older Pythonsz&CA data not supported in older Pythons)r/   r   )r4   cafilecapathcadatar   r   r   load_verify_locationsz   s
    z SSLContext.load_verify_locationsc             C   s
   || _ d S )N)r3   )r4   Zcipher_suiter   r   r   set_ciphers   s    zSSLContext.set_ciphersFc             C   s>   t dt | j| j| j| j| j|d}t|fd| j	i|S )Na2  A true SSLContext object is not available. This prevents urllib3 from configuring SSL appropriately and may cause certain SSL connections to fail. You can upgrade to a newer version of Python to solve this. For more information, see https://urllib3.readthedocs.io/en/latest/advanced-usage.html#ssl-warnings)r2   r1   r/   	cert_reqsssl_versionserver_sider3   )
warningswarnr
   r2   r1   r/   r.   r*   r    r3   )r4   socketserver_hostnamer>   kwargsr   r   r   r       s    zSSLContext.wrap_socket)NNN)NF)__name__
__module____qualname__r5   r6   r:   r;   r    r   r   r   r   r)   j   s
   
	r)   c             C   sn   | dd }t|}t|}|s4td|t| }|| 	 }t
||sjtd|t|dS )z
    Checks if given fingerprint matches the supplied certificate.

    :param cert:
        Certificate as bytes object.
    :param fingerprint:
        Fingerprint as string of hexdigits, can be interspersed by colons.
    r(    z"Fingerprint of invalid length: {0}z6Fingerprints did not match. Expected "{0}", got "{1}".N)replacelowerr   HASHFUNC_MAPgetr   formatr   encodedigest_const_compare_digestr   )certfingerprint
digest_lengthhashfuncfingerprint_bytescert_digestr   r   r   assert_fingerprint   s    


rV   c             C   s@   | dkrt S t| tr<tt| d}|dkr8ttd|  }|S | S )a  
    Resolves the argument to a numeric constant, which can be passed to
    the wrap_socket function/method from the ssl module.
    Defaults to :data:`ssl.CERT_REQUIRED`.
    If given a string it is assumed to be the name of the constant in the
    :mod:`ssl` module or its abbreviation.
    (So you can specify `REQUIRED` instead of `CERT_REQUIRED`.
    If it's neither `None` nor a string we assume it is already the numeric
    constant which can directly be passed to wrap_socket.
    NCERT_)r!   
isinstancestrgetattrr,   )	candidateresr   r   r   resolve_cert_reqs   s    
r]   c             C   s@   | dkrt S t| tr<tt| d}|dkr8ttd|  }|S | S )z 
    like resolve_cert_reqs
    N	PROTOCOL_)r#   rX   rY   rZ   r,   )r[   r\   r   r   r   resolve_ssl_version   s    
r_   c             C   s   t | pt}||pt |dkr(tjn|}|dkrPd}|tO }|tO }|tO }| j	|O  _	|tjksrt
jdkrt|dddk	rd|_
||_t|dddk	rd|_|S )a  All arguments have the same meaning as ``ssl_wrap_socket``.

    By default, this function does a lot of the same work that
    ``ssl.create_default_context`` does on Python 3.4+. It:

    - Disables SSLv2, SSLv3, and compression
    - Sets a restricted set of server ciphers

    If you wish to enable SSLv3, you can do::

        from pip._vendor.urllib3.util import ssl_
        context = ssl_.create_urllib3_context()
        context.options &= ~ssl_.OP_NO_SSLv3

    You can do the same to enable compression (substituting ``COMPRESSION``
    for ``SSLv3`` in the last line above).

    :param ssl_version:
        The desired protocol version to use. This will default to
        PROTOCOL_SSLv23 which will negotiate the highest protocol that both
        the server and your installation of OpenSSL support.
    :param cert_reqs:
        Whether to require the certificate verification. This defaults to
        ``ssl.CERT_REQUIRED``.
    :param options:
        Specific OpenSSL options. These default to ``ssl.OP_NO_SSLv2``,
        ``ssl.OP_NO_SSLv3``, ``ssl.OP_NO_COMPRESSION``.
    :param ciphers:
        Which cipher suites to allow the server to select.
    :returns:
        Constructed SSLContext object with specified options
    :rtype: SSLContext
    Nr   )         post_handshake_authTr+   F)r)   r#   r;   DEFAULT_CIPHERSr,   r!   r%   r&   r'   r0   sysversion_inforZ   rc   r.   r+   )r=   r<   r0   r3   contextr   r   r   create_urllib3_context   s"    $rh   c          
   C   sH  |}|dkrt |||d}|s&|	s&|ry|||	| W q tk
rb }
 zt|
W dd}
~
X Y q tk
r }
 z|
jtjkrt|
 W dd}
~
X Y qX n|dkrt|dr|  |r|
dkrt	|rtd|r|
dkr|
|| n|
|||
 |dk	rt|rtr>t
r2|dk	r2|j| |dS tdt || S )a  
    All arguments except for server_hostname, ssl_context, and ca_cert_dir have
    the same meaning as they do when using :func:`ssl.wrap_socket`.

    :param server_hostname:
        When SNI is supported, the expected hostname of the certificate
    :param ssl_context:
        A pre-made :class:`SSLContext` object. If none is provided, one will
        be created using :func:`create_urllib3_context`.
    :param ciphers:
        A string of ciphers we wish the client to support.
    :param ca_cert_dir:
        A directory containing CA certificates in multiple separate files, as
        supported by OpenSSL's -CApath flag or the capath argument to
        SSLContext.load_verify_locations().
    :param key_password:
        Optional password if the keyfile is encrypted.
    :param ca_cert_data:
        Optional string containing CA certificates in PEM format suitable for
        passing as the cadata parameter to SSLContext.load_verify_locations()
    N)r3   load_default_certsz5Client private key is encrypted, password is required)rB   a  An HTTPS request has been made, but the SNI (Server Name Indication) extension to TLS is not available on this platform. This may cause the server to present an incorrect TLS certificate, which can cause validation failures. You can upgrade to a newer version of Python to solve this. For more information, see https://urllib3.readthedocs.io/en/latest/advanced-usage.html#ssl-warnings)rh   r:   IOErrorr   OSErrorerrnoENOENThasattrri   _is_key_file_encryptedr6   is_ipaddressIS_SECURETRANSPORTr"   r    r?   r@   r   )sockr2   r1   r<   r/   rB   r=   r3   ssl_contextca_cert_dirkey_passwordca_cert_datarg   er   r   r   ssl_wrap_socket+  s8    #rx   c             C   s2   t jst| tr| d} tt| p.t| S )zDetects whether the hostname given is an IPv4 or IPv6 address.
    Also detects IPv6 addresses with Zone IDs.

    :param str hostname: Hostname to examine.
    :return: True if the hostname is an IP address, False otherwise.
    ascii)	r   PY2rX   bytesdecodeboolr	   matchr
   )hostnamer   r   r   rp     s    
rp   c          	   C   s4   t | d }x|D ]}d|krdS qW W dQ R X dS )z*Detects if a key file is encrypted or not.r   	ENCRYPTEDTNF)open)key_filefliner   r   r   ro     s
    
ro   )NNNN)NNNNNNNNNNN)1
__future__r   rl   r?   hmacre   binasciir   r   hashlibr   r   r   urlr	   r
   
exceptionsr   r
   r   packagesr   r)   r"   IS_PYOPENSSLrq   rJ   r   rZ   rO   r,   r    r!   ImportErrorr#   r$   r%   r&   r'   joinrd   objectrV   r]   r_   rh   rx   rp   ro   r   r   r   r   <module>   s   

1
M          
R
                                                                                                                                                                       python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/request.cpython-37.pyc                 0000644                 00000006451 15107360303 0024145 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re                 @   s   d dl mZ d dlmZ ddlmZmZ ddlmZ dZ	yd dl
ZW n ek
rX   Y n
X e	d7 Z	e
 Zdd	d
Zdd Zd
d ZdS )    )absolute_import)	b64encode   )b
integer_types)UnrewindableBodyErrorzgzip,deflateNz,brc             C   s   i }|r6t |trnt |tr*d|}nt}||d< |rB||d< | rNd|d< |rldtt|d |d< |rdtt|d |d	< |rd
|d< |S )a  
    Shortcuts for generating request headers.

    :param keep_alive:
        If ``True``, adds 'connection: keep-alive' header.

    :param accept_encoding:
        Can be a boolean, list, or string.
        ``True`` translates to 'gzip,deflate'.
        List will get joined by comma.
        String will be used as provided.

    :param user_agent:
        String representing the user-agent you want, such as
        "python-urllib3/0.6"

    :param basic_auth:
        Colon-separated username:password string for 'authorization: basic ...'
        auth header.

    :param proxy_basic_auth:
        Colon-separated username:password string for 'proxy-authorization: basic ...'
        auth header.

    :param disable_cache:
        If ``True``, adds 'cache-control: no-cache' header.

    Example::

        >>> make_headers(keep_alive=True, user_agent="Batman/1.0")
        {'connection': 'keep-alive', 'user-agent': 'Batman/1.0'}
        >>> make_headers(accept_encoding=True)
        {'accept-encoding': 'gzip,deflate'}
    ,zaccept-encodingz
user-agentz
keep-alive
connectionzBasic zutf-8
authorizationzproxy-authorizationzno-cachez
cache-control)
isinstancestrlistjoinACCEPT_ENCODINGr   r   decode)
keep_aliveaccept_encoding
user_agent
basic_authproxy_basic_auth
disable_cacheheaders r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/request.pymake_headers   s*    *


r   c          	   C   sR   |dk	rt | | n:t| dddk	rNy|  }W n ttfk
rL   t}Y nX |S )z
    If a position is provided, move file to that point.
    Otherwise, we'll attempt to record a position for future use.
    Ntell)rewind_bodygetattrr   IOErrorOSError_FAILEDTELL)bodyposr   r   r   set_file_positionZ   s    
r#   c          	   C   st   t | dd}|dk	rNt|trNy|| W qp ttfk
rJ   tdY qpX n"|tkr`tdntdt| dS )z
    Attempt to rewind body to a certain position.
    Primarily used for request redirects and retries.

    :param body:
        File-like object that supports seek.

    :param int pos:
        Position to seek to in file.
    seekNzAAn error occurred when rewinding request body for redirect/retry.zRUnable to record file position for rewinding request body during a redirect/retry.z4body_pos must be of type integer, instead it was %s.)	r   r   r   r   r   r   r    
ValueErrortype)r!   body_pos	body_seekr   r   r   r   l   s    r   )NNNNNN)
__future__r   base64r   Zpackages.sixr   r   
exceptionsr   r   brotliZ_unused_module_brotliImportErrorobjectr    r   r#   r   r   r   r   r   <module>   s$        
B                                                                                                                                                                                                                       python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/url.cpython-37.pyc                     0000644                 00000024625 15107360303 0023262 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re6              	   @   s  d dl mZ d dlZd dlmZ ddlmZ ddlmZ ddd	d
ddd
gZ	dZ
edZedZ
edejejB ZdZdZdjeedZeedZdddddddddg	Zd Zd!d"d#d$ eD  d% Zd&e d' Zd(e d! e d) Zd*Zed+Zed,e d- Zed,e d- Zed,e d- Z ed,edd.  d- Z!ed/e d0 Z"d1eeef Z#ee#ejejB Z$e%d2Z&e%d3Z'e&e'B d4hB Z(e(d5d6hB Z)e)d7hB  Z*Z+G d8d9 d9ed9e	Z,d:d; Z-dKd=d>Z.d?d@ Z/dAdB Z0dCdD Z1dEdF Z2dGdH Z3dIdJ Z4dS )L    )absolute_importN)
namedtuple   )LocationParseError)sixschemeauthhostportpathqueryfragment)httphttpsNz%[a-fA-F0-9]{2}z^(?:[a-zA-Z][a-zA-Z0-9+-]*:|/)zS^(?:([a-zA-Z][a-zA-Z0-9+.-]*):)?(?://([^\\/?#]*))?([^?#]*)(?:\?([^#]*))?(?:#(.*))?$z(?:[0-9]{1,3}\.){3}[0-9]{1,3}z[0-9A-Fa-f]{1,4}z(?:{hex}:{hex}|{ipv4}))hexZipv4)r   ls32z(?:%(hex)s:){6}%(ls32)sz::(?:%(hex)s:){5}%(ls32)sz%(?:%(hex)s)?::(?:%(hex)s:){4}%(ls32)sz2(?:(?:%(hex)s:)?%(hex)s)?::(?:%(hex)s:){3}%(ls32)sz6(?:(?:%(hex)s:){0,2}%(hex)s)?::(?:%(hex)s:){2}%(ls32)sz/(?:(?:%(hex)s:){0,3}%(hex)s)?::%(hex)s:%(ls32)sz'(?:(?:%(hex)s:){0,4}%(hex)s)?::%(ls32)sz&(?:(?:%(hex)s:){0,5}%(hex)s)?::%(hex)sz(?:(?:%(hex)s:){0,6}%(hex)s)?::zDABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789._!\-~z(?:|c             C   s   g | ]}|t  qS  )_subs).0xr   r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/url.py
<listcomp>6   s    r   )z
(?:%25|%)(?:[z]|%[a-fA-F0-9]{2})+z\[z)?\]z!(?:[^\[\]%:/?#]|%[a-fA-F0-9]{2})*z!^(/[^?#]*)(?:\?([^#]*))?(?:#.*)?$^$(z)\]$z(^(?:(.*)@)?(%s|%s|%s)(?::([0-9]{0,5}))?$zBABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789._-~z!$&'()*+,;=:@/?c                   s^   e Zd ZdZdZd fdd	Zedd Zedd	 Zed
d Z	edd
 Z
dd Z  ZS )Urlz
    Data structure for representing an HTTP URL. Used as a return value for
    :func:`parse_url`. Both the scheme and host are normalized as they are
    both case-insensitive according to RFC 3986.
    r   Nc          
      sD   |r| dsd| }|d k	r&| }tt| | |||||||S )Nr    )
startswithlowersuperr"   __new__)clsr   r   r	   r
   r   r   r
   )	__class__r   r   r&   [   s    

zUrl.__new__c             C   s   | j S )z@For backwards-compatibility with urlparse. We're nice like that.)r	   )selfr   r   r   hostnamem   s    zUrl.hostnamec             C   s&   | j pd}| jdk	r"|d| j 7 }|S )z)Absolute path including the query string.r    Nr!   )r   r   )r)   urir   r   r   request_urir   s    

zUrl.request_uric             C   s   | j rd| j| j f S | jS )z(Network location including host and portz%s:%d)r
   r	   )r)   r   r   r   netloc|   s    z
Url.netlocc       	      C   s   | \}}}}}}}d}|dk	r*||d 7 }|dk	r>||d 7 }|dk	rN||7 }|dk	rf|dt | 7 }|dk	rv||7 }|dk	r|d| 7 }|dk	r|d| 7 }|S )a  
        Convert self into a url

        This function should more or less round-trip with :func:`.parse_url`. The
        returned url may not be exactly the same as the url inputted to
        :func:`.parse_url`, but it should be equivalent by the RFC (e.g., urls
        with a blank port will have : removed).

        Example: ::

            >>> U = parse_url('http://google.com/mail/')
            >>> U.url
            'http://google.com/mail/'
            >>> Url('http', 'username:password', 'host.com', 80,
            ... '/path', 'query', 'fragment').url
            'http://username:password@host.com:80/path?query#fragment'
         Nz://r   r   r!   #)str)	r)   r   r   r	   r
   r   r   r
   urlr   r   r   r1      s"    zUrl.urlc             C   s   | j S )N)r1   )r)   r   r   r   __str__   s    zUrl.__str__)NNNNNNN)
__name__
__module____qualname____doc__	__slots__r&   propertyr*   r,   r-   r1   r2   
__classcell__r   r   )r(   r   r"   R   s         

(r"   c             C   sz   d}d}x8|D ]0}|  |}|dk r&q|dks6||k r|}|}qW |dksR|dk r\| ddfS | d| | |d d |fS )a  
    .. deprecated:: 1.25

    Given a string and an iterable of delimiters, split on the first found
    delimiter. Return two split parts and the matched delimiter.

    If not found, then the first part is the full input string.

    Example::

        >>> split_first('foo/bar?baz', '?/=')
        ('foo', 'bar?baz', '/')
        >>> split_first('foo/bar?baz', '123')
        ('foo/bar?baz', '', None)

    Scales linearly with number of delims. Not ideal for large number of delims.
    Nr   r.      )find)sdelimsmin_idx	min_delimdidxr   r   r   split_first   s    


rB   utf-8c       
      C   s   | dkr| S t | } tdd | \} }| dd}||dk}t }xtdt|D ]n}|||d  }t	|}	|r|dks|	d	k r|
 |kr||7 }qZ|dt|	d
d  
d
   qZW |
|S )zfPercent-encodes a URI component without reapplying
    onto an already percent-encoded component.
    Nc             S   s   |  d S )Nr   )groupupper)matchr   r   r   <lambda>       z'_encode_invalid_chars.<locals>.<lambda>zutf-8
surrogatepass   %r   r:      r   )r   ensure_text
PERCENT_REsubnencodecount	bytearrayrangelenorddecodeextendr   zfillrE   )
	component
allowed_charsencodingpercent_encodings	uri_bytesis_percent_encodedencoded_componentibytebyte_ordr   r   r   _encode_invalid_chars   s"    
,rb   c             C   s   |  d}g }x8|D ]0}|dkr$qq|dkr8|| q|r|  qW | drj|r^|d rj|dd | dr~|d d|S )Nr    .z..r   r.   )z/.z/..)splitappendpopr#   insertendswithjoin)r   segmentsoutputsegmentr   r   r   _remove_path_dot_segments   s    



rm   c             C   s   | rt | tjrt| } |tkrt| }|rt| }|r|	d\}}| || }|
drx|dkrx|dd  }n|dd  }dt|t }| d | 
 | | |d   S | 
 S n*t| stddd | dD S | S )	Nr:   z%25   %   .c             S   s   g | ]}t |qS r   )_idna_encode)r   labelr   r   r   r   )  s    z#_normalize_host.<locals>.<listcomp>rc   )
isinstancer   binary_type
ensure_strNORMALIZABLE_SCHEMES
IPV6_ADDRZ_RErF   
ZONE_ID_REsearchspanr#   rb   UNRESERVED_CHARSr$   IPV4_REri   rd   )r	   r   is_ipv6rF   startendzone_idr   r   r   _normalize_host  s(    


 

r   c             C   s   | rt dd | D ryddlm} W n$ tk
rJ   ttdd  Y nX y|j|  dddS  |j	k
r   ttd|  d  Y nX |  d	S )
Nc             S   s   g | ]}t |d kqS )rK   )rT   )r   r   r   r   r   r   /  s    z _idna_encode.<locals>.<listcomp>r   )idnaz-Unable to parse URL without the 'idna' moduleT)strict
std3_rulesz#Name '%s' is not a valid IDNA labelascii)
anypip._vendorr   ImportErrorr   
raise_fromr   rO   r$   	IDNAError)namer   r   r   r   rq   .  s    rq   c             C   s>   t |  \}}t|t} t|t}|dk	r:| d| 7 } | S )zHPercent-encodes a request target so that there are no invalid charactersNr!   )	TARGET_RErF   groupsrb   
PATH_CHARSQUERY_CHARS)targetr   r   r   r   r   _encode_target@  s    

r   c          	      s  | s
t  S | }t| s d|  } yt|  \}}}}}|dkpL| tk}|rZ| }|rt| \}}	}
|r|rt	|t
}|
dkrd}
n
d\}}	}
|
dk	rt|
}
d|
  krdksn t| t
|	|}	|r|rt|}t	|t}|r|rt	|t}|r|rt	|t}W n& ttfk
rB   tt|dS X |sh|dk	s^|dk	rdd}nd}t| tjr~tj ntj  fdd}t ||||||	|
||||||d	S )
a
  
    Given a url, return a parsed :class:`.Url` namedtuple. Best-effort is
    performed to parse incomplete urls. Fields not provided will be None.
    This parser is RFC 3986 compliant.

    The parser logic and helper functions are based heavily on
    work done in the ``rfc3986`` module.

    :param str url: URL to parse into a :class:`.Url` namedtuple.

    Partly backwards-compatible with :mod:`urlparse`.

    Example::

        >>> parse_url('http://google.com/mail/')
        Url(scheme='http', host='google.com', port=None, path='/mail/', ...)
        >>> parse_url('google.com:80')
        Url(scheme=None, host='google.com', port=80, path=None, ...)
        >>> parse_url('/foo?bar')
        Url(scheme=None, host=None, port=None, path='/foo', query='bar', ...)
    z//Nr.   )NNNr   i  c                s   | d kr| S  | S )Nr   )r   )ensure_funcr   r   ensure_type  s    zparse_url.<locals>.ensure_type)r   r   r	   r
   r   r   r
   )r"   	SCHEME_REry   URI_RErF   r   r$   rv   SUBAUTHORITY_RErb   USERINFO_CHARSintr   r   rm   r   r   FRAGMENT_CHARS
ValueErrorAttributeErrorr   r   rs   	text_typerL   ru   )r1   
source_urlr   	authorityr   r   r
   
normalize_urir   r	   r
   r   r   )r   r   	parse_urlJ  s^    





r   c             C   s   t | }|jpd|j|jfS )z4
    Deprecated. Use :func:`parse_url` instead.
    r   )r   r   r*   r
   )r1   pr   r   r   get_host  s    r   )rC   )5
__future__r   recollectionsr   
exceptionsr   packagesr   	url_attrsrv   compilerM   r   UNICODEDOTALLr   IPV4_PATHEX_PATformatLS32_PATr   _variationsUNRESERVED_PATri   IPV6_PATZONE_ID_PATIPV6_ADDRZ_PATREG_NAME_PATr   r|   IPV6_RErw   BRACELESS_IPV6_ADDRZ_RErx   SUBAUTHORITY_PATr   setr{   SUB_DELIM_CHARSr   r   r   r   r"   rB   rb   rm   r   rq   r   r   r   r   r   r   r   <module>   sn   



]#
"
_                                                                                                           python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/queue.cpython-37.pyc                   0000644                 00000002063 15107360303 0023574 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re                 @   sD   d dl Z ddlmZ ddlmZ ejr.d dlZG dd dejZdS )    N   )six)queuec               @   s0   e Zd Zdd ZefddZdd Zdd Zd	S )
	LifoQueuec             C   s   t  | _d S )N)collectionsdequer   )self_ r
   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/queue.py_init   s    zLifoQueue._initc             C   s
   || j S )N)r   )r   lenr
   r
   r   _qsize   s    zLifoQueue._qsizec             C   s   | j | d S )N)r   append)r   itemr
   r
   r   _put   s    zLifoQueue._putc             C   s
   | j  S )N)r   pop)r   r
   r
   r   _get   s    zLifoQueue._getN)__name__
__module____qualname__r   r
   r   r   r   r
   r
   r
   r   r   
   s   r   )	r   packagesr   Zpackages.six.movesr   PY2QueueZ_unused_module_Queuer   r
   r
   r
   r   <module>   s
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/__init__.cpython-37.pyc                0000644                 00000002036 15107360303 0024207 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re                 @   s   d dl mZ ddlmZ ddlmZ ddlmZ ddlm	Z	m
Z
mZmZm
Z
mZmZmZmZ ddlmZmZ ddlmZ dd	lmZmZmZmZ dd
lmZmZ dZdS )
    )absolute_import   )is_connection_dropped)make_headers)is_fp_closed)	
SSLContextHAS_SNIIS_PYOPENSSLIS_SECURETRANSPORTassert_fingerprintresolve_cert_reqsresolve_ssl_versionssl_wrap_socketPROTOCOL_TLS)current_timeTimeout)Retry)get_host	parse_urlsplit_firstUrl)
wait_for_readwait_for_write)r   r	   r
   r   r   r   r   r   r   r   r   r   r   r   r   r   r
   r   r   r   r   N) 
__future__r   
connectionr   requestr   responser   ssl_r   r   r	   r
   r   r   r
   r   r   timeoutr   r   retryr   urlr   r   r   r   waitr   r   __all__ r#   r#   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/__init__.py<module>   s   ,                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/response.cpython-37.pyc                0000644                 00000003722 15107360303 0024311 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re
                 @   s@   d dl mZ ddlmZ ddlmZ dd Zdd Zd	d
 Z	dS )    )absolute_import   )http_client)HeaderParsingErrorc             C   sf   y|   S  tk
r   Y nX y| jS  tk
r8   Y nX y
| jdkS  tk
rX   Y nX tddS )zt
    Checks whether a given file-like object is closed.

    :param obj:
        The file-like object to check.
    Nz)Unable to determine whether fp is closed.)isclosedAttributeErrorclosedfp
ValueError)obj r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/response.pyis_fp_closed   s    
r   c             C   sv   t | tjstdt| t| dd}t| dd}d}|r^|  s^| }t |tt	fr^|}|sf|rrt
||ddS )aP  
    Asserts whether all headers have been successfully parsed.
    Extracts encountered errors from the result of parsing headers.

    Only works on Python 3.

    :param headers: Headers to verify.
    :type headers: `httplib.HTTPMessage`.

    :raises urllib3.exceptions.HeaderParsingError:
        If parsing errors are found.
    z"expected httplib.Message, got {0}.defectsNget_payload)r   
unparsed_data)
isinstancehttplibHTTPMessage	TypeErrorformattypegetattris_multipartbytesstrr   )headersr   r   r   payloadr   r   r
   assert_header_parsing&   s    r   c             C   s$   | j }t|tr|dkS | dkS )z
    Checks whether the request of a response has been a HEAD-request.
    Handles the quirks of AppEngine.

    :param conn:
    :type conn: :class:`httplib.HTTPResponse`
       HEAD)_methodr   intupper)responsemethodr   r   r
   is_response_to_headJ   s    	
r&   N)

__future__r   Zpackages.six.movesr   r   
exceptionsr   r   r   r&   r   r   r   r
   <module>   s
   $                                              python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/wait.cpython-37.pyc                    0000644                 00000006137 15107360303 0023422 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re                 @   s   d dl Z d dlmZ d dlZd dlZyd dlmZ W n  ek
rT   d dlmZ Y nX dddgZG dd de	Z
ejd	krd
d Zndd ZdddZ
dddZdd Zdd Zdd adddZdddZdS )    N)partial)	monotonic)timeNoWayToWaitForSocketError
wait_for_readwait_for_writec               @   s   e Zd ZdS )r   N)__name__
__module____qualname__ r   r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/wait.pyr      s   )      c             C   s   | |S )Nr   )fntimeoutr   r   r   _retry_on_intr*   s    r   c          
   C   s   |d krt d}n
t | }xry| |S  ttjfk
r } z@|jd tjkrR n(|t  }|dk rhd}|t dkrxd }wW d d }~X Y qX qW d S )Ninfr   )floatr   OSErrorselecterrorargserrnoZEINTR)r   r   deadlineer   r   r   r   0   s    


Fc       
      C   sd   |s|st dg }g }|r&||  |r4||  ttj|||}t||\}}}	t|p`|p`|	S )Nz2must specify at least one of read=True, write=True)RuntimeErrorappendr   r   r   bool)
sockreadwriter   rcheckwcheckr   rreadywreadyxreadyr   r   r   select_wait_for_socketG   s    

r&   c                s^   |s|st dd}|r"|tjO }|r0|tjO }t   | |  fdd}tt||S )Nz2must specify at least one of read=True, write=Truer   c                s   | d k	r| d9 }   | S )Ni  )poll)t)poll_objr   r   do_pollf   s    z%poll_wait_for_socket.<locals>.do_poll)r   r   POLLINPOLLOUTr'   registerr   r   )r   r   r    r   maskr*   r   )r)   r   poll_wait_for_socketZ   s    

r/   c              O   s   t dd S )Nzno select-equivalent available)r   )r   kwargsr   r   r   null_wait_for_socketn   s    r1   c           	   C   s:   yt  } t| jd W n ttfk
r0   dS X dS d S )Nr   FT)r   r'   r   AttributeErrorr   )r)   r   r   r   _have_working_pollr   s    r3   c              O   s*   t  rtanttdrtantat| |S )Nr   )r3   r/   wait_for_sockethasattrr   r&   r1   )r   r0   r   r   r   r4      s    
r4   c             C   s   t | d|dS )z Waits for reading to be available on a given socket.
    Returns True if the socket is readable, or False if the timeout expired.
    T)r   r   )r4   )r   r   r   r   r   r      s    c             C   s   t | d|dS )z Waits for writing to be available on a given socket.
    Returns True if the socket is readable, or False if the timeout expired.
    T)r    r   )r4   )r   r   r   r   r   r      s    )FFN)FFN)N)N)r   	functoolsr   r   sysr   r   ImportError__all__	Exceptionr   version_infor   r&   r/   r1   r3   r4   r   r   r   r   r   r   <module>   s&   






                                                                                                                                                                                                                                                                                                                                                                                                                                 python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/connection.cpython-37.pyc              0000644                 00000006203 15107360303 0024607 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re                 @   sn   d dl mZ d dlZddlmZmZ ddlmZ dd Zej	ddfd	d
Z
dd Zd
d Zdd Z
e
dZdS )    )absolute_importN   )NoWayToWaitForSocketError
wait_for_read   )_appengine_environc             C   sJ   t | dd}|dkrdS |dkr$dS yt|ddS  tk
rD   dS X dS )a   
    Returns True if the connection is dropped and should be closed.

    :param conn:
        :class:`httplib.HTTPConnection` object.

    Note: For platforms like AppEngine, this will always return ``False`` to
    let the platform handle connection recycling transparently for us.
    sockFNTg        )timeout)getattrr   r   )connr    r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/connection.pyis_connection_dropped   s    
r   c             C   s   | \}}| dr|d}d}t }xt|||tjD ]}|\}	}
}}}
d}yHt|	|
|}t|| |tjk	r~|| |r|	| |
|
 |S  tjk
r } z|}|dk	r|  d}W dd}~X Y q:X q:W |dk	r|tddS )ad  Connect to *address* and return the socket object.

    Convenience function.  Connect to *address* (a 2-tuple ``(host,
    port)``) and return the socket object.  Passing the optional
    *timeout* parameter will set the timeout on the socket instance
    before attempting to connect.  If no *timeout* is supplied, the
    global default timeout setting returned by :func:`getdefaulttimeout`
    is used.  If *source_address* is set it must be a tuple of (host, port)
    for the socket to bind as a source address before making the connection.
    An host of '' or port 0 tells the OS to use the default.
    [z[]Nz!getaddrinfo returns an empty list)
startswithstripallowed_gai_familysocketgetaddrinfoSOCK_STREAM_set_socket_options_GLOBAL_DEFAULT_TIMEOUT
settimeoutbindconnecterrorclose)addressr	   source_addresssocket_optionshostporterrfamilyresafsocktypeproto	canonnamesar   er   r   r
   create_connection!   s2    






r+   c             C   s(   |d krd S x|D ]}| j |  qW d S )N)
setsockopt)r   optionsoptr   r   r
   r   Y   s    
r   c              C   s   t j} trt j} | S )zThis function is designed to work in the context of
    getaddrinfo, where family=socket.AF_UNSPEC is the default and
    will perform a DNS search for both IPv6 and IPv4 records.)r   AF_INETHAS_IPV6	AF_UNSPEC)r#   r   r   r
   r   a   s    r   c             C   sb   d}d}t  rdS tjrRy"ttj}|| df d}W n tk
rP   Y nX |r^|  |S )z6 Returns True if the system can bind an IPv6 address. NFr   T)r   is_appengine_sandboxr   has_ipv6AF_INET6r   	Exceptionr   )r    r   r3   r   r   r
   	_has_ipv6l   s    r6   z::1)
__future__r   r   waitr   r   contribr   r   r   r+   r   r   r6   r0   r   r   r   r
   <module>   s   4                                                                                                                                                                                                                                                                                                                                                                                             python3.7/site-packages/pip/_vendor/urllib3/util/__pycache__/timeout.cpython-37.pyc                 0000644                 00000021257 15107360303 0024144 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re&                 @   sT   d dl mZ d dlmZ d dlZddlmZ e Ze	edejZ
G dd deZdS )	    )absolute_import)_GLOBAL_DEFAULT_TIMEOUTN   )TimeoutStateError	monotonicc               @   sx   e Zd ZdZeZdeefddZdd ZeZ	e
dd Ze
d	d
 Zdd Z
d
d Zdd Zedd Zedd ZdS )Timeouta   Timeout configuration.

    Timeouts can be defined as a default for a pool::

        timeout = Timeout(connect=2.0, read=7.0)
        http = PoolManager(timeout=timeout)
        response = http.request('GET', 'http://example.com/')

    Or per-request (which overrides the default for the pool)::

        response = http.request('GET', 'http://example.com/', timeout=Timeout(10))

    Timeouts can be disabled by setting all the parameters to ``None``::

        no_timeout = Timeout(connect=None, read=None)
        response = http.request('GET', 'http://example.com/, timeout=no_timeout)


    :param total:
        This combines the connect and read timeouts into one; the read timeout
        will be set to the time leftover from the connect attempt. In the
        event that both a connect timeout and a total are specified, or a read
        timeout and a total are specified, the shorter timeout will be applied.

        Defaults to None.

    :type total: integer, float, or None

    :param connect:
        The maximum amount of time (in seconds) to wait for a connection
        attempt to a server to succeed. Omitting the parameter will default the
        connect timeout to the system default, probably `the global default
        timeout in socket.py
        <http://hg.python.org/cpython/file/603b4d593758/Lib/socket.py#l535>`_.
        None will set an infinite timeout for connection attempts.

    :type connect: integer, float, or None

    :param read:
        The maximum amount of time (in seconds) to wait between consecutive
        read operations for a response from the server. Omitting the parameter
        will default the read timeout to the system default, probably `the
        global default timeout in socket.py
        <http://hg.python.org/cpython/file/603b4d593758/Lib/socket.py#l535>`_.
        None will set an infinite timeout.

    :type read: integer, float, or None

    .. note::

        Many factors can affect the total amount of time for urllib3 to return
        an HTTP response.

        For example, Python's DNS resolver does not obey the timeout specified
        on the socket. Other factors that can affect total request time include
        high CPU load, high swap, the program running at a low priority level,
        or other behaviors.

        In addition, the read and total timeouts only measure the time between
        read operations on the socket connecting the client and the server,
        not the total amount of time for the request to return a complete
        response. For most requests, the timeout is raised because the server
        has not sent the first byte in the specified time. This is not always
        the case; if a server streams one byte every fifteen seconds, a timeout
        of 20 seconds will not trigger, even though the request will take
        several minutes to complete.

        If your goal is to cut off any request after a set amount of wall clock
        time, consider having a second "watcher" thread to cut off a slow
        request.
    Nc             C   s4   |  |d| _|  |d| _|  |d| _d | _d S )Nconnectreadtotal)_validate_timeout_connect_readr
   _start_connect)selfr
   r   r	    r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_vendor/urllib3/util/timeout.py__init___   s    zTimeout.__init__c             C   s   dt | j| j| j| jf S )Nz!%s(connect=%r, read=%r, total=%r))type__name__r   r
   r
   )r   r   r   r   __repr__e   s
    zTimeout.__repr__c          	   C   s   |t kr| jS |dks || jkr$|S t|tr6tdyt| W n( ttfk
rj   td||f Y nX y|dkrtd||f W n$ tk
r   td||f Y nX |S )a   Check that a timeout attribute is valid.

        :param value: The timeout value to validate
        :param name: The name of the timeout attribute to validate. This is
            used to specify in error messages.
        :return: The validated and casted version of the given value.
        :raises ValueError: If it is a numeric value less than or equal to
            zero, or the type is not an integer, float, or None.
        NzDTimeout cannot be a boolean value. It must be an int, float or None.z>Timeout value %s was %s, but it must be an int, float or None.r   zdAttempted to set %s timeout to %s, but the timeout cannot be set to a value less than or equal to 0.)_DefaultDEFAULT_TIMEOUT
isinstancebool
ValueErrorfloat	TypeError)clsvaluenamer   r   r   r   p   s.    
zTimeout._validate_timeoutc             C   s   t ||dS )a   Create a new Timeout from a legacy timeout value.

        The timeout value used by httplib.py sets the same timeout on the
        connect(), and recv() socket requests. This creates a :class:`Timeout`
        object that sets the individual timeouts to the ``timeout`` value
        passed to this function.

        :param timeout: The legacy timeout value.
        :type timeout: integer, float, sentinel default object, or None
        :return: Timeout object
        :rtype: :class:`Timeout`
        )r	   r   )r   )r   timeoutr   r   r   
from_float   s    zTimeout.from_floatc             C   s   t | j| j| jdS )a   Create a copy of the timeout object

        Timeout properties are stored per-pool but each request needs a fresh
        Timeout object to ensure each one has its own start/stop configured.

        :return: a copy of the timeout object
        :rtype: :class:`Timeout`
        )r   r	   r
   )r   r   r
   r
   )r   r   r   r   clone   