 self.enable_breadcrumbs:
            crumbs = self.context.breadcrumbs.get_buffer()
            if crumbs:
                # Make sure we send the crumbs here as "values" as we use the
                # raven client internally in sentry and the alternative
                # submission option of a list here is not supported by the
                # internal sender.
                data.setdefault('breadcrumbs', {'values': crumbs})

        return data

    def transform(self, data):
        return transform(
            data, list_max_length=self.list_max_length,
            string_max_length=self.string_max_length)

    @property
    def context(self):
        """
        Updates this clients thread-local context for future events.

        >>> def view_handler(view_func, *args, **kwargs):
        >>>     client.context.merge(tags={'key': 'value'})
        >>>     try:
        >>>         return view_func(*args, **kwargs)
        >>>     finally:
        >>>         client.context.clear()
        """
        return self._context

    def user_context(self, data):
        """
        Update the user context for future events.

        >>> client.user_context({'email': 'foo@example.com'})
        """
        return self.context.merge({
            'user': data,
        })

    def http_context(self, data, **kwargs):
        """
        Update the http context for future events.

        >>> client.http_context({'url': 'http://example.com'})
        """
        return self.context.merge({
            'request': data,
        })

    def extra_context(self, data, **kwargs):
        """
        Update the extra context for future events.

        >>> client.extra_context({'foo': 'bar'})
        """
        return self.context.merge({
            'extra': data,
        })

    def tags_context(self, data, **kwargs):
        """
        Update the tags context for future events.

        >>> client.tags_context({'version': '1.0'})

        """
        return self.context.merge({
            'tags': data,
        })

    def capture(self, event_type, data=None, date=None, time_spent=None,
                extra=None, stack=None, tags=None, sample_rate=None,
                **kwargs):
        """
        Captures and processes an event and pipes it off to SentryClient.send.

        To use structured data (interfaces) with capture:

        >>> capture('raven.events.Message', message='foo', data={
        >>>     'request': {
        >>>         'url': '...',
        >>>         'data': {},
        >>>         'query_string': '...',
        >>>         'method': 'POST',
        >>>     },
        >>>     'logger': 'logger.name',
        >>> }, extra={
        >>>     'key': 'value',
        >>> })

        The finalized ``data`` structure contains the following (some optional)
        builtin values:

        >>> {
        >>>     # the culprit and version information
        >>>     'culprit': 'full.module.name', # or /arbitrary/path
        >>>
        >>>     # all detectable installed modules
        >>>     'modules': {
        >>>         'full.module.name': 'version string',
        >>>     },
        >>>
        >>>     # arbitrary data provided by user
        >>>     'extra': {
        >>>         'key': 'value',
        >>>     }
        >>> }

        :param event_type: the module path to the Event class. Builtins can use
                           shorthand class notation and exclude the full module
                           path.
        :param data: the data base, useful for specifying structured data
                           interfaces. Any key which contains a '.' will be
                           assumed to be a data interface.
        :param date: the datetime of this event
        :param time_spent: a integer value representing the duration of the
                           event (in milliseconds)
        :param extra: a dictionary of additional standard metadata
        :param stack: a stacktrace for the event
        :param tags: dict of extra tags
        :param sample_rate: a float in the range [0, 1] to sample this message
        :return: a tuple with a 32-length string identifying this event
        """
        if not self.is_enabled():
            return

        exc_info = kwargs.get('exc_info')
        if exc_info is not None:
            if self.skip_error_for_logging(exc_info):
                return
            elif not self.should_capture(exc_info):
                self.logger.info(
                    'Not capturing exception due to filters: %s', exc_info[0],
                    exc_info=sys.exc_info())
                return
            self.record_exception_seen(exc_info)

        data = self.build_msg(
            event_type, data, date, time_spent, extra, stack, tags=tags,
            **kwargs)

        # should this event be sampled?
        if sample_rate is None:
            sample_rate = self.sample_rate

        if self._random.random() < sample_rate:
            self.send(**data)

        self._local_state.last_event_id = data['event_id']

        return data['event_id']

    def is_enabled(self):
        """
        Return a boolean describing whether the client should attempt to send
        events.
        """
        return self.remote.is_active()

    def _iter_frames(self, data):
        if 'stacktrace' in data:
            for frame in data['stacktrace']['frames']:
                yield frame
        if 'exception' in data:
            for frame in data['exception']['values'][-1]['stacktrace'].get('frames', []):
                yield frame

    def _successful_send(self):
        self.state.set_success()

    def _failed_send(self, exc, url, data):
        retry_after = 0
        if isinstance(exc, APIError):
            if isinstance(exc, RateLimited):
                retry_after = exc.retry_after
            self.error_logger.error(
                'Sentry responded with an API error: %s(%s)',
                type(exc).__name__, exc.message)
        else:
            self.error_logger.error(
                'Sentry responded with an error: %s (url: %s)',
                exc, url,
                exc_info=True,
                extra={'data': data}
            )

        self._log_failed_submission(data)
        self.state.set_fail(retry_after=retry_after)

    def _log_failed_submission(self, data):
        """
        Log a reasonable representation of an event that should have been sent
        to Sentry
        """
        message = data.pop('message', '<no message value>')
        output = [message]
        if 'exception' in data and 'stacktrace' in data['exception']['values'][-1]:
            # try to reconstruct a reasonable version of the exception
            for frame in data['exception']['values'][-1]['stacktrace'].get('frames', []):
                output.append('  File "%(fn)s", line %(lineno)s, in %(func)s' % {
                    'fn': frame.get('filename', 'unknown_filename'),
                    'lineno': frame.get('lineno', -1),
                    'func': frame.get('function', 'unknown_function'),
                })

        self.uncaught_logger.error(output)

    def send_remote(self, url, data, headers=None):
        # If the client is configured to raise errors on sending,
        # the implication is that the backoff and retry strategies
        # will be handled by the calling application
        if headers is None:
            headers = {}

        if not self.raise_send_errors and not self.state.should_try():
            data = self.decode(data)
            self._log_failed_submission(data)
            return

        self.logger.debug('Sending message of length %d to %s', len(data), url)

        def failed_send(e):
            self._failed_send(e, url, self.decode(data))

        try:
            transport = self.remote.get_transport()
            if transport.is_async:
                transport.async_send(url, data, headers, self._successful_send,
                                     failed_send)
            else:
                transport.send(url, data, headers)
                self._successful_send()
        except Exception as e:
            if self.raise_send_errors:
                raise
            failed_send(e)

    def send(self, auth_header=None, **data):
        """
        Serializes the message and passes the payload onto ``send_encoded``.
        """
        message = self.encode(data)

        return self.send_encoded