python3.7/site-packages/raven/contrib/awslambda/__init__.py                                         0000644                 00000013430 15107360263 0017660 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.awslambda
~~~~~~~~~~~~~~~~~~~~

Raven wrapper for AWS Lambda handlers.

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""
# flake8: noqa

from __future__ import absolute_import

import os
import logging
import functools
from types import FunctionType

from raven.base import Client
from raven.transport.http import HTTPTransport

logger = logging.getLogger('sentry.errors.client')


def get_default_tags():
    return {
        'lambda': 'AWS_LAMBDA_FUNCTION_NAME',
        'version': 'AWS_LAMBDA_FUNCTION_VERSION',
        'memory_size': 'AWS_LAMBDA_FUNCTION_MEMORY_SIZE',
        'log_group': 'AWS_LAMBDA_LOG_GROUP_NAME',
        'log_stream': 'AWS_LAMBDA_LOG_STREAM_NAME',
        'region': 'AWS_REGION'
    }


class LambdaClient(Client):
    """
    Raven decorator for AWS Lambda.

    By default, the lambda integration will capture unhandled exceptions and instrument logging.

    Usage:

    >>> from raven.contrib.awslambda import LambdaClient
    >>>
    >>>
    >>> client = LambdaClient()
    >>>
    >>> @client.capture_exceptions
    >>> def handler(event, context):
    >>>    ...
    >>>    raise Exception('I will be sent to sentry!')

    """

    def __init__(self, *args, **kwargs):
        transport = kwargs.get('transport', HTTPTransport)
        super(LambdaClient, self).__init__(*args, transport=transport, **kwargs)

    def capture(self, *args, **kwargs):
        if 'data' not in kwargs:
            kwargs['data'] = data = {}
        else:
            data = kwargs['data']
        event = kwargs.get('event', None)
        context = kwargs.get('context', None)
        user_info = self._get_user_interface(event)
        if user_info:
            data.update(user_info)
        if event:
            http_info = self._get_http_interface(event)
            if http_info:
                data.update(http_info)
            data['extra'] = self._get_extra_data(event, context)
        return super(LambdaClient, self).capture(*args, **kwargs)

    def build_msg(self, *args, **kwargs):

        data = super(LambdaClient, self).build_msg(*args, **kwargs)
        for option, default in get_default_tags().items():
            data['tags'].setdefault(option, os.environ.get(default))
        data.setdefault('release', os.environ.get('SENTRY_RELEASE'))
        data.setdefault('environment', os.environ.get('SENTRY_ENVIRONMENT'))
        return data

    def capture_exceptions(self, f=None, exceptions=None):  # TODO: Ash fix kwargs in base
        """
        Wrap a function or code block in try/except and automatically call
        ``.captureException`` if it raises an exception, then the exception
        is reraised.

        By default, it will capture ``Exception``

        >>> @client.capture_exceptions
        >>> def foo():
        >>>     raise Exception()

        >>> with client.capture_exceptions():
        >>>    raise Exception()

        You can also specify exceptions to be caught specifically

        >>> @client.capture_exceptions((IOError, LookupError))
        >>> def bar():
        >>>     ...

        ``kwargs`` are passed through to ``.captureException``.
        """
        if not isinstance(f, FunctionType):
            # when the decorator has args which is not a function we except
            # f to be the exceptions tuple
            return functools.partial(self.capture_exceptions, exceptions=f)

        exceptions = exceptions or (Exception,)

        @functools.wraps(f)
        def wrapped(event, context, *args, **kwargs):
            try:
                return f(event, context, *args, **kwargs)
            except exceptions:
                self.captureException(event=event, context=context, **kwargs)
                self.context.clear()
                raise
        return wrapped

    @staticmethod
    def _get_user_interface(event):
        if event.get('requestContext'):
            identity = event['requestContext']['identity']
            if identity:
                user = {
                    'id': identity.get('cognitoIdentityId', None) or identity.get('user', None),
                    'username': identity.get('user', None),
                    'ip_address': identity.get('sourceIp', None),
                    'cognito_identity_pool_id': identity.get('cognitoIdentityPoolId', None),
                    'cognito_authentication_type': identity.get('cognitoAuthenticationType', None),
                    'user_agent': identity.get('userAgent')
                }
                return {'user': user}

    @staticmethod
    def _get_http_interface(event):
        if event.get('path') and event.get('httpMethod'):
            request = {
                "url": event.get('path'),
                "method": event.get('httpMethod'),
                "query_string": event.get('queryStringParameters', None),
                "headers": event.get('headers', None) or [],
            }
            return {'request': request}

    @staticmethod
    def _get_extra_data(event, context):
        extra_context = {
            'event': event,
            'aws_request_id': context.aws_request_id,
            'context': vars(context),
        }

        if context.client_context:
            extra_context['client_context'] = {
                'client.installation_id': context.client_context.client.installation_id,
                'client.app_title': context.client_context.client.app_title,
                'client.app_version_name': context.client_context.client.app_version_name,
                'client.app_version_code': context.client_context.client.app_version_code,
                'client.app_package_name': context.client_context.client.app_package_name,
                'custom': context.client_context.custom,
                'env': context.client_context.env,
            }
        return extra_context



                                                                                                                                                                                                                                        python3.7/site-packages/raven/contrib/__pycache__/__init__.cpython-37.pyc                           0000644                 00000000570 15107360263 0022213 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                  @   s   d Z ddlmZ dS )z
raven.contrib
~~~~~~~~~~~~~

:copyright: (c) 2010-2013 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importN)__doc__
__future__r    r   r   G/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/__init__.py<module>   s                                                                                                                                           python3.7/site-packages/raven/contrib/__pycache__/flask.cpython-37.pyc                              0000644                 00000020606 15107360263 0021556 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Yw)                 @   s  d Z ddlmZ yddlmZ W n ek
r8   dZY nX dZddlZddlZddl	m
Z
mZmZ ddl
mZmZ dd	lmZ dd
lmZ ddlmZ ddlmZ dd
lmZ ddlmZ ddlmZ ddlm Z m!Z! ddl"m#Z# e$ Z%e%&dZ'dddZ(G dd de)ZdS )z
raven.contrib.flask
~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)current_userFTN)requestcurrent_appg)got_request_exceptionrequest_finished)ClientDisconnected)
setup_logging)Client)Sentry)
SentryHandler)urlparse)
to_unicode)get_headersget_environ)convert_optionslogging_configuredc          
   C   sF   | f t |j|t|jdg t|jgB |jdg d|iddS )NZSENTRY_INCLUDE_PATHSZRAVEN_IGNORE_EXCEPTIONSapp)dsnZ
include_pathsZignore_exceptionsextra)defaults)r   configsetgetZimport_name)
client_clsr   r    r   D/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/flask.pymake_client'   s    r   c            	   @   s   e Zd ZdZddedddejddf	ddZedd Z	e	j
d	d Z	d
d Zdd
 Zdd Z
dd Zdd Zdd Zd(ddZdd Zdd Zd)ddZdd Zd d! Zd"d# Zd$d% Zd&d' ZdS )*r   a  
    Flask application for Sentry.

    Look up configuration from ``os.environ['SENTRY_DSN']``::

    >>> sentry = Sentry(app)

    Pass an arbitrary DSN::

    >>> sentry = Sentry(app, dsn='http://public:secret@example.com/1')

    Pass an explicit client::

    >>> sentry = Sentry(app, client=client)

    Automatically configure logging::

    >>> sentry = Sentry(app, logging=True, level=logging.ERROR)

    Capture an exception::

    >>> try:
    >>>     1 / 0
    >>> except ZeroDivisionError:
    >>>     sentry.captureException()

    Capture a message::

    >>> sentry.captureMessage('hello, world!')

    By default, the Flask integration will do the following:

    - Hook into the `got_request_exception` signal. This can be disabled by
      passing `register_signal=False`.
    - Wrap the WSGI application. This can be disabled by passing
      `wrap_wsgi=False`.
    - Capture information from Flask-Login (if available).
    NFTc
       
      C   sX   |rt |tstd|| _|| _|| _|| _|| _|| _|| _	|	| _
|rT| | d S )Nz&client should be an instance of Client)
isinstancer   	TypeErrorr   logginglogging_exclusionsr   clientlevel	wrap_wsgiregister_signalinit_app)
selfr   r#   r   r   r!   r"   r$   r%   r&   r   r   r   __init__e   s    zSentry.__init__c             C   s(   yt jS  tk
r   Y nX t| dd S )N_last_event_id)r   sentry_event_id	Exceptiongetattr)r(   r   r   r   
last_event_idw   s
    zSentry.last_event_idc             C   s*   || _ y
|t_W n tk
r$   Y nX d S )N)r*   r   r+   r,   )r(   valuer   r   r   r.      s
    
c             O   s    | j s
d S | j|dd d S )Nexc_info)r0   )r#   captureExceptionr   )r(   argskwargsr   r   r   handle_exception   s    zSentry.handle_exceptionc             C   s   i }y|j d }W n tk
r,   |j}Y nX |r:||d< tsB|S ttdsP|S y
tj}W n tk
rn   |S X t	|r~| }|s|S t
 |d< dtjkrx*tjd D ]}tt|rtt|||< qW |S )zu
        Requires Flask-Login (https://pypi.python.org/pypi/Flask-Login/)
        to be installed and setup.
        r   
ip_addressZ
login_manageridZSENTRY_USER_ATTRS)
Zaccess_route
IndexErrorZremote_addrhas_flask_loginhasattrr   r   is_authenticatedAttributeErrorcallableZget_idr   r-   )r(   r   Z	user_infor5   r:   attrr   r   r   
get_user_info   s2    



zSentry.get_user_infoc             C   s&   |  |jr| j}n| j}| ||S )zR
        Determine how to retrieve actual data by using request.mimetype.
        )is_json_typeZmimetype
get_json_data
get_form_dataget_http_info_with_retriever)r(   r   	retrieverr   r   r   
get_http_info   s    zSentry.get_http_infoc             C   s   |dkS )Nzapplication/jsonr   )r(   content_typer   r   r   r?      s    zSentry.is_json_typec             C   s   |j S )N)Zform)r(   r   r   r   r   rA      s    zSentry.get_form_datac             C   s   |j S )N)data)r(   r   r   r   r   r@      s    zSentry.get_json_datac             C   sz   |dkr| j }t|j}y||}W n tk
r>   i }Y nX d|j|j|jf |j|j	|t
t|jt
t
|jdS )zT
        Exact method for getting http_info but with form data work around.
        Nz	%s://%s%s)urlZquery_stringmethodrF   headersenv)rA   r   urlsplitrG   r	   schemenetlocpathqueryrH   dictr   environr   )r(   r   rC   ZurlpartsrF   r   r   r   rB      s    
z#Sentry.get_http_info_with_retrieverc          
   O   s   d | _ tjr| jjtjj y| j| t W n4 t	k
rh } z| jj
t| W d d }~X Y nX y| j
| t W n4 t	k
r } z| jj
t| W d d }~X Y nX d S )N)r.   r   url_ruler#   transactionpushruleZhttp_contextrD   r,   logger	exceptionr   user_contextr>   )r(   r2   r3   er   r   r   before_request   s    $zSentry.before_requestc             O   s:   | j r| j |jd< | jj  tjr6| jjtjj	 |S )NzX-Sentry-ID)
r.   rI   r#   contextclearr   rR   rS   poprU   )r(   senderresponser2   r3   r   r   r   
after_request   s    zSentry.after_requestc       
      C   sd  |d k	r|| _ |d k	r|| _|d k	r,|| _n"| jd krN|rH|jrHd| _nd| _|d k	r\|| _|d k	rj|| _|d k	rx|| _| jst| j	|| j | _| jri }| jd k	r| j|d< t
| j| jd}	t|	f| |jj
dkr|j|	 tj| fdt
i| | jrt|j| j|_|| j tj| j|d | jrDtj| j|d t|dsVi |_| |jd< d S )	NFTexclude)r$   Zsentry_handler)r^   
extensionsZsentry)r   r$   r%   debugr&   r!   r"   r#   r   r   r
   r
   rV   	propagate
addHandlerr   sendSentryMiddlewareZwsgi_apprZ   r   connectr`   r   r4   r9   rb   )
r(   r   r   r!   r$   r"   r%   r&   r3   handlerr   r   r   r'      sJ    



zSentry.init_appc             O   s:   | j std| j j||}|r0| j || _nd | _|S )Nz5captureException called before application configured)r#   AssertionErrorr1   	get_identr.   )r(   r2   r3   resultr   r   r   r1   3  s    zSentry.captureExceptionc             O   s:   | j std| j j||}|r0| j || _nd | _|S )Nz3captureMessage called before application configured)r#   rj   captureMessagerk   r.   )r(   r2   r3   rl   r   r   r   rm   <  s    zSentry.captureMessagec             O   s   | j std| j j||S )Nz1user_context called before application configured)r#   rj   rX   )r(   r2   r3   r   r   r   rX   E  s    zSentry.user_contextc             O   s   | j std| j j||S )Nz1tags_context called before application configured)r#   rj   tags_context)r(   r2   r3   r   r   r   rn   I  s    zSentry.tags_contextc             O   s   | j std| j j||S )Nz2extra_context called before application configured)r#   rj   
extra_context)r(   r2   r3   r   r   r   ro   M  s    zSentry.extra_context)N)NNNNNN)__name__
__module____qualname____doc__r   r!   NOTSETr)   propertyr.   setterr4   r>   rD   r?   rA   r@   rB   rZ   r`   r'   r1   rm   rX   rn   ro   r   r   r   r   r   ;   s.   &,

  
7		r   )N)*rs   
__future__r   Zflask_loginr   ImportErrorr8   r!   Zblinkerflaskr   r   r   Z
flask.signalsr   r   Zwerkzeug.exceptionsr	   Z
raven.confr
   Z
raven.baser   Zraven.middlewarer   rg   Zraven.handlers.loggingr
   Zraven.utils.compatr   Zraven.utils.encodingr   Zraven.utils.wsgir   r   Zraven.utils.confr   	NamespaceZ
raven_signalssignalr   r   objectr   r   r   r   <module>   s.   


                                                                                                                          python3.7/site-packages/raven/contrib/__pycache__/paste.cpython-37.pyc                              0000644                 00000001201 15107360263 0021560 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                 @   s4   d Z ddlmZ ddlmZ ddlmZ dd ZdS )z
raven.contrib.paste
~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)Sentry)Clientc             K   s   t f |}t| |S )N)r   r   )ZappZglobal_confkwargsZclient r   D/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/paste.pysentry_filter_factory   s    
r   N)__doc__
__future__r   Zraven.middlewarer   Z
raven.baser   r   r   r   r   r   <module>   s                                                                                                                                                                                                                                                                                                                                                                                                  python3.7/site-packages/raven/contrib/__pycache__/async.cpython-37.pyc                              0000644                 00000002573 15107360263 0021576 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y[                 @   sD   d Z ddlmZ ddlZddlmZ ddlmZ G dd deZdS )z
raven.contrib.async
~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importN)Client)AsyncWorkerc                   s6   e Zd ZdZd	 fdd	Z fddZdd Z  ZS )
AsyncClientzI
    This client uses a single background thread to dispatch errors.
    Nc                s.   t dt |pt | _tt| j|| d S )NzCAsyncClient is deprecated. Use the threaded+http transport instead.)warningswarnDeprecationWarningr   workersuperr   __init__)selfr	   argskwargs)	__class__ D/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/async.pyr      s    zAsyncClient.__init__c                s   t t| jf | d S )N)r
   r   send)r   r   )r   r   r   	send_sync   s    zAsyncClient.send_syncc             K   s   | j j| jf| d S )N)r	   queuer   )r   r   r   r   r   r      s    zAsyncClient.send)N)__name__
__module____qualname____doc__r   r   r   
__classcell__r   r   )r   r   r      s   r   )	r   
__future__r   r   Z
raven.baser   Zraven.transport.threadedr   r   r   r   r   r   <module>   s
                                                                                                                                        python3.7/site-packages/raven/contrib/tornado/__pycache__/__init__.cpython-37.pyc                   0000644                 00000020107 15107360263 0023657 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y2                  @   st   d Z ddlmZ ddlmZ ddlmZ ddlmZm	Z	 ddl
m	Z ddlm
Z
 G dd	 d	e
ZG d
d deZdS )
z
raven.contrib.tornado
~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2012 by the Sentry Team, see AUTHORS for more details
:license: BSD, see LICENSE for more details
    )absolute_import)partial)ioloop)AsyncHTTPClient	HTTPError)r   )Clientc                   sN   e Zd ZdZ fddZdd ZdddZdd	d
Zdd Zdd
dZ	  Z
S )AsyncSentryClientz
    A mixin class that could be used along with request handlers to
    asynchronously send errors to sentry. The client also captures the
    information from the request handlers
    c                s$   | dd| _tt| j|| d S )N
validate_certT)popr	   superr   __init__)selfargskwargs)	__class__ O/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/tornado/__init__.pyr      s    zAsyncSentryClient.__init__c             O   s@   |   sdS | j||}| jf d|ddi|}|d |fS )a	  
        Takes the same arguments as the super function in :py:class:`Client`
        and extracts the keyword argument callback which will be called on
        asynchronous sending of the request

        :return: a 32-length string identifying this event
        NcallbackZevent_id)
is_enabledZ	build_msgsendget)r
   r   r   datafuturer   r   r   capture   s
    zAsyncSentryClient.captureNc             K   s   |  |}| j|||dS )zV
        Serializes the message and passes the payload onto ``send_encoded``.
        )auth_headerr   )encodeZsend_encoded)r
   r   r   r   messager   r   r   r   /   s    
zAsyncSentryClient.sendc             C   s`   |d kri }| j  s.| |}| | d S | j||||d}tj |t	| j
|| |S )N)urlr   headersr   )stateZ
should_trydecodeZ_log_failed_submission_send_remoter   ZIOLoopcurrentZ
add_futurer   _handle_result)r
   r   r   r   r   r   r   r   r   send_remote7   s    


zAsyncSentryClient.send_remotec          
   C   s   y|   W nr tk
rF } z| |}| ||| W d d }~X Y nD tk
r~ } z| |}| ||| W d d }~X Y nX | j  d S )N)resultr   r    Z_failed_send	Exceptionr   Zset_success)r
   r   r   r   er   r   r   r#   F   s    

 z AsyncSentryClient._handle_resultc             C   s&   |dkri }t  j||d||| jdS )z
        Initialise a Tornado AsyncClient and send the reuqest to the sentry
        server. If the callback is a callable, it will be called with the
        response.
        NZPOST)methodbodyr   r	   )r   Zfetchr	   )r
   r   r   r   r   r   r   r   r!   R   s
    
zAsyncSentryClient._send_remote)NN)NN)NN)__name__
__module____qualname____doc__r   r   r   r$   r#   r!   
__classcell__r   r   )r   r   r      s   

r   c                   sr   e Zd ZdZdd Zdd Zdd Zdd	 Zd
d Zdd
dZ	dddZ
dd Z fddZd fdd	Z
  ZS )SentryMixina  
    A mixin class that extracts information from the Request in a Request
    Handler to capture and send to sentry. This mixin class is designed to be
    used along with `tornado.web.RequestHandler`

    .. code-block:: python
        :emphasize-lines: 6

        class MyRequestHandler(SentryMixin, tornado.web.RequestHandler):
            def get(self):
                try:
                    fail()
                except Exception as e:
                    self.captureException()


    While the above example would result in sequential execution, an example
    for asynchronous use would be

    .. code-block:: python
        :emphasize-lines: 6

        class MyRequestHandler(SentryMixin, tornado.web.RequestHandler):

            @tornado.web.asynchronous
            @tornado.gen.engine
            def get(self):
                # Do something and record a message in sentry
                response = yield tornado.gen.Task(
                    self.captureMessage, "Did something really important"
                )
                self.write("Your request to do something important is done")
                self.finish()


    The mixin assumes that the application will have an attribute called
    `sentry_client`, which should be an instance of
    :py:class:`AsyncSentryClient`. This can be changed by implementing your
    own get_sentry_client method on your request handler.
    c             C   s   | j jS )z
        Returns the sentry client configured in the application. If you need
        to change the behaviour to do something else to get the client, then
        subclass this method
        )ZapplicationZ
sentry_client)r
   r   r   r   get_sentry_client   s    zSentryMixin.get_sentry_clientc          	   C   s<   d| j  | j j| j j| j j| j jddt| j jdiS )z
        Extracts the data required for 'sentry.interfaces.Http' from the
        current request being handled by the request handler

        :param return: A dictionary.
        requestCookieN)r   r(   r   Zquery_stringZcookiesr   )r1   Zfull_urlr(   r)   queryr   r   dict)r
   r   r   r   get_sentry_data_from_request   s    z(SentryMixin.get_sentry_data_from_requestc             C   s6   y|   }W n tk
r    i S X dd|r.dndiiS )z
        Data for sentry.interfaces.User

        Default implementation only sends `is_authenticated` by checking if
        `tornado.web.RequestHandler.get_current_user` tests postitively for on
        Truth calue testing
        userZis_authenticatedTF)Zget_current_userr&   )r
   r6   r   r   r   get_sentry_user_info   s    z SentryMixin.get_sentry_user_infoc             C   s   di iS )zf
        Subclass and implement this method if you need to send any extra
        information
        extrar   )r
   r   r   r   get_sentry_extra_info   s    z!SentryMixin.get_sentry_extra_infoc             C   s2   i }| |   | |   | |   |S )N)updater5   r7   r9   )r
   r   r   r   r   get_default_context   s
    zSentryMixin.get_default_contextNc             K   s`   |d kr|   }n.|   }t|tr0|| n||d d< |}|  }t||f d|i|S )Nr8   
extra_datar   )r;   
isinstancer4   r:   r0   getattr)r
   Z	call_namer   r   Zdefault_contextZclientr   r   r   _capture   s    

zSentryMixin._capturec             K   s   | j dd|i|S )NcaptureExceptionexc_info)r@   )r?   )r
   rA   r   r   r   r   r@      s    zSentryMixin.captureExceptionc             K   s   | j dd|i|S )NcaptureMessager   )rB   )r?   )r
   r   r   r   r   r   rB      s    zSentryMixin.captureMessagec                sL   t t| |||}t|tr6|jdk s2|jdkr6|S | j|||fd |S )zvOverride implementation to report all exceptions to sentry.
        log_exception() is added in Tornado v3.1.
        i  iW  )rA   )r   r/   
log_exceptionr=   WebHTTPErrorstatus_coder@   )r
   typvaluetbrv)r   r   r   rC      s
    zSentryMixin.log_exception  c                sj   t tt| dr$tt| j|f|S tt| j|f|}d|  krLdkrbn n| j|dd |S dS )zOverride implementation to report all exceptions to sentry, even
        after self.flush() or self.finish() is called, for pre-v3.1 Tornado.
        rC   i  iW  rA   )rA   N)hasattrr   r/   
send_errorr@   r   )r
   rE   r   rI   )r   r   r   rL      s    zSentryMixin.send_error)N)N)rJ   )r*   r+   r,   r-   r0   r5   r7   r9   r;   r?   r@   rB   rC   rL   r.   r   r   )r   r   r/   a   s   (


r/   N)r-   
__future__r   	functoolsr   Ztornador   Ztornado.httpclientr   r   Ztornado.webrD   Z
raven.baser   r   objectr/   r   r   r   r   <module>   s   N                                                                                                                                                                                                                                                                                                                                                                                                                                                         python3.7/site-packages/raven/contrib/tornado/__init__.py                                           0000644                 00000020062 15107360263 0017372 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.tornado
~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2012 by the Sentry Team, see AUTHORS for more details
:license: BSD, see LICENSE for more details
"""
from __future__ import absolute_import

from functools import partial

from tornado import ioloop
from tornado.httpclient import AsyncHTTPClient, HTTPError
from tornado.web import HTTPError as WebHTTPError

from raven.base import Client


class AsyncSentryClient(Client):
    """
    A mixin class that could be used along with request handlers to
    asynchronously send errors to sentry. The client also captures the
    information from the request handlers
    """

    def __init__(self, *args, **kwargs):
        self.validate_cert = kwargs.pop('validate_cert', True)
        super(AsyncSentryClient, self).__init__(*args, **kwargs)

    def capture(self, *args, **kwargs):
        """
        Takes the same arguments as the super function in :py:class:`Client`
        and extracts the keyword argument callback which will be called on
        asynchronous sending of the request

        :return: a 32-length string identifying this event
        """
        if not self.is_enabled():
            return

        data = self.build_msg(*args, **kwargs)

        future = self.send(callback=kwargs.get('callback', None), **data)

        return (data['event_id'], future)

    def send(self, auth_header=None, callback=None, **data):
        """
        Serializes the message and passes the payload onto ``send_encoded``.
        """
        message = self.encode(data)

        return self.send_encoded(message, auth_header=auth_header, callback=callback)

    def send_remote(self, url, data, headers=None, callback=None):
        if headers is None:
            headers = {}

        if not self.state.should_try():
            data = self.decode(data)
            self._log_failed_submission(data)
            return

        future = self._send_remote(
            url=url, data=data, headers=headers, callback=callback
        )
        ioloop.IOLoop.current().add_future(future, partial(self._handle_result, url, data))
        return future

    def _handle_result(self, url, data, future):
        try:
            future.result()
        except HTTPError as e:
            data = self.decode(data)
            self._failed_send(e, url, data)
        except Exception as e:
            data = self.decode(data)
            self._failed_send(e, url, data)
        else:
            self.state.set_success()

    def _send_remote(self, url, data, headers=None, callback=None):
        """
        Initialise a Tornado AsyncClient and send the reuqest to the sentry
        server. If the callback is a callable, it will be called with the
        response.
        """
        if headers is None:
            headers = {}

        return AsyncHTTPClient().fetch(
            url, callback, method="POST", body=data, headers=headers,
            validate_cert=self.validate_cert
        )


class SentryMixin(object):
    """
    A mixin class that extracts information from the Request in a Request
    Handler to capture and send to sentry. This mixin class is designed to be
    used along with `tornado.web.RequestHandler`

    .. code-block:: python
        :emphasize-lines: 6

        class MyRequestHandler(SentryMixin, tornado.web.RequestHandler):
            def get(self):
                try:
                    fail()
                except Exception as e:
                    self.captureException()


    While the above example would result in sequential execution, an example
    for asynchronous use would be

    .. code-block:: python
        :emphasize-lines: 6

        class MyRequestHandler(SentryMixin, tornado.web.RequestHandler):

            @tornado.web.asynchronous
            @tornado.gen.engine
            def get(self):
                # Do something and record a message in sentry
                response = yield tornado.gen.Task(
                    self.captureMessage, "Did something really important"
                )
                self.write("Your request to do something important is done")
                self.finish()


    The mixin assumes that the application will have an attribute called
    `sentry_client`, which should be an instance of
    :py:class:`AsyncSentryClient`. This can be changed by implementing your
    own get_sentry_client method on your request handler.
    """

    def get_sentry_client(self):
        """
        Returns the sentry client configured in the application. If you need
        to change the behaviour to do something else to get the client, then
        subclass this method
        """
        return self.application.sentry_client

    def get_sentry_data_from_request(self):
        """
        Extracts the data required for 'sentry.interfaces.Http' from the
        current request being handled by the request handler

        :param return: A dictionary.
        """
        return {
            'request': {
                'url': self.request.full_url(),
                'method': self.request.method,
                'data': self.request.body,
                'query_string': self.request.query,
                'cookies': self.request.headers.get('Cookie', None),
                'headers': dict(self.request.headers),
            }
        }

    def get_sentry_user_info(self):
        """
        Data for sentry.interfaces.User

        Default implementation only sends `is_authenticated` by checking if
        `tornado.web.RequestHandler.get_current_user` tests postitively for on
        Truth calue testing
        """
        try:
            user = self.get_current_user()
        except Exception:
            return {}
        return {
            'user': {
                'is_authenticated': True if user else False
            }
        }

    def get_sentry_extra_info(self):
        """
        Subclass and implement this method if you need to send any extra
        information
        """
        return {
            'extra': {
            }
        }

    def get_default_context(self):
        data = {}

        # Update request data
        data.update(self.get_sentry_data_from_request())

        # update user data
        data.update(self.get_sentry_user_info())

        # Update extra data
        data.update(self.get_sentry_extra_info())

        return data

    def _capture(self, call_name, data=None, **kwargs):
        if data is None:
            data = self.get_default_context()
        else:
            default_context = self.get_default_context()
            if isinstance(data, dict):
                default_context.update(data)
            else:
                default_context['extra']['extra_data'] = data
            data = default_context

        client = self.get_sentry_client()

        return getattr(client, call_name)(data=data, **kwargs)

    def captureException(self, exc_info=None, **kwargs):
        return self._capture('captureException', exc_info=exc_info, **kwargs)

    def captureMessage(self, message, **kwargs):
        return self._capture('captureMessage', message=message, **kwargs)

    def log_exception(self, typ, value, tb):
        """Override implementation to report all exceptions to sentry.
        log_exception() is added in Tornado v3.1.
        """
        rv = super(SentryMixin, self).log_exception(typ, value, tb)
        # Do not capture tornado.web.HTTPErrors outside the 500 range.
        if isinstance(value, WebHTTPError) and (value.status_code < 500 or value.status_code > 599):
            return rv
        self.captureException(exc_info=(typ, value, tb))
        return rv

    def send_error(self, status_code=500, **kwargs):
        """Override implementation to report all exceptions to sentry, even
        after self.flush() or self.finish() is called, for pre-v3.1 Tornado.
        """
        if hasattr(super(SentryMixin, self), 'log_exception'):
            return super(SentryMixin, self).send_error(status_code, **kwargs)
        else:
            rv = super(SentryMixin, self).send_error(status_code, **kwargs)
            if 500 <= status_code <= 599:
                self.captureException(exc_info=kwargs.get('exc_info'))
            return rv
                                                                                                                                                                                                                                                                                                                                                                                                                                                                              python3.7/site-packages/raven/contrib/bottle/__pycache__/__init__.cpython-37.pyc                    0000644                 00000005351 15107360263 0023506 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y
                 @   s\   d Z ddlmZ ddlZddlmZ ddlmZ ddlm	Z	 ddl
mZ G dd	 d	eZ
dS )
z
raven.contrib.bottle
~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2013 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importN)request)
setup_logging)get_data_from_request)
SentryHandlerc               @   s:   e Zd ZdZdddZdd Zdd Zd	d
 Zdd Zd
S )Sentryav  
    Bottle application for Sentry.

    >>> sentry = Sentry(app, client)

    Automatically configure logging::

    >>> sentry = Sentry(app, client, logging=True)

    Capture an exception::

    >>> try:
    >>>     1 / 0
    >>> except ZeroDivisionError:
    >>>     sentry.captureException()

    Capture a message::

    >>> sentry.captureMessage('hello, world!')
    Fc             C   s2   || _ || _|| _| jr&tt| j | | j _d S )N)appclientloggingr   r   Zsentry)selfr   r	   r
    r   N/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/bottle/__init__.py__init__)   s    zSentry.__init__c             O   s&   | j j|dttd| jid d S )Nexc_infor   )r   dataextra)r	   captureExceptiongetr   r   r   )r   argskwargsr   r   r
   handle_exception1   s    zSentry.handle_exceptionc                sH   d fdd	}y  ||S  tk
rB    jt d  Y nX d S )Nc                s    |d k	r j |d | ||S )N)r   )r   )statusheadersr   )r   start_responser   r
   session_start_response;   s    z/Sentry.__call__.<locals>.session_start_response)r   )N)r   	Exceptionr   sysr   )r   environr   r   r   )r   r   r
   __call__:   s    zSentry.__call__c             O   sT   | j std|d}|d krFytt|d< W n tk
rD   Y nX | j j||S )Nz5captureException called before application configuredr   )r	   AssertionErrorr   r   r   RuntimeErrorr   )r   r   r   r   r   r   r
   r   H   s    
zSentry.captureExceptionc             O   sT   | j std|d}|d krFytt|d< W n tk
rD   Y nX | j j||S )Nz3captureMessage called before application configuredr   )r	   r   r   r   r   r    captureMessage)r   r   r   r   r   r   r
   r!   S   s    
zSentry.captureMessageN)F)	__name__
__module____qualname____doc__r   r   r   r   r!   r   r   r   r
   r      s   
	r   )r%   
__future__r   r   bottler   Z
raven.confr   Zraven.contrib.bottle.utilsr   Zraven.handlers.loggingr   objectr   r   r   r   r
   <module>   s                                                                                                                                                                                                                                                                                          python3.7/site-packages/raven/contrib/bottle/__pycache__/utils.cpython-37.pyc                       0000644                 00000002303 15107360263 0023101 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                 @   sJ   d Z ddlmZ ddlZddlmZ ddlmZmZ e	e
Zdd ZdS )z
raven.contrib.bottle.utils
~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importN)urlparse)get_headersget_environc          	      s   t | j}y"| jj t fdd D }W n tk
rF   i }Y nX dd|j|j|jf |j	| j
|tt| jtt
| jdi}|S )Nc                s   g | ]}| | d  fqS ) ).0k)	form_dictr   K/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/bottle/utils.py
<listcomp>   s    z)get_data_from_request.<locals>.<listcomp>requestz	%s://%s%s)urlZquery_stringmethoddataheadersenv)r   urlsplitr   Zformsdict	Exceptionschemenetlocpathqueryr   r   environr   )r
   ZurlpartsZformdatar   r   )r
   r   get_data_from_request   s    
r   )
__doc__
__future__r   loggingZraven.utils.compatr   Zraven.utils.wsgir   r   	getLogger__name__loggerr   r   r   r   r   <module>   s   
                                                                                                                                                                                                                                                                                                                             python3.7/site-packages/raven/contrib/bottle/utils.py                                               0000644                 00000002023 15107360263 0016613 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.bottle.utils
~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""
from __future__ import absolute_import

import logging
from raven.utils.compat import urlparse

from raven.utils.wsgi import get_headers, get_environ

logger = logging.getLogger(__name__)


def get_data_from_request(request):
    urlparts = urlparse.urlsplit(request.url)

    try:
        form_dict = request.forms.dict
        # we only are about the most recent one
        formdata = dict([(k, form_dict[k][-1]) for k in form_dict])
    except Exception:
        formdata = {}

    data = {
        'request': {
            'url': '%s://%s%s' % (urlparts.scheme, urlparts.netloc, urlparts.path),
            'query_string': urlparts.query,
            'method': request.method,
            'data': formdata,
            'headers': dict(get_headers(request.environ)),
            'env': dict(get_environ(request.environ)),
        }
    }

    return data
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             python3.7/site-packages/raven/contrib/bottle/__init__.py                                            0000644                 00000005311 15107360263 0017215 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.bottle
~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2013 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""

from __future__ import absolute_import

import sys

from bottle import request
from raven.conf import setup_logging
from raven.contrib.bottle.utils import get_data_from_request
from raven.handlers.logging import SentryHandler


class Sentry(object):
    """
    Bottle application for Sentry.

    >>> sentry = Sentry(app, client)

    Automatically configure logging::

    >>> sentry = Sentry(app, client, logging=True)

    Capture an exception::

    >>> try:
    >>>     1 / 0
    >>> except ZeroDivisionError:
    >>>     sentry.captureException()

    Capture a message::

    >>> sentry.captureMessage('hello, world!')
    """

    def __init__(self, app, client, logging=False):
        self.app = app
        self.client = client
        self.logging = logging
        if self.logging:
            setup_logging(SentryHandler(self.client))
        self.app.sentry = self

    def handle_exception(self, *args, **kwargs):
        self.client.captureException(
            exc_info=kwargs.get('exc_info'),
            data=get_data_from_request(request),
            extra={
                'app': self.app,
            },
        )

    def __call__(self, environ, start_response):
        def session_start_response(status, headers, exc_info=None):
            if exc_info is not None:
                self.handle_exception(exc_info=exc_info)
            return start_response(status, headers, exc_info)

        try:
            return self.app(environ, session_start_response)
        # catch ANY exception that goes through...
        except Exception:
            self.handle_exception(exc_info=sys.exc_info())
            # re-raise the exception to let parent handlers deal with it
            raise

    def captureException(self, *args, **kwargs):
        assert self.client, 'captureException called before application configured'
        data = kwargs.get('data')
        if data is None:
            try:
                kwargs['data'] = get_data_from_request(request)
            except RuntimeError:
                # app is probably not configured yet
                pass
        return self.client.captureException(*args, **kwargs)

    def captureMessage(self, *args, **kwargs):
        assert self.client, 'captureMessage called before application configured'
        data = kwargs.get('data')
        if data is None:
            try:
                kwargs['data'] = get_data_from_request(request)
            except RuntimeError:
                # app is probably not configured yet
                pass
        return self.client.captureMessage(*args, **kwargs)
                                                                                                                                                                                                                                                                                                                       python3.7/site-packages/raven/contrib/pylons/__pycache__/__init__.cpython-37.pyc                    0000644                 00000002306 15107360263 0023536 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Yv                 @   sD   d Z ddlmZ ddlmZ ddlmZ dd ZG dd deZd	S )
z
raven.contrib.pylons
~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)Sentry)Clientc             C   s   |  |}|sd S | S )N)getsplit)configZsettingvalue r	   N/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/pylons/__init__.pylist_from_setting   s    
r   c                   s    e Zd Zef fdd	Z  ZS )r   c                sF   || d| d| dt|dt|dd}tt| || d S )Nz
sentry.dsnzsentry.namezsentry.sitezsentry.include_pathszsentry.exclude_paths)ZdsnnamesiteZ
include_pathsZ
exclude_paths)r   r   superr   __init__)selfZappr   Z
client_clsZclient)	__class__r	   r
   r      s    zSentry.__init__)__name__
__module____qualname__r   r   
__classcell__r	   r	   )r   r
   r      s   r   N)	__doc__
__future__r   Zraven.middlewarer   Z
MiddlewareZ
raven.baser   r   r	   r	   r	   r
   <module>   s
                                                                                                                                                                                                                                                                                                                             python3.7/site-packages/raven/contrib/pylons/__init__.py                                            0000644                 00000001566 15107360263 0017260 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.pylons
~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""
from __future__ import absolute_import

from raven.middleware import Sentry as Middleware
from raven.base import Client


def list_from_setting(config, setting):
    value = config.get(setting)
    if not value:
        return None
    return value.split()


class Sentry(Middleware):
    def __init__(self, app, config, client_cls=Client):
        client = client_cls(
            dsn=config.get('sentry.dsn'),
            name=config.get('sentry.name'),
            site=config.get('sentry.site'),
            include_paths=list_from_setting(config, 'sentry.include_paths'),
            exclude_paths=list_from_setting(config, 'sentry.exclude_paths'),
        )
        super(Sentry, self).__init__(app, client)
                                                                                                                                          python3.7/site-packages/raven/contrib/django/apps.py                                                0000644                 00000000464 15107360263 0016376 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # -*- coding: utf-8 -*-
from __future__ import absolute_import

from django.apps import AppConfig


class RavenConfig(AppConfig):
    name = 'raven.contrib.django'
    label = 'raven_contrib_django'
    verbose_name = 'Raven'

    def ready(self):
        from .models import initialize
        initialize()
                                                                                                                                                                                                            python3.7/site-packages/raven/contrib/django/logging.py                                             0000644                 00000000722 15107360264 0017057 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.django.logging
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""

from __future__ import absolute_import

import warnings

warnings.warn('raven.contrib.django.logging is deprecated. Use raven.contrib.django.handlers instead.', DeprecationWarning)

from raven.contrib.django.handlers import SentryHandler  # NOQA

__all__ = ('SentryHandler',)
                                              python3.7/site-packages/raven/contrib/django/client.py                                              0000644                 00000024211 15107360264 0016706 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # -*- coding: utf-8 -*-
"""
raven.contrib.django.client
~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""

from __future__ import absolute_import

import time
import logging

from django import VERSION as DJANGO_VERSION
from django.conf import settings
from django.core.exceptions import SuspiciousOperation
from django.http import HttpRequest
from django.template import TemplateSyntaxError
from django.utils.datastructures import MultiValueDict

try:
    # support Django 1.9
    from django.template.base import Origin
except ImportError:
    # backward compatibility
    from django.template.loader import LoaderOrigin as Origin

from raven.base import Client
from raven.contrib.django.utils import get_data_from_template, get_host
from raven.contrib.django.middleware import SentryMiddleware
from raven.utils.compat import string_types, binary_type, iterlists
from raven.contrib.django.resolver import RouteResolver
from raven.utils.wsgi import get_headers, get_environ, get_client_ip
from raven.utils import once
from raven import breadcrumbs

__all__ = ('DjangoClient',)


if DJANGO_VERSION < (1, 10):
    def is_authenticated(request_user):
        return request_user.is_authenticated()
else:
    def is_authenticated(request_user):
        return request_user.is_authenticated


class _FormatConverter(object):

    def __init__(self, param_mapping):
        self.param_mapping = param_mapping
        self.params = []

    def __getitem__(self, val):
        self.params.append(self.param_mapping.get(val))
        return '%s'


def format_sql(sql, params):
    rv = []

    if isinstance(params, dict):
        conv = _FormatConverter(params)
        if params:
            sql = sql % conv
            params = conv.params
        else:
            params = ()

    for param in params or ():
        if param is None:
            rv.append('NULL')
        elif isinstance(param, string_types):
            if isinstance(param, binary_type):
                param = param.decode('utf-8', 'replace')
            if len(param) > 256:
                param = param[:256] + u'…'
            rv.append("'%s'" % param.replace("'", "''"))
        else:
            rv.append(repr(param))

    return sql, rv


@once
def install_sql_hook():
    """If installed this causes Django's queries to be captured."""
    try:
        from django.db.backends.utils import CursorWrapper
    except ImportError:
        from django.db.backends.util import CursorWrapper

    try:
        real_execute = CursorWrapper.execute
        real_executemany = CursorWrapper.executemany
    except AttributeError:
        # XXX(mitsuhiko): On some very old django versions (<1.6) this
        # trickery would have to look different but I can't be bothered.
        return

    def record_sql(vendor, alias, start, duration, sql, params):
        def processor(data):
            real_sql, real_params = format_sql(sql, params)
            if real_params:
                real_sql = real_sql % tuple(real_params)
            # maybe category to 'django.%s.%s' % (vendor, alias or
            #   'default') ?
            data.update({
                'message': real_sql,
                'category': 'query',
            })
        breadcrumbs.record(processor=processor)

    def record_many_sql(vendor, alias, start, sql, param_list):
        duration = time.time() - start
        for params in param_list:
            record_sql(vendor, alias, start, duration, sql, params)

    def execute(self, sql, params=None):
        start = time.time()
        try:
            return real_execute(self, sql, params)
        finally:
            record_sql(self.db.vendor, getattr(self.db, 'alias', None),
                       start, time.time() - start, sql, params)

    def executemany(self, sql, param_list):
        start = time.time()
        try:
            return real_executemany(self, sql, param_list)
        finally:
            record_many_sql(self.db.vendor, getattr(self.db, 'alias', None),
                            start, sql, param_list)

    CursorWrapper.execute = execute
    CursorWrapper.executemany = executemany
    breadcrumbs.ignore_logger('django.db.backends')


class DjangoClient(Client):
    logger = logging.getLogger('sentry.errors.client.django')
    resolver = RouteResolver()

    def __init__(self, *args, **kwargs):
        install_sql_hook = kwargs.pop('install_sql_hook', True)
        Client.__init__(self, *args, **kwargs)
        if install_sql_hook:
            self.install_sql_hook()

    def install_sql_hook(self):
        install_sql_hook()

    def get_user_info(self, request):

        user_info = {
            'ip_address': get_client_ip(request.META),
        }
        user = getattr(request, 'user', None)
        if user is None:
            return user_info

        try:
            authenticated = is_authenticated(user)
            if not authenticated:
                return user_info
            user_info['id'] = user.pk

            if hasattr(user, 'email'):
                user_info['email'] = user.email

            if hasattr(user, 'get_username'):
                user_info['username'] = user.get_username()
            elif hasattr(user, 'username'):
                user_info['username'] = user.username
        except Exception:
            # We expect that user objects can be somewhat broken at times
            # and try to just handle as much as possible and ignore errors
            # as good as possible here.
            pass

        return user_info

    def get_data_from_request(self, request):
        result = {}

        result['user'] = self.get_user_info(request)

        try:
            uri = request.build_absolute_uri()
        except SuspiciousOperation:
            # attempt to build a URL for reporting as Django won't allow us to
            # use get_host()
            if request.is_secure():
                scheme = 'https'
            else:
                scheme = 'http'
            host = get_host(request)
            uri = '%s://%s%s' % (scheme, host, request.path)

        if request.method not in ('GET', 'HEAD'):
            try:
                data = request.body
            except Exception:
                try:
                    data = request.raw_post_data
                except Exception:
                    # assume we had a partial read.
                    try:
                        data = request.POST or '<unavailable>'
                    except Exception:
                        data = '<unavailable>'
                    else:
                        if isinstance(data, MultiValueDict):
                            data = dict(
                                (k, v[0] if len(v) == 1 else v)
                                for k, v in iterlists(data))
        else:
            data = None

        environ = request.META

        result.update({
            'request': {
                'method': request.method,
                'url': uri,
                'query_string': request.META.get('QUERY_STRING'),
                'data': data,
                'cookies': dict(request.COOKIES),
                'headers': dict(get_headers(environ)),
                'env': dict(get_environ(environ)),
            }
        })

        return result

    def build_msg(self, *args, **kwargs):
        data = super(DjangoClient, self).build_msg(*args, **kwargs)

        for frame in self._iter_frames(data):
            module = frame.get('module')
            if not module:
                continue

            if module.startswith('django.'):
                frame['in_app'] = False

        if not self.site and 'django.contrib.sites' in settings.INSTALLED_APPS:
            try:
                from django.contrib.sites.models import Site
                site = Site.objects.get_current()
                site_name = site.name or site.domain
                data['tags'].setdefault('site', site_name)
            except Exception:
                # Database error? Fallback to the id
                try:
                    data['tags'].setdefault('site', settings.SITE_ID)
                except AttributeError:
                    # SITE_ID wasn't set, so just ignore
                    pass

        return data

    def capture(self, event_type, request=None, **kwargs):
        if 'data' not in kwargs:
            kwargs['data'] = data = {}
        else:
            data = kwargs['data']

        if request is None:
            request = getattr(SentryMiddleware.thread, 'request', None)

        is_http_request = isinstance(request, HttpRequest)
        if is_http_request:
            data.update(self.get_data_from_request(request))

        if kwargs.get('exc_info'):
            exc_value = kwargs['exc_info'][1]
            # As of r16833 (Django) all exceptions may contain a
            # ``django_template_source`` attribute (rather than the legacy
            # ``TemplateSyntaxError.source`` check) which describes
            # template information.  As of Django 1.9 or so the new
            # template debug thing showed up.
            if hasattr(exc_value, 'django_template_source') or \
               ((isinstance(exc_value, TemplateSyntaxError) and
                isinstance(getattr(exc_value, 'source', None),
                           (tuple, list)) and
                isinstance(exc_value.source[0], Origin))) or \
               hasattr(exc_value, 'template_debug'):
                source = getattr(exc_value, 'django_template_source',
                                 getattr(exc_value, 'source', None))
                debug = getattr(exc_value, 'template_debug', None)
                if source is None:
                    self.logger.info('Unable to get template source from exception')
                data.update(get_data_from_template(source, debug))

        result = super(DjangoClient, self).capture(event_type, **kwargs)

        if is_http_request and result:
            # attach the sentry object to the request
            request.sentry = {
                'project_id': data.get('project', self.remote.project),
                'id': result,
            }

        return result

    def get_transaction_from_request(self, request):
        return self.resolver.resolve(request.path)
                                                                                                                                                                                                                                                                                                                                                                                       python3.7/site-packages/raven/contrib/django/models.py                                              0000644                 00000022305 15107360264 0016715 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.django.models
~~~~~~~~~~~~~~~~~~~~~~~~~~~

Acts as an implicit hook for Django installs.

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""
# flake8: noqa

from __future__ import absolute_import, unicode_literals

import logging
import sys
import warnings

import django
from django.conf import settings
from django.core.signals import got_request_exception, request_started
from threading import Lock

from raven.utils.conf import convert_options
from raven.utils.compat import PY2, binary_type, text_type
from raven.utils.imports import import_string

logger = logging.getLogger('sentry.errors.client')


def get_installed_apps():
    """
    Modules in settings.INSTALLED_APPS as a set.
    """
    return set(settings.INSTALLED_APPS)


_client = (None, None)


class ProxyClient(object):
    """
    A proxy which represents the currently client at all times.
    """
    # introspection support:
    __members__ = property(lambda x: x.__dir__())

    # Need to pretend to be the wrapped class, for the sake of objects that care
    # about this (especially in equality tests)
    __class__ = property(lambda x: get_client().__class__)

    __dict__ = property(lambda o: get_client().__dict__)

    __repr__ = lambda x: repr(get_client())
    __getattr__ = lambda x, o: getattr(get_client(), o)
    __setattr__ = lambda x, o, v: setattr(get_client(), o, v)
    __delattr__ = lambda x, o: delattr(get_client(), o)

    __lt__ = lambda x, o: get_client() < o
    __le__ = lambda x, o: get_client() <= o
    __eq__ = lambda x, o: get_client() == o
    __ne__ = lambda x, o: get_client() != o
    __gt__ = lambda x, o: get_client() > o
    __ge__ = lambda x, o: get_client() >= o
    if PY2:
        __cmp__ = lambda x, o: cmp(get_client(), o)  # NOQA
    __hash__ = lambda x: hash(get_client())
    # attributes are currently not callable
    # __call__ = lambda x, *a, **kw: get_client()(*a, **kw)
    __nonzero__ = lambda x: bool(get_client())
    __len__ = lambda x: len(get_client())
    __getitem__ = lambda x, i: get_client()[i]
    __iter__ = lambda x: iter(get_client())
    __contains__ = lambda x, i: i in get_client()
    __getslice__ = lambda x, i, j: get_client()[i:j]
    __add__ = lambda x, o: get_client() + o
    __sub__ = lambda x, o: get_client() - o
    __mul__ = lambda x, o: get_client() * o
    __floordiv__ = lambda x, o: get_client() // o
    __mod__ = lambda x, o: get_client() % o
    __divmod__ = lambda x, o: get_client().__divmod__(o)
    __pow__ = lambda x, o: get_client() ** o
    __lshift__ = lambda x, o: get_client() << o
    __rshift__ = lambda x, o: get_client() >> o
    __and__ = lambda x, o: get_client() & o
    __xor__ = lambda x, o: get_client() ^ o
    __or__ = lambda x, o: get_client() | o
    __div__ = lambda x, o: get_client().__div__(o)
    __truediv__ = lambda x, o: get_client().__truediv__(o)
    __neg__ = lambda x: -(get_client())
    __pos__ = lambda x: +(get_client())
    __abs__ = lambda x: abs(get_client())
    __invert__ = lambda x: ~(get_client())
    __complex__ = lambda x: complex(get_client())
    __int__ = lambda x: int(get_client())
    if PY2:
        __long__ = lambda x: long(get_client())  # NOQA
    __float__ = lambda x: float(get_client())
    __str__ = lambda x: binary_type(get_client())
    __unicode__ = lambda x: text_type(get_client())
    __oct__ = lambda x: oct(get_client())
    __hex__ = lambda x: hex(get_client())
    __index__ = lambda x: get_client().__index__()
    __coerce__ = lambda x, o: x.__coerce__(x, o)
    __enter__ = lambda x: x.__enter__()
    __exit__ = lambda x, *a, **kw: x.__exit__(*a, **kw)

client = ProxyClient()


def get_client(client=None, reset=False):
    global _client

    tmp_client = client is not None
    if not tmp_client:
        client = getattr(settings, 'SENTRY_CLIENT', 'raven.contrib.django.DjangoClient')

    if _client[0] != client or reset:
        options = convert_options(
            settings,
            defaults={
                'include_paths': get_installed_apps(),
            },
        )

        try:
            Client = import_string(client)
        except ImportError:
            logger.exception('Failed to import client: %s', client)
            if not _client[1]:
                # If there is no previous client, set the default one.
                client = 'raven.contrib.django.DjangoClient'
                _client = (client, get_client(client))
        else:
            instance = Client(**options)
            if not tmp_client:
                _client = (client, instance)
            return instance
    return _client[1]


def sentry_exception_handler(request=None, **kwargs):
    try:
        client.captureException(exc_info=sys.exc_info(), request=request)
    except Exception as exc:
        try:
            logger.exception('Unable to process log entry: %s' % (exc,))
        except Exception as exc:
            warnings.warn('Unable to process log entry: %s' % (exc,))


class SentryDjangoHandler(object):
    def __init__(self, client=client):
        self.client = client

        try:
            import celery
        except ImportError:
            self.has_celery = False
        else:
            self.has_celery = celery.VERSION >= (2, 5)

        self.celery_handler = None

    def install_celery(self):
        from raven.contrib.celery import (
            SentryCeleryHandler, register_logger_signal
        )

        ignore_expected = getattr(settings,
                                  'SENTRY_CELERY_IGNORE_EXPECTED',
                                  False)

        self.celery_handler = SentryCeleryHandler(client,
                                                  ignore_expected=ignore_expected)\
                                                  .install()

        # try:
        #     ga = lambda x, d=None: getattr(settings, 'SENTRY_%s' % x, d)
        #     options = getattr(settings, 'RAVEN_CONFIG', {})
        #     loglevel = options.get('celery_loglevel',
        #                            ga('CELERY_LOGLEVEL', logging.ERROR))

        #     register_logger_signal(client, loglevel=loglevel)
        # except Exception:
        #     logger.exception('Failed to install Celery error handler')

    def install(self):
        request_started.connect(self.before_request, weak=False)
        got_request_exception.connect(self.exception_handler, weak=False)

        if self.has_celery:
            try:
                self.install_celery()
            except Exception:
                logger.exception('Failed to install Celery error handler')

    def uninstall(self):
        request_started.disconnect(self.before_request)
        got_request_exception.disconnect(self.exception_handler)

        if self.celery_handler:
            self.celery_handler.uninstall()

    def exception_handler(self, request=None, **kwargs):
        try:
            self.client.captureException(exc_info=sys.exc_info(), request=request)
        except Exception as exc:
            try:
                logger.exception('Unable to process log entry: %s' % (exc,))
            except Exception as exc:
                warnings.warn('Unable to process log entry: %s' % (exc,))

    def before_request(self, *args, **kwargs):
        self.client.context.activate()


def register_serializers():
    # force import so serializers can call register
    import raven.contrib.django.serializers  # NOQA


def install_middleware(middleware_name, lookup_names=None):
    """
    Install specified middleware
    """
    if lookup_names is None:
        lookup_names = (middleware_name,)
    # default settings.MIDDLEWARE is None
    middleware_attr = 'MIDDLEWARE' if getattr(settings,
                                              'MIDDLEWARE',
                                              None) is not None \
        else 'MIDDLEWARE_CLASSES'
    # make sure to get an empty tuple when attr is None
    middleware = getattr(settings, middleware_attr, ()) or ()
    if set(lookup_names).isdisjoint(set(middleware)):
        setattr(settings,
                middleware_attr,
                type(middleware)((middleware_name,)) + middleware)


_setup_lock = Lock()

_initialized = False

def initialize():
    global _initialized

    with _setup_lock:
        if _initialized:
            return

        # mark this as initialized immediatley to avoid recursive import issues
        _initialized = True

        try:
            register_serializers()
            install_middleware(
                'raven.contrib.django.middleware.SentryMiddleware',
                (
                    'raven.contrib.django.middleware.SentryMiddleware',
                    'raven.contrib.django.middleware.SentryLogMiddleware'))
            install_middleware(
                'raven.contrib.django.middleware.DjangoRestFrameworkCompatMiddleware')

            # XXX(dcramer): maybe this setting should disable ALL of this?
            if not getattr(settings, 'DISABLE_SENTRY_INSTRUMENTATION', False):
                handler = SentryDjangoHandler()
                handler.install()

            # instantiate client so hooks get registered
            get_client()  # NOQA
        except Exception:
            _initialized = False

# Django 1.8 uses ``raven.contrib.apps.RavenConfig``
if django.VERSION < (1, 7, 0):
    initialize()
                                                                                                                                                                                                                                                                                                                           python3.7/site-packages/raven/contrib/django/celery/models.py                                       0000644                 00000001007 15107360264 0020174 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.django.celery.models
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""
from __future__ import absolute_import

from django.conf import settings
from django.core.exceptions import ImproperlyConfigured

if 'djcelery' not in settings.INSTALLED_APPS:
    raise ImproperlyConfigured(
        "Put 'djcelery' in your INSTALLED_APPS setting in order to use the "
        "sentry celery client.")
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         python3.7/site-packages/raven/contrib/django/celery/__pycache__/__init__.cpython-37.pyc             0000644                 00000002736 15107360264 0024747 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                 @   s   d Z ddlmZ ddlmZ yddlmZ W n  ek
rL   ddlmZ Y nX G dd deZ	eddd	d
 Z
edddd Zd
S )z
raven.contrib.django.celery
~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)DjangoClient)taskc               @   s   e Zd Zdd Zdd ZdS )CeleryClientc             C   s
   t |S )N)send_raw_integrateddelay)selfkwargs r
   U/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/celery/__init__.pysend_integrated   s    zCeleryClient.send_integratedc             O   s   t j||S )N)send_rawr   )r   argsr	   r
   r
   r   send_encoded   s    zCeleryClient.send_encodedN)__name__
__module____qualname__r   r   r
   r
   r
   r   r      s   r   Zsentry)Zrouting_keyc             C   s"   ddl m} tt| |  d S )Nr   )
get_client)raven.contrib.django.modelsr   superr   r   )r	   r   r
   r
   r   r      s    r   c              O   s$   ddl m} tt| j| | d S )Nr   )r   )r   r   r   r   r   )r   r	   r   r
   r
   r   r
      s    r
   N)__doc__
__future__r   Zraven.contrib.django.clientr   Zcelery.taskr   ImportErrorZcelery.decoratorsr   r   r
   r
   r
   r
   r   <module>   s                                     python3.7/site-packages/raven/contrib/django/celery/__pycache__/tasks.cpython-37.pyc                0000644                 00000000745 15107360264 0024333 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y%                 @   s    d Z ddlmZ ddlmZ dS )z
raven.contrib.django.celery.tasks
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)clientN)__doc__
__future__r   Zraven.contrib.django.modelsr    r   r   R/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/celery/tasks.py<module>   s                              python3.7/site-packages/raven/contrib/django/celery/__pycache__/models.cpython-37.pyc               0000644                 00000001251 15107360264 0024462 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                 @   s>   d Z ddlmZ ddlmZ ddlmZ dejkr:eddS )z
raven.contrib.django.celery.models
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)settings)ImproperlyConfiguredZdjceleryzWPut 'djcelery' in your INSTALLED_APPS setting in order to use the sentry celery client.N)__doc__
__future__r   Zdjango.confr   Zdjango.core.exceptionsr   ZINSTALLED_APPS r   r   S/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/celery/models.py<module>   s   
                                                                                                                                                                                                                                                                                                                                                       python3.7/site-packages/raven/contrib/django/celery/tasks.py                                        0000644                 00000000445 15107360264 0020043 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.django.celery.tasks
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""
from __future__ import absolute_import

from raven.contrib.django.models import client  # NOQA
                                                                                                                                                                                                                           python3.7/site-packages/raven/contrib/django/celery/__init__.py                                     0000644                 00000001706 15107360264 0020456 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.django.celery
~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""
from __future__ import absolute_import

from raven.contrib.django.client import DjangoClient
try:
    from celery.task import task
except ImportError:
    from celery.decorators import task  # NOQA


class CeleryClient(DjangoClient):
    def send_integrated(self, kwargs):
        return send_raw_integrated.delay(kwargs)

    def send_encoded(self, *args, **kwargs):
        return send_raw.delay(*args, **kwargs)


@task(routing_key='sentry')
def send_raw_integrated(kwargs):
    from raven.contrib.django.models import get_client
    super(DjangoClient, get_client()).send_integrated(kwargs)


@task(routing_key='sentry')
def send_raw(*args, **kwargs):
    from raven.contrib.django.models import get_client
    super(DjangoClient, get_client()).send_encoded(*args, **kwargs)
                                                          python3.7/site-packages/raven/contrib/django/templatetags/__pycache__/__init__.cpython-37.pyc       0000644                 00000000664 15107360264 0026154 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                  @   s   d Z ddlmZ dS )z
raven.contrib.django.templatetags
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2013 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importN)__doc__
__future__r    r   r   [/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/templatetags/__init__.py<module>   s                                                                               python3.7/site-packages/raven/contrib/django/templatetags/__pycache__/raven.cpython-37.pyc          0000644                 00000001350 15107360264 0025521 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                 @   s8   d Z ddlmZ ddlmZ e ZejdddZdS )z
raven.contrib.django.templatetags.raven
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2013 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)templateNc             C   s   ddl m} || pdS )Nr   )client )Zraven.contrib.django.modelsr   Zget_public_dsn)schemer    r   X/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/templatetags/raven.pysentry_public_dsn   s    r	   )N)	__doc__
__future__r   Zdjangor   LibraryregisterZ
simple_tagr	   r   r   r   r   <module>   s
                                                                                                                                                                                                                                                                                           python3.7/site-packages/raven/contrib/django/templatetags/__init__.py                               0000644                 00000000355 15107360264 0021664 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.django.templatetags
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2013 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""
from __future__ import absolute_import
                                                                                                                                                                                                                                                                                   python3.7/site-packages/raven/contrib/django/templatetags/raven.py                                  0000644                 00000000723 15107360264 0021237 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.django.templatetags.raven
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2013 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""

from __future__ import absolute_import

from django import template

register = template.Library()


@register.simple_tag
def sentry_public_dsn(scheme=None):
    from raven.contrib.django.models import client
    return client.get_public_dsn(scheme) or ''
                                             python3.7/site-packages/raven/contrib/django/handlers.py                                            0000644                 00000002016 15107360264 0017227 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """
raven.contrib.django.handlers
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
"""

from __future__ import absolute_import

import logging

from raven.handlers.logging import SentryHandler as BaseSentryHandler
from raven.utils import memoize


class SentryHandler(BaseSentryHandler):
    def __init__(self, *args, **kwargs):
        # TODO(dcramer): we'd like to avoid this duplicate code, but we need
        # to currently defer loading client due to Django loading patterns.
        self.tags = kwargs.pop('tags', None)

        logging.Handler.__init__(self, level=kwargs.get('level', logging.NOTSET))

    @memoize
    def client(self):
        # Import must be lazy for deffered Django loading
        from raven.contrib.django.models import client
        return client

    def _emit(self, record):
        request = getattr(record, 'request', None)
        return super(SentryHandler, self)._emit(record, request=request)
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  python3.7/site-packages/raven/contrib/django/__pycache__/apps.cpython-37.pyc                        0000644                 00000001241 15107360264 0022656 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y4                 @   s,   d dl mZ d dlmZ G dd deZdS )    )absolute_import)	AppConfigc               @   s    e Zd ZdZdZdZdd ZdS )RavenConfigzraven.contrib.djangoZraven_contrib_djangoZRavenc             C   s   ddl m} |  d S )N   )
initialize)Zmodelsr   )selfr    r   J/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/apps.pyready   s    zRavenConfig.readyN)__name__
__module____qualname__namelabelZverbose_namer
   r   r   r   r	   r      s   r   N)
__future__r   Zdjango.appsr   r   r   r   r   r	   <module>   s                                                                                                                                                                                                                                                                                                                                                                  python3.7/site-packages/raven/contrib/django/__pycache__/urls.cpython-37.pyc                        0000644                 00000001337 15107360264 0022706 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    YR                 @   sv   d Z ddlmZ yddlmZ W n  ek
r@   ddlmZ Y nX ddlZedej	j
jjddedej	j
jjfZ
dS )	z
raven.contrib.django.urls
~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)urlNz$^api/(?P<project_id>[\w_-]+)/store/$zraven-report)namez^report/)__doc__
__future__r   Zdjango.conf.urlsr   ImportErrorZdjango.conf.urls.defaultsZraven.contrib.django.viewsravenZcontribZdjangoZviewsreportZurlpatterns r
   r
   J/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/urls.py<module>   s                                                                                                                                                                                                                                                                                                    python3.7/site-packages/raven/contrib/django/__pycache__/__init__.cpython-37.pyc                    0000644                 00000000776 15107360264 0023466 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y;                 @   s$   d Z ddlmZ dZddlmZ dS )z
raven.contrib.django
~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importz%raven.contrib.django.apps.RavenConfig   )DjangoClientN)__doc__
__future__r   Zdefault_app_configZclientr    r   r   N/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/__init__.py<module>   s     python3.7/site-packages/raven/contrib/django/__pycache__/resolver.cpython-37.pyc                    0000644                 00000005057 15107360265 0023566 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y\                 @   sZ   d dl mZ d dlZyd dlmZ W n  ek
rD   d dlmZ Y nX G dd deZdS )    )absolute_importN)get_resolverc               @   s^   e Zd ZedZedZedZedZedZ	i Z
dd Zd
d	d
ZdddZ
dS )
RouteResolverz\(\?\:([^\)]+)\)z\(\?P<(\w+)>[^\)]+\)z
\([^\)]+\)z\[([^\]]+)\|([^\]]+)\]z([A-Z]+)([a-z])c             C   st   | j dd |}| jdd |}| jd|}| jdd |}|ddddd	dd
ddd}|S )
z
        Clean up urlpattern regexes into something readable by humans:

        From:
        > "^(?P<sport_slug>\w+)/athletes/(?P<athlete_slug>\w+)/$"

        To:
        > "{sport_slug}/athletes/{athlete_slug}/"
        c             S   s   d|  d S )Nz%s   )group)m r   N/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/resolver.py<lambda>#       z)RouteResolver._simplify.<locals>.<lambda>c             S   s   d|  d S )Nz{%s}r   )r   )r   r   r   r	   r
   &   r   z{var}c             S   s
   |  dS )Nr   )r   )r   r   r   r	   r
   ,   r   ^ $?z///\)_optional_group_matchersub_named_group_matcher_non_named_group_matcher_either_option_matcherreplace)selfpatternresultr   r   r	   	_simplify   s    zRouteResolver._simplifyNc       	   	      s   |j |}|sd S |d kr$|g}n||kr6||g }|| d  }x|jD ]}|jsr |||}|rN|S qNn|j |sqNy
 j| S  tk
r   Y nX d fdd|D }| 	|j j
 }|dsd| }| j|< |S W d S )Nr
   c             3   s   | ]}  |jjV  qd S )N)r   regexr   ).0p)r   r   r	   	<genexpr>O   s    z)RouteResolver._resolve.<locals>.<genexpr>r   )r   searchendZurl_patternscallback_resolve_cacheKeyErrorjoinr   r   
startswith)	r   resolverpathparentsmatchnew_pathr   prefixr   r   )r   r	   r#   4   s4    



zRouteResolver._resolvec             C   s   t |}| ||}|p|S )N)r   r#   )r   r)   Zurlconfr(   r+   r   r   r	   resolveV   s    zRouteResolver.resolve)N)N)__name__
__module____qualname__recompiler   r   r   r   Z	_camel_rer$   r   r#   r.   r   r   r   r	   r      s   





"r   )	
__future__r   r2   Zdjango.urlsr   ImportErrorZdjango.core.urlresolversobjectr   r   r   r   r	   <module>   s                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    python3.7/site-packages/raven/contrib/django/__pycache__/serializers.cpython-37.pyc                 0000644                 00000004610 15107360265 0024253 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Yw                 @   s   d Z ddlmZ ddlmZ ddlmZ ddlmZ ddlm	Z	 ddl
mZmZ ddl
mZ d	ZG d
d deZee G dd
 d
eZee eeddrddlmZ G dd deZee dS )z
raven.contrib.django.serializers
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_import)unicode_literals)settings)HttpRequest)Promise)
Serializerregister)	text_type)PromiseSerializerc                   s*   e Zd ZefZ fddZdd Z  ZS )r
   c                sV   t t| |sdS |jjdd  }t|d| sRt|d| sRt|d| sRdS dS )NF   z%s__funcz%s__unicode_castz
%s__text_castT)superr
   can	__class____name__hasattr)selfvaluepre)r    Q/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/serializers.pyr
      s    zPromiseSerializer.canc             K   sd   |j jdd  }t|d| rHt|d| t|d| t|d| }n| t|S | j|f|S )Nr   z%s__funcz%s__argsz%s__kw)r   r   r   getattrZrecurser	   )r   r   kwargsr   r   r   r   	serialize#   s    zPromiseSerializer.serialize)r   
__module____qualname__r   typesr
   r   
__classcell__r   r   )r   r   r
      s   r
   c               @   s   e Zd ZefZdd ZdS )HttpRequestSerializerc             K   s   dt |jt|f S )Nz<%s at 0x%s>)typer   id)r   r   r   r   r   r   r   7   s    zHttpRequestSerializer.serializeN)r   r   r   r   r   r   r   r   r   r   r   4   s   r   Z	DATABASESN)QuerySetc               @   s   e Zd ZefZdd ZdS )QuerySetSerializerc             K   s*   t |j}|jr d||jjf S d|f S )Nz<%s: model=%s>z<%s: (Unbound)>)r   r   model)r   r   r   Zqs_namer   r   r   r   D   s    
zQuerySetSerializer.serializeN)r   r   r   r    r   r   r   r   r   r   r!   A   s   r!   )__doc__
__future__r   r   Zdjango.confr   Zdjango.httpr   Zdjango.utils.functionalr   Zraven.utils.serializerr   r   Zraven.utils.compatr	   __all__r
   r   r   Zdjango.db.models.queryr    r!   r   r   r   r   <module>   s    	                                                                                                                        python3.7/site-packages/raven/contrib/django/__pycache__/handlers.cpython-37.pyc                    0000644                 00000002504 15107360265 0023517 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                 @   sD   d Z ddlmZ ddlZddlmZ ddlmZ G dd deZdS )z
raven.contrib.django.handlers
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importN)
SentryHandler)memoizec                   s0   e Zd Zdd Zedd Z fddZ  ZS )r   c             O   s,   | dd | _tjj| |dtjd d S )Ntagslevel)r   )popr   loggingHandler__init__getNOTSET)selfargskwargs r   N/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/handlers.pyr
      s    zSentryHandler.__init__c             C   s   ddl m} |S )Nr   )client)Zraven.contrib.django.modelsr   )r
   r   r   r   r   r      s    zSentryHandler.clientc                s    t |dd }tt| j||dS )Nrequest)r   )getattrsuperr   _emit)r
   recordr   )	__class__r   r   r      s    zSentryHandler._emit)__name__
__module____qualname__r
   r   r   r   
__classcell__r   r   )r   r   r      s   r   )	__doc__
__future__r   r   Zraven.handlers.loggingr   ZBaseSentryHandlerZraven.utilsr   r   r   r   r   <module>   s
                                                                                                                                                                                               python3.7/site-packages/raven/contrib/django/__pycache__/logging.cpython-37.pyc                     0000644                 00000001203 15107360265 0023340 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                 @   s8   d Z ddlmZ ddlZede ddlmZ dZdS )z
raven.contrib.django.logging
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importNzVraven.contrib.django.logging is deprecated. Use raven.contrib.django.handlers instead.)
SentryHandler)	__doc__
__future__r   warningswarnDeprecationWarningZraven.contrib.django.handlersr   __all__ r
   r
   M/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/logging.py<module>   s
                                                                                                                                                                                                                                                                                                                                                                                                python3.7/site-packages/raven/contrib/django/__pycache__/client.cpython-37.pyc                      0000644                 00000020032 15107360265 0023171 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y(                 @   sf  d Z ddlmZ ddlZddlZddlmZ ddlm	Z	 ddl
mZ ddlm
Z
 ddlmZ dd	lmZ ydd
lmZ W n  ek
r   ddlmZ Y nX ddlmZ dd
lmZmZ ddlmZ ddlmZm Z m!Z! ddl"m#Z# ddl$m%Z%m&Z&m'Z' ddl(m)Z) ddl*m+Z+ dZ,edk r&dd Z-ndd Z-G dd de.Z/dd Z0e)dd Z1G dd  d eZ2dS )!z
raven.contrib.django.client
~~~~~~~~~~~~~~~~~~~~~~~~~~~~

:copyright: (c) 2010-2012 by the Sentry Team, see AUTHORS for more details.
:license: BSD, see LICENSE for more details.
    )absolute_importN)VERSION)settings)SuspiciousOperation)HttpRequest)TemplateSyntaxError)MultiValueDict)Origin)LoaderOrigin)Client)get_data_from_templateget_host)SentryMiddleware)string_typesbinary_type	iterlists)
RouteResolver)get_headersget_environ
get_client_ip)once)breadcrumbs)DjangoClient)   
   c             C   s   |   S )N)is_authenticated)request_user r   L/opt/alt/python37/lib/python3.7/site-packages/raven/contrib/django/client.pyr   *   s    r   c             C   s   | j S )N)r   )r   r   r   r   r   -   s    c               @   s   e Zd Zdd Zdd ZdS )_FormatConverterc             C   s   || _ g | _d S )N)
param_mappingparams)selfr    r   r   r   __init__3   s    z_FormatConverter.__init__c             C   s   | j | j| dS )Nz%s)r!   appendr    get)r"   valr   r   r   __getitem__7   s    z_FormatConverter.__getitem__N)__name__
__module____qualname__r#   r'   r   r   r   r   r   1   s   r   c             C   s   g }t |tr.t|}|r*| | } |j}nd}x|p6dD ]z}|d krP|d q8t |trt |trp|dd}t|dkr|d d d }|d|	dd	  q8|t
| q8W | |fS )
Nr   NULLzutf-8replace   u   …z'%s''z'')
isinstancedictr   r!   r$   r   r   decodelenr,   repr)sqlr!   rvconvparamr   r   r   
format_sql<   s$    


r8   c                 s   yddl m}  W n  tk
r0   ddlm}  Y nX y| j | jW n tk
rV   dS X dd fddd
 fdd		}fd
d}|| _|| _td dS )z9If installed this causes Django's queries to be captured.r   )
CursorWrapperNc                s    fdd}t j|d d S )Nc                s2   t  \}}|r|t| }| |dd d S )Nquery)messagecategory)r8   tupleupdate)dataZreal_sqlZreal_params)r!   r4   r   r   	processorg   s    z7install_sql_hook.<locals>.record_sql.<locals>.processor)r@   )r   record)vendoraliasstartdurationr4   r!   r@   r   )r!   r4   r   
record_sqlf   s    
z$install_sql_hook.<locals>.record_sqlc                s0   t   | }x|D ]} | ||||| qW d S )N)time)rB   rC   rD   r4   
param_listrE   r!   )rF   r   r   record_many_sqls   s    
z)install_sql_hook.<locals>.record_many_sqlc          
      sD   t   }z | ||S | jjt| jdd |t   | || X d S )NrC   )rG   dbrB   getattr)r"   r4   r!   rD   )real_executerF   r   r   executex   s
    z!install_sql_hook.<locals>.executec                s:   t   }z | ||S | jjt| jdd ||| X d S )NrC   )rG   rJ   rB   rK   )r"   r4   rH   rD   )real_executemanyrI   r   r   executemany   s
    z%install_sql_hook.<locals>.executemanyzdjango.db.backends)N)	Zdjango.db.backends.utilsr9   ImportErrorZdjango.db.backends.utilrM   rO   AttributeErrorr   Z
ignore_logger)r9   rM   rO   r   )rL   rN   rI   rF   r   install_sql_hookV   s     

rR   c                   sb   e Zd ZedZe Zdd Zdd Z	dd Z
dd	 Z fd
dZd fd
d	Z
dd Z  ZS )r   zsentry.errors.client.djangoc             O   s.   | dd}tj| f|| |r*|   d S )NrR   T)popr   r#   rR   )r"   argskwargsrR   r   r   r   r#      s    zDjangoClient.__init__c             C   s
   t   d S )N)rR   )r"   r   r   r   rR      s    zDjangoClient.install_sql_hookc             C   s   dt |ji}t|dd }|d kr&|S y^t|}|s8|S |j|d< t|drV|j|d< t|drn| |d< nt|dr|j|d< W n t	k
r   Y nX |S )NZ
ip_addressuseridemailget_usernameusername)
r   METArK   r   pkhasattrrX   rY   rZ   	Exception)r"   requestZ	user_inforV   Z
authenticatedr   r   r   
get_user_info   s$    




zDjangoClient.get_user_infoc             C   s>  i }|  ||d< y| }W n> tk
r\   | r<d}nd}t|}d|||jf }Y nX |jdkry
|j}W q tk
r   y
|j	}W n\ tk
r   y|j
pd}W n tk
r   d}Y n"X t|trt
dd t|D }Y nX Y qX nd }|j}|d	|j||jd
|t
|jt
t|t
t|di |S )NrV   httpshttpz	%s://%s%s)GETZHEADz
<unavailable>c             s   s.   | ]&\}}|t |d kr |d n|fV  qdS )r   r   N)r2   ).0kvr   r   r   	<genexpr>   s   z5DjangoClient.get_data_from_request.<locals>.<genexpr>r_   ZQUERY_STRING)methodurlZquery_stringr?   Zcookiesheadersenv)r`   Zbuild_absolute_urir   Z	is_securer
   pathrh   bodyr^   Z
raw_post_dataZPOSTr/   r   r0   r   r[   r>   r%   ZCOOKIESr   r   )r"   r_   resultZurischemehostr?   environr   r   r   get_data_from_request   sJ    






z"DjangoClient.get_data_from_requestc       	         s   t t| j||}x4| |D ]&}|d}|s2q|drd|d< qW | jsdtjkry6ddl	m
} |j }|j
pz|j}|d d	| W n@ tk
r   y|d d	tj W n tk
r   Y nX Y nX |S )
Nmodulezdjango.FZin_appzdjango.contrib.sitesr   )SiteZtagssite)superr   	build_msgZ_iter_framesr%   
startswithru   r   ZINSTALLED_APPSZdjango.contrib.sites.modelsrt   objectsZget_currentnamedomain
setdefaultr^   ZSITE_IDrQ   )	r"   rT   rU   r?   framers   rt   ru   Z	site_name)	__class__r   r   rw      s&    


zDjangoClient.build_msgNc       
         s,  d|kri  |d< }n|d }|d kr4t tjdd }t|t}|rR|| | |dr|d d }t|dst|t	rtt |dd t
tfrt|jd t
st|drt |dt |dd }t |dd }|d kr| jd	 |t|| tt| j|f|}	|r(|	r(|d
| jj|	d|_|	S )Nr?   r_   exc_infor   Zdjango_template_sourcesourcer   Ztemplate_debugz,Unable to get template source from exceptionproject)Z
project_idrW   )rK   r   threadr/   r   r>   rr   r%   r]   r   r=   listr   r	   loggerinfor   rv   r   captureZremoter   Zsentry)
r"   Z
event_typer_   rU   r?   Zis_http_request	exc_valuer   debugrn   )r~   r   r   r     s6    






zDjangoClient.capturec             C   s   | j |jS )N)resolverresolverl   )r"   r_   r   r   r   get_transaction_from_request4  s    z)DjangoClient.get_transaction_from_request)N)r(   r)   r*   logging	getLoggerr   r   r   r#   rR   r`   rr   rw   r   r   
__classcell__r   r   )r~   r   r      s   
5,r   )3__doc__
__future__r   rG   r   Zdjangor   ZDJANGO_VERSIONZdjango.confr   Zdjango.core.exceptionsr   Zdjango.httpr   Zdjango.templater   Zdjango.utils.datastructuresr   Zdjango.template.baser	   rP   Zdjango.template.loaderr
   Z
raven.baser   Zraven.contrib.django.utilsr   r
   Zraven.contrib.django.middlewarer   Zraven.utils.compatr   r   r   Zraven.contrib.django.resolverr   Zraven.utils.wsgir   r   r   Zraven.utilsr   ravenr   __all__r   objectr   r8   rR   r   r   r   r   r   <module>   s:   

7                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      python3.7/site-packages/raven/contrib/django/__pycache__/views.cpython-37.pyc                       0000644                 00000005502 15107360265 0023055 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Y                 @   s   d Z ddlmZ ddlmZ ddlmZ ddlmZm	Z	m
Z
 ddlmZ ddl
mZ ddlmZ dd	lmZ dd
lmZ ddlmZ dd
 Zdd Zdd ZeedddgeedddZdS )z
raven.contrib.django.views
~~~~~~~~~~~~~~~~~~~~~~~