"""Cache Management
"""

import hashlib
import json
import logging
import os

from pip._vendor.packaging.tags import interpreter_name, interpreter_version
from pip._vendor.packaging.utils import canonicalize_name

from pip._internal.exceptions import InvalidWheelFilename
from pip._internal.models.link import Link
from pip._internal.models.wheel import Wheel
from pip._internal.utils.temp_dir import TempDirectory, tempdir_kinds
from pip._internal.utils.typing import MYPY_CHECK_RUNNING
from pip._internal.utils.urls import path_to_url

if MYPY_CHECK_RUNNING:
    from typing import Optional, Set, List, Any, Dict

    from pip._vendor.packaging.tags import Tag

    from pip._internal.models.format_control import FormatControl

logger = logging.getLogger(__name__)


def _hash_dict(d):
    # type: (Dict[str, str]) -> str
    """Return a stable sha224 of a dictionary."""
    s = json.dumps(d, sort_keys=True, separators=(",", ":"), ensure_ascii=True)
    return hashlib.sha224(s.encode("ascii")).hexdigest()


class Cache(object):
    """An abstract class - provides cache directories for data from links


        :param cache_dir: The root of the cache.
        :param format_control: An object of FormatControl class to limit
            binaries being read from the cache.
        :param allowed_formats: which formats of files the cache should store.
            ('binary' and 'source' are the only allowed values)
    """

    def __init__(self, cache_dir, format_control, allowed_formats):
        # type: (str, FormatControl, Set[str]) -> None
        super(Cache, self).__init__()
        assert not cache_dir or os.path.isabs(cache_dir)
        self.cache_dir = cache_dir or None
        self.format_control = format_control
        self.allowed_formats = allowed_formats

        _valid_formats = {"source", "binary"}
        assert self.allowed_formats.union(_valid_formats) == _valid_formats

    def _get_cache_path_parts_legacy(self, link):
        # type: (Link) -> List[str]
        """Get parts of part that must be os.path.joined with cache_dir

        Legacy cache key (pip < 20) for compatibility with older caches.
        """

        # We want to generate an url to use as our cache key, we don't want to
        # just re-use the URL because it might have other items in the fragment
        # and we don't care about those.
        key_parts = [link.url_without_fragment]
        if link.hash_name is not None and link.hash is not None:
            key_parts.append("=".join([link.hash_name, link.hash]))
        key_url = "#".join(key_parts)

        # Encode our key url with sha224, we'll use this because it has similar
        # security properties to sha256, but with a shorter total output (and
        # thus less secure). However the differences don't make a lot of
        # difference for our use case here.
        hashed = hashlib.sha224(key_url.encode()).hexdigest()

        # We want to nest the directories some to prevent having a ton of top
        # level directories where we might run out of sub directories on some
        # FS.
        parts = [hashed[:2], hashed[2:4], hashed[4:6], hashed[6:]]

        return parts

    def _get_cache_path_parts(self, link):
        # type: (Link) -> List[str]
        """Get parts of part that must be os.path.joined with cache_dir
        """

        # We want to generate an url to use as our cache key, we don't want to
        # just re-use the URL because it might have other items in the fragment
        # and we don't care about those.
        key_parts = {"url": link.url_without_fragment}
        if link.hash_name is not None and link.hash is not None:
            key_parts[link.hash_name] = link.hash
        if link.subdirectory_fragment:
            key_parts["subdirectory"] = link.subdirectory_fragment

        # Include interpreter name, major and minor version in cache key
        # to cope with ill-behaved sdists that build a different wheel
        # depending on the python version their setup.py is being run on,
        # and don't encode the difference in compatibility tags.
        # https://github.com/pypa/pip/issues/7296
        key_parts["interpreter_name"] = interpreter_name()
        key_parts["interpreter_version"] = interpreter_version()

        # Encode our key url with sha224, we'll use this because it has similar
        # security properties to sha256, but with a shorter total output (and
        # thus less secure). However the differences don't make a lot of
        # difference for our use case here.
        hashed = _hash_dict(key_parts)

        # We want to nest the directories some to prevent having a ton of top
        # level directories where we might run out of sub directories on some
        # FS.
        parts = [hashed[:2], hashed[2:4], hashed[4:6], hashed[6:]]

        return parts

    def _get_candidates(self, link, canonical_package_name):
        # type: (Link, str) -> List[Any]
        can_not_cache = (
            not self.cache_dir or
            not canonical_package_name or
            not link
        )
        if can_not_cache:
            return []

        formats = self.format_control.get_allowed_formats(
            canonical_package_name
        )
        if not self.allowed_formats.intersection(formats):
            return []

        candidates = []
        path = self.get_path_for_link(link)
        if os.path.isdir(path):
            for candidate in os.listdir(path):
                candidates.append((candidate, path))
        # TODO remove legacy path lookup in pip>=21
        legacy_path = self.get_path_for_link_legacy(link)
        if os.path.isdir(legacy_path):
            for candidate in os.listdir(legacy_path):
                candidates.append((candidate, legacy_path))
        return candidates

    def get_path_for_link_legacy(self, link):
        # type: (Link) -> str
        raise NotImplementedError()

    def get_path_for_link(self, link):
        # type: (Link) -> str
        """Return a directory to store cached items in for link.
        """
        raise NotImplementedError()

    def get(
        self,
        link,            # type: Link
        package_name,    # type: Optional[str]
        supported_tags,  # type: List[Tag]
    ):
        # type: (...) -> Link
        """Returns a link to a cached item if it exists, otherwise returns the
        passed link.
        """
        raise NotImplementedError()


class SimpleWheelCache(Cache):
    """A cache of wheels for future installs.
    """

    def __init__(self, cache_dir, format_control):
        # type: (str, FormatControl) -> None
        super(SimpleWheelCache, self).__init__(
            cache_dir, format_control, {"binary"}
        )

    def get_path_for_link_legacy(self, link):
        # type: (Link) -> str
        parts = self._get_cache_path_parts_legacy(link)
        assert self.cache_dir
        return os.path.join(self.cache_dir, "wheels", *parts)

    def get_path_for_link(self, link):
        # type: (Link) -> str
        """Return a directory to store cached wheels for link

        Because there are M wheels for any one sdist, we provide a directory
        to cache them in, and then consult that directory when looking up
        cache hits.

        We only insert things into the cache if they have plausible version
        numbers, so that we don't contaminate the cache with things that were
        not unique. E.g. ./package might have dozens of installs done for it
        and build a version of 0.0...and if we built and cached a wheel, we'd
        end up using the same wheel even if the source has been edited.

        :param link: The link of the sdist for which this will cache wheels.
        """
        parts = self._get_cache_path_parts(link)
        assert self.cache_dir
        # Store wheels within the root cache_dir
        return os.path.join(self.cache_dir, "wheels", *parts)

    def get(
        self,
        link,            # type: Link
        package_name,    # type: Optional[str]
        supported_tags,  # type: List[Tag]
    ):
        # type: (...) -> Link
        candidates = []

        if not package_name:
            return link

        canonical_package_name = canonicalize_name(package_name)
        for wheel_name, wheel_dir in self._get_candidates(
            link, canonical_package_name
        ):
            try:
                wheel = Wheel(wheel_name)
            except InvalidWheelFilename:
                continue
            if canonicalize_name(wheel.name) != canonical_package_name:
                logger.debug(
                    "Ignoring cached wheel %s for %s as it "
                    "does not match the expected distribution name %s.",
                    wheel_name, link, package_name,
                )
                continue
            if not wheel.supported(supported_tags):
                # Built for a different python/arch/etc
                continue
            candidates.append(
                (
                    wheel.support_index_min(supported_tags),
                    wheel_name,
                    wheel_dir,
                )
            )

        if not candidates:
            return link

        _, wheel_name, wheel_dir = min(candidates)
        return Link(path_to_url(os.path.join(wheel_dir, wheel_name)))


class EphemWheelCache(SimpleWheelCache):
    """A SimpleWheelCache that creates it's own temporary cache directory
    """

    def __init__(self, format_control):
        # type: (FormatControl) -> None
        self._temp_dir = TempDirectory(
            kind=tempdir_kinds.EPHEM_WHEEL_CACHE,
            globally_managed=True,
        )

        super(EphemWheelCache, self).__init__(
            self._temp_dir.path, format_control
        )


class CacheEntry(object):
    def __init__(
        self,
        link,  # type: Link
        persistent,  # type: bool
    ):
        self.link = link
        self.persistent = persistent


class WheelCache(Cache):
    """Wraps EphemWheelCache and SimpleWheelCache into a single Cache

    This Cache allows for gracefully degradation, using the ephem wheel cache
    when a certain link is not found in the simple wheel cache first.
    """

    def __init__(self, cache_dir, format_control):
        # type: (str, FormatControl) -> None
        super(WheelCache, self).__init__(
            cache_dir, format_control, {'binary'}
        )
        self._wheel_cache = SimpleWheelCache(cache_dir, format_control)
        self._ephem_cache = EphemWheelCache(format_control)

    def get_path_for_link_legacy(self, link):
        # type: (Link) -> str
        return self._wheel_cache.get_path_for_link_legacy(link)

    def get_path_for_link(self, link):
        # type: (Link) -> str
        return self._wheel_cache.get_path_for_link(link)

    def get_ephem_path_for_link(self, link):
        # type: (Link) -> str
        return self._ephem_cache.get_path_for_link(link)

    def get(
        self,
        link,            # type: Link
        package_name,    # type: Optional[str]
        supported_tags,  # type: List[Tag]
    ):
        # type: (...) -> Link
        cache_entry = self.get_cache_entry(link, package_name, supported_tags)
        if cache_entry is None:
            return link
        return cache_entry.link

    def get_cache_entry(
        self,
        link,            # type: Link
        package_name,    # type: Optional[str]
        supported_tags,  # type: List[Tag]
    ):
        # type: (...) -> Optional[CacheEntry]
        """Returns a CacheEntry with a link to a cached item if it exists or
        None. The cache entry indicates if the item was found in the persistent
        or ephemeral cache.
        """
        retval = self._wheel_cache.get(
            link=link,
            package_name=package_name,
            supported_tags=supported_tags,
        )
        if retval is not link:
            return CacheEntry(retval, persistent=True)

        retval = self._ephem_cache.get(
            link=link,
            package_name=package_name,
            supported_tags=supported_tags,
        )
        if retval is not link:
            return CacheEntry(retval, persistent=False)

        return None
                                       python3.7/site-packages/pip/_internal/wheel_builder.py                                              0000644                 00000022462 15107360303 0016770 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """Orchestrator for building wheels from InstallRequirements.
"""

import logging
import os.path
import re
import shutil

from pip._internal.models.link import Link
from pip._internal.operations.build.wheel import build_wheel_pep517
from pip._internal.operations.build.wheel_legacy import build_wheel_legacy
from pip._internal.utils.logging import indent_log
from pip._internal.utils.misc import ensure_dir, hash_file, is_wheel_installed
from pip._internal.utils.setuptools_build import make_setuptools_clean_args
from pip._internal.utils.subprocess import call_subprocess
from pip._internal.utils.temp_dir import TempDirectory
from pip._internal.utils.typing import MYPY_CHECK_RUNNING
from pip._internal.utils.urls import path_to_url
from pip._internal.vcs import vcs

if MYPY_CHECK_RUNNING:
    from typing import (
        Any, Callable, Iterable, List, Optional, Tuple,
    )

    from pip._internal.cache import WheelCache
    from pip._internal.req.req_install import InstallRequirement

    BinaryAllowedPredicate = Callable[[InstallRequirement], bool]
    BuildResult = Tuple[List[InstallRequirement], List[InstallRequirement]]

logger = logging.getLogger(__name__)

_egg_info_re = re.compile(r'([a-z0-9_.]+)-([a-z0-9_.!+-]+)', re.IGNORECASE)


def _contains_egg_info(s):
    # type: (str) -> bool
    """Determine whether the string looks like an egg_info.

    :param s: The string to parse. E.g. foo-2.1
    """
    return bool(_egg_info_re.search(s))


def _should_build(
    req,  # type: InstallRequirement
    need_wheel,  # type: bool
    check_binary_allowed,  # type: BinaryAllowedPredicate
):
    # type: (...) -> bool
    """Return whether an InstallRequirement should be built into a wheel."""
    if req.constraint:
        # never build requirements that are merely constraints
        return False
    if req.is_wheel:
        if need_wheel:
            logger.info(
                'Skipping %s, due to already being wheel.', req.name,
            )
        return False

    if need_wheel:
        # i.e. pip wheel, not pip install
        return True

    # From this point, this concerns the pip install command only
    # (need_wheel=False).

    if req.editable or not req.source_dir:
        return False

    if not check_binary_allowed(req):
        logger.info(
            "Skipping wheel build for %s, due to binaries "
            "being disabled for it.", req.name,
        )
        return False

    if not req.use_pep517 and not is_wheel_installed():
        # we don't build legacy requirements if wheel is not installed
        logger.info(
            "Using legacy 'setup.py install' for %s, "
            "since package 'wheel' is not installed.", req.name,
        )
        return False

    return True


def should_build_for_wheel_command(
    req,  # type: InstallRequirement
):
    # type: (...) -> bool
    return _should_build(
        req, need_wheel=True, check_binary_allowed=_always_true
    )


def should_build_for_install_command(
    req,  # type: InstallRequirement
    check_binary_allowed,  # type: BinaryAllowedPredicate
):
    # type: (...) -> bool
    return _should_build(
        req, need_wheel=False, check_binary_allowed=check_binary_allowed
    )


def _should_cache(
    req,  # type: InstallRequirement
):
    # type: (...) -> Optional[bool]
    """
    Return whether a built InstallRequirement can be stored in the persistent
    wheel cache, assuming the wheel cache is available, and _should_build()
    has determined a wheel needs to be built.
    """
    if req.editable or not req.source_dir:
        # never cache editable requirements
        return False

    if req.link and req.link.is_vcs:
        # VCS checkout. Do not cache
        # unless it points to an immutable commit hash.
        assert not req.editable
        assert req.source_dir
        vcs_backend = vcs.get_backend_for_scheme(req.link.scheme)
        assert vcs_backend
        if vcs_backend.is_immutable_rev_checkout(req.link.url, req.source_dir):
            return True
        return False

    assert req.link
    base, ext = req.link.splitext()
    if _contains_egg_info(base):
        return True

    # Otherwise, do not cache.
    return False


def _get_cache_dir(
    req,  # type: InstallRequirement
    wheel_cache,  # type: WheelCache
):
    # type: (...) -> str
    """Return the persistent or temporary cache directory where the built
    wheel need to be stored.
    """
    cache_available = bool(wheel_cache.cache_dir)
    assert req.link
    if cache_available and _should_cache(req):
        cache_dir = wheel_cache.get_path_for_link(req.link)
    else:
        cache_dir = wheel_cache.get_ephem_path_for_link(req.link)
    return cache_dir


def _always_true(_):
    # type: (Any) -> bool
    return True


def _build_one(
    req,  # type: InstallRequirement
    output_dir,  # type: str
    build_options,  # type: List[str]
    global_options,  # type: List[str]
):
    # type: (...) -> Optional[str]
    """Build one wheel.

    :return: The filename of the built wheel, or None if the build failed.
    """
    try:
        ensure_dir(output_dir)
    except OSError as e:
        logger.warning(
            "Building wheel for %s failed: %s",
            req.name, e,
        )
        return None

    # Install build deps into temporary directory (PEP 518)
    with req.build_env:
        return _build_one_inside_env(
            req, output_dir, build_options, global_options
        )


def _build_one_inside_env(
    req,  # type: InstallRequirement
    output_dir,  # type: str
    build_options,  # type: List[str]
    global_options,  # type: List[str]
):
    # type: (...) -> Optional[str]
    with TempDirectory(kind="wheel") as temp_dir:
        assert req.name
        if req.use_pep517:
            assert req.metadata_directory
            wheel_path = build_wheel_pep517(
                name=req.name,
                backend=req.pep517_backend,
                metadata_directory=req.metadata_directory,
                build_options=build_options,
                tempd=temp_dir.path,
            )
        else:
            wheel_path = build_wheel_legacy(
                name=req.name,
                setup_py_path=req.setup_py_path,
                source_dir=req.unpacked_source_directory,
                global_options=global_options,
                build_options=build_options,
                tempd=temp_dir.path,
            )

        if wheel_path is not None:
            wheel_name = os.path.basename(wheel_path)
            dest_path = os.path.join(output_dir, wheel_name)
            try:
                wheel_hash, length = hash_file(wheel_path)
                shutil.move(wheel_path, dest_path)
                logger.info('Created wheel for %s: '
                            'filename=%s size=%d sha256=%s',
                            req.name, wheel_name, length,
                            wheel_hash.hexdigest())
                logger.info('Stored in directory: %s', output_dir)
                return dest_path
            except Exception as e:
                logger.warning(
                    "Building wheel for %s failed: %s",
                    req.name, e,
                )
        # Ignore return, we can't do anything else useful.
        if not req.use_pep517:
            _clean_one_legacy(req, global_options)
        return None


def _clean_one_legacy(req, global_options):
    # type: (InstallRequirement, List[str]) -> bool
    clean_args = make_setuptools_clean_args(
        req.setup_py_path,
        global_options=global_options,
    )

    logger.info('Running setup.py clean for %s', req.name)
    try:
        call_subprocess(clean_args, cwd=req.source_dir)
        return True
    except Exception:
        logger.error('Failed cleaning build dir for %s', req.name)
        return False


def build(
    requirements,  # type: Iterable[InstallRequirement]
    wheel_cache,  # type: WheelCache
    build_options,  # type: List[str]
    global_options,  # type: List[str]
):
    # type: (...) -> BuildResult
    """Build wheels.

    :return: The list of InstallRequirement that succeeded to build and
        the list of InstallRequirement that failed to build.
    """
    if not requirements:
        return [], []

    # Build the wheels.
    logger.info(
        'Building wheels for collected packages: %s',
        ', '.join(req.name for req in requirements),  # type: ignore
    )

    with indent_log():
        build_successes, build_failures = [], []
        for req in requirements:
            cache_dir = _get_cache_dir(req, wheel_cache)
            wheel_file = _build_one(
                req, cache_dir, build_options, global_options
            )
            if wheel_file:
                # Update the link for this.
                req.link = Link(path_to_url(wheel_file))
                req.local_file_path = req.link.file_path
                assert req.link.is_wheel
                build_successes.append(req)
            else:
                build_failures.append(req)

    # notify success/failure
    if build_successes:
        logger.info(
            'Successfully built %s',
            ' '.join([req.name for req in build_successes]),  # type: ignore
        )
    if build_failures:
        logger.info(
            'Failed to build %s',
            ' '.join([req.name for req in build_failures]),  # type: ignore
        )
    # Return a list of requirements that failed to build
    return build_successes, build_failures
                                                                                                                                                                                                              python3.7/site-packages/pip/_internal/vcs/__pycache__/__init__.cpython-37.pyc                       0000644                 00000001007 15107360303 0022765 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Rei                 @   s<   d dl Zd dlZd dlZd dlZd dlmZmZmZm	Z	 dS )    N)RemoteNotFoundErroris_urlmake_vcs_requirement_urlvcs)
pip._internal.vcs.bazaarpippip._internal.vcs.gitpip._internal.vcs.mercurialpip._internal.vcs.subversion pip._internal.vcs.versioncontrolr   r   r   r    r   r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_internal/vcs/__init__.py<module>   s                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            python3.7/site-packages/pip/_internal/vcs/__pycache__/mercurial.cpython-37.pyc                      0000644                 00000011574 15107360303 0023223 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re*                 @   s   d dl mZ d dlZd dlZd dlmZ d dlmZmZ d dl	m
Z
 d dlmZ d dl
mZ d dlmZ d d	lmZ d d
lmZmZmZ erd dl	mZ d dlmZ eeZG d
d deZee dS )    )absolute_importN)configparser)
BadCommandSubProcessError)display_path)make_command)
TempDirectory)MYPY_CHECK_RUNNING)path_to_url)VersionControl!find_path_to_setup_from_repo_rootvcs)
HiddenText)
RevOptionsc                   s   e Zd ZdZdZdZdZedd Zdd Z	d	d
 Z
dd Zd
d Ze
dd Ze
dd Ze
dd Ze
dd Ze
dd Ze
 fddZ  ZS )	Mercurialhgz.hgclone)r   zhg+filezhg+httpzhg+httpszhg+sshzhg+static-httpc             C   s   | gS )N )revr   r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_internal/vcs/mercurial.pyget_base_rev_args'   s    zMercurial.get_base_rev_argsc          	   C   s>   t dd*}| j|j|d | jd|g|jd W dQ R X dS )z?Export the Hg repository at the url to the destination locationexport)kind)urlarchive)cwdN)r   unpackpathrun_command)selflocationr   temp_dirr   r   r   r   +   s    zMercurial.exportc             C   sP   |  }td||t| | tddd|| | jtdd| |d d S )NzCloning hg %s%s to %sr   z
--noupdatez-qupdate)r   )
to_displayloggerinfor   r   r   to_args)r   destr   rev_optionsrev_displayr   r   r   	fetch_new5   s    
zMercurial.fetch_newc       	   
   C   s   t j|| jd}t }y>|| |dd|j t	|d}|
| W d Q R X W n6 ttjfk
r } zt
d|| W d d }~X Y n X tdd| }| j||d d S )	Nhgrcpathsdefaultwz/Could not switch Mercurial repository to %s: %sr"   z-q)r   )osr   joindirnamer   RawConfigParserreadsetsecretopenwriteOSErrorNoSectionErrorr$   warningr   r&   r   )	r   r'   r   r(   repo_configconfigconfig_fileexccmd_argsr   r   r   switchD   s    
zMercurial.switchc             C   s4   | j ddg|d tdd| }| j ||d d S )Npullz-q)r   r"   )r   r   r&   )r   r'   r   r(   r?   r   r   r   r"   U   s    zMercurial.updatec             C   s0   | j ddg|d }| |r(t|}| S )N
showconfigz
paths.default)r   )r   strip_is_local_repositoryr
   )clsr    r   r   r   r   get_remote_url[   s    
zMercurial.get_remote_urlc             C   s   | j ddg|d }|S )zW
        Return the repository-local changeset revision number, as an integer.
        parentsz--template={rev})r   )r   rC   )rE   r    current_revisionr   r   r   get_revisiond   s    zMercurial.get_revisionc             C   s   | j ddg|d }|S )zh
        Return the changeset identification hash, as a 40-character
        hexadecimal string
        rG   z--template={node})r   )r   rC   )rE   r    current_rev_hashr   r   r   get_requirement_revisionm   s    z"Mercurial.get_requirement_revisionc             C   s   dS )z&Always assume the versions don't matchFr   )rE   r'   namer   r   r   is_commit_id_equalx   s    zMercurial.is_commit_id_equalc             C   s@   | j dg|d }tj|s6tjtj||}t||S )z~
        Return the path to setup.py, relative to the repo root.
        Return None if setup.py is in the repo root.
        root)r   )r   rC   r/   r   isabsabspathr0   r   )rE   r    	repo_rootr   r   r   get_subdirectory}   s
    zMercurial.get_subdirectoryc                st   t t| |}|r|S y| jdg|dd}W n2 tk
rN   td| d S  tk
r`   d S X tj	
|dS )NrN   F)r   log_failed_cmdzIcould not determine if %s is under hg control because hg is not availablez
)superr   get_repository_rootr   r   r$   debugr   r/   r   normpathrstrip)rE   r    locr)	__class__r   r   rU      s    zMercurial.get_repository_root)__name__
__module____qualname__rL   r1   	repo_nameschemesstaticmethodr   r   r*   r@   r"   classmethodrF   rI   rK   rM   rR   rU   
__classcell__r   r   )r[   r   r      s   
		
r   )
__future__r   loggingr/   pip._vendor.six.movesr   pip._internal.exceptionsr   r   pip._internal.utils.miscr   pip._internal.utils.subprocessr   pip._internal.utils.temp_dirr   pip._internal.utils.typingr	   pip._internal.utils.urlsr
    pip._internal.vcs.versioncontrolr   r   r
   r   r   	getLoggerr\   r$   r   registerr   r   r   r   <module>   s    
                                                                                                                                    python3.7/site-packages/pip/_internal/vcs/__pycache__/versioncontrol.cpython-37.pyc                 0000644                 00000050670 15107360303 0024326 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Rene                 @   s  d Z ddlmZ ddlZddlZddlZddlZddlZddlZddl	m
Z
 ddlmZ
 ddlmZmZmZ ddlmZmZ ddlmZ dd	lmZmZmZmZmZmZ dd
lmZm Z m!Z!m"Z" ddl#m$Z$ ddl%m&Z& e$r>dd
l'm(Z(m)Z)m*Z*m+Z+m,Z,m-Z-m.Z.m/Z/m0Z0m1Z1m2Z2 ddlm3Z3 ddlm4Z4 e.e,e5 e,e5 f Z6dgZ7e8e9Z:dd Z;d"ddZ<d#ddZ=dd Z>G dd de?Z@G dd deAZBG dd deAZCeC ZDG d d! d!eAZEdS )$z)Handles all VCS (version control) support    )absolute_importN)
pkg_resources)parse)
BadCommandInstallationErrorSubProcessError)console_to_strsamefile)subprocess_logger)ask_path_exists
backup_dirdisplay_pathhide_url
hide_valuermtree)format_command_argsmake_commandmake_subprocess_output_errorreveal_command_args)MYPY_CHECK_RUNNING)get_url_scheme)DictIterableIteratorListOptionalTextTupleTypeUnionMappingAny)
HiddenText)CommandArgsvcsc             C   s*   t | }|dkrdS |ddddgtj kS )z3
    Return true if the name looks like a URL.
    NFhttphttpsfileftp)r   r$   all_schemes)namescheme r,   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_internal/vcs/versioncontrol.pyis_url8   s    r.   c             C   s.   t |}d| ||}|r*|d|7 }|S )z
    Return the URL for a VCS requirement.

    Args:
      repo_url: the remote VCS url, with any needed VCS prefix (e.g. "git+").
      project_name: the (unescaped) project name.
    z{}@{}#egg={}z&subdirectory={})r   to_filenameformat)repo_urlrevproject_namesubdiregg_project_namereqr,   r,   r-   make_vcs_requirement_urlC   s
    	
r7   Tc          
   C   sd  |dkrg }t j}tj }|r*|| d}t| }y.tjt	| tj
tj
|d}	|	jrb|	j  W n6 t
k
r }
 z|rt d|
|  W dd}
~
X Y nX g }x@d}|	jrt|	j }|sP | }||d  || qW z|	  W d|	jr|	j  X |	jo|	j|k}
|
rZ|sD|rDt| |||	jd}t | d|	j|}t|d|S )	z
    Args:
      extra_ok_returncodes: an iterable of integer return codes that are
        acceptable, in addition to 0. Defaults to None, which means [].
      log_failed_cmd: if false, failed commands are not logged,
        only raised.
    NT)stdoutstderrcwdz#Error %s while executing command %s
)cmd_argsr:   linesexit_statuszSCommand errored out with exit status {}: {} Check the logs for full command output. )r
   debugosenvironcopyupdater   
subprocessPopenr   PIPEstdinclose	Exceptioncriticalr8   r   readlinerstripappendwait
returncoder   errorr0   r   join)cmdr:   
extra_environextra_ok_returncodeslog_failed_cmdlog_subprocessenvshowing_subprocesscommand_descprocexc
all_outputlineproc_had_errormsgexc_msgr,   r,   r-   call_subprocessT   s`    





rb   c             C   sd   | }xBt jt j| dsF| }t j| } | |krtd| dS qW t|| rVdS t j| |S )z
    Find the path to `setup.py` by searching up the filesystem from `location`.
    Return the path to `setup.py` relative to `repo_root`.
    Return None if `setup.py` is in `repo_root` or cannot be found.
    zsetup.pyzGCould not find setup.py for directory %s (tried all parent directories)N)	rA   pathexistsrR   dirnameloggerwarningr	   relpath)location	repo_root
orig_location
last_locationr,   r,   r-   !find_path_to_setup_from_repo_root   s    
rm   c               @   s   e Zd ZdS )RemoteNotFoundErrorN)__name__
__module____qualname__r,   r,   r,   r-   rn      s   rn   c               @   sF   e Zd ZdZdddZdd Zedd Zd	d
 Zdd Z	d
d Z
dS )
RevOptionsz
    Encapsulates a VCS-specific revision to install, along with any VCS
    install options.

    Instances of this class should be treated as if immutable.
    Nc             C   s(   |dkrg }|| _ || _|| _d| _dS )z
        Args:
          vc_class: a VersionControl subclass.
          rev: the name of the revision to install.
          extra_args: a list of extra options.
        N)
extra_argsr2   vc_classbranch_name)selfrt   r2   rs   r,   r,   r-   __init__   s    
zRevOptions.__init__c             C   s   d | jj| jS )Nz<RevOptions {}: rev={!r}>)r0   rt   r*   r2   )rv   r,   r,   r-   __repr__   s    zRevOptions.__repr__c             C   s   | j d kr| jjS | j S )N)r2   rt   default_arg_rev)rv   r,   r,   r-   arg_rev   s    
zRevOptions.arg_revc             C   s0   g }| j }|dk	r"|| j|7 }|| j7 }|S )z<
        Return the VCS-specific command arguments.
        N)rz   rt   get_base_rev_argsrs   )rv   argsr2   r,   r,   r-   to_args   s    
zRevOptions.to_argsc             C   s   | j s
dS d| j S )Nr?   z (to revision {}))r2   r0   )rv   r,   r,   r-   
to_display  s    zRevOptions.to_displayc             C   s   | j j|| jdS )z
        Make a copy of the current instance, but with a new rev.

        Args:
          rev: the name of the revision for the new object.
        )rs   )rt   make_rev_optionsrs   )rv   r2   r,   r,   r-   make_new  s    zRevOptions.make_new)NN)ro   rp   rq   __doc__rw   rx   propertyrz   r}   r~   r   r,   r,   r,   r-   rr      s    

rr   c                   s   e Zd Zi ZddddddgZ fddZd	d
 Zedd Zed
d Z	edd Z
dd Zdd Zdd Z
dd Zdd Z  ZS )
VcsSupportsshgithgbzrsftpsvnc                s:   t j| j tt dd r(t j| j tt|   d S )N
uses_fragment)	urllib_parseuses_netlocextendschemesgetattrr   superr   rw   )rv   )	__class__r,   r-   rw     s    zVcsSupport.__init__c             C   s
   | j  S )N)	_registry__iter__)rv   r,   r,   r-   r   !  s    zVcsSupport.__iter__c             C   s   t | j S )N)listr   values)rv   r,   r,   r-   backends%  s    zVcsSupport.backendsc             C   s   dd | j D S )Nc             S   s   g | ]
}|j qS r,   )re   ).0backendr,   r,   r-   
<listcomp>-  s    z'VcsSupport.dirnames.<locals>.<listcomp>)r   )rv   r,   r,   r-   dirnames*  s    zVcsSupport.dirnamesc             C   s$   g }x| j D ]}||j qW |S )N)r   r   r   )rv   r   r   r,   r,   r-   r)   /  s    zVcsSupport.all_schemesc             C   sH   t |dstd|j d S |j| jkrD| | j|j< td|j d S )Nr*   zCannot register VCS %szRegistered VCS backend: %s)hasattrrf   rg   ro   r*   r   r@   )rv   clsr,   r,   r-   register7  s    
zVcsSupport.registerc             C   s   || j kr| j |= d S )N)r   )rv   r*   r,   r,   r-   
unregister@  s    
zVcsSupport.unregisterc             C   s\   i }x:| j  D ],}||}|s$qtd||j |||< qW |sHdS t|td}|| S )zv
        Return a VersionControl object if a repository of that type is found
        at the given directory.
        zDetermine that %s uses VCS: %sN)key)r   r   get_repository_rootrf   r@   r*   maxlen)rv   ri   vcs_backendsvcs_backend	repo_pathinner_most_repo_pathr,   r,   r-   get_backend_for_dirE  s    

zVcsSupport.get_backend_for_dirc             C   s&   x | j  D ]}||jkr|S qW dS )z9
        Return a VersionControl object or None.
        N)r   r   r   )rv   r+   r   r,   r,   r-   get_backend_for_scheme^  s    
z!VcsSupport.get_backend_for_schemec             C   s   |  }| j|S )z9
        Return a VersionControl object or None.
        )lowerr   get)rv   r*   r,   r,   r-   get_backendh  s    zVcsSupport.get_backend)ro   rp   rq   r   r   rw   r   r   r   r   r)   r   r   r   r   r   
__classcell__r,   r,   )r   r-   r     s   
	
r   c               @   s@  e Zd ZdZdZdZdZdZdZe	dd Z
e	dd Ze	dd	 Ze	d
d Z
edd
 Zdd Ze	d9ddZe	dd Zdd Ze	dd Ze	dd Zedd Zdd Zedd Ze	d d! Zd"d# Zd$d% Zd&d' Ze	d(d) Zd*d+ Zd,d- Ze	d.d/ Z e	d0d1 Z!e	d:d3d4Z"e	d5d6 Z#e	d7d8 Z$dS );VersionControlr?   r,   Nc             C   s   |  d| j S )z
        Return whether the vcs prefix (e.g. "git+") should be added to a
        repository's remote url when used in a requirement.
        z{}:)r   
startswithr0   r*   )r   
remote_urlr,   r,   r-   should_add_vcs_url_prefix~  s    z(VersionControl.should_add_vcs_url_prefixc             C   s   dS )z~
        Return the path to setup.py, relative to the repo root.
        Return None if setup.py is in the repo root.
        Nr,   )r   ri   r,   r,   r-   get_subdirectory  s    zVersionControl.get_subdirectoryc             C   s
   |  |S )zR
        Return the revision string that should be used in a requirement.
        )get_revision)r   repo_dirr,   r,   r-   get_requirement_revision  s    z'VersionControl.get_requirement_revisionc             C   sV   |  |}|dkrdS | |r.d| j|}| |}| |}t||||d}|S )aC  
        Return the requirement string to use to redownload the files
        currently at the given repository directory.

        Args:
          project_name: the (unescaped) project name.

        The return value has a form similar to the following:

            {repository_url}@{revision}#egg={project_name}
        Nz{}+{})r4   )get_remote_urlr   r0   r*   r   r   r7   )r   r   r3   r1   revisionr4   r6   r,   r,   r-   get_src_requirement  s    



z"VersionControl.get_src_requirementc             C   s   t dS )z
        Return the base revision arguments for a vcs command.

        Args:
          rev: the name of a revision to install.  Cannot be None.
        N)NotImplementedError)r2   r,   r,   r-   r{     s    	z VersionControl.get_base_rev_argsc             C   s   dS )aZ  
        Return true if the commit hash checked out at dest matches
        the revision in url.

        Always return False, if the VCS does not support immutable commit
        hashes.

        This method does not check if there are local uncommitted changes
        in dest after checkout, as pip currently has no use case for that.
        Fr,   )rv   urldestr,   r,   r-   is_immutable_rev_checkout  s    z(VersionControl.is_immutable_rev_checkoutc             C   s   t | ||dS )z
        Return a RevOptions object.

        Args:
          rev: the name of a revision to install.
          extra_args: a list of extra options.
        )rs   )rr   )r   r2   rs   r,   r,   r-   r     s    
zVersionControl.make_rev_optionsc             C   s&   t j|\}}|t jjp$t|S )zy
           posix absolute paths start with os.path.sep,
           win32 ones start with drive (like c:\folder)
        )rA   rc   
splitdriver   sepbool)r   repodrivetailr,   r,   r-   _is_local_repository  s    z#VersionControl._is_local_repositoryc             C   s   t dS )z
        Export the repository at the url to the destination location
        i.e. only download the files, without vcs informations

        :param url: the repository URL starting with a vcs prefix.
        N)r   )rv   ri   r   r,   r,   r-   export  s    zVersionControl.exportc             C   s   |dfS )aZ  
        Parse the repository URL's netloc, and return the new netloc to use
        along with auth information.

        Args:
          netloc: the original repository URL netloc.
          scheme: the repository URL's scheme without the vcs prefix.

        This is mainly for the Subversion class to override, so that auth
        information can be provided via the --username and --password options
        instead of through the URL.  For other subclasses like Git without
        such an option, auth information must stay in the URL.

        Returns: (netloc, (username, password)).
        )NNr,   )r   netlocr+   r,   r,   r-   get_netloc_and_auth  s    z"VersionControl.get_netloc_and_authc       	      C   s   t |\}}}}}d|kr*td||ddd }| ||\}}d}d|krx|dd\}}|sxtd|t ||||df}|||fS )z
        Parse the repository URL to use, and return the URL, revision,
        and auth info to use.

        Returns: (url, rev, (username, password)).
        +zvSorry, {!r} is a malformed VCS url. The format is <vcs>+<protocol>://<url>, e.g. svn+http://myrepo/svn/MyApp#egg=MyApp   N@zyThe URL {!r} has an empty revision (after @) which is not supported. Include a revision after @ or remove @ from the URL.r?   )	r   urlsplit
ValueErrorr0   splitr   rsplitr   
urlunsplit)	r   r   r+   r   rc   queryfrag	user_passr2   r,   r,   r-   get_url_rev_and_auth  s     	z#VersionControl.get_url_rev_and_authc             C   s   g S )zM
        Return the RevOptions "extra arguments" to use in obtain().
        r,   )usernamepasswordr,   r,   r-   
make_rev_args   s    zVersionControl.make_rev_argsc       
      C   sT   |  |j\}}}|\}}d}|dk	r.t|}| ||}| j||d}	t||	fS )z
        Return the URL and RevOptions object to use in obtain() and in
        some cases export(), as a tuple (url, rev_options).
        N)rs   )r   secretr   r   r   r   )
rv   r   
secret_urlr2   r   r   secret_passwordr   rs   rev_optionsr,   r,   r-   get_url_rev_options(  s    z"VersionControl.get_url_rev_optionsc             C   s   t | dS )zi
        Normalize a URL for comparison by unquoting it and removing any
        trailing slash.
        /)r   unquoterM   )r   r,   r,   r-   
normalize_url8  s    zVersionControl.normalize_urlc             C   s   |  ||  |kS )zV
        Compare two repo URLs for identity, ignoring incidental differences.
        )r   )r   url1url2r,   r,   r-   compare_urlsA  s    zVersionControl.compare_urlsc             C   s   t dS )z
        Fetch a revision from a repository, in the case that this is the
        first fetch from the repository.

        Args:
          dest: the directory to fetch the repository to.
          rev_options: a RevOptions object.
        N)r   )rv   r   r   r   r,   r,   r-   	fetch_newI  s    
zVersionControl.fetch_newc             C   s   t dS )z}
        Switch the repo at ``dest`` to point to ``URL``.

        Args:
          rev_options: a RevOptions object.
        N)r   )rv   r   r   r   r,   r,   r-   switchU  s    zVersionControl.switchc             C   s   t dS )z
        Update an already-existing repo to the given ``rev_options``.

        Args:
          rev_options: a RevOptions object.
        N)r   )rv   r   r   r   r,   r,   r-   rD   _  s    zVersionControl.updatec             C   s   t dS )z
        Return whether the id of the current commit equals the given name.

        Args:
          dest: the repository directory.
          name: a string name.
        N)r   )r   r   r*   r,   r,   r-   is_commit_id_equali  s    
z!VersionControl.is_commit_id_equalc       	      C   s  |  |\}}tj|s,| ||| dS | }| |r| |}| ||j	rt
d| j
 t|| | ||jst
dt|| j| | ||| n
t
d dS t
d| j| jt|| d}nt
d|| j| j d}t
d	| j| td
|d |d }|d
kr&td |dkrZt
dt| t| | ||| dS |dkrt|}t
dt|| t|| | ||| dS |dkrt
d| jt||| | ||| dS )a  
        Install or update in editable mode the package represented by this
        VersionControl object.

        :param dest: the repository directory in which to install or update.
        :param url: the repository URL starting with a vcs prefix.
        Nz)%s in %s exists, and has correct URL (%s)zUpdating %s %s%sz$Skipping because already up-to-date.z%s %s in %s exists with URL %s)z%(s)witch, (i)gnore, (w)ipe, (b)ackup )siwbz0Directory %s already exists, and is not a %s %s.)z(i)gnore, (w)ipe, (b)ackup )r   r   r   z+The plan is to install the %s repository %szWhat to do?  {}r   r   ar   zDeleting %sr   zBacking up %s to %sr   zSwitching %s %s to %s%s)r   rA   rc   rd   r   r~   is_repository_directoryr   r   r   rf   r@   	repo_nametitler
   r   r2   inforD   rg   r*   r   r0   sysexitr   r   shutilmover   )	rv   r   r   r   rev_displayexisting_urlpromptresponsedest_dirr,   r,   r-   obtainu  s~    	







zVersionControl.obtainc             C   s&   t j|rt| | j||d dS )z
        Clean up current location and download the url repository
        (and vcs infos) into location

        :param url: the repository URL starting with a vcs prefix.
        )r   N)rA   rc   rd   r   r   )rv   ri   r   r,   r,   r-   unpack  s    zVersionControl.unpackc             C   s   t dS )z
        Return the url used at location

        Raises RemoteNotFoundError if the repository does not have a remote
        url configured.
        N)r   )r   ri   r,   r,   r-   r     s    	zVersionControl.get_remote_urlc             C   s   t dS )zR
        Return the current commit id of the files at the given location.
        N)r   )r   ri   r,   r,   r-   r     s    zVersionControl.get_revisionTc          
   C   sl   t | jf| }yt|||||dS  tk
rf } z&|jtjkrTtdjf t n W dd}~X Y nX dS )z
        Run a VCS subcommand
        This is simply a wrapper around call_subprocess that adds the VCS
        command name, and checks that the VCS is available
        )rT   rU   rV   zWCannot find command {cls.name!r} - do you have {cls.name!r} installed and in your PATH?N)	r   r*   rb   OSErrorerrnoENOENTr   r0   locals)r   rS   r:   rT   rU   rV   er,   r,   r-   run_command  s    zVersionControl.run_commandc             C   s,   t d|| j| j tjtj|| jS )zL
        Return whether a directory path is a repository directory.
        zChecking in %s for %s (%s)...)rf   r@   re   r*   rA   rc   rd   rR   )r   rc   r,   r,   r-   r     s    z&VersionControl.is_repository_directoryc             C   s   |  |r|S dS )ay  
        Return the "root" (top-level) directory controlled by the vcs,
        or `None` if the directory is not in any.

        It is meant to be overridden to implement smarter detection
        mechanisms for specific vcs.

        This can do more than is_repository_directory() alone. For
        example, the Git override checks that Git is actually available.
        N)r   )r   ri   r,   r,   r-   r     s    
z"VersionControl.get_repository_root)NN)NNNT)%ro   rp   rq   r*   re   r   r   
unset_environry   classmethodr   r   r   r   staticmethodr{   r   r   r   r   r   r   r   r   r   r   r   r   rD   r   r   r   r   r   r   r   r   r,   r,   r,   r-   r   t  sJ   		

	

^   
r   )N)NNNT)Fr   
__future__r   r   loggingrA   r   rE   r   pip._vendorr   Zpip._vendor.six.moves.urllibr   r   pip._internal.exceptionsr   r   r   pip._internal.utils.compatr   r	   pip._internal.utils.loggingr
   pip._internal.utils.miscr   r   r
   r   r   r   pip._internal.utils.subprocessr   r   r   r   pip._internal.utils.typingr   pip._internal.utils.urlsr   typingr   r   r   r   r   r   r   r   r   r    r!   r"   r#   strZAuthInfo__all__	getLoggerro   rf   r.   r7   rb   rm   rJ   rn   objectrr   r   r$   r   r,   r,   r,   r-   <module>   sF    4

   
QI^                                                                        python3.7/site-packages/pip/_internal/vcs/__pycache__/subversion.cpython-37.pyc                     0000644                 00000020431 15107360303 0023427 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Reo0                 @   s   d dl mZ d dlZd dlZd dlZd dlmZ d dlmZm	Z	m
Z
mZ d dlm
Z
 d dlmZ d dlmZmZ edZed	Zed
ZedZerd dlmZmZ d d
lmZ d dlmZ d dlmZmZ ee Z!G dd deZ"e#e" dS )    )absolute_importN)
indent_log)display_pathis_console_interactivermtreesplit_auth_from_netloc)make_command)MYPY_CHECK_RUNNING)VersionControlvcsz
url="([^"]+)"zcommitted-rev="(\d+)"z\s*revision="(\d+)"z<url>(.*)</url>)OptionalTuple)CommandArgs)
HiddenText)AuthInfo
RevOptionsc                   s   e Zd ZdZdZdZdZedd Ze	dd Z
ed	d
 Ze fddZe fd
dZ
e	dd Zedd Zedd Zedd Zd( fdd	Zdd Zdd Zdd Zd d! Zd"d# Zd$d% Zd&d' Z  ZS ))
Subversionsvnz.svncheckout)r   zsvn+sshzsvn+httpz	svn+httpszsvn+svnc             C   s   dS )NT )cls
remote_urlr   r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_internal/vcs/subversion.pyshould_add_vcs_url_prefix+   s    z$Subversion.should_add_vcs_url_prefixc             C   s   d| gS )Nz-rr   )revr   r   r   get_base_rev_args/   s    zSubversion.get_base_rev_argsc       	      C   s   d}xt |D ]\}}}| j|kr2g |dd< q|| j t j|| jd}t j|s^q| |\}}||kr~|d }n|r||sg |dd< qt	||}qW |S )zR
        Return the maximum revision for all files under a given location
        r   Nentries/)
oswalkdirnameremovepathjoinexists_get_svn_url_rev
startswithmax)	r   locationrevisionbasedirs_
entries_fndirurllocalrevr   r   r   get_revision3   s"    

zSubversion.get_revisionc                s"   |dkrt t| ||S t|S )z
        This override allows the auth information to be passed to svn via the
        --username and --password options instead of via the URL.
        ssh)superr   get_netloc_and_authr   )r   netlocscheme)	__class__r   r   r3   O   s    zSubversion.get_netloc_and_authc                s2   t t| |\}}}|dr(d| }|||fS )Nzssh://zsvn+)r2   r   get_url_rev_and_authr&   )r   urlr   	user_pass)r6   r   r   r7   \   s    
zSubversion.get_url_rev_and_authc             C   s(   g }| r|d| g7 }|r$|d|g7 }|S )Nz
--usernamez
--passwordr   )usernamepassword
extra_argsr   r   r   
make_rev_argse   s    zSubversion.make_rev_argsc             C   sV   |}xBt jt j|dsF|}t j|}||krtd| d S qW | |d S )Nzsetup.pyzGCould not find setup.py for directory %s (tried all parent directories)r   )r   r"   r$   r#   r    loggerwarningr%   )r   r(   
orig_location
last_locationr   r   r   get_remote_urlp   s    zSubversion.get_remote_urlc          	   C   sr  ddl m} tj|| jd}tj|rHt|}| }W d Q R X nd}|	dsj|	dsj|	drt
ttj
|d}|d d= |d d	 }d
d |D dg }n|	drt|}|std
jf t |d}dd t|D dg }nZy8| dd|g}	t|	d}dd t|	D }W n  |k
rT   d g  }}Y nX |rft|}
nd}
||
fS )Nr   )SubProcessErrorr    8910z

   c             S   s,   g | ]$}t |d kr|d  rt|d  qS )	   )lenint).0dr   r   r   
<listcomp>   s    z/Subversion._get_svn_url_rev.<locals>.<listcomp>z<?xmlzBadly formatted data: {data!r}   c             S   s   g | ]}t |d qS )rO   )rK   group)rL   mr   r   r   rN      s    infoz--xmlc             S   s   g | ]}t |d qS )rO   )rK   rP   )rL   rQ   r   r   r   rN      s    )pip._internal.exceptionsrC   r   r"   r#   r    r$   openreadr&   listmapstr
splitlinessplit_svn_xml_url_research
ValueErrorformatlocalsrP   _svn_rev_refinditerrun_command_svn_info_xml_url_re_svn_info_xml_rev_rer'   )r   r(   rC   entries_pathfdatar8   revsmatchxmlr   r   r   r   r%      s>    








zSubversion._get_svn_url_revc             C   s   dS )z&Always assume the versions don't matchFr   )r   destnamer   r   r   is_commit_id_equal   s    zSubversion.is_commit_id_equalNc                s,   |d krt  }|| _d | _tt|   d S )N)r   use_interactive_vcs_versionr2   r   __init__)selfrn   )r6   r   r   rp      s
    zSubversion.__init__c             C   sv   d}|  dg}||sdS |t|d  d }|dd d}yttt|}W n tk
rp   dS X |S )a  Query the version of the currently installed Subversion client.

        :return: A tuple containing the parts of the version information or
            ``()`` if the version returned from ``svn`` could not be parsed.
        :raises: BadCommand: If ``svn`` is not installed.
        z
svn, version z	--versionr   Nr   -.)	rb   r&   rJ   rZ   	partitiontuplerW   rK   r]   )rq   version_prefixversionversion_listparsed_versionr   r   r   call_vcs_version   s    
zSubversion.call_vcs_versionc             C   s"   | j dk	r| j S |  }|| _ |S )a  Return the version of the currently installed Subversion client.

        If the version of the Subversion client has already been queried,
        a cached value will be used.

        :return: A tuple containing the parts of the version information or
            ``()`` if the version returned from ``svn`` could not be parsed.
        :raises: BadCommand: If ``svn`` is not installed.
        N)ro   rz   )rq   vcs_versionr   r   r   get_vcs_version   s
    
zSubversion.get_vcs_versionc             C   s&   | j sdgS |  }|dkr"dgS g S )aX  Return options to be used on calls to Subversion that contact the server.

        These options are applicable for the following ``svn`` subcommands used
        in this class.

            - checkout
            - export
            - switch
            - update

        :return: A list of command line arguments to pass to ``svn``.
        z--non-interactive)rO      z--force-interactive)rn   r|   )rq   svn_versionr   r   r   get_remote_call_options   s    	z"Subversion.get_remote_call_optionsc          	   C   sh   |  |\}}td|| t < tj|r8t| td| 	 |
 ||}| | W dQ R X dS )z@Export the svn repository at the url to the destination locationz!Exporting svn repository %s to %sexportN)get_url_rev_optionsr>   rR   r   r   r"   r$   r   r   r   to_argsrb   )rq   r(   r8   rev_optionscmd_argsr   r   r   r     s    zSubversion.exportc             C   sD   |  }td||t| tdd|  | ||}| | d S )NzChecking out %s%s to %sr   z-q)
to_displayr>   rR   r   r   r   r   rb   )rq   rk   r8   r   rev_displayr   r   r   r   	fetch_new0  s    

zSubversion.fetch_newc             C   s&   t d|  | ||}| | d S )Nswitch)r   r   r   rb   )rq   rk   r8   r   r   r   r   r   r   ?  s    zSubversion.switchc             C   s$   t d|  | |}| | d S )Nupdate)r   r   r   rb   )rq   rk   r8   r   r   r   r   r   r   G  s    zSubversion.update)N)__name__
__module____qualname__rl   r    	repo_nameschemesclassmethodr   staticmethodr   r0   r3   r7   r=   rB   r%   rm   rp   rz   r|   r   r   r   r   r   
__classcell__r   r   )r6   r   r   %   s*   
	2!r   )$
__future__r   loggingr   repip._internal.utils.loggingr   pip._internal.utils.miscr   r   r   r   pip._internal.utils.subprocessr   pip._internal.utils.typingr	    pip._internal.vcs.versioncontrolr
   r   compiler[   r`   rd   rc   typingr   r
   r   r   r   r   	getLoggerr   r>   r   registerr   r   r   r   <module>   s,   




  -                                                                                                                                                                                                                                       python3.7/site-packages/pip/_internal/vcs/__pycache__/git.cpython-37.pyc                            0000644                 00000022473 15107360303 0022023 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re6                 @   s  d dl mZ d dlZd dlZd dlZd dlmZ d dl	mZ
 d dl	mZ d dl
mZmZ d dlmZmZ d dlmZ d dlmZ d d	lmZ d d
lmZmZmZmZ erd dlmZm Z  d dlm!Z! d d
lm"Z"m#Z# e
j$Z$e
j%Z%e&e'Z(e)dZ*dd Z+G dd deZ,e-e, dS )    )absolute_importN)parse)request)
BadCommandSubProcessError)display_pathhide_url)make_command)
TempDirectory)MYPY_CHECK_RUNNING)RemoteNotFoundErrorVersionControl!find_path_to_setup_from_repo_rootvcs)OptionalTuple)
HiddenText)AuthInfo
RevOptionsz^[a-fA-F0-9]{40}$c             C   s   t t| S )N)bool
HASH_REGEXmatch)sha r   /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_internal/vcs/git.pylooks_like_hash*   s    r   c                   s   e Zd ZdZdZdZdZdZdZe	dd Z
d	d
 Zdd Ze
d
d Zdd Ze
dd Ze
dd Ze
dd Zdd Zdd Zdd Ze
dd Ze
d*d d!Ze
d"d# Ze
 fd$d%Ze
d&d' Ze
 fd(d)Z  ZS )+Gitgitz.gitclone)r   zgit+httpz	git+httpszgit+sshzgit+gitzgit+file)GIT_DIR
GIT_WORK_TREEHEADc             C   s   | gS )Nr   )revr   r   r   get_base_rev_args:   s    zGit.get_base_rev_argsc             C   sJ   |  t|\}}|jsdS | ||js.dS t| ||jd }| S )NFr   )get_url_rev_optionsr   r"   is_commit_id_equalr   get_revision_sha)selfurldest_rev_optionsis_tag_or_branchr   r   r   is_immutable_rev_checkout>   s    zGit.is_immutable_rev_checkoutc             C   sX   d}|  dg}||r4|t|d   d }nd}d|dd d }t|S )Nzgit version versionr    .   )run_command
startswithlensplitjoin
parse_version)r'   VERSION_PFXr.   r   r   r   get_git_versionO   s    
zGit.get_git_versionc             C   s@   dddg}| j |d|d}| }|dr<|tdd S dS )zl
        Return the current branch, or None if HEAD isn't at a branch
        (e.g. detached HEAD).
        zsymbolic-refz-qr!   )   )extra_ok_returncodescwdzrefs/heads/N)r2   stripr3   r4   )clslocationargsoutputrefr   r   r   get_current_branch\   s    


zGit.get_current_branchc          	   C   sV   | ds|d }tdd0}| j|j|d | jdddd|g|jd	 W d
Q R X d
S )z@Export the Git repository at the url to the destination location/export)kind)r(   zcheckout-indexz-az-fz--prefix)r<   N)endswithr
   unpackpathr2   )r'   r?   r(   temp_dirr   r   r   rE   q   s    
z
Git.exportc       
   	   C   s   d}y| j d|g|d}W n tk
r.   Y nX i }xP|  D ]@}y| \}}W n" tk
rx   td|Y nX |||< qBW d|}d|}	||}|dk	r|dfS ||	}|d	fS )
z
        Return (sha_or_none, is_branch), where sha_or_none is a commit hash
        if the revision names a remote branch or tag, otherwise None.

        Args:
          dest: the repository directory.
          rev: the revision name.
        r/   zshow-ref)r<   zunexpected show-ref line: {!r}zrefs/remotes/origin/{}zrefs/tags/{}NTF)r2   r   r=   
splitlinesr5   
ValueErrorformatget)
r>   r)   r"   rA   refsliner   rB   
branch_reftag_refr   r   r   r&   ~   s&    



zGit.get_revision_shac             C   s   |j }|dk	st| ||\}}|dk	rF||}|r<|nd|_|S t|sZtd| |dsh|S | j	t
dd|| |d | j|dd}||}|S )	z
        Resolve a revision to a new RevOptions object with the SHA1 of the
        branch, tag, or ref if found.

        Args:
          rev_options: a RevOptions object.
        Nz:Did not find branch or tag '%s', assuming revision or ref.zrefs/fetchz-q)r<   
FETCH_HEAD)r"   )
arg_revAssertionErrorr&   make_newbranch_namer   loggerwarningr3   r2   r	   to_argsget_revision)r>   r)   r(   r+   r"   r   	is_branchr   r   r   resolve_revision   s&    



zGit.resolve_revisionc             C   s   |sdS |  ||kS )z
        Return whether the current commit hash equals the given name.

        Args:
          dest: the repository directory.
          name: a string name.
        F)r\   )r>   r)   namer   r   r   r%      s    	zGit.is_commit_id_equalc             C   s   |  }td||t| | tdd|| |jr| |||}t|dd }|d kr| 	||jstdd|
 }| j||d n4| ||krd|}dd|d	|g}| j||d | 
| d S )
NzCloning %s%s to %sr   z-qrX   checkout)r<   z	origin/{}z-bz--track)
to_displayrY   infor   r2   r	   r"   r^   getattrr%   r[   rC   rM   update_submodules)r'   r)   r(   r+   rev_displayrX   cmd_argstrack_branchr   r   r   	fetch_new   s     
z
Git.fetch_newc             C   sB   | j tdd||d tdd| }| j ||d | | d S )Nconfigzremote.origin.url)r<   r`   z-q)r2   r	   r[   rd   )r'   r)   r(   r+   rf   r   r   r   switch   s    
z
Git.switchc             C   st   |   tdkr&| jdddg|d n| jddg|d | |||}tddd| }| j||d | | d S )Nz1.9.0rS   z-qz--tags)r<   resetz--hard)r9   r7   r2   r^   r	   r[   rd   )r'   r)   r(   r+   rf   r   r   r   update  s    z
Git.updatec             C   sx   | j dddgd|d}| }y|d }W n tk
rB   tY nX x|D ]}|drJ|}P qJW |dd	 }| S )
z
        Return URL of the first remote encountered.

        Raises RemoteNotFoundError if the repository does not have a remote
        url configured.
        ri   z--get-regexpzremote\..*\.url)r:   )r;   r<   r   zremote.origin.url  r:   )r2   rK   
IndexErrorr   r3   r5   r=   )r>   r?   stdoutremotesfound_remoteremoter(   r   r   r   get_remote_url  s    




zGit.get_remote_urlNc             C   s&   |d krd}| j d|g|d}| S )Nr!   z	rev-parse)r<   )r2   r=   )r>   r?   r"   current_revr   r   r   r\   5  s
    zGit.get_revisionc             C   sP   | j ddg|d }tj|s0tj||}tjtj|d}t||S )z~
        Return the path to setup.py, relative to the repo root.
        Return None if setup.py is in the repo root.
        z	rev-parsez	--git-dir)r<   z..)r2   r=   osrI   isabsr6   abspathr   )r>   r?   git_dir	repo_rootr   r   r   get_subdirectory>  s    zGit.get_subdirectoryc                s   t |\}}}}}|dr|dt|d  }|t|ddd }t|||||f}|dd }	|d|	 t||	d ||||f }d|krd|kst	|d	d
}t
t| |\}}
}|dd}nt
t| |\}}
}||
|fS )
a9  
        Prefixes stub URLs like 'user@hostname:user/repo.git' with 'ssh://'.
        That's required because although they use SSH they sometimes don't
        work with a ssh:// scheme (e.g. GitHub). But we need a scheme for
        parsing. Hence we remove it again afterwards and return it as a stub.
        fileNrD   \+r:   z://zfile:zgit+z
git+ssh://zssh://r/   )
urlsplitrG   r4   lstripurllib_requesturl2pathnamereplace
urlunsplitfindrV   superr   get_url_rev_and_auth)r>   r(   schemenetlocrI   queryfragmentinitial_slashesnewpath
after_plusr"   	user_pass)	__class__r   r   r   M  s"    

zGit.get_url_rev_and_authc             C   s6   t jt j|dsd S | jdddddg|d d S )Nz.gitmodules	submodulerl   z--initz--recursivez-q)r<   )ru   rI   existsr6   r2   )r>   r?   r   r   r   rd   p  s
    zGit.update_submodulesc                sv   t t| |}|r|S y| jddg|dd}W n2 tk
rP   td| d S  tk
rb   d S X tj	
|dS )Nz	rev-parsez--show-toplevelF)r<   log_failed_cmdzKcould not determine if %s is under git control because git is not availablez
)r   r   get_repository_rootr2   r   rY   debugr   ru   rI   normpathrstrip)r>   r?   locr)r   r   r   r   y  s    zGit.get_repository_root)N)__name__
__module____qualname__r_   dirname	repo_nameschemes
unset_environdefault_arg_revstaticmethodr#   r-   r9   classmethodrC   rE   r&   r^   r%   rh   rj   rl   rs   r\   rz   r   rd   r   
__classcell__r   r   )r   r   r   .   s0   

(-#	r   ).
__future__r   loggingos.pathru   repip._vendor.packaging.versionr   r7   Zpip._vendor.six.moves.urlliburllib_parser   r   pip._internal.exceptionsr   r   pip._internal.utils.miscr   r   pip._internal.utils.subprocessr	   pip._internal.utils.temp_dirr
   pip._internal.utils.typingr    pip._internal.vcs.versioncontrolr   r
   r   r   typingr   r   r   r   r   r~   r   	getLoggerr   rY   compiler   r   r   registerr   r   r   r   <module>   s2   

  a                                                                                                                                                                                                     python3.7/site-packages/pip/_internal/vcs/__pycache__/bazaar.cpython-37.pyc                         0000644                 00000007206 15107360303 0022475 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       B

    Re/                 @   s   d dl mZ d dlZd dlZd dlmZ d dlmZm	Z	 d dl
mZ d dlm
Z
 d dlmZ d dlmZmZ e
rd d	lmZmZ d d
lmZ d dlmZmZ eeZG dd
 d
eZee dS )    )absolute_importN)parse)display_pathrmtree)make_command)MYPY_CHECK_RUNNING)path_to_url)VersionControlvcs)OptionalTuple)
HiddenText)AuthInfo
RevOptionsc                   s   e Zd ZdZdZdZdZ fddZedd Z	d	d
 Z
dd Zd
d Zdd Z
e fddZedd Zedd Zedd Z  ZS )Bazaarbzrz.bzrbranch)r   zbzr+httpz	bzr+httpszbzr+sshzbzr+sftpzbzr+ftpzbzr+lpc                s0   t t| j|| ttdd r,tjdg d S )N
uses_fragmentlp)superr   __init__getattrurllib_parser   extend)selfargskwargs)	__class__ /builddir/build/BUILDROOT/alt-python37-pip-20.2.4-6.el8.x86_64/opt/alt/python37/lib/python3.7/site-packages/pip/_internal/vcs/bazaar.pyr   #   s    zBazaar.__init__c             C   s   d| gS )Nz-rr   )revr   r   r   get_base_rev_args*   s    zBazaar.get_base_rev_argsc             C   s>   t j|rt| | |\}}| td|||  dS )zU
        Export the Bazaar repository at the url to the destination location
        exportN)ospathexistsr   get_url_rev_optionsrun_commandr   to_args)r   locationurlrev_optionsr   r   r   r"   .   s
    z
Bazaar.exportc             C   s>   |  }td||t| tdd| ||}| | d S )NzChecking out %s%s to %sr   z-q)
to_displayloggerinfor   r   r(   r'   )r   destr*   r+   rev_displaycmd_argsr   r   r   	fetch_new<   s    
zBazaar.fetch_newc             C   s   | j td||d d S )Nswitch)cwd)r'   r   )r   r/   r*   r+   r   r   r   r3   J   s    z
Bazaar.switchc             C   s"   t dd| }| j||d d S )Npullz-q)r4   )r   r(   r'   )r   r/   r*   r+   r1   r   r   r   updateN   s    z
Bazaar.updatec                s2   t t| |\}}}|dr(d| }|||fS )Nzssh://zbzr+)r   r   get_url_rev_and_auth
startswith)clsr*   r    	user_pass)r   r   r   r7   S   s    
zBazaar.get_url_rev_and_authc             C   sj   | j dg|d}xT| D ]H}| }x:dD ]2}||r,||d }| |rZt|S |S q,W qW d S )Nr.   )r4   )zcheckout of branch: zparent branch:    )r'   
splitlinesstripr8   split_is_local_repositoryr   )r9   r)   urlslinexrepor   r   r   get_remote_url\   s    


zBazaar.get_remote_urlc             C   s   | j dg|d}| d S )Nrevno)r4   )r'   r<   )r9   r)   revisionr   r   r   get_revisionj   s    zBazaar.get_revisionc             C   s   dS )z&Always assume the versions don't matchFr   )r9   r/   namer   r   r   is_commit_id_equalq   s    zBazaar.is_commit_id_equal)__name__
__module____qualname__rI   dirname	repo_nameschemesr   staticmethodr!   r"   r2   r3   r6   classmethodr7   rD   rH   rJ   
__classcell__r   r   )r   r   r      s   	r   )
__future__r   loggingr#   Zpip._vendor.six.moves.urllibr   r   pip._internal.utils.miscr   r   pip._internal.utils.subprocessr   pip._internal.utils.typingr   pip._internal.utils.urlsr    pip._internal.vcs.versioncontrolr	   r
   typingr   r   r
   r   r   	getLoggerrK   r-   r   registerr   r   r   r   <module>   s   
]                                                                                                                                                                                                                                                                                                                                                                                          python3.7/site-packages/pip/_internal/vcs/versioncontrol.py                                         0000644                 00000062556 15107360303 0020047 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       """Handles all VCS (version control) support"""

from __future__ import absolute_import

import errno
import logging
import os
import shutil
import subprocess
import sys

from pip._vendor import pkg_resources
from pip._vendor.six.moves.urllib import parse as urllib_parse

from pip._internal.exceptions import (
    BadCommand,
    InstallationError,
    SubProcessError,
)
from pip._internal.utils.compat import console_to_str, samefile
from pip._internal.utils.logging import subprocess_logger
from pip._internal.utils.misc import (
    ask_path_exists,
    backup_dir,
    display_path,
    hide_url,
    hide_value,
    rmtree,
)
from pip._internal.utils.subprocess import (
    format_command_args,
    make_command,
    make_subprocess_output_error,
    reveal_command_args,
)
from pip._internal.utils.typing import MYPY_CHECK_RUNNING
from pip._internal.utils.urls import get_url_scheme

if MYPY_CHECK_RUNNING:
    from typing import (
        Dict, Iterable, Iterator, List, Optional, Text, Tuple,
        Type, Union, Mapping, Any
    )
    from pip._internal.utils.misc import HiddenText
    from pip._internal.utils.subprocess import CommandArgs

    AuthInfo = Tuple[Optional[str], Optional[str]]


__all__ = ['vcs']


logger = logging.getLogger(__name__)


def is_url(name):
    # type: (Union[str, Text]) -> bool
    """
    Return true if the name looks like a URL.
    """
    scheme = get_url_scheme(name)
    if scheme is None:
        return False
    return scheme in ['http', 'https', 'file', 'ftp'] + vcs.all_schemes


def make_vcs_requirement_url(repo_url, rev, project_name, subdir=None):
    # type: (str, str, str, Optional[str]) -> str
    """
    Return the URL for a VCS requirement.

    Args:
      repo_url: the remote VCS url, with any needed VCS prefix (e.g. "git+").
      project_name: the (unescaped) project name.
    """
    egg_project_name = pkg_resources.to_filename(project_name)
    req = '{}@{}#egg={}'.format(repo_url, rev, egg_project_name)
    if subdir:
        req += '&subdirectory={}'.format(subdir)

    return req


def call_subprocess(
    cmd,  # type: Union[List[str], CommandArgs]
    cwd=None,  # type: Optional[str]
    extra_environ=None,  # type: Optional[Mapping[str, Any]]
    extra_ok_returncodes=None,  # type: Optional[Iterable[int]]
    log_failed_cmd=True  # type: Optional[bool]
):
    # type: (...) -> Text
    """
    Args:
      extra_ok_returncodes: an iterable of integer return codes that are
        acceptable, in addition to 0. Defaults to None, which means [].
      log_failed_cmd: if false, failed commands are not logged,
        only raised.
    """
    if extra_ok_returncodes is None:
        extra_ok_returncodes = []

    # log the subprocess output at DEBUG level.
    log_subprocess = subprocess_logger.debug

    env = os.environ.copy()
    if extra_environ:
        env.update(extra_environ)

    # Whether the subprocess will be visible in the console.
    showing_subprocess = True

    command_desc = format_command_args(cmd)
    try:
        proc = subprocess.Popen(
            # Convert HiddenText objects to the underlying str.
            reveal_command_args(cmd),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            cwd=cwd
        )
        if proc.stdin:
            proc.stdin.close()
    except Exception as exc:
        if log_failed_cmd:
            subprocess_logger.critical(
                "Error %s while executing command %s", exc, command_desc,
            )
        raise
    all_output = []
    while True:
        # The "line" value is a unicode string in Python 2.
        line = None
        if proc.stdout:
            line = console_to_str(proc.stdout.readline())
        if not line:
            break
        line = line.rstrip()
        all_output.append(line + '\n')

        # Show the line immediately.
        log_subprocess(line)
    try:
        proc.wait()
    finally:
        if proc.stdout:
            proc.stdout.close()

    proc_had_error = (
        proc.returncode and proc.returncode not in extra_ok_returncodes
    )
    if proc_had_error:
        if not showing_subprocess and log_failed_cmd:
            # Then the subprocess streams haven't been logged to the
            # console yet.
            msg = make_subprocess_output_error(
                cmd_args=cmd,
                cwd=cwd,
                lines=all_output,
                exit_status=proc.returncode,
            )
            subprocess_logger.error(msg)
        exc_msg = (
            'Command errored out with exit status {}: {} '
            'Check the logs for full command output.'
        ).format(proc.returncode, command_desc)
        raise SubProcessError(exc_msg)
    return ''.join(all_output)


def find_path_to_setup_from_repo_root(location, repo_root):
    # type: (str, str) -> Optional[str]
    """
    Find the path to `setup.py` by searching up the filesystem from `location`.
    Return the path to `setup.py` relative to `repo_root`.
    Return None if `setup.py` is in `repo_root` or cannot be found.
    """
    # find setup.py
    orig_location = location
    while not os.path.exists(os.path.join(location, 'setup.py')):
        last_location = location
        location = os.path.dirname(location)
        if location == last_location:
            # We've traversed up to the root of the filesystem without
            # finding setup.py
            logger.warning(
                "Could not find setup.py for directory %s (tried all "
                "parent directories)",
                orig_location,
            )
            return None

    if samefile(repo_root, location):
        return None

    return os.path.relpath(location, repo_root)


class RemoteNotFoundError(Exception):
    pass


class RevOptions(object):

    """
    Encapsulates a VCS-specific revision to install, along with any VCS
    install options.

    Instances of this class should be treated as if immutable.
    """

    def __init__(
        self,
        vc_class,  # type: Type[VersionControl]
        rev=None,  # type: Optional[str]
        extra_args=None,  # type: Optional[CommandArgs]
    ):
        # type: (...) -> None
        """
        Args:
          vc_class: a VersionControl subclass.
          rev: the name of the revision to install.
          extra_args: a list of extra options.
        """
        if extra_args is None:
            extra_args = []

        self.extra_args = extra_args
        self.rev = rev
        self.vc_class = vc_class
        self.branch_name = None  # type: Optional[str]

    def __repr__(self):
        # type: () -> str
        return '<RevOptions {}: rev={!r}>'.format(self.vc_class.name, self.rev)

    @property
    def arg_rev(self):
        # type: () -> Optional[str]
        if self.rev is None:
            return self.vc_class.default_arg_rev

        return self.rev

    def to_args(self):
        # type: () -> CommandArgs
        """
        Return the VCS-specific command arguments.
        """
        args = []  # type: CommandArgs
        rev = self.arg_rev
        if rev is not None:
            args += self.vc_class.get_base_rev_args(rev)
        args += self.extra_args

        return args

    def to_display(self):
        # type: () -> str
        if not self.rev:
            return ''

        return ' (to revision {})'.format(self.rev)

    def make_new(self, rev):
        # type: (str) -> RevOptions
        """
        Make a copy of the current instance, but with a new rev.

        Args:
          rev: the name of the revision for the new object.
        """
        return self.vc_class.make_rev_options(rev, extra_args=self.extra_args)


class VcsSupport(object):
    _registry = {}  # type: Dict[str, VersionControl]
    schemes = ['ssh', 'git', 'hg', 'bzr', 'sftp', 'svn']

    def __init__(self):
        # type: () -> None
        # Register more schemes with urlparse for various version control
        # systems
        urllib_parse.uses_netloc.extend(self.schemes)
        # Python >= 2.7.4, 3.3 doesn't have uses_fragment
        if getattr(urllib_parse, 'uses_fragment', None):
            urllib_parse.uses_fragment.extend(self.schemes)
        super(VcsSupport, self).__init__()

    def __iter__(self):
        # type: () -> Iterator[str]
        return self._registry.__iter__()

    @property
    def backends(self):
        # type: () -> List[VersionControl]
        return list(self._registry.values())

    @property
    def dirnames(self):
        # type: () -> List[str]
        return [backend.dirname for backend in self.backends]

    @property
    def all_schemes(self):
        # type: () -> List[str]
        schemes = []  # type: List[str]
        for backend in self.backends:
            schemes.extend(backend.schemes)
        return schemes

    def register(self, cls):
        # type: (Type[VersionControl]) -> None
        if not hasattr(cls, 'name'):
            logger.warning('Cannot register VCS %s', cls.__name__)
            return
        if cls.name not in self._registry:
            self._registry[cls.name] = cls()
            logger.debug('Registered VCS backend: %s', cls.name)

    def unregister(self, name):
        # type: (str) -> None
        if name in self._registry:
            del self._registry[name]

    def get_backend_for_dir(self, location):
        # type: (str) -> Optional[VersionControl]
        """
        Return a VersionControl object if a repository of that type is found
        at the given directory.
        """
        vcs_backends = {}
        for vcs_backend in self._registry.values():
            repo_path = vcs_backend.get_repository_root(location)
            if not repo_path:
                continue
            logger.debug('Determine that %s uses VCS: %s',
                         location, vcs_backend.name)
            vcs_backends[repo_path] = vcs_backend

        if not vcs_backends:
            return None

        # Choose the VCS in the inner-most directory. Since all repository
        # roots found here would be either `location` or one of its
        # parents, the longest path should have the most path components,
        # i.e. the backend representing the inner-most repository.
        inner_most_repo_path = max(vcs_backends, key=len)
        return vcs_backends[inner_most_repo_path]

    def get_backend_for_scheme(self, scheme):
        # type: (str) -> Optional[VersionControl]
        """
        Return a VersionControl object or None.
        """
        for vcs_backend in self._registry.values():
            if scheme in vcs_backend.schemes:
                return vcs_backend
        return None

    def get_backend(self, name):
        # type: (str) -> Optional[VersionControl]
        """
        Return a VersionControl object or None.
        """
        name = name.lower()
        return self._registry.get(name)


vcs = VcsSupport()


class VersionControl(object):
    name = ''
    dirname = ''
    repo_name = ''
    # List of supported schemes for this Version Control
    schemes = ()  # type: Tuple[str, ...]
    # Iterable of environment variable names to pass to call_subprocess().
    unset_environ = ()  # type: Tuple[str, ...]
    default_arg_rev = None  # type: Optional[str]

    @classmethod
    def should_add_vcs_url_prefix(cls, remote_url):
        # type: (str) -> bool
        """
        Return whether the vcs prefix (e.g. "git+") should be added to a
        repository's remote url when used in a requirement.
        """
        return not remote_url.lower().startswith('{}:'.format(cls.name))

    @classmethod
    def get_subdirectory(cls, location):
        # type: (str) -> Optional[str]
        """
        Return the path to setup.py, relative to the repo root.
        Return None if setup.py is in the repo root.
        """
        return None

    @classmethod
    def get_requirement_revision(cls, repo_dir):
        # type: (str) -> str
        """
        Return the revision string that should be used in a requirement.
        """
        return cls.get_revision(repo_dir)

    @classmethod
    def get_src_requirement(cls, repo_dir, project_name):
        # type: (str, str) -> Optional[str]
        """
        Return the requirement string to use to redownload the files
        currently at the given repository directory.

        Args:
          project_name: the (unescaped) project name.

        The return value has a form similar to the following:

            {repository_url}@{revision}#egg={project_name}
        """
        repo_url = cls.get_remote_url(repo_dir)
        if repo_url is None:
            return None

        if cls.should_add_vcs_url_prefix(repo_url):
            repo_url = '{}+{}'.format(cls.name, repo_url)

        revision = cls.get_requirement_revision(repo_dir)
        subdir = cls.get_subdirectory(repo_dir)
        req = make_vcs_requirement_url(repo_url, revision, project_name,
                                       subdir=subdir)

        return req

    @staticmethod
    def get_base_rev_args(rev):
        # type: (str) -> List[str]
        """
        Return the base revision arguments for a vcs command.

        Args:
          rev: the name of a revision to install.  Cannot be None.
        """
        raise NotImplementedError

    def is_immutable_rev_checkout(self, url, dest):
        # type: (str, str) -> bool
        """
        Return true if the commit hash checked out at dest matches
        the revision in url.

        Always return False, if the VCS does not support immutable commit
        hashes.

        This method does not check if there are local uncommitted changes
        in dest after checkout, as pip currently has no use case for that.
        """
        return False

    @classmethod
    def make_rev_options(cls, rev=None, extra_args=None):
        # type: (Optional[str], Optional[CommandArgs]) -> RevOptions
        """
        Return a RevOptions object.

        Args:
          rev: the name of a revision to install.
          extra_args: a list of extra options.
        """
        return RevOptions(cls, rev, extra_args=extra_args)

    @classmethod
    def _is_local_repository(cls, repo):
        # type: (str) -> bool
        """
           posix absolute paths start with os.path.sep,
           win32 ones start with drive (like c:\\folder)
        """
        drive, tail = os.path.splitdrive(repo)
        return repo.startswith(os.path.sep) or bool(drive)

    def export(self, location, url):
        # type: (str, HiddenText) -> None
        """
        Export the repository at the url to the destination location
        i.e. only download the files, without vcs informations

        :param url: the repository URL starting with a vcs prefix.
        """
        raise NotImplementedError

    @classmethod
    def get_netloc_and_auth(cls, netloc, scheme):
        # type: (str, str) -> Tuple[str, Tuple[Optional[str], Optional[str]]]
        """
        Parse the repository URL's netloc, and return the new netloc to use
        along with auth information.

        Args:
          netloc: the original repository URL netloc.
          scheme: the repository URL's scheme without the vcs prefix.

        This is mainly for the Subversion class to override, so that auth
        information can be provided via the --username and --password options
        instead of through the URL.  For other subclasses like Git without
        such an option, auth information must stay in the URL.

        Returns: (netloc, (username, password)).
        """
        return netloc, (None, None)

    @classmethod
    def get_url_rev_and_auth(cls, url):
        # type: (str) -> Tuple[str, Optional[str], AuthInfo]
        """
        Parse the repository URL to use, and return the URL, revision,
        and auth info to use.

        Returns: (url, rev, (username, password)).
        """
        scheme, netloc, path, query, frag = urllib_parse.urlsplit(url)
        if '+' not in scheme:
            raise ValueError(
                "Sorry, {!r} is a malformed VCS url. "
                "The format is <vcs>+<protocol>://<url>, "
                "e.g. svn+http://myrepo/svn/MyApp#egg=MyApp".format(url)
            )
        # Remove the vcs prefix.
        scheme = scheme.split('+', 1)[1]
        netloc, user_pass = cls.get_netloc_and_auth(netloc, scheme)
        rev = None
        if '@' in path:
            path, rev = path.rsplit('@', 1)
            if not rev:
                raise InstallationError(
                    "The URL {!r} has an empty revision (after @) "
                    "which is not supported. Include a revision after @ "
                    "or remove @ from the URL.".format(url)
                )
        url = urllib_parse.urlunsplit((scheme, netloc, path, query, ''))
        return