<?php

/**
 * Joomla! Content Management System
 *
 * @copyright  (C) 2006 Open Source Matters, Inc. <https://www.joomla.org>
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\CMS\Pagination;

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Router\Route;
use Joomla\Filter\InputFilter;

// phpcs:disable PSR1.Files.SideEffects
\defined('JPATH_PLATFORM') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Pagination Class. Provides a common interface for content pagination for the Joomla! CMS.
 *
 * @since  1.5
 */
class Pagination
{
    /**
     * @var    integer  The record number to start displaying from.
     * @since  1.5
     */
    public $limitstart = null;

    /**
     * @var    integer  Number of rows to display per page.
     * @since  1.5
     */
    public $limit = null;

    /**
     * @var    integer  Total number of rows.
     * @since  1.5
     */
    public $total = null;

    /**
     * @var    integer  Prefix used for request variables.
     * @since  1.6
     */
    public $prefix = null;

    /**
     * @var    integer  Value pagination object begins at
     * @since  3.0
     */
    public $pagesStart;

    /**
     * @var    integer  Value pagination object ends at
     * @since  3.0
     */
    public $pagesStop;

    /**
     * @var    integer  Current page
     * @since  3.0
     */
    public $pagesCurrent;

    /**
     * @var    integer  Total number of pages
     * @since  3.0
     */
    public $pagesTotal;

    /**
     * @var    boolean  The flag indicates whether to add limitstart=0 to URL
     * @since  3.9.0
     */
    public $hideEmptyLimitstart = false;

    /**
     * @var    boolean  View all flag
     * @since  3.0
     */
    protected $viewall = false;

    /**
     * Additional URL parameters to be added to the pagination URLs generated by the class.  These
     * may be useful for filters and extra values when dealing with lists and GET requests.
     *
     * @var    array
     * @since  3.0
     */
    protected $additionalUrlParams = [];

    /**
     * List of parameters that will be added from request automatically.
     * When exists they will be added to the $additionalUrlParams list, while pagination initialisation.
     *
     * In format key => filter
     *
     * @var  string[]
     *
     * @since  4.4.9
     */
    protected $paramsFromRequest = [
        'format'        => 'CMD',
        'option'        => 'CMD',
        'controller'    => 'CMD',
        'view'          => 'CMD',
        'layout'        => 'STRING',
        'task'          => 'CMD',
        'template'      => 'CMD',
        'templateStyle' => 'INT',
        'tmpl'          => 'CMD',
        'tpl'           => 'CMD',
        'id'            => 'STRING',
        'Itemid'        => 'INT',
    ];

    /**
     * @var    CMSApplication  The application object
     * @since  3.4
     */
    protected $app = null;

    /**
     * Pagination data object
     *
     * @var    object
     * @since  3.4
     */
    protected $data;

    /**
     * Constructor.
     *
     * @param   integer         $total       The total number of items.
     * @param   integer         $limitstart  The offset of the item to start at.
     * @param   integer         $limit       The number of items to display per page.
     * @param   string          $prefix      The prefix used for request variables.
     * @param   CMSApplication  $app         The application object
     *
     * @since   1.5
     */
    public function __construct($total, $limitstart, $limit, $prefix = '', CMSApplication $app = null)
    {
        // Value/type checking.
        $this->total      = (int) $total;
        $this->limitstart = (int) max($limitstart, 0);
        $this->limit      = (int) max($limit, 0);
        $this->prefix     = $prefix;
        $this->app        = $app ?: Factory::getApplication();

        if ($this->limit > $this->total) {
            $this->limitstart = 0;
        }

        if (!$this->limit) {
            $this->limit      = $total;
            $this->limitstart = 0;
        }

        /*
         * If limitstart is greater than total (i.e. we are asked to display records that don't exist)
         * then set limitstart to display the last natural page of results
         */
        if ($this->limitstart > $this->total - $this->limit) {
            $this->limitstart = max(0, (int) (ceil($this->total / $this->limit) - 1) * $this->limit);
        }

        // Set the total pages and current page values.
        if ($this->limit > 0) {
            $this->pagesTotal   = (int) ceil($this->total / $this->limit);
            $this->pagesCurrent = (int) ceil(($this->limitstart + 1) / $this->limit);
        }

        // Set the pagination iteration loop values.
        $displayedPages   = 10;
        $this->pagesStart = $this->pagesCurrent - ($displayedPages / 2);

        if ($this->pagesStart < 1) {
            $this->pagesStart = 1;
        }

        if ($this->pagesStart + $displayedPages > $this->pagesTotal) {
            $this->pagesStop = $this->pagesTotal;

            if ($this->pagesTotal < $displayedPages) {
                $this->pagesStart = 1;
            } else {
                $this->pagesStart = $this->pagesTotal - $displayedPages + 1;
            }
        } else {
            $this->pagesStop = $this->pagesStart + $displayedPages - 1;
        }

        // If we are viewing all records set the view all flag to true.
        if ($limit === 0) {
            $this->viewall = true;
        }

        $this->setUrlParamsFromRequest();
    }

    /**
     * Set URL parameters from request.
     *
     * @return void
     *
     * @since  4.4.9
     */
    protected function setUrlParamsFromRequest()
    {
        // Get the requested parameters from the router
        $client = $this->app->getName();
        $router = Factory::getContainer()->get(ucfirst($client) . 'Router');
        $filter = new InputFilter();

        // It is applicable only for CMS router. API router works differently.
        if (!$router instanceof \Joomla\CMS\Router\Router) {
            return;
        }

        // Filter them and add to the params list
        foreach ($router->getVars() as $key => $value) {
            // Check if the parameter is allowed
            if (empty($this->paramsFromRequest[$key])) {
                continue;
            }

            $filterMethod = $this->paramsFromRequest[$key];

            $this->setAdditionalUrlParam($key, $filter->clean($value, $filterMethod));
        }
    }

    /**
     * Method to set an additional URL parameter to be added to all pagination class generated
     * links.
     *
     * @param   string  $key    The name of the URL parameter for which to set a value.
     * @param   mixed   $value  The value to set for the URL parameter.
     *
     * @return  mixed  The old value for the parameter.
     *
     * @since   1.6
     */
    public function setAdditionalUrlParam($key, $value)
    {
        // Get the old value to return and set the new one for the URL parameter.
        $result = $this->additionalUrlParams[$key] ?? null;

        // If the passed parameter value is null unset the parameter, otherwise set it to the given value.
        if ($value === null) {
            unset($this->additionalUrlParams[$key]);
        } else {
            $this->additionalUrlParams[$key] = $value;
        }

        return $result;
    }

    /**
     * Method to get an additional URL parameter (if it exists) to be added to
     * all pagination class generated links.
     *
     * @param   string  $key  The name of the URL parameter for which to get the value.
     *
     * @return  mixed  The value if it exists or null if it does not.
     *
     * @since   1.6
     */
    public function getAdditionalUrlParam($key)
    {
        return $this->additionalUrlParams[$key] ?? null;
    }

    /**
     * Return the rationalised offset for a row with a given index.
     *
     * @param   integer  $index  The row index
     *
     * @return  integer  Rationalised offset for a row with a given index.
     *
     * @since   1.5
     */
    public function getRowOffset($index)
    {
        return $index + 1 + $this->limitstart;
    }

    /**
     * Return the pagination data object, only creating it if it doesn't already exist.
     *
     * @return  \stdClass  Pagination data object.
     *
     * @since   1.5
     */
    public function getData()
    {
        if (!$this->data) {
            $this->data = $this->_buildDataObject();
        }

        return $this->data;
    }

    /**
     * Create and return the pagination pages counter string, ie. Page 2 of 4.
     *
     * @return  string   Pagination pages counter string.
     *
     * @since   1.5
     */
    public function getPagesCounter()
    {
        $html = null;

        if ($this->pagesTotal > 1) {
            $html .= Text::sprintf('JLIB_HTML_PAGE_CURRENT_OF_TOTAL', $this->pagesCurrent, $this->pagesTotal);
        }

        return $html;
    }

    /**
     * Create and return the pagination result set counter string, e.g. Results 1-10 of 42
     *
     * @return  string   Pagination result set counter string.
     *
     * @since   1.5
     */
    public function getResultsCounter()
    {
        $html       = null;
        $fromResult = $this->limitstart + 1;

        // If the limit is reached before the end of the list.
        if ($this->limitstart + $this->limit < $this->total) {
            $toResult = $this->limitstart + $this->limit;
        } else {
            $toResult = $this->total;
        }

        // If there are results found.
        if ($this->total > 0) {
            $msg = Text::sprintf('JLIB_HTML_RESULTS_OF', $fromResult, $toResult, $this->total);
            $html .= "\n" . $msg;
        } else {
            $html .= "\n" . Text::_('JLIB_HTML_NO_RECORDS_FOUND');
        }

        return $html;
    }

    /**
     * Create and return the pagination page list string, ie. Previous, Next, 1 2 3 ... x.
     *
     * @return  string  Pagination page list string.
     *
     * @since   1.5
     */
    public function getPagesLinks()
    {
        // Build the page navigation list.
        $data = $this->_buildDataObject();

        $list           = [];
        $list['prefix'] = $this->prefix;

        $chromePath = JPATH_THEMES . '/' . $this->app->getTemplate() . '/html/pagination.php';

        if (is_file($chromePath)) {
            include_once $chromePath;
        }

        // Build the select list
        if ($data->all->base !== null) {
            $list['all']['active'] = true;
            $list['all']['data']   = $this->_item_active($data->all);
        } else {
            $list['all']['active'] = false;
            $list['all']['data']   = $this->_item_inactive($data->all);
        }

        if ($data->start->base !== null) {
            $list['start']['active'] = true;
            $list['start']['data']   = $this->_item_active($data->start);
        } else {
            $list['start']['active'] = false;
            $list['start']['data']   = $this->_item_inactive($data->start);
        }

        if ($data->previous->base !== null) {
            $list['previous']['active'] = true;
            $list['previous']['data']   = $this->_item_active($data->previous);
        } else {
            $list['previous']['active'] = false;
            $list['previous']['data']   = $this->_item_inactive($data->previous);
        }

        // Make sure it exists
        $list['pages'] = [];

        foreach ($data->pages as $i => $page) {
            if ($page->base !== null) {
                $list['pages'][$i]['active'] = true;
                $list['pages'][$i]['data']   = $this->_item_active($page);
            } else {
                $list['pages'][$i]['active'] = false;
                $list['pages'][$i]['data']   = $this->_item_inactive($page);
            }
        }

        if ($data->next->base !== null) {
            $list['next']['active'] = true;
            $list['next']['data']   = $this->_item_active($data->next);
        } else {
            $list['next']['active'] = false;
            $list['next']['data']   = $this->_item_inactive($data->next);
        }

        if ($data->end->base !== null) {
            $list['end']['active'] = true;
            $list['end']['data']   = $this->_item_active($data->end);
        } else {
            $list['end']['active'] = false;
            $list['end']['data']   = $this->_item_inactive($data->end);
        }

        if ($this->total > $this->limit) {
            return $this->_list_render($list);
        } else {
            return '';
        }
    }

    /**
     * Get the pagination links
     *
     * @param   string  $layoutId  Layout to render the links
     * @param   array   $options   Optional array with settings for the layout
     *
     * @return  string  Pagination links.
     *
     * @since   3.3
     */
    public function getPaginationLinks($layoutId = 'joomla.pagination.links', $options = [])
    {
        // Allow to receive a null layout
        $layoutId = $layoutId ?? 'joomla.pagination.links';

        $list = [
            'prefix'       => $this->prefix,
            'limit'        => $this->limit,
            'limitstart'   => $this->limitstart,
            'total'        => $this->total,
            'limitfield'   => $this->getLimitBox(),
            'pagescounter' => $this->getPagesCounter(),
            'pages'        => $this->getPaginationPages(),
            'pagesTotal'   => $this->pagesTotal,
        ];

        return LayoutHelper::render($layoutId, ['list' => $list, 'options' => $options]);
    }

    /**
     * Create and return the pagination pages list, ie. Previous, Next, 1 2 3 ... x.
     *
     * @return  array  Pagination pages list.
     *
     * @since   3.3
     */
    public function getPaginationPages()
    {
        $list = [];

        if ($this->total > $this->limit) {
            // Build the page navigation list.
            $data = $this->_buildDataObject();

            // All
            $list['all']['active'] = $data->all->base !== null;
            $list['all']['data']   = $data->all;

            // Start
            $list['start']['active'] = $data->start->base !== null;
            $list['start']['data']   = $data->start;

            // Previous link
            $list['previous']['active'] = $data->previous->base !== null;
            $list['previous']['data']   = $data->previous;

            // Make sure it exists
            $list['pages'] = [];

            foreach ($data->pages as $i => $page) {
                $list['pages'][$i]['active'] = $page->base !== null;
                $list['pages'][$i]['data']   = $page;
            }

            $list['next']['active'] = $data->next->base !== null;
            $list['next']['data']   = $data->next;

            $list['end']['active'] = $data->end->base !== null;
            $list['end']['data']   = $data->end;
        }

        return $list;
    }

    /**
     * Return the pagination footer.
     *
     * @return  string  Pagination footer.
     *
     * @since   1.5
     */
    public function getListFooter()
    {
        // Keep B/C for overrides done with chromes
        $chromePath = JPATH_THEMES . '/' . $this->app->getTemplate() . '/html/pagination.php';

        if (is_file($chromePath)) {
            include_once $chromePath;

            if (\function_exists('pagination_list_footer')) {
                @trigger_error(
                    'pagination_list_footer is deprecated. Use the layout joomla.pagination.links instead.',
                    E_USER_DEPRECATED
                );

                $list = [
                    'prefix'       => $this->prefix,
                    'limit'        => $this->limit,
                    'limitstart'   => $this->limitstart,
                    'total'        => $this->total,
                    'limitfield'   => $this->getLimitBox(),
                    'pagescounter' => $this->getPagesCounter(),
                    'pageslinks'   => $this->getPagesLinks(),
                ];

                return pagination_list_footer($list);
            }
        }

        return $this->getPaginationLinks();
    }

    /**
     * Creates a dropdown box for selecting how many records to show per page.
     *
     * @return  string  The HTML for the limit # input box.
     *
     * @since   1.5
     */
    public function getLimitBox()
    {
        $limits = [];

        // Make the option list.
        for ($i = 5; $i <= 30; $i += 5) {
            $limits[] = HTMLHelper::_('select.option', "$i");
        }

        $limits[] = HTMLHelper::_('select.option', '50', Text::_('J50'));
        $limits[] = HTMLHelper::_('select.option', '100', Text::_('J100'));
        $limits[] = HTMLHelper::_('select.option', '0', Text::_('JALL'));

        $selected = $this->viewall ? 0 : $this->limit;

        // Build the select list.
        if ($this->app->isClient('administrator')) {
            $html = HTMLHelper::_(
                'select.genericlist',
                $limits,
                $this->prefix . 'limit',
                'class="form-select" onchange="Joomla.submitform();"',
                'value',
                'text',
                $selected
            );
        } else {
            $html = HTMLHelper::_(
                'select.genericlist',
                $limits,
                $this->prefix . 'limit',
                'class="form-select" onchange="this.form.submit()"',
                'value',
                'text',
                $selected
            );
        }

        return $html;
    }

    /**
     * Return the icon to move an item UP.
     *
     * @param   integer  $i          The row index.
     * @param   boolean  $condition  True to show the icon.
     * @param   string   $task       The task to fire.
     * @param   string   $alt        The image alternative text string.
     * @param   boolean  $enabled    An optional setting for access control on the action.
     * @param   string   $checkbox   An optional prefix for checkboxes.
     *
     * @return  string   Either the icon to move an item up or a space.
     *
     * @since   1.5
     */
    public function orderUpIcon($i, $condition = true, $task = 'orderup', $alt = 'JLIB_HTML_MOVE_UP', $enabled = true, $checkbox = 'cb')
    {
        if (($i > 0 || ($i + $this->limitstart > 0)) && $condition) {
            return HTMLHelper::_('jgrid.orderUp', $i, $task, '', $alt, $enabled, $checkbox);
        } else {
            return '&#160;';
        }
    }

    /**
     * Return the icon to move an item DOWN.
     *
     * @param   integer  $i          The row index.
     * @param   integer  $n          The number of items in the list.
     * @param   boolean  $condition  True to show the icon.
     * @param   string   $task       The task to fire.
     * @param   string   $alt        The image alternative text string.
     * @param   boolean  $enabled    An optional setting for access control on the action.
     * @param   string   $checkbox   An optional prefix for checkboxes.
     *
     * @return  string   Either the icon to move an item down or a space.
     *
     * @since   1.5
     */
    public function orderDownIcon($i, $n, $condition = true, $task = 'orderdown', $alt = 'JLIB_HTML_MOVE_DOWN', $enabled = true, $checkbox = 'cb')
    {
        if (($i < $n - 1 || $i + $this->limitstart < $this->total - 1) && $condition) {
            return HTMLHelper::_('jgrid.orderDown', $i, $task, '', $alt, $enabled, $checkbox);
        } else {
            return '&#160;';
        }
    }

    /**
     * Create the HTML for a list footer
     *
     * @param   array  $list  Pagination list data structure.
     *
     * @return  string  HTML for a list footer
     *
     * @since   1.5
     */
    protected function _list_footer($list)
    {
        $html = "<div class=\"list-footer\">\n";

        $html .= "\n<div class=\"limit\">" . Text::_('JGLOBAL_DISPLAY_NUM') . $list['limitfield'] . "</div>";
        $html .= $list['pageslinks'];
        $html .= "\n<div class=\"counter\">" . $list['pagescounter'] . "</div>";

        $html .= "\n<input type=\"hidden\" name=\"" . $list['prefix'] . "limitstart\" value=\"" . $list['limitstart'] . "\">";
        $html .= "\n</div>";

        return $html;
    }

    /**
     * Create the html for a list footer
     *
     * @param   array  $list  Pagination list data structure.
     *
     * @return  string  HTML for a list start, previous, next,end
     *
     * @since   1.5
     */
    protected function _list_render($list)
    {
        return LayoutHelper::render('joomla.pagination.list', ['list' => $list]);
    }

    /**
     * Method to create an active pagination link to the item
     *
     * @param   PaginationObject  $item  The object with which to make an active link.
     *
     * @return  string  HTML link
     *
     * @since   1.5
     */
    protected function _item_active(PaginationObject $item)
    {
        return LayoutHelper::render('joomla.pagination.link', ['data' => $item, 'active' => true]);
    }

    /**
     * Method to create an inactive pagination string
     *
     * @param   PaginationObject  $item  The item to be processed
     *
     * @return  string
     *
     * @since   1.5
     */
    protected function _item_inactive(PaginationObject $item)
    {
        return LayoutHelper::render('joomla.pagination.link', ['data' => $item, 'active' => false]);
    }

    /**
     * Create and return the pagination data object.
     *
     * @return  \stdClass  Pagination data object.
     *
     * @since   1.5
     */
    protected function _buildDataObject()
    {
        $data = new \stdClass();

        // Prepare the routes
        $params = [];

        if (!empty($this->additionalUrlParams)) {
            foreach ($this->additionalUrlParams as $key => $value) {
                $params[$key] = $value;
            }
        }

        $params = http_build_query($params);

        $data->all = new PaginationObject(Text::_('JLIB_HTML_VIEW_ALL'), $this->prefix);

        if (!$this->viewall) {
            $data->all->base = '0';
            $data->all->link = Route::_('index.php?' . $params . '&' . $this->prefix . 'limitstart=');
        }

        // Set the start and previous data objects.
        $data->start    = new PaginationObject(Text::_('JLIB_HTML_START'), $this->prefix);
        $data->previous = new PaginationObject(Text::_('JPREV'), $this->prefix);

        if ($this->pagesCurrent > 1) {
            $page = ($this->pagesCurrent - 2) * $this->limit;

            if ($this->hideEmptyLimitstart) {
                $data->start->link = Route::_('index.php?' . $params . '&' . $this->prefix . 'limitstart=');
            } else {
                $data->start->link = Route::_('index.php?' . $params . '&' . $this->prefix . 'limitstart=0');
            }

            $data->start->base    = '0';
            $data->previous->base = $page;

            if ($page === 0 && $this->hideEmptyLimitstart) {
                $data->previous->link = $data->start->link;
            } else {
                $data->previous->link = Route::_('index.php?' . $params . '&' . $this->prefix . 'limitstart=' . $page);
            }
        }

        // Set the next and end data objects.
        $data->next = new PaginationObject(Text::_('JNEXT'), $this->prefix);
        $data->end  = new PaginationObject(Text::_('JLIB_HTML_END'), $this->prefix);

        if ($this->pagesCurrent < $this->pagesTotal) {
            $next = $this->pagesCurrent * $this->limit;
            $end  = ($this->pagesTotal - 1) * $this->limit;

            $data->next->base = $next;
            $data->next->link = Route::_('index.php?' . $params . '&' . $this->prefix . 'limitstart=' . $next);
            $data->end->base  = $end;
            $data->end->link  = Route::_('index.php?' . $params . '&' . $this->prefix . 'limitstart=' . $end);
        }

        $data->pages = [];
        $stop        = $this->pagesStop;

        for ($i = $this->pagesStart; $i <= $stop; $i++) {
            $offset = ($i - 1) * $this->limit;

            $data->pages[$i] = new PaginationObject($i, $this->prefix);

            if ($i != $this->pagesCurrent || $this->viewall) {
                $data->pages[$i]->base = $offset;

                if ($offset === 0 && $this->hideEmptyLimitstart) {
                    $data->pages[$i]->link = $data->start->link;
                } else {
                    $data->pages[$i]->link = Route::_(
                        'index.php?' . $params . '&' . $this->prefix . 'limitstart=' . $offset
                    );
                }
            } else {
                $data->pages[$i]->active = true;
            }
        }

        return $data;
    }
}
