<?php
 /**
 * Jamroom Followers module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrFollower_meta()
{
    return array(
        'name'        => 'Followers',
        'url'         => 'follow',
        'version'     => '2.1.8',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Users can &quot;follow&quot; other users Profiles',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/3600/followers',
        'category'    => 'profiles',
        'requires'    => 'jrCore:6.5.0',
        'license'     => 'mpl'
    );
}

/**
 * init
 */
function jrFollower_init()
{
    // Pulse Key support
    jrCore_register_module_feature('jrProfile', 'pulse_key', 'jrFollower', 'profile_jrFollower_item_count', 'followers');

    // Integrity Check
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrFollower', 'integrity_check', array('Integrity Check', 'Validate and Update follower counts for Profiles'));

    // Register our custom JS
    jrCore_register_module_feature('jrCore', 'javascript', 'jrFollower', 'jrFollower.js');
    jrCore_register_module_feature('jrCore', 'css', 'jrFollower', 'jrFollower.css');

    // Let the core Action System know we are adding actions to followers Support
    jrCore_register_module_feature('jrCore', 'action_support', 'jrFollower', 'create', 'item_action.tpl');

    // follower notifications
    $_tmp = array(
        'label' => 9,  // 'new pending follower'
        'help'  => 23  // 'If you are approving new followers, do you want to be notified when a new follower is waiting to be approved?'
    );
    jrCore_register_module_feature('jrUser', 'notification', 'jrFollower', 'follower_pending', $_tmp);

    $_tmp = array(
        'label' => 10, // 'new follower'
        'help'  => 24  // 'Do you want to be notified when you get a new follower?'
    );
    jrCore_register_module_feature('jrUser', 'notification', 'jrFollower', 'new_follower', $_tmp);

    $_tmp = array(
        'label' => 11, // 'follow approved'
        'help'  => 25  // 'Do you want to be notified if your pending follow request for another profile is approved?'
    );
    jrCore_register_module_feature('jrUser', 'notification', 'jrFollower', 'follow_approved', $_tmp);

    // Skin menu link to 'following'
    $_tmp = array(
        'group' => 'user',
        'label' => 37, // 'Profiles I Follow'
        'url'   => 'following'
    );
    jrCore_register_module_feature('jrCore', 'skin_menu_item', 'jrFollower', 'following', $_tmp);

    // add follower information into the {jrAction_stats} template function
    jrCore_register_event_listener('jrAction', 'action_stats', 'jrFollower_action_stats_listener');

    // We provide some search items params
    jrCore_register_event_listener('jrCore', 'db_search_params', 'jrFollower_db_search_params_listener');
    jrCore_register_event_listener('jrCore', 'db_search_items', 'jrFollower_db_search_items_listener');

    // Add "Share Follows" to User Account
    jrCore_register_event_listener('jrCore', 'form_display', 'jrFollower_form_display_listener');

    // Add action on follow approve
    jrCore_register_event_listener('jrAction', 'create', 'jrFollower_action_create_listener');

    // Cleanup skin menu item
    jrCore_register_event_listener('jrCore', 'verify_module', 'jrFollower_verify_module_listener');

    // Add in action_item data
    jrCore_register_event_listener('jrCore', 'template_variables', 'jrFollower_template_variables_listener');

    // We have tips for followers
    jrCore_register_module_feature('jrTips', 'tip', 'jrFollower', 'tip');

    // Recycle Bin
    jrCore_register_module_feature('jrCore', 'recycle_bin_user_id_table', 'jrFollower', 'follow', 'follow_user_id');
    jrCore_register_module_feature('jrCore', 'recycle_bin_profile_id_table', 'jrFollower', 'follow', 'follow_profile_id');

    // event triggers
    jrCore_register_event_trigger('jrFollower', 'follower_created', 'Fired when a new follower is created');
    jrCore_register_event_trigger('jrFollower', 'follower_updated', 'Fired when a new follower is updated');
    jrCore_register_event_trigger('jrFollower', 'follower_deleted', 'Fired when a new follower is deleted');

    return true;
}

//---------------------
// EVENT LISTENERS
//---------------------

/**
 * adds the follower stats into the jrAction_stats call that retrieves
 * action information twitter style - i.e. 200 TWEETS | 300 FOLLOWERS | 400 FOLLOWING
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrFollower_action_stats_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_args['profile_id']) && jrCore_checktype($_args['profile_id'], 'number_nz')) {

        $pid = (int) $_args['profile_id'];
        $key = "jrfollower_action_stats_{$pid}";
        if (!$_rt = jrCore_is_cached('jrFollower', $key)) {

            $_rt = array(
                'followers' => jrFollower_get_profile_follower_count($pid, false),
                'following' => 0
            );

            // Get all profiles users of this profile are following
            $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
            $req = "SELECT `user_id` FROM {$tbl} WHERE profile_id = {$pid}";
            $_us = jrCore_db_query($req, 'user_id');
            if ($_us && is_array($_us)) {
                $tbl = jrCore_db_table_name('jrFollower', 'follow');
                $req = "SELECT follow_profile_id AS p FROM {$tbl} WHERE follow_user_id IN(" . implode(',', array_keys($_us)) . ')';
                $_pf = jrCore_db_query($req, 'p');
                if ($_pf && is_array($_pf)) {
                    $_rt['following'] = count($_pf);
                }
                unset($_pf);
            }
            jrCore_add_to_cache('jrFollower', $key, $_rt, 0, $pid);

        }
        $_data['followers'] = $_rt['followers'];
        $_data['following'] = $_rt['following'];
    }
    return $_data;
}

/**
 * Adds a "show tips" field to the User Settings
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrFollower_form_display_listener($_data, $_user, $_conf, $_args, $event)
{
    if ($_data['form_view'] == 'jrUser/account') {
        if (empty($_user['quota_jrFollower_share_type']) || $_user['quota_jrFollower_share_type'] == 'user') {
            $_lng = jrUser_load_lang_strings();
            $_tmp = array(
                'name'          => 'user_jrFollower_share',
                'type'          => 'checkbox',
                'default'       => 'on',
                'validate'      => 'onoff',
                'label'         => $_lng['jrFollower'][38],
                'help'          => $_lng['jrFollower'][39],
                'required'      => false,
                'form_designer' => false // no form designer or we can't turn it off
            );
            jrCore_form_field_create($_tmp);
        }
    }

    // Is this the jrProfile/settings form?
    elseif ($_data['form_view'] == 'jrProfile/settings') {
        $_ln = jrUser_load_lang_strings();
        $_tm = array(
            'name'          => 'profile_jrFollower_approve',
            'label'         => $_ln['jrFollower'][3],
            'help'          => $_ln['jrFollower'][4],
            'sublabel'      => $_ln['jrFollower'][41],
            'type'          => 'checkbox',
            'default'       => 'off',
            'validate'      => 'onoff',
            'required'      => 'on',
            'form_designer' => false
        );
        jrCore_form_field_create($_tm);
    }
    return $_data;
}

/**
 * Remove old skin menu item
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrFollower_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // Bad menu entry
    $tbl = jrCore_db_table_name('jrCore', 'menu');
    $req = "DELETE FROM {$tbl} WHERE menu_module = 'jrFollower'";
    jrCore_db_query($req);

    // Bad form designer entry
    $tbl = jrCore_db_table_name('jrCore', 'form');
    $req = "DELETE FROM {$tbl} WHERE `module` = 'jrProfile' AND `view` = 'settings' AND `name` = 'profile_jrFollower_approve' LIMIT 1";
    jrCore_db_query($req);

    // Convert to NEW database format
    if (jrCore_db_table_exists('jrFollower', 'item_key') && jrCore_db_table_exists('jrFollower', 'follow')) {
        // We still have a datastore - we have not converted yet
        $iid = 0;
        $tot = 0;
        while (true) {
            $_sc = array(
                'search'         => array(
                    "_item_id > {$iid}"
                ),
                'skip_triggers'  => true,
                'ignore_pending' => true,
                'privacy_check'  => false,
                'no_cache'       => true,
                'limit'          => 500
            );
            $_rt = jrCore_db_search_items('jrFollower', $_sc);
            if ($_rt && is_array($_rt) && is_array($_rt['_items'])) {
                // Get array of existing profiles and make sure profile id's are good
                $_pids = array();
                foreach ($_rt['_items'] as $v) {
                    $_pids["{$v['follow_profile_id']}"] = $v['follow_profile_id'];
                }
                $_pids = jrCore_db_get_multiple_items('jrProfile', $_pids, array('_profile_id'), true, '_profile_id');
                $_up   = array();
                foreach ($_rt['_items'] as $v) {
                    if (isset($_pids["{$v['follow_profile_id']}"])) {
                        $_up[] = '(' . intval($v['_created']) . ',' . intval($v['_user_id']) . ',' . intval($v['follow_profile_id']) . ',' . intval($v['follow_active']) . ')';
                        $tot++;
                    }
                    $iid = $v['_item_id'];
                }
                if (count($_up) > 0) {
                    $tbl = jrCore_db_table_name('jrFollower', 'follow');
                    $req = "INSERT IGNORE INTO {$tbl} (follow_created, follow_user_id, follow_profile_id, follow_active) VALUES " . implode(',', $_up);
                    jrCore_db_query($req);
                }
            }
            else {
                // No more profiles...
                break;
            }
        }
        if ($tot > 0) {
            jrCore_logger('INF', "Followers: successfully convert {$tot} follower entries to new database format");
        }
        // Delete the datastore
        jrCore_db_delete_datastore('jrFollower');
    }
    return $_data;
}

/**
 * Updates a created action with proper user and profile info
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrFollower_action_create_listener($_data, $_user, $_conf, $_args, $event)
{
    $_sv = jrCore_get_flag('follower_approved');
    if ($_sv && is_array($_sv) && isset($_args['_item_id'])) {
        $aid = (int) $_args['_item_id'];
        $_cr = array('_user_id' => $_sv['_user_id']);
        unset($_sv['_user_id']);
        jrCore_db_update_item('jrAction', $aid, $_sv, $_cr);
        jrCore_delete_flag('follower_approved');
    }
    return $_data;
}

/**
 * Add support for "include_followed" jrCore_list param for jrAction lists
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrFollower_db_search_params_listener($_data, $_user, $_conf, $_args, $event)
{
    switch ($_args['module']) {

        case 'jrUser':

            // Watch for "follow_profile_id" and "follow_active"
            if (isset($_data['follow_profile_id']) && jrCore_checktype($_data['follow_profile_id'], 'number_nz')) {
                $pid = (int) $_data['follow_profile_id'];
                $add = '';
                if (isset($_data['follow_active']) && ($_data['follow_active'] == 1 || $_data['follow_active'] == 0)) {
                    $add = " AND follow_active = " . intval($_data['follow_active']);
                }
                $tbl = jrCore_db_table_name('jrFollower', 'follow');
                $req = "SELECT * FROM {$tbl} WHERE follow_profile_id = {$pid}{$add}";
                $_rt = jrCore_db_query($req, 'follow_user_id');
                if ($_rt && is_array($_rt)) {
                    // Order followers from newest to oldest - this way when
                    // it is passed via IN() that is the order they will return in
                    uasort($_rt, function ($a, $b) {
                        return ($a['follow_created'] < $b['follow_created']) ? 1 : -1;
                    });
                    if (!isset($_data['search'])) {
                        $_data['search'] = array();
                    }
                    $_data['search'][] = "_item_id in " . implode(',', array_keys($_rt));
                    jrCore_set_flag('db_search_param_follower_info', $_rt);
                    if (!isset($_data['include_jrProfile_keys'])) {
                        $_data['include_jrProfile_keys'] = true;
                    }
                    if (!isset($_data['exclude_jrProfile_quota_keys'])) {
                        $_data['exclude_jrProfile_quota_keys'] = true;
                    }
                }
                else {
                    // No followers - short circuit
                    $_data['full_result_set'] = false;
                    return $_data;
                }
            }
            break;

        case 'jrAction':

            // include_followed="true"
            if (isset($_data['include_followed']) && (intval($_data['include_followed']) === 1 || $_data['include_followed'] == 'true')) {

                // Check for profile_id
                $pid = false;
                if (isset($_data['profile_id'])) {
                    // see if jrProfile has expanded this profile_id - if so remove it
                    if (isset($_data['search']) && is_array($_data['search'])) {
                        foreach ($_data['search'] as $k => $v) {
                            if (strpos(' ' . $v, '_profile_id = ')) {
                                unset($_data['search'][$k]);
                                break;
                            }
                        }
                    }
                    $pid = (int) $_data['profile_id'];
                    unset($_data['profile_id']);
                }
                elseif (isset($_data['search']) && is_array($_data['search'])) {
                    foreach ($_data['search'] as $k => $v) {
                        list($sf, $so, $sv) = explode(' ', $v, 3);
                        if (trim($sf) == '_profile_id') {
                            if ($so == '=') {
                                $pid = $sv;
                            }
                            elseif (strtolower($so) == 'in') {
                                $pid = array();
                                foreach (explode(',', $sv) as $pi) {
                                    $pi = intval($pi);
                                    if ($pi > 0) {
                                        $pid[] = $pi;
                                    }
                                }
                            }
                            unset($_data['search'][$k]);
                        }
                    }
                }

                if ($pid && jrCore_checktype($pid, 'number_nz')) {
                    // Get the users associated with this profile
                    $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
                    $req = "SELECT user_id, profile_id FROM {$tbl} WHERE profile_id = {$pid}";
                    $_us = jrCore_db_query($req, 'user_id', false, 'profile_id');

                    if (!isset($_data['search'])) {
                        $_data['search'] = array();
                    }
                    if ($_us && is_array($_us)) {
                        $tbl = jrCore_db_table_name('jrFollower', 'follow');
                        $req = "SELECT follow_profile_id AS p FROM {$tbl} WHERE follow_user_id IN(" . implode(',', array_keys($_us)) . ')';
                        $_pr = jrCore_db_query($req, 'p', false, 'p', false);
                        if (!$_pr || !is_array($_pr)) {
                            // No user on this profile is following any other users - make sure an include OUR posts
                            $_pr = $_us;
                        }
                        else {
                            $_pr = array_merge($_pr, $_us);
                        }
                        $_data['search'][] = "_profile_id in " . implode(',', $_pr);
                    }
                    else {
                        // There are NO profiles associated with the requested $pid - set a no result condition
                        $_data['search'][] = "_profile_id < 0";

                    }
                    // We can turn of quota checking here since "include_followed" is always used on
                    // a profile, and we would not be here if jrAction was not allowed.  This
                    // saves an extra set of _profile_id lookups in our query
                    $_data['quota_check'] = false;
                }
            }
            break;

        case 'jrProfile':

            // followed_by="user_id[,user_id]"
            if (isset($_data['followed_by']) && strlen($_data['followed_by']) > 0) {
                // See if we have an individual user_id or group of user_id's
                $_ui = array();
                if (strpos($_data['followed_by'], ',')) {
                    foreach (explode(',', $_data['followed_by']) as $v) {
                        $v = (int) $v;
                        if (jrCore_checktype($v, 'number_nz')) {
                            $_ui[] = $v;
                        }
                    }
                }
                else {
                    if (jrCore_checktype($_data['followed_by'], 'number_nz')) {
                        $_ui[] = (int) $_data['followed_by'];
                    }
                }
                if (count($_ui) > 0) {
                    $tbl = jrCore_db_table_name('jrFollower', 'follow');
                    $req = "SELECT follow_profile_id AS p FROM {$tbl} WHERE follow_user_id IN(" . implode(',', $_ui) . ')';
                    $_pr = jrCore_db_query($req, 'p', false, 'p', false);
                    if ($_pr && count($_pr) > 0) {
                        if (!isset($_data['search'])) {
                            $_data['search'] = array();
                        }
                        $_data['search'][] = '_profile_id in ' . implode(',', $_pr);
                    }
                }
            }
            break;

    }
    return $_data;
}

/**
 * Insert follower info into search results
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrFollower_db_search_items_listener($_data, $_user, $_conf, $_args, $event)
{
    switch ($_args['module']) {
        case 'jrUser':
            if ($_fi = jrCore_get_flag('db_search_param_follower_info')) {
                foreach ($_data['_items'] as $k => $v) {
                    $uid = (int) $v['_item_id'];
                    if (isset($_fi[$uid])) {
                        $_data['_items'][$k] = array_merge($v, $_fi[$uid]);
                    }
                }
                jrCore_delete_flag('db_search_param_follower_info');
            }
            break;
    }
    return $_data;
}

/**
 * Add in action_data arrays if necessary
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrFollower_template_variables_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_data['item']['action_module']) && $_data['item']['action_module'] == 'jrFollower' && jrCore_checktype($_data['item']['action_item_id'], 'number_nz') && isset($_data['item']['action_original_data']) && !isset($_data['item']['action_data'])) {
        $_data['item']['action_data'] = array();
        foreach ($_data['item'] as $key => $val) {
            if (strpos($key, 'profile') === 0 || strpos($key, 'user') === 0 || strpos($key, 'quota') === 0 || strpos($key, '_') === 0) {
                $_data['item']['action_data']["{$key}"] = $val;
            }
        }
        $_data['item']['action_data']['_item_id']          = $_data['item']['action_item_id'];
        $_data['item']['action_data']['follow_profile_id'] = $_data['item']['action_original_data']['_profile_id'];
    }
    return $_data;
}

//---------------------
// Functions
//---------------------

/**
 * Create a new follow in the follow table
 * @param int $user_id
 * @param int $profile_id
 * @param int $active 0|1
 * @return bool
 */
function jrFollower_create_follow($user_id, $profile_id, $active)
{
    $uid   = (int) $user_id;
    $pid   = (int) $profile_id;
    $act   = (int) $active;
    $tbl   = jrCore_db_table_name('jrFollower', 'follow');
    $req   = "INSERT IGNORE INTO {$tbl} (follow_created, follow_user_id, follow_profile_id, follow_active) VALUES (UNIX_TIMESTAMP(), {$uid}, {$pid}, {$act})";
    $iid   = jrCore_db_query($req, 'INSERT_ID');
    $_data = array(
        'follow_user_id'    => $uid,
        'follow_profile_id' => $pid,
        'follow_active'     => $act,
        'follow_id'         => $iid
    );
    jrCore_trigger_event('jrFollower', 'follower_created', $_data);
    return $iid;
}

/**
 * Get follow info
 * @param int $user_id
 * @param int $profile_id
 * @return array|false
 */
function jrFollower_get_follow($user_id, $profile_id)
{
    $uid = (int) $user_id;
    $pid = (int) $profile_id;
    $tbl = jrCore_db_table_name('jrFollower', 'follow');
    $req = "SELECT * FROM {$tbl} WHERE follow_user_id = {$uid} AND follow_profile_id = {$pid}";
    return jrCore_db_query($req, 'SINGLE', false, null, false);
}

/**
 * Delete an existing follow from the DB
 * @param int $user_id
 * @param int $profile_id
 * @return bool
 */
function jrFollower_delete_follow($user_id, $profile_id)
{
    $uid = (int) $user_id;
    $pid = (int) $profile_id;
    $tbl = jrCore_db_table_name('jrFollower', 'follow');
    $req = "DELETE FROM {$tbl} WHERE follow_user_id = {$uid} AND follow_profile_id = {$pid}";
    jrCore_db_query($req);
    $_data = array(
        'follow_user_id'    => $uid,
        'follow_profile_id' => $pid
    );
    jrCore_trigger_event('jrFollower', 'follower_deleted', $_data);
    return true;
}

/**
 * See if a Timeline Action is enabled for a follow
 * @param array $_user
 * @return bool
 */
function jrFollower_is_action_enabled($_user)
{
    if (empty($_user['quota_jrFollower_share_type'])) {
        // Default to User Control
        $_user['quota_jrFollower_share_type'] = 'user';
    }
    if ($_user['quota_jrFollower_share_type'] == 'always') {
        // Always enabled
        return true;
    }
    elseif ($_user['quota_jrFollower_share_type'] == 'never') {
        // Explicitly disabled
        return false;
    }
    elseif (empty($_user['user_jrFollower_share']) || $_user['user_jrFollower_share'] == 'on') {
        // User control is enabled and User has sharing enabled
        return true;
    }
    return false;
}

/**
 * Get number of pending Followers
 * @note This is a SKIN MENU function - it will always return pending count for VIEWING user
 * @return int Number of pending followers
 */
function jrFollower_pending_count()
{
    $pid = jrUser_get_profile_home_key('_profile_id');
    $key = "jrfollower_pending_count{$pid}";
    if (!$key = jrCore_is_cached('jrFollower', $key)) {
        $tbl = jrCore_db_table_name('jrFollower', 'follow');
        $req = "SELECT * FROM {$tbl} WHERE follow_profile_id = {$pid} AND follow_active = 0";
        $num = jrCore_db_query($req, 'NUM_ROWS', false, null, false);
        jrCore_add_to_cache('jrFollower', $key, $num, 0, $pid);
        return $num;
    }
    return 0;
}

/**
 * Returns an array of profiles a given user_id follows
 * @note: If $active_only is FALSE the resulting array includes the $active status
 * @param string $user_id User ID
 * @param bool $active_only set to FALSE to include inactive followers
 * @return mixed Array of profile IDs or false if none
 */
function jrFollower_get_profiles_followed($user_id, $active_only = true)
{
    $uid = (int) $user_id;
    $key = "jrfollower_profiles_followed_{$uid}_" . intval($active_only);
    if (!$_rt = jrCore_is_cached('jrFollower', $key)) {
        $tbl = jrCore_db_table_name('jrFollower', 'follow');
        if ($active_only) {
            $req = "SELECT follow_profile_id AS i FROM {$tbl} WHERE follow_user_id = {$uid} AND follow_active = 1";
            $_rt = jrCore_db_query($req, 'i', false, 'i', false);
        }
        else {
            $req = "SELECT follow_profile_id AS i, follow_active AS a FROM {$tbl} WHERE follow_user_id = {$uid}";
            $_rt = jrCore_db_query($req, 'i', false, 'a', false);
        }
        if (!$_rt || !is_array($_rt)) {
            $_rt = 'no_profiles';
        }
        jrCore_add_to_cache('jrFollower', $key, $_rt);
    }
    if ($_rt == 'no_profiles') {
        return false;
    }
    return $_rt;
}

/**
 * Returns an array (user_id => user_name) of users following the given profile_id
 * @param int $profile_id
 * @param bool $user_id_only set to TRUE to only return User ID
 * @param bool $active_only set to FALSE to include pending followers
 * @param bool $order_by_created set to TRUE to order results by created DESC
 * @return bool|mixed
 */
function jrFollower_get_users_following($profile_id, $user_id_only = false, $active_only = true, $order_by_created = false)
{
    $pid = (int) $profile_id;
    $key = "jrfollower_users_following_{$pid}_" . intval($user_id_only) . intval($active_only) . intval($order_by_created);
    if (!$_rt = jrCore_is_cached('jrFollower', $key)) {
        $tbl = jrCore_db_table_name('jrFollower', 'follow');
        $req = "SELECT follow_user_id AS i FROM {$tbl} WHERE follow_profile_id = {$pid}";
        if ($active_only) {
            $req .= " AND follow_active = 1";
        }
        if ($order_by_created) {
            $req .= " ORDER BY follow_created DESC";
        }
        $_rt = jrCore_db_query($req, 'i', false, 'i', false);
        if ($_rt && is_array($_rt)) {
            if (!$user_id_only) {
                if ($_us = jrCore_db_get_multiple_items('jrUser', $_rt, array('_item_id', 'user_name'), false, '_item_id')) {
                    foreach ($_rt as $uid => $v) {
                        if (isset($_us[$uid])) {
                            $_rt[$uid] = $_us[$uid]['user_name'];
                        }
                    }
                }
            }
        }
        else {
            $_rt = 'no_users';
        }
        jrCore_add_to_cache('jrFollower', $key, $_rt, 0, $pid);
    }
    if ($_rt == 'no_users') {
        return false;
    }
    return $_rt;
}

/**
 * Return follow info if user is a follower
 * @param $user_id string User ID
 * @param $profile_id string Profile ID
 * @return bool
 */
function jrFollower_is_follower($user_id, $profile_id)
{
    $uid = (int) $user_id;
    $pid = (int) $profile_id;
    $key = "jrfollower_is_follower_{$pid}_{$uid}";
    if (!$_rt = jrCore_is_cached('jrFollower', $key)) {
        $tbl = jrCore_db_table_name('jrFollower', 'follow');
        $req = "SELECT * FROM {$tbl} WHERE follow_profile_id = {$pid} AND follow_user_id = {$uid}";
        $_rt = jrCore_db_query($req, 'SINGLE', false, null, false);
        if (!$_rt || !is_array($_rt)) {
            $_rt = 'no_data';
        }
        jrCore_add_to_cache('jrFollower', $key, $_rt, 0, $pid);
    }
    if ($_rt == 'no_data') {
        return false;
    }
    return $_rt;
}

/**
 * Get number of users following a profile
 * @param int $profile_id
 * @param bool $active_only set to FALSE to get count of all followers regardless if they are active
 * @return int
 */
function jrFollower_get_profile_follower_count($profile_id, $active_only = true)
{
    $pid = (int) $profile_id;
    $key = "jrfollower_profile_count_{$pid}_" . intval($active_only);
    if (!$num = jrCore_get_flag($key)) {
        $num = 0;
        $tbl = jrCore_db_table_name('jrFollower', 'follow');
        $req = "SELECT COUNT(*) AS cnt FROM {$tbl} WHERE follow_profile_id = {$pid}";
        if ($active_only) {
            $req .= " AND follow_active = 1";
        }
        $_rt = jrCore_db_query($req, 'SINGLE', false, null, false);
        if ($_rt && is_array($_rt)) {
            $num = intval($_rt['cnt']);
        }
        jrCore_set_flag($key, $num);
    }
    return $num;
}

/**
 * Get number of profiles a user is following
 * @param int $user_id
 * @param bool $active_only set to FALSE to get count of all followers regardless if they are active
 * @return int
 */
function jrFollower_get_user_follower_count($user_id, $active_only = true)
{
    $uid = (int) $user_id;
    $key = "jrfollower_user_count_{$uid}_" . intval($active_only);
    if (!$num = jrCore_get_flag($key)) {
        $num = 0;
        $tbl = jrCore_db_table_name('jrFollower', 'follow');
        $req = "SELECT COUNT(*) AS cnt FROM {$tbl} WHERE follow_user_id = {$uid}";
        if ($active_only) {
            $req .= " AND follow_active = 1";
        }
        $_rt = jrCore_db_query($req, 'SINGLE', false, null, false);
        if ($_rt && is_array($_rt)) {
            $num = intval($_rt['cnt']);
        }
        jrCore_set_flag($key, $num);
    }
    return $num;
}

/**
 * Get all profile_id's followed by all user_id's associated with a profile_id
 * @param int $profile_id
 * @return mixed
 */
function jrFollower_get_profiles_followed_by_profile_id($profile_id)
{
    $pid = (int) $profile_id;
    $key = "jrfollower_followed_by_profile_{$pid}";
    if (!$_rt = jrCore_is_cached('jrFollower', $key)) {
        $tb1 = jrCore_db_table_name('jrFollower', 'follow');
        $tb2 = jrCore_db_table_name('jrProfile', 'profile_link');
        $req = "SELECT f.follow_profile_id AS p FROM {$tb1} f LEFT JOIN {$tb2} l ON (l.user_id = f.follow_user_id) WHERE l.profile_id = {$pid}";
        $_rt = jrCore_db_query($req, 'p', false, 'p', false);
        if (!$_rt && !is_array($_rt)) {
            $_rt = 'no_items';
        }
        jrCore_add_to_cache('jrFollower', $key, $_rt, 0, $pid);
    }
    if ($_rt == 'no_items') {
        return false;
    }
    return $_rt;
}

//---------------------
// Smarty
//---------------------

/**
 * Return the number of profiles a user is following
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrFollower_following_count($params, $smarty)
{
    if (!isset($params['user_id'])) {
        return jrCore_smarty_missing_error('user_id');
    }
    if (!jrCore_checktype($params['user_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('user_id');
    }
    $num = jrFollower_get_user_follower_count($params['user_id']);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $num);
        return '';
    }
    return $num;
}

/**
 * Creates a Follow/Unfollow button for logged in users on a profile
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrFollower_button($params, $smarty)
{
    global $_conf, $_user;
    if (!jrUser_is_logged_in()) {
        return '';
    }
    if (!jrCore_module_is_active('jrFollower')) {
        return '';
    }
    // we must get a profile id
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    // If we are viewing our own profile....
    if (jrUser_get_profile_home_key('_profile_id') == $params['profile_id']) {
        return '';
    }
    $params['profile_id'] = (int) $params['profile_id'];
    $_lang                = jrUser_load_lang_strings();

    // Figure template
    $tpl = 'button_follow.tpl';
    $val = $_lang['jrFollower'][1];
    if ($_rt = jrFollower_is_follower($_user['_user_id'], $params['profile_id'])) {
        // See if we are pending or active...
        if (isset($_rt['follow_active']) && $_rt['follow_active'] != '1') {
            $tpl = 'button_pending.tpl';
            $val = $_lang['jrFollower'][5];
        }
        else {
            $tpl = 'button_following.tpl';
            $val = $_lang['jrFollower'][2];
        }
    }
    $params['value'] = $val;
    if (!isset($params['title'])) {
        $params['title'] = $val;
    }
    if (isset($params['title']) && jrCore_checktype($params['title'], 'number_nz') && isset($_lang["{$_conf['jrCore_active_skin']}"]["{$params['title']}"])) {
        $params['title'] = $_lang["{$_conf['jrCore_active_skin']}"]["{$params['title']}"];
    }
    $params['title'] = jrCore_entity_string($params['title']);

    // process and return
    $out = jrCore_parse_template($tpl, $params, 'jrFollower');
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Returns a list of users that the profile is currently not following as a suggestion list.
 * usage: {jrFollower_who_to_follow profile_id=$_profile_id}
 * @param $params
 * @param $smarty
 * @return bool|string
 */
function smarty_function_jrFollower_who_to_follow($params, $smarty)
{
    global $_conf;

    // we must get a profile id
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    $pid = (int) $params['profile_id'];

    $limit = 5;
    if (isset($params['limit']) && jrCore_checktype($params['limit'], 'number_nz')) {
        $limit = $params['limit'];
    }

    $template = 'who_to_follow.tpl';
    $tpl_dir  = 'jrFollower';
    if (isset($params['template']) && strlen($params['template']) > 0) {
        $template = $params['template'];
        $tpl_dir  = $_conf['jrCore_active_skin'];
    }

    // First - get all profiles that are already being followed
    $key = "jrfollower_who_to_follow_{$pid}";
    if (!$out = jrCore_is_cached('jrFollower', $key)) {
        if ($_id = jrFollower_get_profiles_followed_by_profile_id($pid)) {
            // Add in own profile ID
            $_id["{$pid}"] = $pid;
            $out           = '';
            $_it           = array(
                'search'        => array(
                    "_profile_id not_in " . implode(',', $_id)
                ),
                'limit'         => $limit,
                'skip_triggers' => true,
                'order_by'      => array('_profile_id' => 'desc')
            );
            foreach ($params as $k => $v) {
                if (strpos($k, 'search') === 0) {
                    $_it['search'][] = $v;
                }
            }
            $_it = jrCore_db_search_items('jrProfile', $_it);
            if ($_it && is_array($_it) && isset($_it['_items'])) {
                $out = jrCore_parse_template($template, $_it, $tpl_dir);
            }
        }
        else {
            $out = 'no_items';
        }
        jrCore_add_to_cache('jrFollower', $key, $out, 0, $pid);
    }
    if ($out == 'no_items') {
        $out = '';
    }
    if (!empty($params['assign'])) {
        /** @noinspection PhpUndefinedMethodInspection */
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}
