<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Formatters
 * @copyright 2022 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 * @noinspection PhpElementIsNotAvailableInCurrentPhpVersionInspection
 */

defined('APP_DIR') or exit();

/**
 * Load formatter plugins for a given 2 digit Country Code
 * @see https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2
 * @param string $country_code 2 Digit ISO 3166 Country Code
 * @return string
 */
function jrCore_load_formatter_plugins($country_code)
{
    if (is_null($country_code) || strlen("{$country_code}") !== 2) {
        $country = 'US';
    }
    else {
        $country = strtoupper("{$country_code}");
    }
    $func = "jrCore_plugin_format_phone_number_{$country}";
    if (!function_exists($func)) {
        @include_once APP_DIR . "/modules/jrCore/plugins/formatter/{$country}.php";
    }
    if (!function_exists($func)) {
        $func = "jrCore_plugin_format_phone_number_US";
        if (!function_exists($func)) {
            require_once APP_DIR . "/modules/jrCore/plugins/formatter/US.php";
        }
    }
    return $country;
}

/**
 * Format a phone number
 * @see https://en.wikipedia.org/wiki/National_conventions_for_writing_telephone_numbers
 * @param string $string input phone number
 * @param string $country optional 2 digit country code
 * @param bool $include_country_code Set to TRUE to return Country code as part of number
 * @return string
 */
function jrCore_format_phone_number($string, $country = null, $include_country_code = false)
{
    $country = jrCore_load_formatter_plugins($country);
    $pn_func = "jrCore_plugin_format_phone_number_{$country}";
    // Should never see this..
    if (!function_exists($pn_func)) {
        return $string;
    }

    $c_code = '';
    $string = trim(preg_replace('/[^0-9+]/', '', $string));
    $string = $pn_func($string);
    if ($include_country_code) {
        $c_code = 1;
        $c_func = "jrCore_plugin_format_phone_number_{$country}_country_code";
        if (function_exists($c_func)) {
            $c_code = $c_func();
        }
    }
    return "{$c_code}{$string}";
}

/**
 * Format an integer to "readable" size
 * @param int $number Number to format
 * @return string Returns formatted number
 */
function jrCore_format_size($number)
{
    // make sure we get a number
    if (!is_numeric($number)) {
        return false;
    }
    $kb = 1024;
    $mb = 1024 * $kb;
    $gb = 1024 * $mb;
    $tb = 1024 * $gb;

    // if it's less than a kb we just return
    // the size, otherwise we keep going until
    // the size is in the appropriate measurement range.
    if ($number < $kb) {
        return $number . 'B';
    }
    elseif ($number < $mb) {
        return round($number / $kb) . 'KB';
    }
    elseif ($number < $gb) {
        return round($number / $mb, 1) . 'MB';
    }
    elseif ($number < $tb) {
        return round($number / $gb, 2) . 'GB';
    }
    return round($number / $tb, 2) . 'TB';
}

/**
 * Format an integer of seconds to a "readable" timestamp in HH:MM:SS
 * @param int $length Time number to format
 * @return string Returns formatted time
 */
function jrCore_format_seconds($length = 0)
{
    $length = round($length); // no decimals
    $numh   = (int) ($length / 3600);
    $hour   = str_pad($numh, 2, '0', STR_PAD_LEFT);
    $mins   = str_pad(floor(($length - ($numh * 3600)) / 60), 2, '0', STR_PAD_LEFT);
    $secs   = str_pad(($length % 60), 2, '0', STR_PAD_LEFT);
    return "{$hour}:{$mins}:{$secs}";
}

/**
 * Formats an Epoch Time Stamp to the format specified by the system
 * @param int $timestamp Epoch Time Stamp to format
 * @param bool $date_only Set to true to just return DATE portion (instead of DATE TIME)
 * @param string $format Date Format for Display
 * @param bool $adjust set to FALSE to not adjust for DLS
 * @return string
 */
function jrCore_format_time($timestamp, $date_only = false, $format = null, $adjust = true)
{
    global $_user, $_conf;
    if (!jrCore_checktype($timestamp, 'number_nz')) {
        return '';
    }
    if ($date_only) {
        if (is_null($format)) {
            $format = $_conf['jrCore_date_format'];
        }
    }
    else {
        if (is_null($format)) {
            $format = "{$_conf['jrCore_date_format']} {$_conf['jrCore_hour_format']}";
        }
    }

    $dlsoffset = 0;
    if ($adjust) {
        $dlsoffset = jrCore_get_timezone_dls_offset($timestamp, $adjust);
        $timestamp = ($timestamp + $dlsoffset);
    }
    if ($format == 'relative') {

        $_lang = jrUser_load_lang_strings();

        $time = (time() + $dlsoffset);

        $diff = ($time - $timestamp);
        if ($diff >= 0) {
            // Past relative times

            // Seconds
            if ($diff < 60) {
                return sprintf($diff > 1 ? "%s {$_lang['jrCore'][51]}" : $_lang['jrCore'][52], $diff);
            }

            // Minutes
            $diff = floor($diff / 60);
            if ($diff < 60) {
                return sprintf($diff > 1 ? "%s {$_lang['jrCore'][53]}" : $_lang['jrCore'][54], $diff);
            }

            // Hours
            $diff = floor($diff / 60);
            if ($diff < 24) {
                return sprintf($diff > 1 ? "%s {$_lang['jrCore'][55]}" : $_lang['jrCore'][56], $diff);
            }

            // Days
            $diff = round($diff / 24);
            if ($diff < 7) {
                return sprintf($diff > 1 ? "%s {$_lang['jrCore'][57]}" : $_lang['jrCore'][58], $diff);
            }

            // Months
            if ($diff < 30) {
                $diff = round($diff / 7);
                return sprintf($diff > 1 ? "%s {$_lang['jrCore'][59]}" : $_lang['jrCore'][60], $diff);
            }

            // Years
            $diff = round($diff / 30);
            if ($diff < 12) {
                return sprintf($diff > 1 ? "%s {$_lang['jrCore'][61]}" : $_lang['jrCore'][62], $diff);
            }
            $diff = date('Y', $time) - date('Y', $timestamp);
            return sprintf($diff > 1 ? "%s {$_lang['jrCore'][63]}" : $_lang['jrCore'][64], $diff);
        }
        else {
            // Future relative times
            $diff = abs($diff);

            // Seconds
            if ($diff < 60) {
                return sprintf($diff > 1 ? "{$_lang['jrCore'][177]} %s {$_lang['jrCore'][178]}" : $_lang['jrCore'][179], $diff);
            }

            // Minutes
            $diff = floor($diff / 60);
            if ($diff < 60) {
                return sprintf($diff > 1 ? "{$_lang['jrCore'][177]} %s {$_lang['jrCore'][180]}" : $_lang['jrCore'][181], $diff);
            }

            // Hours
            $diff = floor($diff / 60);
            if ($diff < 24) {
                return sprintf($diff > 1 ? "{$_lang['jrCore'][177]} %s {$_lang['jrCore'][182]}" : $_lang['jrCore'][183], $diff);
            }

            // Days
            $diff = round($diff / 24);
            if ($diff < 7) {
                return sprintf($diff > 1 ? "{$_lang['jrCore'][177]} %s {$_lang['jrCore'][184]}" : $_lang['jrCore'][185], $diff);
            }

            // Months
            if ($diff < 30) {
                $diff = round($diff / 7);
                return sprintf($diff > 1 ? "{$_lang['jrCore'][177]} %s {$_lang['jrCore'][186]}" : $_lang['jrCore'][187], $diff);
            }

            // Years
            $diff = round($diff / 30);
            if ($diff < 12) {
                return sprintf($diff > 1 ? "{$_lang['jrCore'][177]} %s {$_lang['jrCore'][188]}" : $_lang['jrCore'][189], $diff);
            }
            $diff = date('Y', $timestamp) - date('Y', $time);
            return sprintf($diff > 1 ? "{$_lang['jrCore'][177]} %s {$_lang['jrCore'][190]}" : $_lang['jrCore'][191], $diff);
        }
    }
    $lang = (isset($_conf['jrUser_default_language'])) ? $_conf['jrUser_default_language'] : 'en_US';
    if (jrUser_is_logged_in() && !empty($_user['user_language'])) {
        $lang = str_replace('-', '_', $_user['user_language']);
    }
    setlocale(LC_TIME, $lang . '.UTF-8');
    $format = jrCore_get_converted_strftime_modifiers($format);
    return gmdate($format, $timestamp);
}

/**
 * Get a modifier string converted from strftime to date format
 * @param string $string date modifier string
 * @return string
 */
function jrCore_get_converted_strftime_modifiers($string)
{
    if (!strpos(' ' . $string, '%')) {
        return $string;
    }
    $_map = array(
        '<br>' => '\<\b\\r\>',
        '%D'   => 'm/d/y',
        '%T'   => 'H:i:s',
        '%a'   => 'D',
        '%b'   => 'M',
        '%A'   => 'l',
        '%B'   => 'F',
        '%d'   => 'd',
        '%u'   => 'N',
        '%w'   => 'w',
        '%m'   => 'm',
        '%e'   => 'j',
        '%V'   => 'W',
        '%Y'   => 'Y',
        '%y'   => 'y',
        '%m'   => 'm',
        '%d'   => 'd',
        '%H'   => 'H',
        '%k'   => 'G',
        '%I'   => 'h',
        '%l'   => 'g',
        '%M'   => 'i',
        '%p'   => 'A',
        '%P'   => 'a',
        '%S'   => 's',
        '%z'   => 'O',
        '%x'   => 'm/d/y',
        '%r'   => 'H:i:s A',
        '%F'   => 'Y-m-d'
    );
    return str_replace(array_keys($_map), $_map, $string);
}

/**
 * Format a readable weight
 * @param string $weight A number or weight - i.e. "35" or "25k"
 * @param string $output p|k|detect
 * @param int $precision
 * @return string
 */
function jrCore_format_weight($weight, $output = 'detect', $precision = 1)
{
    if ($output == 'detect') {
        if (strpos("{$weight}", 'k')) {
            $output = 'k';
        }
        else {
            $output = 'p';
        }
    }
    if ($output == 'p') {
        if (strpos("{$weight}", 'k')) {
            $number = jrCore_convert_kilos_to_pounds($weight, $precision);
        }
        else {
            $number = floatval(str_replace('p', '', "{$weight}"));
            $number = round($number, $precision);
        }
        return "{$number} lb";
    }
    if (strpos("{$weight}", 'p')) {
        $number = jrCore_convert_pounds_to_kilos($weight, $precision);
    }
    else {
        $number = floatval(str_replace('k', '', "{$weight}"));
        $number = round($number, $precision);
    }
    $number = number_format($number);
    return "{$number} kg";
}

/**
 * Convert a number in POUNDS to KILOGRAMS
 * @param float|int $number
 * @param int $precision
 * @return float
 */
function jrCore_convert_pounds_to_kilos($number, $precision = 1)
{
    // Are we already in kilos?
    if (strpos("{$number}", 'k')) {
        return round(floatval(str_replace('k', '', "{$number}")), $precision);
    }
    $number = floatval(str_replace('p', '', "{$number}"));
    if ($number <= 0) {
        return 0;
    }
    return round(($number / 2.204623), intval($precision));
}

/**
 * Convert a number in KILOGRAMS to POUNDS
 * @param float|int $number
 * @param int $precision
 * @return float
 */
function jrCore_convert_kilos_to_pounds($number, $precision = 1)
{
    // Are we already in pounds?
    if (strpos("{$number}", 'p')) {
        return round(floatval(str_replace('p', '', "{$number}")), $precision);
    }
    $number = floatval(str_replace('k', '', "{$number}"));
    if ($number <= 0) {
        return 0;
    }
    return round(($number * 2.204623), intval($precision));
}

/**
 * Format a number based on Locale information
 * @param $number number to convert
 * @param $dec int Number of decimal places
 * @return string
 */
function jrCore_number_format($number, $dec = 0)
{
    if (!is_numeric($number)) {
        return $number;
    }
    if ($dec > 0) {
        $_tmp = localeconv();
        $dsep = (strlen("{$_tmp['decimal_point']}") > 0) ? $_tmp['decimal_point'] : '.';
        $tsep = (strlen("{$_tmp['thousands_sep']}") > 0) ? $_tmp['thousands_sep'] : ',';
        return number_format($number, $dec, $dsep, $tsep);
    }
    return number_format($number);
}

/**
 * Format currency to locale
 * @param $amount string Monetary amount
 * @param null $format string format
 * @return string
 * @deprecated
 */
function jrCore_money_format($amount, $format = null)
{
    if (function_exists('money_format')) {
        if (is_null($format)) {
            $format = '%n';
        }
        return money_format($format, $amount);
    }
    return false;
}
