<?php
 /**
 * Jamroom Site Builder module
 *
 * copyright 2021 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *

// make sure we are not being called directly
defined('APP_DIR') or exit();

 * meta
 */
function jrSiteBuilder_meta()
{
    return array(
        'name'        => 'Site Builder',
        'url'         => 'sbcore',
        'version'     => '2.4.2',
        'developer'   => 'The Jamroom Network, &copy;' . strftime('%Y'),
        'description' => 'Site Builder Core provides support for all Site Builder functions',
        'category'    => 'site',
        'requires'    => 'jrCore:6.4.1b3',
        'priority'    => 5,
        'license'     => 'mpl'
    );
}

/**
 * init
 */
function jrSiteBuilder_init()
{
    // Events
    jrCore_register_module_feature('jrCore', 'javascript', 'jrSiteBuilder', 'jrSiteBuilder_admin.js', 'admin');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrSiteBuilder', 'jrSiteBuilder.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrSiteBuilder', 'jquery-ui.min.js', 'admin');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrSiteBuilder', 'jquery.mjs.nestedSortable.js', 'admin');
    jrCore_register_module_feature('jrCore', 'css', 'jrSiteBuilder', 'jrSiteBuilder.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrSiteBuilder', 'jquery.nouislider.css');

    // Site Builder Core provided widgets
    jrCore_register_module_feature('jrSiteBuilder', 'widget', 'jrSiteBuilder', 'widget_html', 'HTML Editor');
    jrCore_register_module_feature('jrSiteBuilder', 'widget', 'jrSiteBuilder', 'widget_code', 'Template Code');

    // Insert our "Edit this page" option for masters
    jrCore_register_event_listener('jrCore', 'index_template', 'jrSiteBuilder_view_template_listener');
    jrCore_register_event_listener('jrCore', 'skin_template', 'jrSiteBuilder_view_template_listener');
    jrCore_register_event_listener('jrCore', 'profile_template', 'jrSiteBuilder_view_template_listener');
    jrCore_register_event_listener('jrCore', 'module_view', 'jrSiteBuilder_view_template_listener');

    jrCore_register_event_listener('jrCore', 'view_results', 'jrSiteBuilder_view_results_listener');

    jrCore_register_event_listener('jrCore', 'verify_module', 'jrSiteBuilder_verify_module_listener');
    jrCore_register_event_listener('jrCore', 'verify_skin', 'jrSiteBuilder_verify_skin_listener');

    // For a 404 not found page, if it is a master and looks good let them create it
    jrCore_register_event_listener('jrCore', '404_not_found', 'jrSiteBuilder_404_not_found_listener');

    // widget specifies a template, but its not found
    jrCore_register_event_listener('jrCore', 'tpl_404', 'jrSiteBuilder_tpl_404_listener');

    // Add pages to Sitemap
    jrCore_register_event_listener('jrSitemap', 'sitemap_site_pages', 'jrSiteBuilder_sitemap_site_pages_listener');

    // tools
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrSiteBuilder', 'browser', array('Page Browser', 'View and Delete existing Site Builder pages'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrSiteBuilder', 'reset', array('Reset Site Builder', 'Delete all Menus, Panels, and Widgets from the system'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrSiteBuilder', 'import', array('Backup Browser', 'View and Import existing Site Buildet Backups'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrSiteBuilder', 'export', array('Create Backup', 'Create a new Site Builder backup of Pages, Widgets and Menus'));

    // Once a day we export the layout as a backup
    jrCore_register_event_listener('jrCore', 'daily_maintenance', 'jrSiteBuilder_daily_maintenance_listener');

    // clear the default menu array on clear caches
    jrCore_register_event_listener('jrCore', 'template_cache_reset', 'jrSiteBuilder_template_cache_reset_listener');

    return true;
}

//------------------------------------
// WIDGETS
//------------------------------------

/**
 * Display CONFIG screen for HTML Editor Widget
 * @param $_post array Post info
 * @param $_user array User array
 * @param $_conf array Global Config
 * @param $_wg array Widget info
 * @return bool
 */
function jrSiteBuilder_widget_html_config($_post, $_user, $_conf, $_wg)
{
    $html = jrCore_parse_template('widget_html_config.tpl', $_wg, 'jrSiteBuilder');
    jrCore_page_custom($html);
    return true;
}

/**
 * Get Widget results from posted Config data
 * @param $_post array Post info
 * @return array
 */
function jrSiteBuilder_widget_html_config_save($_post)
{
    // Did we get some content?
    if (!isset($_post['html_content_editor_contents']) || strlen($_post['html_content_editor_contents']) === 0) {
        jrCore_set_form_notice('error', 'You forgot to entet the HTML content');
        jrCore_form_result();
    }

    // Make sure some smarty constructs are properly setup
    $_rp                                   = array(
        '&amp;&amp;' => '&&',
        '&gt;'       => '>',
        '&lt;'       => '<'
    );
    $_post['html_content_editor_contents'] = str_replace(array_keys($_rp), $_rp, $_post['html_content_editor_contents']);

    // We need to test this HTML and make sure it does not cause any Smarty errors
    if (jrSiteBuilder_template_code_contains_errors($_post['html_content_editor_contents'])) {
        jrCore_set_form_notice('error', 'There is a Smarty syntax error in your HTML - please fix and try again');
        jrCore_form_result();
    }
    return array('html_content' => trim($_post['html_content_editor_contents']));
}

/**
 * HTML Editor Widget DISPLAY
 * @param $_widget array Page Widget info
 * @return string
 */
function jrSiteBuilder_widget_html_display($_widget)
{
    $_widget['html_content'] = '<div class="item">' . smarty_modifier_jrCore_format_string($_widget['html_content'], 'allow_all_formatters', null, 'nl2br,hash_tags') . '</div>';
    return jrCore_parse_template($_widget['html_content'], $_widget, 'jrSiteBuilder');
}

/**
 * Display CONFIG screen for Template Code Widget
 * @param $_post array Post info
 * @param $_user array User array
 * @param $_conf array Global Config
 * @param $_wg array Widget info
 * @return bool
 */
function jrSiteBuilder_widget_code_config($_post, $_user, $_conf, $_wg)
{
    $_wg['code_content'] = htmlspecialchars($_wg['code_content'], ENT_NOQUOTES);
    $html                = jrCore_parse_template('widget_code_config.tpl', $_wg, 'jrSiteBuilder');
    jrCore_page_custom($html);
    return true;
}

/**
 * Get Widget results from posted Config data
 * @param $_post array Post info
 * @return array
 */
function jrSiteBuilder_widget_code_config_save($_post)
{
    // We need to test this HTML and make sure it does not cause any Smarty errors
    if (jrSiteBuilder_template_code_contains_errors($_post['code_content'])) {
        jrCore_set_form_notice('error', 'There is a Smarty syntax error in your Template Code - please fix and try again');
        jrCore_form_result();
    }
    return array('code_content' => $_post['code_content']);
}

/**
 * Template Code Widget
 * @param $_widget array Page Widget info
 * @return string
 */
function jrSiteBuilder_widget_code_display($_widget)
{
    global $_post;
    return jrCore_parse_template($_widget['code_content'], $_post, 'jrSiteBuilder');
}

//------------------------------------
// EVENT LISTENERS
//------------------------------------

/**
 * Profile Template - we listen for custom pages
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return mixed
 */
function jrSiteBuilder_sitemap_site_pages_listener($_data, $_user, $_conf, $_args, $event)
{
    // Add in custom site builder pages
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'page');
    $req = "SELECT page_uri FROM {$tbl} WHERE page_active = 1";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt)) {
        foreach ($_rt as $p) {
            $_data[] = $p['page_uri'];
        }
    }
    return $_data;
}

/**
 * Create a new page on a 404
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrSiteBuilder_404_not_found_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    // Make sure this is a not a profile page...
    if (jrUser_is_master() && !empty($_post['module_url'])) {
        if (empty($_post['module'])) {

            // Let the Master CREATE this new page via Site Builder
            $_post['create_notice'] = '';

            $temp = jrCore_parse_template('page_create.tpl', $_post, 'jrSiteBuilder');
            jrCore_send_response_and_detach($temp);
        }
    }
    return $_data;
}

/**
 * Return SB page if enabled for URI
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return mixed
 */
function jrSiteBuilder_view_template_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_mods, $_post;

    // Check if this is a module view function - not module index
    if (isset($_post['module']) && isset($_mods["{$_post['module']}"]) && isset($_post['option']) && strlen($_post['option']) > 0) {
        return $_data;
    }

    switch ($event) {
        case 'index_template':
            $page_uri = '/';
            break;
        default:
            $page_uri = jrSiteBuilder_get_page_uri();
            break;
    }
    if ($page_uri) {

        if ($page_uri == '/index') {
            jrCore_location($_conf['jrCore_base_url']);
        }

        // Get our page info - first by DB
        $_pg = jrSiteBuilder_get_page_by_uri($page_uri);
        if ($_pg && is_array($_pg)) {
            // We have a DB SB page
            if (isset($_pg['page_active']) && $_pg['page_active'] == '1') {
                if (jrUser_is_master() || !isset($_pg['page_groups']) || jrCore_user_is_part_of_group($_pg['page_groups'])) {
                    if ($_temp = jrSiteBuilder_get_page($_pg)) {
                        if (strlen($_temp) > 5) {
                            $_data = $_temp;
                        }
                    }
                }
            }
        }
        else {
            // See if we have a JSON file for this page
            $_pg = jrSiteBuilder_get_page_by_json_file($_conf['jrCore_active_skin'], $page_uri);
            if ($_pg && is_array($_pg)) {
                if (!isset($_pg['page_active']) || $_pg['page_active'] == '1') {
                    if (jrUser_is_master() || !isset($_pg['page_groups']) || jrCore_user_is_part_of_group($_pg['page_groups'])) {
                        if ($_temp = jrSiteBuilder_get_page($_pg)) {
                            if (strlen($_temp) > 5) {
                                $_data = $_temp;
                            }
                        }
                        $_pg['notice'] = "This URL is using a default Site Builder layout - click OK to create a new Site Builder page.\\n\\nIf you change your mind, delete the new page and the default layout will return.";
                    }
                }
            }
        }

        if (jrUser_is_master()) {

            // If we don't have a page, show options
            if (!is_array($_pg)) {
                jrCore_set_flag('jrsitebuilder_show_page_notices', $page_uri);
            }
            // If we DO have a page, add our SB buttons
            if (!is_array($_data) && strpos($_data, '<body>')) {
                if (!strpos($_data, 'jrSiteBuilder_no_sitebuilder')) {
                    $temp  = jrCore_parse_template('page_editor_include.tpl', $_pg, 'jrSiteBuilder');
                    $_data = str_replace('</body>', "{$temp}\n</body>", $_data);
                }
            }

        }
    }
    return $_data;
}

/**
 * Insert "Edit This Page" option for a master admin
 * @param $_data string HTML results
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return string
 */
function jrSiteBuilder_view_results_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post, $_mods;
    if ($page_uri = jrCore_get_flag('jrsitebuilder_show_page_notices')) {

        if (jrCore_get_flag('jrprofile_view_is_active')) {
            // This is a valid profile view - return
            return $_data;
        }

        // Are we being hidden on this view?
        if (jrCore_get_flag('jrsitebuilder_hide_sitebuilder')) {
            return $_data;
        }

        // OK - we either have a MODULE INDEX or a SKIN TEMPLATE - let the site
        // admin OVERRIDE these if they want to, but let them know about it
        $_pg = array();
        $mod = false;
        $tpl = false;
        $_mt = jrCore_skin_meta_data($_conf['jrCore_active_skin']);
        if ($page_uri == '/') {
            $mod = $_mt['title'];
            $tpl = 'index.tpl';
        }
        elseif (isset($_post['module']) && isset($_mods["{$_post['module']}"]) && empty($_post['option']) && !strpos($page_uri, '/')) {
            $mod = $_mods["{$_post['module']}"]['module_name'] . ' module';
            $tpl = "index.tpl";
        }
        elseif (isset($_post['module']) && isset($_mods["{$_post['module']}"]) && !strpos($page_uri, '/')) {
            $mod = $_mods["{$_post['module']}"]['module_name'] . ' module';
            $tpl = "item_index.tpl";
        }
        elseif (is_file(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/{$_post['module_url']}.tpl")) {
            $mod = $_mt['title'] . ' skin';
            $tpl = "{$_post['module_url']}.tpl";
        }
        if ($mod) {
            $_pg['notice'] = "This URL is using the {$mod} {$tpl} template - click OK to create a new Site Builder page.\\n\\nIf you change your mind, delete the new page and the {$mod} {$tpl} template content will return.";
        }
        // Add SB buttons
        if (jrUser_is_master() && strpos($_data, '</body>') && !strpos($_data, 'jrSiteBuilder_no_sitebuilder')) {
            $temp  = jrCore_parse_template('page_editor_include.tpl', $_pg, 'jrSiteBuilder');
            $_data = str_replace('</body>', "{$temp}\n</body>", $_data);
        }
    }
    return $_data;
}

/**
 * clear the default skin menu when skins change.
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrSiteBuilder_verify_skin_listener($_data, $_user, $_conf, $_args, $event)
{
    // on 'verify_skin' clear the default menu temp values
    jrCore_delete_temp_value('jrSiteBuilder', 'default_menu');
    return $_data;
}

/**
 * Fired when the integrity check runs.  Used to turn off jrPanels, jrWidget, jrMenu so jrSiteBuilder is main.
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrSiteBuilder_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // correct for /index and rename it to /
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'page');
    $req = "SELECT page_id, page_uri FROM {$tbl} WHERE page_uri =  '/'";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if (!$_rt || $_rt == null) {
        // do the update
        $req = "SELECT page_id, page_uri FROM {$tbl} WHERE page_uri =  '/index'";
        $_rt = jrCore_db_query($req, 'SINGLE');
        if ($_rt && is_array($_rt)) {
            $req = "UPDATE {$tbl} SET page_uri = '/' WHERE page_id =  '{$_rt['page_id']}'";
            jrCore_db_query($req);
        }

        $tbl = jrCore_db_table_name('jrSiteBuilder', 'menu');
        $req = "SELECT menu_id, menu_url FROM {$tbl} WHERE menu_url = 'index' ";
        $_rt = jrCore_db_query($req, 'SINGLE');
        if ($_rt && is_array($_rt)) {
            $req = "UPDATE {$tbl} SET menu_url = '' WHERE menu_id =  '{$_rt['menu_id']}'";
            jrCore_db_query($req);
        }
    }

    // make sure each widget has a unique id
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'widget');
    $req = "SELECT * FROM {$tbl} WHERE widget_unique = ''";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if (is_array($_rt)) {
        $i = 0;
        foreach ($_rt as $_w) {
            $unique = jrCore_create_unique_string(20);
            $req    = "UPDATE {$tbl} SET widget_unique = '{$unique}' WHERE widget_id = '{$_w['widget_id']}'";
            jrCore_db_query($req);
            $i++;
        }
        if ($i > 0) {
            jrCore_logger('INF', 'SiteBuilder: added ' . $i . ' unique ids to widgets.');
        }
    }

    // Move any existing Site Builder backups to NEW DB format
    $dir = jrCore_get_media_directory(0, FORCE_LOCAL);
    if ($_fl = glob("{$dir}/sb_backup_*.json")) {
        $cnt = 0;
        $tbl = jrCore_db_table_name('jrSiteBuilder', 'backup');
        foreach ($_fl as $file) {
            $name = basename($file);
            $time = (int) str_replace(array('sb_backup_', '.json'), '', $name);
            $req  = "INSERT IGNORE INTO {$tbl} (backup_time, backup_data) VALUES ({$time}, '" . jrCore_db_escape(file_get_contents($file)) . "')";
            jrCore_db_query($req);
            $cnt++;
            unlink($file);
        }
        if ($cnt > 0) {
            jrCore_logger('INF', 'SiteBuilder: migrated ' . jrCore_number_format($cnt) . ' backup files to new backup format');
        }
    }

    return $_data;
}

/**
 * cleare the default menu items when the cache is cleared.
 * @param $_data array trigger array
 * @param $_user array User info
 * @param $_conf array Global Config
 * @param $_args array Extra arguments from trigger
 * @param $event string Event name
 * @return array
 */
function jrSiteBuilder_template_cache_reset_listener($_data, $_user, $_conf, $_args, $event)
{

    jrCore_delete_temp_value('jrSiteBuilder', 'default_menu');
    return $_data;
}

/**
 * pick up on the template 404 and show a custom page so the widgets don't exit().
 * @param array $_data
 * @param array $_user
 * @param array $_conf
 * @param array $_args
 * @param string $event
 * @return mixed
 */
function jrSiteBuilder_tpl_404_listener($_data, $_user, $_conf, $_args, $event)
{
    // Module needs to be active
    if (!jrProfile_is_profile_view() && !jrCore_is_ajax_request()) {
        $_data['file'] = $_conf['jrCore_base_dir'] . '/modules/jrSiteBuilder/templates/404.tpl';
    }
    return $_data;
}

/**
 * jrSiteBuilder_daily_maintenance_listener
 * <code>
 * Uses the EXPORT tool to add a package called sb_backup_1421982941.json where the number is a timestamp.
 * Old auto-created EXPORT packages are deleted.
 * </code>
 * @param $_data array incoming data
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array $_data
 */
function jrSiteBuilder_daily_maintenance_listener($_data, $_user, $_conf, $_args, $event)
{
    if (jrCore_get_config_value('jrSiteBuilder', 'backup', 'on') == 'on') {
        jrSiteBuilder_backup();
        jrSiteBuilder_prune_backups();
    }
    return $_data;
}

//------------------------------------
// FUNCTIONS
//------------------------------------

/**
 * Backup Site Builder system
 * @param mixed $page_id
 * @return bool
 */
function jrSiteBuilder_backup($page_id = null)
{
    @ini_set('memory_limit', '512M');
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'page');
    if (!is_null($page_id)) {
        if (is_array($page_id)) {
            $_id = array();
            foreach ($page_id as $p) {
                $_id[] = (int) $p;
            }
            $req = "SELECT page_id FROM {$tbl} WHERE page_id IN(" . implode(',', $_id) . ')';
        }
        else {
            $req = "SELECT page_id FROM {$tbl} WHERE page_id = " . intval($page_id);
        }
    }
    else {
        $req = "SELECT page_id FROM {$tbl}";
    }
    $_rt = jrCore_db_query($req, 'page_id', false, 'page_id');
    if (!$_rt || !is_array($_rt)) {
        // No Site Builder content, return.
        return true;
    }
    // Create Backup
    if ($_rt = jrSiteBuilder_export($_rt)) {
        $_rt = jrCore_db_escape(json_encode($_rt));
        $tbl = jrCore_db_table_name('jrSiteBuilder', 'backup');
        $req = "INSERT INTO {$tbl} (backup_time, backup_data) VALUES (UNIX_TIMESTAMP(), '{$_rt}')";
        jrCore_db_query($req);
    }
    return true;
}

/**
 * Prune old site builder backups
 * @return bool
 */
function jrSiteBuilder_prune_backups()
{
    $old = jrCore_get_config_value('jrSiteBuilder', 'backup_retain_days', 30);
    if ($old > 0) {
        $old = ($old * 86400);
        $tbl = jrCore_db_table_name('jrSiteBuilder', 'backup');
        $req = "DELETE FROM {$tbl} WHERE backup_time < (UNIX_TIMESTAMP() - {$old})";
        $cnt = jrCore_db_query($req, 'COUNT');
        if ($cnt > 0) {
            jrCore_logger('INF', "SiteBuilder: removed {$cnt} expired backup entries");
        }
    }
    return true;
}

/**
 * Get backup data by backup_id
 * @param int $backup_id
 * @return mixed
 */
function jrSiteBuilder_get_backup_by_id($backup_id)
{
    $bid = (int) $backup_id;
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'backup');
    $req = "SELECT * FROM {$tbl} WHERE backup_id = {$bid}";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Check if a given bit of template code is valid
 * @param $code string Smarty Template code
 * @return bool
 */
function jrSiteBuilder_template_code_contains_errors($code)
{
    $err = jrCore_test_template_for_errors('jrSiteBuilder', $code);
    if ($err && strpos($err, 'error') === 0) {
        return true;
    }
    return false;
}

/**
 * Get the title of a widget
 * @param $title mixed array or string
 * @return string
 */
function jrSiteBuilder_get_widget_title($title)
{
    return (is_array($title)) ? $title['title'] : $title;
}

/**
 * Construct the Page Builder URI for the current URL
 * @return string
 */
function jrSiteBuilder_get_page_uri()
{
    global $_post;
    $uri = '/';
    if (isset($_post['module_url']) && strlen($_post['module_url']) > 0) {
        $uri .= "{$_post['module_url']}";
        if (isset($_post['option']) && strlen($_post['option']) > 0) {
            $uri .= "/{$_post['option']}";
            $num = 1;
            while (true) {
                if (isset($_post["_{$num}"])) {
                    $uri .= '/' . $_post["_{$num}"];
                    $num++;
                }
                else {
                    break;
                }
            }
        }
    }
    return $uri;
}

/**
 * Construct a good Menu URL from a given URL
 * @param $url string URL as given in form
 * @return string
 */
function jrSiteBuilder_get_menu_url($url)
{
    $url = trim(trim($url), '/');
    if (strpos($url, '/')) {
        $_ot = array();
        foreach (explode('/', $url) as $part) {
            $_ot[] = jrCore_url_string($part);
        }
        return implode('/', $_ot);
    }
    return jrCore_url_string($url);
}

/**
 * Get Site menu entries
 * @param bool $cache set to FALSE to bypass caching
 * @param bool|true $group_filter
 * @return array|mixed
 */
function jrSiteBuilder_get_menu_entries($cache = true, $group_filter = true)
{
    $ckey = 'sb-menu' . intval($group_filter);
    if (!$cache || !$_rp = jrCore_is_cached('jrSiteBuilder', $ckey)) {

        // Get existing menu entries
        $tbl = jrCore_db_table_name('jrSiteBuilder', 'menu');
        $req = "SELECT * FROM {$tbl} ORDER BY menu_order ASC";
        $_rt = jrCore_db_query($req, 'menu_id');

        $_rp = array(
            '_list'  => array(),
            '_items' => $_rt
        );
        if ($_rt && is_array($_rt)) {

            // First - get TOP level menu items setup
            foreach ($_rt as $mid => $_m) {
                // Remove any we are not allowed to view
                if (isset($_m['menu_group']) && $_m['menu_group'] != 'all' && !jrCore_user_is_part_of_group($_m['menu_group']) && $group_filter) {
                    unset($_rt[$mid]);
                    continue;
                }
                $pid = (int) $_m['menu_parent_id'];
                if ($pid === 0) {
                    $_rp['_list'][$mid] = $_m;
                    unset($_rt[$mid]);
                }
            }

            // Next get First Level menu items
            if (count($_rt) > 0) {
                $_mp = array();
                foreach ($_rt as $mid => $_m) {
                    $pid = (int) $_m['menu_parent_id'];
                    if (isset($_rp['_list'][$pid])) {
                        // We are a First Level menu item
                        if (!isset($_rp['_list'][$pid]['_children']) || !is_array($_rp['_list'][$pid]['_children'])) {
                            $_rp['_list'][$pid]['_children'] = array();
                        }
                        $_rp['_list'][$pid]['_children'][$_m['menu_id']] = $_m;
                        $_mp[$mid]                                       = $pid;
                        unset($_rt[$mid]);
                    }
                }
            }

            // Finally get Second Level options
            if (count($_rt) > 0) {
                foreach ($_rt as $mid => $_m) {
                    $pid = (int) $_m['menu_parent_id'];
                    if (isset($_mp) && isset($_mp[$pid])) {
                        $tid = $_mp[$pid];
                        if (isset($_rp['_list'][$tid]['_children'][$pid])) {
                            if (!isset($_rp['_list'][$tid]['_children'][$pid]['_children']) || !is_array($_rp['_list'][$tid]['_children'][$pid]['_children'])) {
                                $_rp['_list'][$tid]['_children'][$pid]['_children'] = array();
                            }
                            $ord                                                      = (int) $_m['menu_order'];
                            $_rp['_list'][$tid]['_children'][$pid]['_children'][$ord] = $_m;
                        }
                    }
                }
            }

        }
        jrCore_add_to_cache('jrSiteBuilder', $ckey, $_rp);
    }
    return $_rp;
}

/**
 * Get Data for a page based on URI
 * @param $uri string URI
 * @return mixed
 */
function jrSiteBuilder_get_page_by_uri($uri)
{
    // We need to remove any URL variables from the URI
    if (!$_rt = jrCore_get_flag('jrsitebuilder_get_page_by_uri')) {
        $tbl = jrCore_db_table_name('jrSiteBuilder', 'page');
        $req = "SELECT * FROM {$tbl} WHERE page_uri = '" . jrCore_db_escape($uri) . "' LIMIT 1";
        $_rt = jrCore_db_query($req, 'SINGLE');
        if (!$_rt || !is_array($_rt)) {
            $_rt = 'not_found';
        }
        jrCore_set_flag('jrsitebuilder_get_page_by_uri', $_rt);
    }
    return ($_rt == 'not_found') ? false : $_rt;
}

/**
 * Get Data for a page based on filename
 * @param $skin string Skin
 * @param $uri string URI
 * @return mixed
 */
function jrSiteBuilder_get_page_by_json_file($skin, $uri)
{
    // $uri is like "/welcome" and corresponds to SkinName/sitebuilder/welcome.json
    $filename = trim($uri, '/') . '.json';
    if ($uri == '/') {
        $filename = 'index.json';
    }
    $path = APP_DIR . "/skins/{$skin}/sitebuilder/{$filename}";
    if (is_file($path)) {
        $contents = file_get_contents($path);
        if (jrCore_checktype($contents, 'json')) {
            $_cont = json_decode($contents, true);
            $_pg   = $_cont['_page'];

            // sort by weight http://stackoverflow.com/questions/2699086/sort-multi-dimensional-array-by-value
            usort($_cont['_widget'], function ($a, $b) {
                return $a['widget_weight'] - $b['widget_weight'];
            });

            $_pg['_widget'] = $_cont['_widget'];
            return $_pg;
        }
    }
    return false;
}

/**
 * Install a page from a sitebuilder JSON file
 * @param string $skin Skin
 * @param string $page JSON file
 * @return int|bool
 */
function jrSiteBuilder_install_page_from_json($skin, $page)
{
    $_pg = jrSiteBuilder_get_page_by_json_file($skin, $page);
    if (!$_pg || !is_array($_pg)) {
        return false;
    }
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'page');
    $uri = jrCore_db_escape($page);
    $ttl = jrCore_db_escape($_pg['page_title']);
    $grp = jrCore_db_escape($_pg['page_groups']);
    $act = (isset($_pg['page_active']) && $_pg['page_active'] == '1') ? 1 : 0;
    $lay = jrCore_db_escape($_pg['page_layout']);
    $cfg = jrCore_db_escape($_pg['page_settings']);
    $mta = jrCore_db_escape($_pg['page_head']);
    $req = "INSERT INTO {$tbl} (page_updated, page_uri, page_title, page_groups, page_active, page_layout, page_settings, page_head)
            VALUES (UNIX_TIMESTAMP(), '{$uri}', '{$ttl}', '{$grp}', '{$act}', '{$lay}', '{$cfg}', '{$mta}')";
    $pid = jrCore_db_query($req, 'INSERT_ID');
    if (!$pid || !jrCore_checktype($pid, 'number_nz')) {
        return false;
    }
    // add in the widgets
    if (isset($_pg['_widget']) && is_array($_pg['_widget'])) {
        $tbl = jrCore_db_table_name('jrSiteBuilder', 'widget');
        foreach ($_pg['_widget'] as $_w) {
            // Create widget
            $ttl = jrCore_db_escape($_w['widget_title']);
            $mod = jrCore_db_escape($_w['widget_module']);
            $nam = jrCore_db_escape($_w['widget_name']);
            $dta = jrCore_db_escape($_w['widget_data']);
            $loc = jrCore_db_escape($_w['widget_location']);
            $wgt = jrCore_db_escape($_w['widget_weight']);
            $grp = jrCore_db_escape($_w['widget_groups']);
            $unq = jrCore_db_escape($_w['widget_unique']);
            $req = "INSERT INTO {$tbl} (widget_updated, widget_page_id, widget_title, widget_module, widget_name, widget_data, widget_location, widget_weight, widget_groups, widget_unique)
                    VALUES (UNIX_TIMESTAMP(), '{$pid}', '{$ttl}', '{$mod}', '{$nam}', '{$dta}', '{$loc}', '{$wgt}', '{$grp}', '{$unq}') ON DUPLICATE KEY UPDATE widget_updated = UNIX_TIMESTAMP()";
            $wid = jrCore_db_query($req, 'INSERT_ID');
            if (!$wid || !jrCore_checktype($wid, 'number_nz')) {
                return false;
            }
        }
    }
    jrCore_logger('INF', "SiteBuilder: successfully installed new {$skin} page: {$page}");

    // Reset caches
    jrCore_delete_all_cache_entries();
    return $pid;
}

/**
 * Get Data for a page based on Page ID
 * @param $id int Page ID
 * @return mixed
 */
function jrSiteBuilder_get_page_by_id($id)
{
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'page');
    $req = "SELECT * FROM {$tbl} WHERE page_id = '" . intval($id) . "' LIMIT 1";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Get Data for a menu based on URI
 * @param $uri string URI
 * @return mixed
 */
function jrSiteBuilder_get_menu_entry_by_uri($uri)
{
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'menu');
    $req = "SELECT * FROM {$tbl} WHERE menu_url = '" . jrCore_db_escape($uri) . "' LIMIT 1";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Get Data for a menu entry based on Menu ID
 * @param $id int Menu ID
 * @return mixed
 */
function jrSiteBuilder_get_menu_entry_by_id($id)
{
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'menu');
    $req = "SELECT * FROM {$tbl} WHERE menu_id = '" . intval($id) . "' LIMIT 1";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Get Data for a widget by ID
 * @param $id int widget ID
 * @return mixed
 */
function jrSiteBuilder_get_widget_by_id($id)
{
    $tbl = jrCore_db_table_name('jrSiteBuilder', 'widget');
    $req = "SELECT * FROM {$tbl} WHERE widget_id = '" . intval($id) . "' LIMIT 1";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if ($_rt && is_array($_rt)) {
        if (isset($_rt['widget_data']) && strpos($_rt['widget_data'], '{') === 0) {
            $_rt['widget_data'] = json_decode($_rt['widget_data'], true);
        }
        return $_rt;
    }
    return false;
}

/**
 * Get Data for a widget by ID
 * @param $html_id int Page ID
 * @return mixed
 */
function jrSiteBuilder_get_widget_by_html_id($html_id)
{
    // ID will look like: "widget_id-18"
    list(, $wid) = explode('-', $html_id);
    if (!isset($wid) || !jrCore_checktype($wid, 'number_nz')) {
        return false;
    }

    $tbl = jrCore_db_table_name('jrSiteBuilder', 'widget');
    $req = "SELECT * FROM {$tbl} WHERE widget_id = '{$wid}' LIMIT 1";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if ($_rt && is_array($_rt)) {
        if (isset($_rt['widget_data']) && strpos($_rt['widget_data'], '{') === 0) {
            $_rt['widget_data'] = json_decode($_rt['widget_data'], true);
        }
        return $_rt;
    }
    return false;
}

/**
 * Create a Site Builder Page
 * @param $_page array Page info
 * @return string
 */
function jrSiteBuilder_get_page($_page)
{
    global $_post, $_conf;
    if (!isset($_post['_uri']) || strlen($_post['_uri']) === 0) {
        $_post['_uri'] = '/';
    }
    else {
        $_post['_uri'] = rtrim($_post['_uri'], '/');
    }
    $key = $_post['_uri'];
    if (!$out = jrCore_is_cached('jrSiteBuilder', $key)) {

        // Get all registered Widgets
        $_rw = jrCore_get_registered_module_features('jrSiteBuilder', 'widget');

        $out = '';
        $_ct = 0;
        // 3-6-3,4-4-4
        $_tm = explode(',', $_page['page_layout']);
        if ($_tm && is_array($_tm)) {

            // array(
            //     0 => '3-6-3',
            //     1 => '4-4-4'
            // )
            $_ly = array();
            $_cf = (isset($_page['page_settings']) && strlen($_page['page_settings']) > 0) ? json_decode($_page['page_settings'], true) : array();

            // Get all widgets for this page
            if ($_page['page_id'] == 0 && isset($_page['_widget']) && is_array($_page['_widget'])) {
                // its a skin provided page we've fallen through to, get it from the file.
                $_wg = $_page['_widget'];

                if (!isset($_post['module_url'])) {
                    // We are the index page, so set index so its not empty
                    jrCore_set_flag('jrsitebuilder_show_override_notice_for_custom_page', "/");
                }
                else {
                    // we're loading the page from a file, if the admin wants to customize it, they need to import to the db
                    jrCore_set_flag('jrsitebuilder_show_override_notice_for_custom_page', $_post['module_url']);
                }

            }
            else {
                $tbl = jrCore_db_table_name('jrSiteBuilder', 'widget');
                $req = "SELECT * FROM {$tbl} WHERE widget_page_id = '{$_page['page_id']}' ORDER BY widget_location ASC, widget_weight ASC";
                $_wg = jrCore_db_query($req, 'NUMERIC');
            }

            if (is_array($_wg)) {
                foreach ($_wg as $k => $w) {

                    // check the module is active, if not show activate module message here
                    if (jrCore_module_is_active($w['widget_module'])) {

                        // Check widget group
                        if (isset($w['widget_groups']) && strlen($w['widget_groups']) > 0 && $w['widget_groups'] != 'all') {
                            if (!jrUser_is_master() && !jrCore_user_is_part_of_group($w['widget_groups'])) {
                                // This user does not have access to any configured group
                                unset($_wg[$k]);
                                continue;
                            }
                        }

                        // Make sure we have a good function
                        $fnc = "{$w['widget_module']}_{$w['widget_name']}_display";
                        if (function_exists($fnc)) {
                            $out = '';
                            if (isset($w['widget_data']) && strlen($w['widget_data']) > 1) {
                                $_wc = (isset($_cf["{$w['widget_location']}"])) ? $_cf["{$w['widget_location']}"] : null;
                                $_dt = json_decode($w['widget_data'], true);
                                if (isset($_post['repair']) && $_post['repair'] == 'true') {
                                    $out = 'REPAIR MODE';
                                }
                                else {
                                    $out = $fnc($_dt, $w, $_wc);
                                    // allow functions in the titles so skins can have "View More" links.
                                    if (isset($w['widget_title']) && strlen($w['widget_title']) > 0) {
                                        $w['widget_title'] = jrCore_parse_template($w['widget_title'], $w, 'jrSiteBuilder');
                                    }
                                }
                            }
                            $_ct++;
                            $w['widget_module_title'] = (is_array($_rw["{$w['widget_module']}"]["{$w['widget_name']}"])) ? $_rw["{$w['widget_module']}"]["{$w['widget_name']}"]['title'] : $_rw["{$w['widget_module']}"]["{$w['widget_name']}"];
                        }
                        else {
                            $out = "unknown function: {$fnc}";
                        }
                    }
                    else {
                        $out = "Module inactive";
                        if (jrUser_is_admin()) {
                            $mkurl = jrCore_get_module_url('jrMarket');
                            $out   = "<div class=\"page_notice error\"> Either activate this module or delete this section. <a href=\"{$_conf['jrCore_base_url']}/{$mkurl}/browse/module?search_string={$w['widget_module']}\">Install {$w['widget_module']} here</a></div>";
                        }
                    }
                    $loc           = (int) $w['widget_location'];
                    $_ly[$loc][$k] = $w;
                    if (strlen($out) > 0) {
                        $_ly[$loc][$k]['content'] = '<div class="widget-item widget-item-' . $w['widget_name'] . '">' . $out . '</div>';
                    }
                    else {
                        $_ly[$loc][$k]['content'] = '';
                    }
                    $_ly[$loc][$k]['widget_display_number'] = ($loc + 1);
                }
            }

            $_rp = array(
                '_page' => $_page,
                '_rows' => array()
            );

            // Get the column structure on the page
            foreach ($_tm as $row => $_cols) {
                $_rp['_rows'][$row] = array(
                    '_cols' => array()
                );
                foreach (explode('-', $_cols) as $k => $col) {
                    $_rp['_rows'][$row]['_cols'][$k] = array(
                        'width' => $col
                    );
                }
            }

            $_rp['layout']              = $_ly;
            $_rp['config']              = $_cf;
            $_rp['_registered_widgets'] = jrCore_get_registered_module_features('jrSiteBuilder', 'widget');
            $out                        = jrCore_parse_template('page_row_column.tpl', $_rp, 'jrSiteBuilder');

            // some javascript for the hashtag tabs change:
            $_js = array("if (window.location.hash){var hash=location.hash.replace('#','');var tab_target=$('[data-widget_hash=\"' + hash + '\"]').data('widget_value');if (typeof tab_target != 'undefined'){var sb=tab_target.split('|'); jrSiteBuilder_load_tab(sb[0],sb[1],sb[2]);}}");
            jrCore_create_page_element('javascript_ready_function', $_js);

        }
        $_rp                       = array(
            '_page'         => $_page,
            'page_content'  => $out,
            '_widget_count' => $_ct
        );
        $_rp                       = jrCore_trigger_event('jrSiteBuilder', 'page_content', $_rp, $_page);
        $_rp['show_widget_notice'] = true;
        if ($pid = jrCore_get_flag('sb-page-layout-reload')) {
            $_rp['show_widget_notice'] = false;
        }
        $out = jrCore_parse_template('page_container.tpl', $_rp, 'jrSiteBuilder');

        // page head html
        if (!isset($_post['repair']) || $_post['repair'] != 'true') {
            if (isset($_page['page_head']) && strlen($_page['page_head']) > 0) {
                $out = str_replace('</head>', $_page['page_head'] . '</head>', $out);
            }
        }

        jrCore_add_to_cache('jrSiteBuilder', $key, $out);
    }
    return $out;
}

/**
 * Get {jrSiteBuilder_default_menu} menu items in nested array order
 * @return array|bool
 */
function jrSiteBuilder_skin_default_menu_items()
{
    $_default = jrCore_get_temp_value('jrSiteBuilder', 'default_menu');
    $_list    = array();
    if ($_default && is_array($_default)) {
        // sort by menu_order if it exists
        uasort($_default, function ($a, $b) {
            return $a['menu_order'] - $b['menu_order'];
        });

        foreach ($_default as $k => $_m) {
            if (!isset($_m['menu_parent_url'])) {
                $_list[$_m['menu_url']] = $_m;
                unset($_default[$k]);
            }
        }
        if (count($_default) > 0) {
            foreach ($_default as $k => $_m) {
                // first level.
                if ($_list[$_m['menu_parent_url']]) {
                    $_list[$_m['menu_parent_url']]['_children'][$_m['menu_url']] = $_m;
                    unset($_default[$k]);
                }
            }
        }
        if (count($_default) > 0) {
            foreach ($_default as $k => $_m) {
                // second level.
                foreach ($_list as $url => $_values) {
                    if (is_array($_values['_children'])) {
                        if (in_array($_m['menu_parent_url'], array_keys($_values['_children']))) {
                            $_list[$url]['_children'][$_m['menu_parent_url']]['_children'][$k] = $_m;
                            unset($_default[$k]);
                        }
                    }
                }
            }
        }
    }
    if (count($_list) > 0) {
        return $_list;
    }
    return false;
}

/**
 * explode 2-8-2,12-0-0 into its row array.
 * @param string $seq
 * @return array
 */
function jrSiteBuilder_explode_sequence($seq)
{
    $rows  = explode(',', $seq);
    $i     = 0;
    $t     = 0;
    $l     = 1;
    $_rows = array();
    foreach ($rows as $row) {
        $layout = explode('-', $row);
        foreach ($layout as $col) {
            if ($col == 0) {
                continue;
            }

            $_rows[$i][] = array(
                'span'     => $col,
                'location' => $l,
            );
            $l++;

            $t += $col;
            if ($t % 12 == 0) {
                $i++;
            }

        }
    }
    return $_rows;
}

/**
 * Takes an array of page ids and returns the sitebuilder export file data in an array.
 * @param $_page_ids
 * @return array|bool
 */
function jrSiteBuilder_export($_page_ids)
{
    if (!is_array($_page_ids)) {
        return false;
    }
    $pid  = implode(',', $_page_ids);
    $tblm = jrCore_db_table_name('jrSiteBuilder', 'menu');
    $tblp = jrCore_db_table_name('jrSiteBuilder', 'page');
    $tblw = jrCore_db_table_name('jrSiteBuilder', 'widget');

    $req    = "SELECT * FROM {$tblp} WHERE page_id IN ($pid)";
    $_pages = jrCore_db_query($req, 'NUMERIC');
    if (!$_pages || !is_array($_pages)) {
        return false;
    }
    $_data = array();

    // parents
    $req     = "SELECT * FROM {$tblm}";
    $_parent = jrCore_db_query($req, 'menu_id');

    foreach ($_pages as $_p) {
        $_data['pages'][$_p['page_id']]['_page'] = $_p;
        // widgets
        $req      = "SELECT * FROM {$tblw} WHERE widget_page_id  = '{$_p['page_id']}'";
        $_widgets = jrCore_db_query($req, 'NUMERIC');
        if (is_array($_widgets)) {
            foreach ($_widgets as $_w) {
                $_data['pages'][$_p['page_id']]['_widget'][$_w['widget_id']]                    = $_w;
                $_data['pages'][$_p['page_id']]['_widget'][$_w['widget_id']]['widget_page_uri'] = $_p['page_uri'];
            }
        }
        // menu
        $req   = "SELECT * FROM {$tblm} WHERE concat('/',menu_url) = '" . jrCore_db_escape($_p['page_uri']) . "'";
        $_menu = jrCore_db_query($req, 'menu_id');

        if (is_array($_menu)) {
            foreach ($_menu as $_m) {
                $_data['pages'][$_p['page_id']]['_menu'][$_m['menu_id']]                    = $_m;
                $_data['pages'][$_p['page_id']]['_menu'][$_m['menu_id']]['menu_parent_url'] = ($_m['menu_parent_id'] > 0) ? $_parent[$_m['menu_parent_id']]['menu_url'] : '0';
            }
        }
    }

    return $_data;
}

//------------------------------------
// SMARTY
//------------------------------------

/**
 * add a default menu item to the menu
 * @param $params array parameters
 * @param $smarty object current Smarty object
 * @return string
 */
function smarty_function_jrSiteBuilder_default_menu($params, $smarty)
{
    if (!isset($params['title'])) {
        return jrCore_smarty_missing_error('title');
    }
    if (!isset($params['url'])) {
        return jrCore_smarty_missing_error('url');
    }
    $_menu = jrCore_get_temp_value('jrSiteBuilder', 'default_menu');

    $_new = array(
        'menu_title' => $params['title'],
        'menu_url'   => $params['url']
    );

    if (isset($params['parent'])) {
        $_new['menu_parent_url'] = $params['parent'];
    }
    if (isset($params['weight'])) {
        $_new['menu_order'] = (int) $params['weight'];
    }

    $_menu[$params['url']] = $_new;

    // need to save a copy of this to populate the jr_jrsitebuilder_menu table if the MENU EDITOR button is clicked.
    jrCore_set_temp_value('jrSiteBuilder', 'default_menu', $_menu);
    return '';
}

/**
 * Show the Site Menu (Desktop)
 * @param $params array parameters
 * @param $smarty object current Smarty object
 * @return string
 */
function smarty_function_jrSiteBuilder_menu($params, $smarty)
{
    global $_conf;
    $_rp = jrSiteBuilder_get_menu_entries(true);
    if (!$_rp['_list'] || count($_rp['_list']) == 0) {
        // get the default instead
        $_rp['_list'] = jrSiteBuilder_skin_default_menu_items();
    }
    $tpl = 'menu.tpl';
    $mod = 'jrSiteBuilder';
    if (isset($params['template']) && strlen($params['template']) > 0) {
        $tpl = $params['template'];
        $mod = $_conf['jrCore_active_skin'];
    }
    $out = jrCore_parse_template($tpl, $_rp, $mod);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Show the Site Menu (Mobile)
 * @param $params array parameters
 * @param $smarty object current Smarty object
 * @return string
 */
function smarty_function_jrSiteBuilder_mobile_menu($params, $smarty)
{
    global $_conf;
    $_rp = jrSiteBuilder_get_menu_entries(true);
    if (!$_rp['_list'] || count($_rp['_list']) == 0) {
        // get the default instead
        $_rp['_list'] = jrSiteBuilder_skin_default_menu_items();
    }
    $tpl = 'menu_mobile.tpl';
    $mod = 'jrSiteBuilder';
    if (isset($params['template']) && strlen($params['template']) > 0) {
        $tpl = $params['template'];
        $mod = $_conf['jrCore_active_skin'];
    }
    $out = jrCore_parse_template($tpl, $_rp, $mod);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * initialize tiny mce editor manager.
 * @param array $params parameters
 * @param object $smarty current Smarty object
 * @return string
 */
function smarty_function_jrSiteBuilder_tinymce_init($params, $smarty)
{
    global $_mods, $_user, $_conf;

    // Initialize fields
    $_rp['theme'] = 'silver';
    $allowed_tags = explode(',', $_user['quota_jrCore_allowed_tags']);
    foreach ($allowed_tags as $tag) {
        $_rp[$tag] = true;
    }

    // See what modules are providing
    $_tm = jrCore_get_registered_module_features('jrCore', 'editor_button');
    if ($_tm && is_array($_tm)) {
        foreach ($_tm as $mod => $_items) {
            $tag       = strtolower($mod);
            $_rp[$tag] = false;
            if (jrCore_module_is_active($mod)) {
                if (!isset($_rp['_sources'])) {
                    $_rp['_sources'] = array();
                }
                $_rp['_sources'][] = "{$_conf['jrCore_base_url']}/modules/{$mod}/tinymce/plugin.min.js?v=" . $_mods[$mod]['module_version'];
                $_rp[$tag]         = true;
            }
        }
    }
    return jrCore_parse_template('form_editor.tpl', $_rp, 'jrSiteBuilder');
}

/**
 * Smarty rotator function
 * (from the construction kit. here to allow for imports from old system.)
 * @param array $params
 * @param object $smarty
 * @return string
 */
function smarty_function_jrSiteBuilder_rotator($params, $smarty)
{

    // default template
    if (!isset($params['template'])) {
        $params['template'] = 'rotator_default_row.tpl';
        $params['tpl_dir']  = 'jrSiteBuilder';
    }
    // default order_by
    if (!isset($params['order_by'])) {
        $params['order_by'] = '_created RAND';
    }

    // only active profiles
    $params['search9'] = "profile_active = 1";

    // require image
    if (!isset($params['require_image'])) {
        $pfx                     = jrCore_db_get_prefix($params['module']);
        $params['require_image'] = $pfx . "_image";
    }

    $out  = smarty_function_jrCore_list($params, $smarty);
    $_rep = array(
        'unique_id'  => jrCore_create_unique_string(6),
        'row_output' => $out,
    );
    return jrCore_parse_template('rotator.tpl', $_rep, 'jrSiteBuilder');
}

/**
 * Exclude Site Builder from running on a template
 * @param array $params
 * @param object $smarty
 * @return string
 */
function smarty_function_jrSiteBuilder_no_sitebuilder($params, &$smarty)
{
    return '<!-- jrSiteBuilder_no_sitebuilder -->';
}
