<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Information about the Jamroom Core
 * @return array
 */
function jrCore_meta()
{
    return array(
        'name'        => 'System Core',
        'url'         => 'core',
        'version'     => '7.0.2',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Provides low level functionality for all system operations',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/2857/system-core',
        'category'    => 'core',
        'license'     => 'mpl',
        'php_minv'    => '7.2.0',
        'locked'      => true,
        'activate'    => true
    );
}

/**
 * Core Initialization
 * @return bool
 */
function jrCore_init()
{
    global $_conf, $_mods, $_urls;

    ob_start();
    mb_internal_encoding('UTF-8');
    ini_set('mbstring.substitute_character', 'none');

    // Load up config file and default values for some items
    jrCore_load_config_file_and_defaults();

    // Core magic views
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'admin', 'view_jrCore_admin');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'admin_save', 'view_jrCore_admin_save');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'stream', 'view_jrCore_stream');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'download', 'view_jrCore_download');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'template_compare', 'view_jrCore_template_compare');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'template_compare_save', 'view_jrCore_template_compare_save');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'template_modify', 'view_jrCore_template_modify');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'template_modify_save', 'view_jrCore_template_modify_save');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'browser', 'view_jrCore_browser');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'browser_item_update', 'view_jrCore_browser_item_update');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'browser_item_update_save', 'view_jrCore_browser_item_update_save');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'browser_item_delete', 'view_jrCore_browser_item_delete');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'license', 'view_jrCore_license');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'form_designer', 'view_jrCore_form_designer');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'form_designer_save', 'view_jrCore_form_designer_save');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'form_field_update', 'view_jrCore_form_field_update');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'form_field_update_save', 'view_jrCore_form_field_update_save');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'dashboard', 'view_jrCore_dashboard');
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrCore', 'item_display_order', 'view_jrCore_item_display_order');

    // Core tool views
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCore', 'cache_reset', array('Reset Caches', 'Reset database and filesystem caches'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCore', 'integrity_check', array('Integrity Check', 'Validate, Optimize and Repair module and skin installs'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCore', 'system_check', array('System Check', 'Display information about your System and installed modules'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCore', 'performance_check', array('Performance Check', 'Run a performance test on your server and optionally share the results'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCore', 'skin_menu', array('User Menu Editor', 'Customize the items and options that appear in the main User drop down Menu'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCore', 'module_detail_features', array('Item Detail Features', 'Set the Order of Item Detail Features provided by modules'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCore', 'advanced_config', array('Advanced Config', 'Add or Delete advanced config options in the config.php file'));

    // Our default view for admins
    jrCore_register_module_feature('jrCore', 'default_admin_view', 'jrCore', 'admin/global');

    // Core checktype plugins
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'allowed_html');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'core_string');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'valid_name');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'unique_name');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'date');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'date_birthday');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'domain');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'email');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'float');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'hex');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'ip_address');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'private_ip_address');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'is_true');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'is_false');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'md5');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'multi_word');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'not_empty');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'signed');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'number');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'number_nn');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'number_nz');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'onoff');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'phone_number');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'price');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'printable');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'sha1');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'string');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'url');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'url_name');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'file_name');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'yesno');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'json');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'array');
    jrCore_register_module_feature('jrCore', 'checktype', 'jrCore', 'age_range');

    // Core form fields supported
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'hidden');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'checkbox');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'checkbox_spambot');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'custom');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'date');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'daterange');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'datetime');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'date_birthday');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'editor');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'file');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'live_search');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'notice');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'optionlist');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'password');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'radio');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'range');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'section_header');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'select');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'select_and_text');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'select_date');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'select_multiple');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'text');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'textarea');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'time');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'weight');
    jrCore_register_module_feature('jrCore', 'form_field', 'jrCore', 'age_range');

    jrCore_register_module_feature('jrTips', 'tip', 'jrCore', 'tip');

    // Bring in core javascript
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/jquery/jquery-3.6.0.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/jquery/jquery-migrate-1.4.1.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/simplemodal/jquery.simplemodal.1.4.4.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/lightbox/lightbox-2.11.0.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/livesearch/jquery.livesearch.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/fileuploader/imageresize.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/fileuploader/fileuploader.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/jplayer/jquery.jplayer.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/jplayer/jquery.jplayer.playlist.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/sweetalert/sweetalert.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/sortable/jquery.sortable.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/slidebars/slidebars.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', APP_DIR . '/modules/jrCore/contrib/colorpicker/jquery.colorpicker.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', 'jrCore_admin.js', 'admin');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCore', 'jrCore.js');

    // When javascript is registered, we have a function that is called
    jrCore_register_module_feature_function('jrCore', 'javascript', 'jrCore_enable_external_javascript');

    // Register our core CSS
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', APP_DIR . '/modules/jrCore/contrib/fileuploader/fileuploader.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', APP_DIR . '/modules/jrCore/contrib/sweetalert/sweetalert.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', APP_DIR . '/modules/jrCore/contrib/livesearch/jquery.livesearch.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', APP_DIR . '/modules/jrCore/contrib/lightbox/lightbox.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', APP_DIR . '/modules/jrCore/contrib/select2/css/select2.min.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', 'jrCore.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', 'jrCore_bbcode.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', 'jrCore_tinymce.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', 'jrCore_dashboard.css');
    jrCore_register_module_feature('jrCore', 'css', 'jrCore', 'jrCore_mobile_override.css');

    // Recycle Bin support
    jrCore_register_module_feature('jrCore', 'recycle_bin_user_id_table', 'jrCore', 'count_ip', 'count_user_id');
    jrCore_register_module_feature('jrCore', 'recycle_bin_user_id_table', 'jrCore', 'stat_unique', 'stat_user_id');

    // When CSS is registered, we have a function that is called
    jrCore_register_module_feature_function('jrCore', 'css', 'jrCore_enable_external_css');

    // BBCode string formatter - MUST COME FIRST!
    $_tmp = array(
        'wl'    => 'bbcode',
        'label' => 'Allow BBCode',
        'help'  => 'If active, BBCode tags will be expanded to HTML tags.'
    );
    jrCore_register_module_feature('jrCore', 'format_string', 'jrCore', 'jrCore_format_string_bbcode', $_tmp);

    // Allowed HTML string formatter
    $_tmp = array(
        'wl'    => 'html',
        'label' => 'Allow HTML',
        'help'  => 'If active, any HTML tags defined in the Allowed HTML Tags setting will be allowed in the text.'
    );
    jrCore_register_module_feature('jrCore', 'format_string', 'jrCore', 'jrCore_format_string_allowed_html', $_tmp);

    // Convert @ tags string formatter
    $_tmp = array(
        'wl'    => 'at_tags',
        'label' => 'Convert @ Tags',
        'help'  => 'If active, links to Profiles written as @profile_name will be linked to the actual Profile.'
    );
    jrCore_register_module_feature('jrCore', 'format_string', 'jrCore', 'jrCore_format_string_convert_at_tags', $_tmp);

    // Clickable URLs string formatter
    $_tmp = array(
        'wl'    => 'click_urls',
        'label' => 'Make URLs Clickable',
        'help'  => 'If active, URLs entered into the text will be hyperlinked so they are clickable.'
    );
    jrCore_register_module_feature('jrCore', 'format_string', 'jrCore', 'jrCore_format_string_clickable_urls', $_tmp);

    // We don't need sessions on a few views
    jrCore_register_module_feature('jrUser', 'skip_session', 'jrCore', 'js');
    jrCore_register_module_feature('jrUser', 'skip_session', 'jrCore', 'css');
    jrCore_register_module_feature('jrUser', 'skip_session', 'jrCore', 'icon');
    jrCore_register_module_feature('jrUser', 'skip_session', 'jrCore', 'icon_css');
    jrCore_register_module_feature('jrUser', 'skip_session', 'jrCore', 'icon_sprite');
    jrCore_register_module_feature('jrUser', 'skip_session', 'jrCore', 'test_template');
    jrCore_register_module_feature('jrUser', 'skip_session', 'jrCore', 'form_modal_status');

    // Core plugins
    jrCore_register_system_plugin('jrCore', 'email', 'activity', 'Log Sent Email to Activity Log');
    jrCore_register_system_plugin('jrCore', 'email', 'debug', 'Log Sent Email to Debug Log');
    jrCore_register_system_plugin('jrCore', 'media', 'local', 'Local File System');
    jrCore_register_system_plugin('jrCore', 'cache', 'mysql', 'Core Data Cache (default)');
    jrCore_register_system_plugin('jrCore', 'datastore', 'mysql', 'Core Datastore (MySQL)');

    // Core event triggers
    jrCore_register_event_trigger('jrCore', 'allowed_html_tags', 'Fired when validating posted HTML in the editor');
    jrCore_register_event_trigger('jrCore', 'approve_pending_item', 'Fired when a pending item is approved by an admin');
    jrCore_register_event_trigger('jrCore', 'reject_pending_item', 'Fired when a pending item is rejected by an admin');
    jrCore_register_event_trigger('jrCore', 'delete_pending_item', 'Fired when a pending item is deleted by an admin');
    jrCore_register_event_trigger('jrCore', 'pending_items_header', 'Fired when when formatting the pending items header in the Dashboard');
    jrCore_register_event_trigger('jrCore', 'pending_items_row', 'Fired when when formatting a pending item row in the Dashboard');
    jrCore_register_event_trigger('jrCore', 'daily_maintenance', 'Fired once a day after midnight server time');
    jrCore_register_event_trigger('jrCore', 'hourly_maintenance', 'Fired once an hour on the first request in the new hour');
    jrCore_register_event_trigger('jrCore', 'ten_minute_maintenance', 'Fired every 10 minutes on the first request in the minute');
    jrCore_register_event_trigger('jrCore', 'minute_maintenance', 'Fired once a minute on the first request in the minute');
    jrCore_register_event_trigger('jrCore', 'maintenance_check', 'Fired once a minute when minute_maintenance lock is acquired');
    jrCore_register_event_trigger('jrCore', 'db_increment_key', 'Fired when a datastore key is incremented via jrCore_db_increment_key');
    jrCore_register_event_trigger('jrCore', 'db_decrement_key', 'Fired when a datastore key is decremented via jrCore_db_decrement_key');
    jrCore_register_event_trigger('jrCore', 'db_create_datastore', 'Fired when a new DataStore is initialized');
    jrCore_register_event_trigger('jrCore', 'db_truncate_datastore', 'Fired when a DataStore is emptied (truncated)');
    jrCore_register_event_trigger('jrCore', 'db_delete_datastore', 'Fired when a DataStore is deleted');
    jrCore_register_event_trigger('jrCore', 'db_create_item_data', 'Fired with data before creating a new DataStore item');
    jrCore_register_event_trigger('jrCore', 'db_create_item', 'Fired with item_id and data before adding keys to new DataStore item');
    jrCore_register_event_trigger('jrCore', 'db_create_item_exit', 'Fired with _item_id and data after creating new DataStore item');
    jrCore_register_event_trigger('jrCore', 'db_update_item', 'Fired when a DataStore item is updated');
    jrCore_register_event_trigger('jrCore', 'db_update_multiple_item', 'Fired AFTER multiple items have been updated in a DataStore');
    jrCore_register_event_trigger('jrCore', 'db_get_item', 'Fired when a DataStore item is retrieved');
    jrCore_register_event_trigger('jrCore', 'db_get_multiple_item', 'Fired when retrieving multiple DataStore items');
    jrCore_register_event_trigger('jrCore', 'db_search_multiple_item', 'Fired when searching for multiple DataStore items');
    jrCore_register_event_trigger('jrCore', 'db_delete_item', 'Fired when a DataStore item is deleted');
    jrCore_register_event_trigger('jrCore', 'db_delete_multiple_item', 'Fired when multiple DataStore items from the same module are deleted');
    jrCore_register_event_trigger('jrCore', 'db_delete_keys', 'Fired when specific keys are deleted from a DataStore item');
    jrCore_register_event_trigger('jrCore', 'db_delete_multiple_item_keys', 'Fired after multiple keys are deleted from a DataStore item');
    jrCore_register_event_trigger('jrCore', 'db_delete_key_from_all_items', 'Fired when a key is deleted from all items in a DataStore');
    jrCore_register_event_trigger('jrCore', 'db_search_items', 'Fired with an array of DataStore items that matched the search criteria  - check _items array');
    jrCore_register_event_trigger('jrCore', 'db_search_cache_check', 'Fired before cache check in DataStore search with unique Cache Key in _args');
    jrCore_register_event_trigger('jrCore', 'db_search_params', 'Fired when doing a DataStore search for search params');
    jrCore_register_event_trigger('jrCore', 'db_search_simple_keys', 'Fired when doing a DataStore search for keys that do not span multiple value rows');
    jrCore_register_event_trigger('jrCore', 'db_search_count_query', 'Fired with SQL Query and Query Time for COUNT query used in pagination');
    jrCore_register_event_trigger('jrCore', 'db_search_query', 'Fired with SQL Query and Query Time for main _items query');
    jrCore_register_event_trigger('jrCore', 'db_search_results', 'Fired with results from jrCore_db_search_items - including cached');
    jrCore_register_event_trigger('jrCore', 'db_query_init', 'Fired in jrCore_db_query() with query to be run');
    jrCore_register_event_trigger('jrCore', 'db_query_exit', 'Fired in jrCore_db_query() with query results');
    jrCore_register_event_trigger('jrCore', 'db_verify_table', 'Fired in jrCore_db_verify_table() with schema as array');
    jrCore_register_event_trigger('jrCore', 'db_connect', 'Fired in jrCore_db_connect() with connection configuration');
    jrCore_register_event_trigger('jrCore', 'db_connect_warning', 'Fired in jrCore_db_connect() when unable to connect to server (initial)');
    jrCore_register_event_trigger('jrCore', 'db_connect_error', 'Fired in jrCore_db_connect() when unable to connect to server (fatal)');
    jrCore_register_event_trigger('jrCore', 'debug_log', 'Fired in fdebug() with data to be written to debug_log');  // OK
    jrCore_register_event_trigger('jrCore', 'display_order', 'Fired with entries during the display_order magic view');
    jrCore_register_event_trigger('jrCore', 'fatal_error', 'Fired with error info when a PHP Fatal error is encountered');
    jrCore_register_event_trigger('jrCore', 'upload_prepare', 'Fired when a file is uploaded and before it is saved');
    jrCore_register_event_trigger('jrCore', 'upload_saved', 'Fired when an uploaded file is saved to the tmp filesystem');
    jrCore_register_event_trigger('jrCore', 'upload_complete', 'Fired when an uploaded file is done uploading');
    jrCore_register_event_trigger('jrCore', 'download_file', 'Fired when a DataStore file is downloaded');
    jrCore_register_event_trigger('jrCore', 'email_addresses', 'Fired before emails are sent with array of email addresses');
    jrCore_register_event_trigger('jrCore', 'email_info', 'Fired before emails are sent with array of email delivery info');
    jrCore_register_event_trigger('jrCore', 'email_prepare', 'Fired just before emails are sent with queue entry data');
    jrCore_register_event_trigger('jrCore', 'email_sent', 'Fired with total sent when email queue entry is complete');
    jrCore_register_event_trigger('jrCore', 'form_create', 'Fired with when a new form session is created');
    jrCore_register_event_trigger('jrCore', 'format_string_display', 'Fired at the of jrCore_format_string with results');
    jrCore_register_event_trigger('jrCore', 'form_validate_init', 'Fired at the beginning of jrCore_form_validate()');
    jrCore_register_event_trigger('jrCore', 'form_validate_exit', 'Fired at the end of jrCore_form_validate()');
    jrCore_register_event_trigger('jrCore', 'form_field_create', 'Fired when a form_field is added to a form session');
    jrCore_register_event_trigger('jrCore', 'form_display', 'Fired when a form is displayed (receives form data)');
    jrCore_register_event_trigger('jrCore', 'form_result', 'Fired when a form target view has completed');
    jrCore_register_event_trigger('jrCore', 'form_session_create', 'Fired when a form session is created');
    jrCore_register_event_trigger('jrCore', 'form_session_delete', 'Fired when a form session is deleted');
    jrCore_register_event_trigger('jrCore', 'form_set_notice', 'Fired when a form notice is set in a form session during submit');
    jrCore_register_event_trigger('jrCore', 'get_save_data', 'Fired on exit of jrCore_form_get_save_data()');
    jrCore_register_event_trigger('jrCore', 'get_item_title', 'Fired with item data when creating an item Title');
    jrCore_register_event_trigger('jrCore', 'get_item_url', 'Fired with item data when creating an item URL');
    jrCore_register_event_trigger('jrCore', 'html_purifier', 'Fired during HTMLPurifier config setup');
    jrCore_register_event_trigger('jrCore', 'index_template', 'Fired when the skin index template is displayed');
    jrCore_register_event_trigger('jrCore', 'json_response', 'Fired with incoming data in jrCore_json_response()');
    jrCore_register_event_trigger('jrCore', 'log_message', 'Fired when a message is logged to the Activity Log');
    jrCore_register_event_trigger('jrCore', 'item_action_button_data', 'Fired with item data when creating an action button');
    jrCore_register_event_trigger('jrCore', 'media_playlist', 'Fired when a playlist is assembled in {jrCore_media_player}');
    jrCore_register_event_trigger('jrCore', 'media_player_params', 'Fired when all media player parameters are assembled in {jrCore_media_player}');
    jrCore_register_event_trigger('jrCore', 'media_allowed_referrer', 'Fired when checking if referrer is allowed in media stream_file');
    jrCore_register_event_trigger('jrCore', 'get_media_function', 'Fired in media functions to get active media system function');
    jrCore_register_event_trigger('jrCore', 'module_view', 'Fired when a module view is going to be processed');
    jrCore_register_event_trigger('jrCore', 'module_activated', 'Fired when a module has been activated from the info tab');
    jrCore_register_event_trigger('jrCore', 'module_deactivated', 'Fired when a module has been made inactive from the info tab');
    jrCore_register_event_trigger('jrCore', 'module_deleted', 'Fired when a module has been deleted from the info tab');
    jrCore_register_event_trigger('jrCore', 'skin_activated', 'Fired when a skin has been activated from the skins tab');
    jrCore_register_event_trigger('jrCore', 'skin_deleted', 'Fired when a skin has been deleted from the info tab');
    jrCore_register_event_trigger('jrCore', 'parse_url', 'Fired when the current URL has been parsed into $_url');
    jrCore_register_event_trigger('jrCore', 'is_local_url', 'Fired when checking if a URL is a local URL');
    jrCore_register_event_trigger('jrCore', 'get_current_url', 'Fired when getting current URL for user');
    jrCore_register_event_trigger('jrCore', 'get_local_referrer', 'Fired when getting referring URL for user');
    jrCore_register_event_trigger('jrCore', 'is_mobile_device', 'Fired when testing if a browser is a mobile device');
    jrCore_register_event_trigger('jrCore', 'is_tablet_device', 'Fired when testing if a browser is a tablet device');
    jrCore_register_event_trigger('jrCore', 'router_init', 'Fired at beginning of view display when determining routing');
    jrCore_register_event_trigger('jrCore', 'process_init', 'Fired when the core has initialized all modules and the active skin');
    jrCore_register_event_trigger('jrCore', 'process_exit', 'Fired when process exits');
    jrCore_register_event_trigger('jrCore', 'profile_template', 'Fired when a profile template is displayed');
    jrCore_register_event_trigger('jrCore', 'run_view_function', 'Fired before a view function is run for a module');
    jrCore_register_event_trigger('jrCore', 'save_media_file', 'Fired when a media file has been saved for a profile');
    jrCore_register_event_trigger('jrCore', 'skin_template', 'Fired when a skin template is displayed');
    jrCore_register_event_trigger('jrCore', 'skin_menu_items', 'Fired with menu items when rendering the user skin menu');
    jrCore_register_event_trigger('jrCore', 'stream_file', 'Fired when a DataStore file is streamed');
    jrCore_register_event_trigger('jrCore', 'stream_url_error', 'Fired when Media Player encounters a URL error');
    jrCore_register_event_trigger('jrCore', 'system_check', 'Fired in System Check so modules can run own checks');
    jrCore_register_event_trigger('jrCore', 'template_cache_reset', 'Fired when Reset Template Cache is fired');
    jrCore_register_event_trigger('jrCore', 'template_variables', 'Fired for replacement variables when parsing a template');
    jrCore_register_event_trigger('jrCore', 'template_file', 'Fired with template file info when a template is being parsed');
    jrCore_register_event_trigger('jrCore', 'template_list', 'Fired with parameters when {jrCore_list} is called in a template');
    jrCore_register_event_trigger('jrCore', 'parsed_template', 'Fired with the content of a parsed template');
    jrCore_register_event_trigger('jrCore', 'verify_module', 'Fired when a module is verified during Integrity Check or installed');
    jrCore_register_event_trigger('jrCore', 'repair_module', 'Fired when a module is repaired during the Integrity Check');
    jrCore_register_event_trigger('jrCore', 'repair_module_db', 'Fired in the repair_module worker during the Integrity Check');
    jrCore_register_event_trigger('jrCore', 'verify_skin', 'Fired when a skin is verified during Integrity Check or activated');
    jrCore_register_event_trigger('jrCore', 'view_results', 'Fired just before results from a view are displayed');
    jrCore_register_event_trigger('jrCore', 'validate_location_url', 'Fired when validating a CSRF protected URL');
    jrCore_register_event_trigger('jrCore', 'results_sent', 'Fired just after results from a view are displayed');
    jrCore_register_event_trigger('jrCore', '404_not_found', 'Fired when a URL results in a 404 not found');
    jrCore_register_event_trigger('jrCore', 'tpl_404', 'Fired when a template can not be found');
    jrCore_register_event_trigger('jrCore', 'page_table_header', 'Fired with cell data for page table header');
    jrCore_register_event_trigger('jrCore', 'page_table_row', 'Fired with cell data for each row in a page table');
    jrCore_register_event_trigger('jrCore', 'page_table_footer', 'Fired with cell data for page table footer');
    jrCore_register_event_trigger('jrCore', 'get_cache_key', 'Fired with cache key');

    jrCore_register_event_trigger('jrCore', 'create_queue_entry', 'Fired when a process tries to create a queue entry');
    jrCore_register_event_trigger('jrCore', 'queue_entry_created', 'Fired after a queue entry has been created with queue_id');
    jrCore_register_event_trigger('jrCore', 'get_queue_entry', 'Fired when a worker tries to get a queue entry');
    jrCore_register_event_trigger('jrCore', 'release_queue_entry', 'Fired when a worker releases a queue entry back to the queue');
    jrCore_register_event_trigger('jrCore', 'sleep_queue_entry', 'Fired when a worker adjusts the sleep of an existing queue entry');
    jrCore_register_event_trigger('jrCore', 'delete_queue_entry', 'Fired when a worker tries to delete a queue entry');
    jrCore_register_event_trigger('jrCore', 'delete_queue_by_item_id', 'Fired when deleting a queue by module/item_id');
    jrCore_register_event_trigger('jrCore', 'get_queue_info', 'Fired in the Queue Viewer tool to get queue info');
    jrCore_register_event_trigger('jrCore', 'check_queues_ready', 'Fired when checking Queue State');
    jrCore_register_event_trigger('jrCore', 'set_queue_status', 'Fired when setting Queue Status');
    jrCore_register_event_trigger('jrCore', 'queue_worker_can_work', 'Fired when checking if a process can become a queue worker');
    jrCore_register_event_trigger('jrCore', 'can_be_queue_worker', 'Fired to check if current process can become a queue worker');
    jrCore_register_event_trigger('jrCore', 'release_queue_worker_slot', 'Fired when a queue worker is releasing a queue slot');
    jrCore_register_event_trigger('jrCore', 'decrement_queue_depth', 'Fired when the depth of a queue is decremented');
    jrCore_register_event_trigger('jrCore', 'check_for_dead_queue_workers_init', 'Fired when preparing to check for dead workers');
    jrCore_register_event_trigger('jrCore', 'check_for_dead_queue_workers', 'Fired with worker info for dead workers');
    jrCore_register_event_trigger('jrCore', 'queue_worker_count', 'Fired when getting count of active queue workers');
    jrCore_register_event_trigger('jrCore', 'validate_queue_info', 'Fired when validating the queue depth and worker count');
    jrCore_register_event_trigger('jrCore', 'validate_queue_data', 'Fired when validating queue entry data');

    jrCore_register_event_trigger('jrCore', 'all_events', 'Fired when any other event trigger is fired');
    jrCore_register_event_trigger('jrCore', 'admin_tabs', 'Fired when tabs are created for modules in the ACP');
    jrCore_register_event_trigger('jrCore', 'skin_tabs', 'Fired when tabs are created for skins in the ACP');
    jrCore_register_event_trigger('jrCore', 'empty_recycle_bin', 'Fired when the Recycle Bin is emptied');
    jrCore_register_event_trigger('jrCore', 'restore_recycle_bin_item', 'Fired when an item is restored from the Recycle Bin');
    jrCore_register_event_trigger('jrCore', 'expire_recycle_bin', 'Fired when expired items are deleted from the Recycle Bin');
    jrCore_register_event_trigger('jrCore', 'dashboard_tabs', 'Fired when the tab bar is created for the Dashboard');
    jrCore_register_event_trigger('jrCore', 'dashboard_panels', 'Fired with registered panels when the Dashboard is viewed');
    jrCore_register_event_trigger('jrCore', 'exclude_item_detail_buttons', 'Fired with button info on Item Detail pages');
    jrCore_register_event_trigger('jrCore', 'exclude_item_index_buttons', 'Fired with button info on Item Index pages');
    jrCore_register_event_trigger('jrCore', 'included_css', 'Fired with CSS files to be included in master CSS file');
    jrCore_register_event_trigger('jrCore', 'create_master_css', 'Fired when creating the site master CSS file');
    jrCore_register_event_trigger('jrCore', 'master_css_src', 'Fired when creating URL to site master CSS file');
    jrCore_register_event_trigger('jrCore', 'included_javascript', 'Fired with Javascript files to be included in master Javascript file');
    jrCore_register_event_trigger('jrCore', 'create_master_javascript', 'Fired when creating the site master Javascript files');
    jrCore_register_event_trigger('jrCore', 'master_javascript_src', 'Fired when creating URL to site master Javascript file');
    jrCore_register_event_trigger('jrCore', 'get_icon_html', 'Fired to get the HTML for an interface Icon');
    jrCore_register_event_trigger('jrCore', 'global_config_created', 'Fired when the Global Config is first created');
    jrCore_register_event_trigger('jrCore', 'global_config_updated', 'Fired when the Global Config is updated');
    jrCore_register_event_trigger('jrCore', 'delete_config_cache', 'Fired when the Core config cache is reset');
    jrCore_register_event_trigger('jrCore', 'caching_is_enabled', 'Fired when checking if caching is enabled');

    // If the tracer module is installed, we have a few events for it
    jrCore_register_module_feature('jrTrace', 'trace_event', 'jrCore', 'download_file', 'A user downloads a file');
    jrCore_register_module_feature('jrTrace', 'trace_event', 'jrCore', 'stream_file', 'A user streams a file');

    // Check for install routine
    if (defined('IN_JAMROOM_INSTALLER')) {
        return true;
    }

    // See if our master data/config/config.php has defined the active skin - it needs to be part of our cache key if it has
    $dom = intval($_conf['jrCore_custom_domain_loaded']);
    $skn = (isset($_conf['jrCore_active_skin'])) ? $_conf['jrCore_active_skin'] : '';
    $key = md5("{$_conf['jrCore_base_url']}/{$dom}/{$skn}");
    jrCore_set_flag('jrcore_config_and_modules_key', $key);

    // Init $_conf, $_mods and $_urls
    $init = false;
    jrCore_set_flag('jrcore_in_module_init', 1);

    // On systems with APCu we are going to cache our config in APC
    // otherwise we will cache the config in MySQL
    if (jrCore_local_cache_is_enabled()) {
        if (!$_rt = jrCore_get_local_cache_key('config_cache')) {
            jrCore_init_conf_mods_and_urls();
            $_rt = array(
                '_conf' => $_conf,
                '_mods' => $_mods,
                '_urls' => $_urls
            );
            // @note: developer mode will be checked in jrCore_set_local_cache_key()
            $exp = jrCore_get_config_value('jrCore', 'default_cache_seconds', 300);
            jrCore_set_local_cache_key('config_cache', $_rt, $exp);
        }
        else {
            // Setup globals
            $_conf = $_rt['_conf'];
            $_mods = $_rt['_mods'];
            $_urls = $_rt['_urls'];
            $init  = true;
            unset($_rt);
        }
    }
    else {
        if (!$_rt = _jrCore_mysql_is_cached('jrCore', $key, false)) {
            jrCore_init_conf_mods_and_urls();
            $_rt = array(
                '_conf' => $_conf,
                '_mods' => $_mods,
                '_urls' => $_urls
            );
            _jrCore_mysql_add_to_cache('jrCore', $key, $_rt, 0, 0, false);
        }
        else {
            // Setup globals
            $_conf = $_rt['_conf'];
            $_mods = $_rt['_mods'];
            $_urls = $_rt['_urls'];
            $init  = true;
            unset($_rt);
        }
    }

    // At this point $_conf, $_mods and $_urls are set
    // If $init is true it means we have a CACHED config - need to include
    // scripts here so actual scripts are initialized
    if ($init) {
        // NOTE: we INCLUDE all modules first, THEN init
        foreach ($_mods as $_md) {
            if ($_md['module_directory'] != 'jrCore' && $_md['module_active'] == '1') {
                // jrCore is already included ;)
                // Error redirect here for users that simply try to delete a module
                // by removing the module directory BEFORE removing the module from the DB!
                @include_once APP_DIR . "/modules/{$_md['module_directory']}/include.php";
            }
        }
        // .. and init
        foreach ($_mods as $k => $_md) {
            if ($_md['module_directory'] != 'jrCore' && $_md['module_active'] == '1') {
                $func = "{$_md['module_directory']}_init";
                if (function_exists($func)) {
                    $func();
                }
                $_mods[$k]['module_initialized'] = 1;
            }
        }
    }

    // NOTE: this is the FIRST triggered event in the system
    $_conf = jrCore_trigger_event('jrCore', 'global_config_created', $_conf);
    jrCore_delete_flag('jrcore_in_module_init');

    // Set our timezone...
    date_default_timezone_set($_conf['jrCore_system_timezone']);

    // Initialize active skin...
    $func = "{$_conf['jrCore_active_skin']}_skin_init";
    if (!function_exists($func)) {
        require_once APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/include.php";
        if (function_exists($func)) {
            $func();
        }
    }
    ob_end_clean();

    // Trigger our process_init event
    jrCore_trigger_event('jrCore', 'process_init', array());

    // Core event listeners - must come after $_mods
    jrCore_register_event_listener('jrCore', 'view_results', 'jrCore_view_results_listener');
    jrCore_register_event_listener('jrCore', 'process_exit', 'jrCore_process_exit_listener');
    jrCore_register_event_listener('jrCore', 'verify_module', 'jrCore_verify_module_listener');
    jrCore_register_event_listener('jrCore', 'repair_module', 'jrCore_repair_module_listener');
    jrCore_register_event_listener('jrCore', 'daily_maintenance', 'jrCore_daily_maintenance_listener');
    jrCore_register_event_listener('jrCore', 'db_create_item_exit', 'jrCore_db_create_item_exit_listener');
    jrCore_register_event_listener('jrCore', 'run_view_function', 'jrCore_run_view_function_listener');
    jrCore_register_event_listener('jrCore', '404_not_found', 'jrCore_404_not_found_listener');
    jrCore_register_event_listener('jrMarket', 'updated_module', 'jrCore_updated_module_listener');

    $_tmp = array(
        'label' => 'pending item',
        'help'  => 'When a new Item is created and is pending review, how do you want to be notified?',
        'group' => 'admin'
    );
    jrCore_register_module_feature('jrUser', 'notification', 'jrCore', 'pending_item', $_tmp);

    $_tmp = array(
        'label' => 97, // pending item approved
        'help'  => 98  // When a pending item is approved, how do you want to be notified?
    );
    jrCore_register_module_feature('jrUser', 'notification', 'jrCore', 'pending_approve', $_tmp);

    // Core item buttons
    $_tmp = array(
        'title'  => 'Item Order Button',
        'icon'   => 'refresh',
        'active' => 'on',
        'group'  => 'owner'
    );
    jrCore_register_module_feature('jrCore', 'item_index_button', 'jrCore', 'jrCore_item_order_button', $_tmp);

    $_tmp = array(
        'title'  => 'Item Create Button',
        'icon'   => 'plus',
        'active' => 'on',
        'group'  => 'owner'
    );
    jrCore_register_module_feature('jrCore', 'item_index_button', 'jrCore', 'jrCore_item_create_button', $_tmp);
    $_tmp['active'] = 'off';
    jrCore_register_module_feature('jrCore', 'item_detail_button', 'jrCore', 'jrCore_item_create_button', $_tmp);

    $_tmp = array(
        'title'  => 'Item Update Button',
        'icon'   => 'gear',
        'active' => 'on',
        'group'  => 'owner'
    );
    jrCore_register_module_feature('jrCore', 'item_list_button', 'jrCore', 'jrCore_item_update_button', $_tmp);
    jrCore_register_module_feature('jrCore', 'item_detail_button', 'jrCore', 'jrCore_item_update_button', $_tmp);

    $_tmp = array(
        'title'  => 'Item Delete Button',
        'icon'   => 'trash',
        'active' => 'on',
        'group'  => 'owner'
    );
    jrCore_register_module_feature('jrCore', 'item_list_button', 'jrCore', 'jrCore_item_delete_button', $_tmp);
    jrCore_register_module_feature('jrCore', 'item_detail_button', 'jrCore', 'jrCore_item_delete_button', $_tmp);

    $_tmp = array(
        'title'  => 'Create Bundle Button',
        'icon'   => 'plus',
        'active' => 'on',
        'group'  => 'owner'
    );
    jrCore_register_module_feature('jrCore', 'item_bundle_index_button', 'jrCore', 'jrCore_bundle_create_button', $_tmp);
    $_tmp['active'] = 'off';
    jrCore_register_module_feature('jrCore', 'item_bundle_list_button', 'jrCore', 'jrCore_bundle_create_button', $_tmp);
    jrCore_register_module_feature('jrCore', 'item_bundle_detail_button', 'jrCore', 'jrCore_bundle_create_button', $_tmp);

    $_tmp = array(
        'title'  => 'Update Bundle Button',
        'icon'   => 'gear',
        'active' => 'on',
        'group'  => 'owner'
    );
    jrCore_register_module_feature('jrCore', 'item_bundle_list_button', 'jrCore', 'jrCore_bundle_update_button', $_tmp);
    jrCore_register_module_feature('jrCore', 'item_bundle_detail_button', 'jrCore', 'jrCore_bundle_update_button', $_tmp);

    $_tmp = array(
        'title'  => 'Delete Bundle Button',
        'icon'   => 'trash',
        'active' => 'on',
        'group'  => 'owner'
    );
    jrCore_register_module_feature('jrCore', 'item_bundle_list_button', 'jrCore', 'jrCore_bundle_delete_button', $_tmp);
    jrCore_register_module_feature('jrCore', 'item_bundle_detail_button', 'jrCore', 'jrCore_bundle_delete_button', $_tmp);

    // We provide some dashboard panels
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', 'queue depth', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', 'memory used', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', 'disk usage', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', 'CPU count', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', 'installed modules', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', 'installed skins', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', '1 minute load', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', '5 minute load', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', '15 minute load', 'jrCore_dashboard_panels');
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrCore', 'pending items', 'jrCore_dashboard_panels');

    // provided widgets
    jrCore_register_module_feature('jrSiteBuilder', 'widget', 'jrCore', 'widget_list', 'Item List');

    // Archive table support
    // NOTE: log_created can never be changed or archiving breaks!
    $_tmp = array(
        'time_field'    => 'log_created',
        'active_months' => 3
    );
    jrCore_register_module_feature('jrCore', 'archive_table', 'jrCore', 'log', $_tmp);

    // We run the core email queue
    $max_workers = 4;
    if (intval(jrCore_get_config_value('jrMailer', 'throttle', 0)) > 0) {
        $max_workers = 1;
    }
    jrCore_register_queue_worker('jrCore', 'send_email', 'jrCore_send_email_queue_worker', 0, $max_workers, 60, HIGH_PRIORITY_QUEUE);
    jrCore_register_queue_worker('jrCore', 'send_email_low_priority', 'jrCore_send_email_queue_worker', 0, $max_workers, 60, LOW_PRIORITY_QUEUE);

    // Our Recycle bin media file cleanup worker
    jrCore_register_queue_worker('jrCore', 'empty_recycle_bin_files', 'jrCore_empty_recycle_bin_files_worker', 0, 1, 28800, LOW_PRIORITY_QUEUE);

    // Delete media files worker
    jrCore_register_queue_worker('jrCore', 'db_delete_item_media', 'jrCore_db_delete_item_media_worker', 0, 4, 3600, LOW_PRIORITY_QUEUE);

    // Ten_Minute/Hourly/Daily maintenance worker
    jrCore_register_queue_worker('jrCore', 'ten_minute_maintenance', 'jrCore_ten_minute_maintenance_worker', 0, 1, 590, LOW_PRIORITY_QUEUE);
    jrCore_register_queue_worker('jrCore', 'hourly_maintenance', 'jrCore_hourly_maintenance_worker', 0, 1, 3500, LOW_PRIORITY_QUEUE);
    jrCore_register_queue_worker('jrCore', 'daily_maintenance', 'jrCore_daily_maintenance_worker', 0, 1, 86000, LOW_PRIORITY_QUEUE);

    // Update DS Row Counts worker
    jrCore_register_queue_worker('jrCore', 'update_ds_counts', 'jrCore_update_ds_counts_worker', 0, 1, 540, LOW_PRIORITY_QUEUE);

    // Verify Module DB updates queue
    jrCore_register_queue_worker('jrCore', 'repair_module_db', 'jrCore_repair_module_db_worker', 0, 1, 86000);

    // Test Queues Worker
    jrCore_register_queue_worker('jrCore', 'test_queue_system', 'jrCore_test_queue_system_worker', 0, 4, 10);
    return true;
}

// Include Library
require_once APP_DIR . '/modules/jrCore/contrib/smarty4/libs/Smarty.class.php';
require_once APP_DIR . '/modules/jrCore/lib/mysql.php';
require_once APP_DIR . '/modules/jrCore/lib/dashboard.php';
require_once APP_DIR . '/modules/jrCore/lib/datastore.php';
require_once APP_DIR . '/modules/jrCore/lib/datastore_mysql.php';
require_once APP_DIR . '/modules/jrCore/lib/formatter.php';
require_once APP_DIR . '/modules/jrCore/lib/module.php';
require_once APP_DIR . '/modules/jrCore/lib/media.php';
require_once APP_DIR . '/modules/jrCore/lib/checktype.php';
require_once APP_DIR . '/modules/jrCore/lib/smarty.php';
require_once APP_DIR . '/modules/jrCore/lib/cache.php';
require_once APP_DIR . '/modules/jrCore/lib/page.php';
require_once APP_DIR . '/modules/jrCore/lib/form.php';
require_once APP_DIR . '/modules/jrCore/lib/skin.php';
require_once APP_DIR . '/modules/jrCore/lib/util.php';
require_once APP_DIR . '/modules/jrCore/lib/misc.php';
require_once APP_DIR . '/modules/jrCore/lib/queue.php';
require_once APP_DIR . '/modules/jrCore/lib/events.php';
require_once APP_DIR . '/modules/jrCore/lib/view.php';
require_once APP_DIR . '/modules/jrCore/lib/widget.php';
