rc           Contents to append.
 * @param[in]   srclen        Length of `src`.
 * @exception   rb_eArgError  `srclen` is negative.
 * @return      The passed `dst`.
 * @pre         `dst` must not be any arbitrary objects except ::RString.
 * @post        `dst` has the contents of `ptr` appended.
 */
VALUE rb_str_cat(VALUE dst, const char *src, long srclen);

/**
 * Identical to rb_str_cat(), except it assumes the passed pointer is a pointer
 * to a C string.
 *
 * @param[out]  dst           Destination object.
 * @param[in]   src           Contents to append.
 * @exception   rb_eArgError  Result string too big.
 * @exception   rb_eArgError  `src` is a null pointer.
 * @return      The passed `dst`.
 * @pre         `dst` must not be any arbitrary objects except ::RString.
 * @pre         `src` must not be a null pointer.
 * @post        `dst` has the contents of `src` appended.
 */
VALUE rb_str_cat_cstr(VALUE dst, const char *src);

/** @alias{rb_str_cat_cstr} */
VALUE rb_str_cat2(VALUE, const char*);

/**
 * Identical to  rb_str_buf_append(), except  it converts  the right  hand side
 * before concatenating.
 *
 * @param[out]  dst                 Destination object.
 * @param[in]   src                 Source object.
 * @exception   rb_eEncCompatError  Can't mix the encodings.
 * @exception   rb_eArgError        Result string too big.
 * @return      The passed `dst`.
 * @pre         `dst` must not be any arbitrary objects except ::RString.
 * @post        `dst`  has  the  contents  of  `src`  appended,  with  encoding
 *              converted into `dst`'s one, into the end of `dst`.
 */
VALUE rb_str_append(VALUE dst, VALUE src);

/**
 * Identical  to  rb_str_append(), except  it  also  accepts  an integer  as  a
 * codepoint.  This resembles `String#<<`.
 *
 * @param[out]  dst                 Destination object.
 * @param[in]   src                 Source object, String or Numeric.
 * @exception   rb_eRangeError      Source numeric is out of range.
 * @exception   rb_eEncCompatError  Source string too long.
 * @exception   rb_eArgError        Result string too big.
 * @return      The passed `dst`.
 * @pre         `dst` must not be any arbitrary objects except ::RString.
 * @post        `dst`  has  the  contents  of  `src`  appended,  with  encoding
 *              converted into `dst`'s one, into the end of `dst`.
 */
VALUE rb_str_concat(VALUE dst, VALUE src);

/* random.c */

/**
 * This is a universal hash function.
 *
 * @warning    This function changes its value per process.
 * @param[in]  ptr  Target message.
 * @param[in]  len  Length of `ptr` in bytes.
 * @return     A pseudorandom number suitable for Hash's hash value.
 * @see        Aumasson,  JP., Bernstein,  D.J., "SipHash:  A Fast  Short-Input
 *             PRF",  In  proceedings  of   13th  International  Conference  on
 *             Cryptology in  India (INDOCRYPT 2012), LNCS  7668, pp.  489-508,
 *             2012.  http://doi.org/10.1007/978-3-642-34931-7_28
*/
st_index_t rb_memhash(const void *ptr, long len);

/**
 * Starts a series of hashing.  Suppose you have a struct:
 *
 * ```CXX
 * struct foo_tag {
 *     unsigned char bar;
 *     uint32_t baz;
 * };
 * ```
 *
 * It is not a  wise idea to call rb_memhash() over it,  because there could be
 * padding bits.  Instead you should explicitly iterate over each fields:
 *
 * ```CXX
 * foo_tag foo = { 0, 0, };
 * st_index_t hash = 0;
 *
 * hash = rb_hash_start(0);
 * hash = rb_hash_uint(hash, foo.bar);
 * hash = rb_hash_uint32(hash, foo.baz);
 * hash = rb_hash_end(hash);
 * ```
 *
 * @param[in]  i  Initial value.
 * @return     A hash value.
 */
st_index_t rb_hash_start(st_index_t i);

/** @alias{st_hash_uint32} */
#define rb_hash_uint32(h, i) st_hash_uint32((h), (i))

/** @alias{st_hash_uint} */
#define rb_hash_uint(h, i) st_hash_uint((h), (i))

/** @alias{st_hash_end} */
#define rb_hash_end(h) st_hash_end(h)

/* string.c */

/**
 * Calculates a hash value of a string.   This is one of the two functions that
 * constructs struct ::st_hash_type.
 *
 * @param[in]  str  An object of ::RString.
 * @return     A hash value.
 * @pre        `str` must not be any arbitrary object except ::RString.
 *
 * @internal
 *
 * Although safe to call, there must be no particular use case of this function
 * for extension libraries.  Only ruby internals must know about it.
 *
 * This is not a simple alias  of rb_memhash(), because it considers the passed
 * string's encoding as well as its contents.
 */
st_index_t rb_str_hash(VALUE str);

/**
 * Compares two  strings.  This  is one  of the  two functions  that constructs
 * struct ::st_hash_type.
 *
 * @param[in]  str1  A string.
 * @param[in]  str2  Another string.
 * @retval     1     They have identical contents, length, and encodings.
 * @retval     0     Otherwise.
 * @pre        Both   objects   must  not  be  any   arbitrary  objects  except
 *             ::RString.
 *
 * @internal
 *
 * In contrast to  rb_str_hash(), this could be handy for  comparison that only
 * concerns equality.  rb_str_cmp() returns 1, 0, -1.
 */
int rb_str_hash_cmp(VALUE str1, VALUE str2);

/**
 * Checks  if  two   strings  are  comparable  each  other   or  not.   Because
 * rb_str_cmp()  must  return  "lesser  than" or  "greater  than"  information,
 * comparing two strings needs a stricter restriction.  Both sides must be in a
 * same set of strings which have total order.  This is to check that property.
 * Intuitive it  sounds?  But they  can have different encodings.   A character
 * and another might or might not appear in the same order in their codepoints.
 * It is complicated than you think.
 *
 * @param[in]  str1  A string.
 * @param[in]  str2  Another string.
 * @retval     1     They agree on a total order.
 * @retval     0     Otherwise.
 * @pre        Both   objects   must  not  be  any   arbitrary  objects  except
 *             ::RString.
 */
int rb_str_comparable(VALUE str1, VALUE str2);

/**
 * Compares two strings, as in `strcmp(3)`.  This does not consider the current
 * locale, but considers the encodings of both sides instead.
 *
 * @param[in]  lhs  A string.
 * @param[in]  rhs  Another string.
 * @retval     -1   `lhs` is "bigger than" `rhs`.
 * @retval      1   `rhs` is "bigger than" `lhs`.
 * @retval      0    Otherwise, e.g. not comparable.
 * @pre        Both   objects   must  not  be  any   arbitrary  objects  except
 *             ::RString.
 */
int rb_str_cmp(VALUE lhs, VALUE rhs);

/**
 * Equality of two strings.
 *
 * If `str2` is not a String, it  resorts to `str2 == str1`.  Otherwise if they
 * are not comparable, returns ::RUBY_Qfalse.   Otherwise if they have the same
 * contents  and   the  length,   returns  ::RUBY_Qtrue.    Otherwise,  returns
 * ::RUBY_Qfalse.
 *
 * @param[in]  str1         A string.
 * @param[in]  str2         Another string.
 * @retval     RUBY_Qtrue   They are equal.
 * @retval     RUBY_Qfalse  They are either different, or not comparable.
 */
VALUE rb_str_equal(VALUE str1, VALUE str2);

/**
 * Shrinks the given string for the given number of bytes.
 *
 * @param[out]  str               String to squash.
 * @param[in]   len               Number of bytes to reduce.
 * @exception   rb_eRuntimeError  `str` is `locktmp`-ed.
 * @exception   rb_eFrozenError   `str` is frozen.
 * @return      The passed `str`.
 * @pre         `str` must not be any arbitrary objects except ::RString.
 * @post        `str` is shrunken.
 * @warning     Can break a multibyte character in middle.
 *
 * @internal
 *
 * What if `len` is negative?
 */
VALUE rb_str_drop_bytes(VALUE str, long len);

/**
 * Replaces some  (or all) of  the contents of the  given string.  This  is the
 * implementation of three-argumented `String#[]=`.
 *
 * @param[out]  dst               Target string to update.
 * @param[in]   beg               Offset of the affected portion.
 * @param[in]   len               Length of the affected portion.
 * @param[in]   src               Object to be assigned.
 * @exception   rb_eTypeError     `src` has no implicit conversion to String.
 * @exception   rb_eIndexError    `len` is negative, or `beg` is OOB.
 * @exception   rb_eRuntimeError  `dst` is `locktmp`-ed.
 * @exception   rb_eFrozenError   `dst` is frozen.
 * @note        Unlike rb_str_substr(), this function raises.
 * @post        A  portion of  `dst`  from  `beg` to  `len`  is the  stringised
 *              representation of `src`.  If that replacement string is not the
 *              same  length as  the portion  it  is replacing,  `dst` will  be
 *              resized accordingly.
 */
void rb_str_update(VALUE dst, long beg, long len, VALUE src);

/**
 * Replaces the contents  of the former object with the  stringised contents of
 * the latter.
 *
 * @param[out]  dst               Destination object.
 * @param[in]   src               Source object.
 * @exception   rb_eTypeError     `src` has no implicit conversion to String.
 * @exception   rb_eRuntimeError  `dst` is `locktmp`-ed.
 * @exception   rb_eFrozenError   `dst` is frozen.
 * @return      The passed `dst`.
 * @pre        `dst` must not be any arbitrary object except ::RString.
 * @post        `dst`'s  former  components  are  abandoned.  It  now  has  the
 *              identical encoding, length, and contents to `src`.
 */
VALUE rb_str_replace(VALUE dst, VALUE src);

/**
 * Generates a "readable" version of the receiver.
 *
 * @warning    The output is _insecure_.  Never feed one to `eval`.
 * @warning    The output is not always in the same encoding as the given one.
 * @warning    A  character might  or might  not be  escaped, depending  on the
 *             result encoding.
 * @param[in]  str  String to inspect.
 * @return     Its inspection, either  in default internal encoding  if any, or
 *             in default external encoding otherwise.
 * @see        rb_str_dump()
 *
 * @internal
 *
 * This is a  (silent) fix of an actual vulnerability  feeding `inspect` output
 * strings to `eval`:
 * https://github.com/hiki/hiki/commit/8771a6e25198e264a2bf9dc1c102fea2cc8ff975
 *
 * ... and its advisory:
 * http://hikiwiki.org/en/advisory20040712.html
 */
VALUE rb_str_inspect(VALUE str);

/**
 * "Inverse" of rb_eval_string().  Returns a quoted version of the string.  All
 * non-printing characters are replaced by  `\uNNNN` or `\xHH` notation and all
 * special characters are escaped.  The result string is guaranteed to render a
 * string of the same contents when passed to `eval` and friends.
 *
 * @param[in]  str               String to dump.
 * 