  ((uint32_t)(2*MI_GiB))

// blocks up to this size are always allocated aligned
#define MI_MAX_ALIGN_GUARANTEE            (8*MI_MAX_ALIGN_SIZE)

// Alignments over MI_ALIGNMENT_MAX are allocated in dedicated huge page segments
#define MI_ALIGNMENT_MAX                  (MI_SEGMENT_SIZE >> 1)


// ------------------------------------------------------
// Mimalloc pages contain allocated blocks
// ------------------------------------------------------

// The free lists use encoded next fields
// (Only actually encodes when MI_ENCODED_FREELIST is defined.)
typedef uintptr_t  mi_encoded_t;

// thread id's
typedef size_t     mi_threadid_t;

// free lists contain blocks
typedef struct mi_block_s {
  _Atomic(mi_encoded_t) next;
} mi_block_t;


// The delayed flags are used for efficient multi-threaded free-ing
typedef enum mi_delayed_e {
  MI_USE_DELAYED_FREE   = 0, // push on the owning heap thread delayed list
  MI_DELAYED_FREEING    = 1, // temporary: another thread is accessing the owning heap
  MI_NO_DELAYED_FREE    = 2, // optimize: push on page local thread free queue if another block is already in the heap thread delayed free list
  MI_NEVER_DELAYED_FREE = 3  // sticky, only resets on page reclaim
} mi_delayed_t;


// The `in_full` and `has_aligned` page flags are put in a union to efficiently
// test if both are false (`full_aligned == 0`) in the `mi_free` routine.
#if !MI_TSAN
typedef union mi_page_flags_s {
  uint8_t full_aligned;
  struct {
    uint8_t in_full : 1;
    uint8_t has_aligned : 1;
  } x;
} mi_page_flags_t;
#else
// under thread sanitizer, use a byte for each flag to suppress warning, issue #130
typedef union mi_page_flags_s {
  uint16_t full_aligned;
  struct {
    uint8_t in_full;
    uint8_t has_aligned;
  } x;
} mi_page_flags_t;
#endif

// Thread free list.
// We use the bottom 2 bits of the pointer for mi_delayed_t flags
typedef uintptr_t mi_thread_free_t;

// A page contains blocks of one specific size (`block_size`).
// Each page has three list of free blocks:
// `free` for blocks that can be allocated,
// `local_free` for freed blocks that are not yet available to `mi_malloc`
// `thread_free` for freed blocks by other threads
// The `local_free` and `thread_free` lists are migrated to the `free` list
// when it is exhausted. The separate `local_free` list is necessary to
// implement a monotonic heartbeat. The `thread_free` list is needed for
// avoiding atomic operations in the common case.
//
//
// `used - |thread_free|` == actual blocks that are in use (alive)
// `used - |thread_free| + |free| + |local_free| == capacity`
//
// We don't count `freed` (as |free|) but use `used` to reduce
// the number of memory accesses in the `mi_page_all_free` function(s).
//
// Notes:
// - Access is optimized for `mi_free` and `mi_page_alloc` (in `alloc.c`)
// - Using `uint16_t` does not seem to slow things down
// - The size is 8 words on 64-bit which helps the page index calculations
//   (and 10 words on 32-bit, and encoded free lists add 2 words. Sizes 10
//    and 12 are still good for address calculation)
// - To limit the structure size, the `xblock_size` is 32-bits only; for
//   blocks > MI_HUGE_BLOCK_SIZE the size is determined from the segment page size
// - `thread_free` uses the bottom bits as a delayed-free flags to optimize
//   concurrent frees where only the first concurrent free adds to the owning
//   heap `thread_delayed_free` list (see `alloc.c:mi_free_block_mt`).
//   The invariant is that no-delayed-free is only set if there is
//   at least one block that will be added, or as already been added, to
//   the owning heap `thread_delayed_free` list. This guarantees that pages
//   will be freed correctly even if only other threads free blocks.
typedef struct mi_page_s {
  // "owned" by the segment
  uint32_t              slice_count;       // slices in this page (0 if not a page)
  uint32_t              slice_offset;      // distance from the actual page data slice (0 if a page)
  uint8_t               is_committed : 1;  // `true` if the page virtual memory is committed
  uint8_t               is_zero_init : 1;  // `true` if the page was initially zero initialized
  uint8_t               use_qsbr : 1;      // delay page freeing using qsbr
  uint8_t               tag : 4;           // tag from the owning heap
  uint8_t               debug_offset;      // number of bytes to preserve when filling freed or uninitialized memory

  // layout like this to optimize access in `mi_malloc` and `mi_free`
  uint16_t              capacity;          // number of blocks committed, must be the first field, see `segment.c:page_clear`
  uint16_t              reserved;          // number of blocks reserved in memory
  mi_page_flags_t       flags;             // `in_full` and `has_aligned` flags (8 bits)
  uint8_t               free_is_zero : 1;  // `true` if the blocks in the free list are zero initialized
  uint8_t               retire_expire : 7; // expiration count for retired blocks

  mi_block_t*           free;              // list of available free blocks (`malloc` allocates from this list)
  uint32_t              used;              // number of blocks in use (including blocks in `local_free` and `thread_free`)
  uint32_t              xblock_size;       // size available in each block (always `>0`)
  mi_block_t*           local_free;        // list of deferred free blocks by this thread (migrates to `free`)

  #if (MI_ENCODE_FREELIST || MI_PADDING)
  uintptr_t             keys[2];           // two random keys to encode the free lists (see `_mi_block_next`) or padding canary
  #endif

  _Atomic(mi_thread_free_t) xthread_free;  // list of deferred free blocks freed by other threads
  _Atomic(uintptr_t)        xheap;

  struct mi_page_s*     next;              // next page owned by this thread with the same `block_size`
  struct mi_page_s*     prev;              // previous page owned by this thread with the same `block_size`

#ifdef Py_GIL_DISABLED
  struct llist_node     qsbr_node;
  uint64_t              qsbr_goal;
#endif

  // 64-bit 9 words, 32-bit 12 words, (+2 for secure)
  #if MI_INTPTR_SIZE==8 && !defined(Py_GIL_DISABLED)
  uintptr_t padding[1];
  #endif
} mi_page_t;



// ------------------------------------------------------
// Mimalloc segments contain mimalloc pages
// ------------------------------------------------------

typedef enum mi_page_kind_e {
  MI_PAGE_SMALL,    // small blocks go into 64KiB pages inside a segment
  MI_PAGE_MEDIUM,   // medium blocks go into medium pages inside a segment
  MI_PAGE_LARGE,    // larger blocks go into a page of just one block
  MI_PAGE_HUGE,     // huge blocks (> 16 MiB) are put into a single page in a single segment.
} mi_page_kind_t;

typedef enum mi_segment_kind_e {
  MI_SEGMENT_NORMAL, // MI_SEGMENT_SIZE size with pages inside.
  MI_SEGMENT_HUGE,   // > MI_LARGE_SIZE_MAX segment with just one huge page inside.
} mi_segment_kind_t;

// ------------------------------------------------------
// A segment holds a commit mask where a bit is set if
// the corresponding MI_COMMIT_SIZE area is committed.
// The MI_COMMIT_SIZE must be a multiple of the slice
// size. If it is equal we have the most fine grained
// decommit (but setting it higher can be more efficient).
// The MI_MINIMAL_COMMIT_SIZE is the minimal amount that will
// be committed in one go which can be set higher than
// MI_COMMIT_SIZE for efficiency (while the decommit mask
// is still tracked in fine-grained MI_COMMIT_SIZE chunks)
// ------------------------------------------------------

#define MI_MINIMAL_COMMIT_SIZE      (1*MI_SEGMENT_SLICE_SIZE)
#define MI_COMMIT_SIZE              (MI_SEGMENT_SLICE_SIZE)              // 64KiB
#define MI_COMMIT_MASK_BITS         (MI_SEGMENT_SIZE / MI_COMMIT_SIZE)
#define MI_COMMIT_MASK_FIELD_BITS    MI_SIZE_BITS
#define MI_COMMIT_MASK_FIELD_COUNT  (MI_COMMIT_MASK_BITS / MI_COMMIT_MASK_FIELD_BITS)

#if (MI_COMMIT_MASK_BITS != (MI_COMMIT_MASK_FIELD_COUNT * MI_COMMIT_MASK_FIELD_BITS))
#error "the segment size must be exactly divisible by the (commit size * size_t bits)"
#endif

typedef struct mi_commit_mask_s {
  size_t mask[MI_COMMIT_MASK_FIELD_COUNT];
} mi_commit_mask_t;

typedef mi_page_t  mi_slice_t;
typedef int64_t    mi_msecs_t;


// Memory can reside in arena's, direct OS allocated, or statically allocated. The memid keeps track of this.
typedef enum mi_memkind_e {
  MI_MEM_NONE,      // not allocated
  MI_MEM_EXTERNAL,  // not owned by mimalloc but provided externally (via `mi_manage_os_memory` for example)
  MI_MEM_STATIC,    // allocated in a static area and should not be freed (for arena meta data for example)
  MI_MEM_OS,        // allocated from the OS
  MI_MEM_OS_HUGE,   // allocated as huge os pages
  MI_MEM_OS_REMAP,  // allocated in a remapable area (i.e. using `mremap`)
  MI_MEM_ARENA      // allocated from an arena (the usual case)
} mi_memkind_t;

static inline bool mi_memkind_is_os(mi_memkind_t memkind) {
  return (memkind >= MI_MEM_OS && memkind <= MI_MEM_OS_REMAP);
}

typedef struct mi_memid_os_info {
  void*         base;               // actual base address of the block (used for offset aligned allocations)
  size_t        alignment;          // alignment at allocation
} mi_memid_os_info_t;

typedef struct mi_memid_arena_info {
  size_t        block_index;        // index in the arena
  mi_arena_id_t id;                 // arena id (>= 1)
  bool          is_exclusive;       // the arena can only be used for specific arena allocations
} mi_memid_arena_info_t;

typedef struct mi_memid_s {
  union {
    mi_memid_os_info_t    os;       // only used for MI_MEM_OS
    mi_memid_arena_info_t arena;    // only used for MI_MEM_ARENA
  } mem;
  bool          is_pinned;          // `true` if we cannot decommit/reset/protect in this memory (e.g. when allocated using large OS pages)
  bool          initially_committed;// `true` if the memory was originally allocated as committed
  bool          initially_zero;     // `true` if the memory was originally zero initialized
  mi_memkind_t  memkind;
} mi_memid_t;


// Segments are large allocated memory blocks (8mb on 64 bit) from
// the OS. Inside segments we allocated fixed size _pages_ that
// contain blocks.
typedef struct mi_segment_s {
  // constant fields
  mi_memid_t        memid;              // memory id for arena allocation
  bool              allow_decommit;
  bool              allow_purge;
  size_t            segment_size;

  // segment fields
  mi_msecs_t        purge_expire;
  mi_commit_mask_t  purge_mask;
  mi_commit_mask_t  commit_mask;

  _Atomic(struct mi_segment_s*) abandoned_next;

  // from here is zero initialized
  struct mi_segment_s* next;            // the list of freed segments in the cache (must be first field, see `segment.c:mi_segment_init`)

  size_t            abandoned;          // abandoned pages (i.e. the original owning thread stopped) (`abandoned <= used`)
  size_t            abandoned_visits;   // count how often this segment is visited in the abandoned list (to force reclaim it it is too long)
  size_t            used;               // count of pages in use
  uintptr_t         cookie;             // verify addresses in debug mode: `mi_ptr_cookie(segment) == segment->cookie`

  size_t            segment_slices;      // for huge segments this may be different from `MI_SLICES_PER_SEGMENT`
  size_t            segment_info_slices; // initial slices we are using segment info and possible guard pages.

  // layout like this to optimize access in `mi_free`
  mi_segment_kind_t kind;
  size_t            slice_entries;       // entries in the `slices` array, at most `MI_SLICES_PER_SEGMENT`
  _Atomic(mi_threadid_t) thread_id;      // unique id of the thread owning this segment

  mi_slice_t        slices[MI_SLICES_PER_SEGMENT+1];  // one more for huge blocks with large alignment
} mi_segment_t;

typedef uintptr_t        mi_tagged_segment_t;

// Segments unowned by any thread are put in a shared pool
typedef struct mi_abandoned_pool_s {
  // This is a list of visited abandoned pages that were full at the time.
  // this list migrates to `abandoned` when that becomes NULL. The use of
  // this list reduces contention and the rate at which segments are visited.
  mi_decl_cache_align _Atomic(mi_segment_t*)       abandoned_visited; // = NULL

  // The abandoned page list (tagged as it supports pop)
  mi_decl_cache_align _Atomic(mi_tagged_segment_t) abandoned;         // = NULL

  // Maintain these for debug purposes (these counts may be a bit off)
  mi_decl_cache_align _Atomic(size_t)           abandoned_count;
  mi_decl_cache_align _Atomic(size_t)           abandoned_visited_count;

  // We also maintain a count of current readers of the abandoned list
  // in order to prevent resetting/decommitting segment memory if it might
  // still be read.
  mi_decl_cache_align _Atomic(size_t)           abandoned_readers; // = 0
} mi_abandoned_pool_t;


// ------------------------------------------------------
// Heaps
// Provide first-class heaps to allocate from.
// A heap just owns a set of pages for allocation and
// can only be allocate/reallocate from the thread that created it.
// Freeing blocks can be done from any thread though.
// Per thread, the segments are shared among its heaps.
// Per thread, there is always a default heap that is
// used for allocation; it is initialized to statically
// point to an empty heap to avoid initialization checks
// in the fast path.
// ------------------------------------------------------

// Thread local data
typedef struct mi_tld_s mi_tld_t;

// Pages of a certain block size are held in a queue.
typedef struct mi_page_queue_s {
  mi_page_t* first;
  mi_page_t* last;
  size_t     block_size;
} mi_page_queue_t;

#define MI_BIN_FULL  (MI_BIN_HUGE+1)

// Random context
typedef struct mi_random_cxt_s {
  uint32_t input[16];
  uint32_t output[16];
  int      output_available;
  bool     weak;
} mi_random_ctx_t;


// In debug mode there is a padding structure at the end of the blocks to check for buffer overflows
#if (MI_PADDING)
typedef struct mi_padding_s {
  uint32_t canary; // encoded block value to check validity of the padding (in case of overflow)
  uint32_t delta;  // padding bytes before the block. (mi_usable_size(p) - delta == exact allocated bytes)
} mi_padding_t;
#define MI_PADDING_SIZE   (sizeof(mi_padding_t))
#define MI_PADDING_WSIZE  ((MI_PADDING_SIZE + MI_INTPTR_SIZE - 1) / MI_INTPTR_SIZE)
#else
#define MI_PADDING_SIZE   0
#define MI_PADDING_WSIZE  0
#endif

#define MI_PAGES_DIRECT   (MI_SMALL_WSIZE_MAX + MI_PADDING_WSIZE + 1)


// A heap owns a set of pages.
struct mi_heap_s {
  mi_tld_t*             tld;
  mi_page_t*            pages_free_direct[MI_PAGES_DIRECT];  // optimize: array where every entry points a page with possibly free blocks in the corresponding queue for that size.
  mi_page_queue_t       pages[MI_BIN_FULL + 1];              // queue of pages for each size class (or "bin")
  _Atomic(mi_block_t*)  thread_delayed_free;
  mi_threadid_t         thread_id;                           // thread this heap belongs too
  mi_arena_id_t         arena_id;                            // arena id if the heap belongs to a specific arena (or 0)
  uintptr_t             cookie;                              // random cookie to verify pointers (see `_mi_ptr_cookie`)
  uintptr_t             keys[2];                             // two random keys used to encode the `thread_delayed_free` list
  mi_random_ctx_t       random;                              // random number context used for secure allocation
  size_t                page_count;                          // total number of pages in the `pages` queues.
  size_t                page_retired_min;                    // smallest retired index (retired pages are fully free, but still in the page queues)
  size_t                page_retired_max;                    // largest retired index into the `pages` array.
  mi_heap_t*            next;                                // list of heaps per thread
  bool                  no_reclaim;                          // `true` if this heap should not reclaim abandoned pages
  uint8_t               tag;                                 // custom identifier for this heap
  uint8_t               debug_offset;                        // number of bytes to preserve when filling freed or uninitialized memory
  bool                  page_use_qsbr;                       // should freeing pages be delayed using QSBR
};



// ------------------------------------------------------
// Debug
// ------------------------------------------------------

#if !defined(MI_DEBUG_UNINIT)
#define MI_DEBUG_UNINIT     (0xD0)
#endif
#if !defined(MI_DEBUG_FREED)
#define MI_DEBUG_FREED      (0xDF)
#endif
#if !defined(MI_DEBUG_PADDING)
#define MI_DEBUG_PADDING    (0xDE)
#endif

#if (MI_DEBUG)
// use our own assertion to print without memory allocation
void _mi_assert_fail(const char* assertion, const char* fname, unsigned int line, const char* func );
#define mi_assert(expr)     ((expr) ? (void)0 : _mi_assert_fail(#expr,__FILE__,__LINE__,__func__))
#else
#define mi_assert(x)
#endif

#if (MI_DEBUG>1)
#define mi_assert_internal    mi_assert
#else
#define mi_assert_internal(x)
#endif

#if (MI_DEBUG>2)
#define mi_assert_expensive   mi_assert
#else
#define mi_assert_expensive(x)
#endif

// ------------------------------------------------------
// Statistics
// ------------------------------------------------------

#ifndef MI_STAT
#if (MI_DEBUG>0)
#define MI_STAT 2
#else
#define MI_STAT 0
#endif
#endif

typedef struct mi_stat_count_s {
  int64_t allocated;
  int64_t freed;
  int64_t peak;
  int64_t current;
} mi_stat_count_t;

typedef struct mi_stat_counter_s {
  int64_t total;
  int64_t count;
} mi_stat_counter_t;

typedef struct mi_stats_s {
  mi_stat_count_t segments;
  mi_stat_count_t pages;
  mi_stat_count_t reserved;
  mi_stat_count_t committed;
  mi_stat_count_t reset;
  mi_stat_count_t purged;
  mi_stat_count_t page_committed;
  mi_stat_count_t segments_abandoned;
  mi_stat_count_t pages_abandoned;
  mi_stat_count_t threads;
  mi_stat_count_t normal;
  mi_stat_count_t huge;
  mi_stat_count_t large;
  mi_stat_count_t malloc;
  mi_stat_count_t segments_cache;
  mi_stat_counter_t pages_extended;
  mi_stat_counter_t mmap_calls;
  mi_stat_counter_t commit_calls;
  mi_stat_counter_t reset_calls;
  mi_stat_counter_t purge_calls;
  mi_stat_counter_t page_no_retire;
  mi_stat_counter_t searches;
  mi_stat_counter_t normal_count;
  mi_stat_counter_t huge_count;
  mi_stat_counter_t large_count;
#if MI_STAT>1
  mi_stat_count_t normal_bins[MI_BIN_HUGE+1];
#endif
} mi_stats_t;


void _mi_stat_increase(mi_stat_count_t* stat, size_t amount);
void _mi_stat_decrease(mi_stat_count_t* stat, size_t amount);
void _mi_stat_counter_increase(mi_stat_counter_t* stat, size_t amount);

#if (MI_STAT)
#define mi_stat_increase(stat,amount)         _mi_stat_increase( &(stat), amount)
#define mi_stat_decrease(stat,amount)         _mi_stat_decrease( &(stat), amount)
#define mi_stat_counter_increase(stat,amount) _mi_stat_counter_increase( &(stat), amount)
#else
#define mi_stat_increase(stat,amount)         (void)0
#define mi_stat_decrease(stat,amount)         (void)0
#define mi_stat_counter_increase(stat,amount) (void)0
#endif

#define mi_heap_stat_counter_increase(heap,stat,amount)  mi_stat_counter_increase( (heap)->tld->stats.stat, amount)
#define mi_heap_stat_increase(heap,stat,amount)  mi_stat_increase( (heap)->tld->stats.stat, amount)
#define mi_heap_stat_decrease(heap,stat,amount)  mi_stat_decrease( (heap)->tld->stats.stat, amount)

// ------------------------------------------------------
// Thread Local data
// ------------------------------------------------------

// A "span" is is an available range of slices. The span queues keep
// track of slice spans of at most the given `slice_count` (but more than the previous size class).
typedef struct mi_span_queue_s {
  mi_slice_t* first;
  mi_slice_t* last;
  size_t      slice_count;
} mi_span_queue_t;

#define MI_SEGMENT_BIN_MAX (35)     // 35 == mi_segment_bin(MI_SLICES_PER_SEGMENT)

// OS thread local data
typedef struct mi_os_tld_s {
  size_t                region_idx;   // start point for next allocation
  mi_stats_t*           stats;        // points to tld stats
} mi_os_tld_t;


// Segments thread local data
typedef struct mi_segments_tld_s {
  mi_span_queue_t     spans[MI_SEGMENT_BIN_MAX+1];  // free slice spans inside segments
  size_t              count;        // current number of segments;
  size_t              peak_count;   // peak number of segments
  size_t              current_size; // current size of all segments
  size_t              peak_size;    // peak size of all segments
  mi_stats_t*         stats;        // points to tld stats
  mi_os_tld_t*        os;           // points to os stats
  mi_abandoned_pool_t* abandoned;   // pool of abandoned segments
} mi_segments_tld_t;

// Thread local data
struct mi_tld_s {
  unsigned long long  heartbeat;     // monotonic heartbeat count
  bool                recurse;       // true if deferred was called; used to prevent infinite recursion.
  mi_heap_t*          heap_backing;  // backing heap of this thread (cannot be deleted)
  mi_heap_t*          heaps;         // list of heaps in this thread (so we can abandon all when the thread terminates)
  mi_segments_tld_t   segments;      // segment tld
  mi_os_tld_t         os;            // os tld
  mi_stats_t          stats;         // statistics
};

#endif
PK       ! %oJ  J  )  python3.13/internal/pycore_opcode_utils.hnu [        #ifndef Py_INTERNAL_OPCODE_UTILS_H
#define Py_INTERNAL_OPCODE_UTILS_H
#ifdef __cplusplus
extern "C" {
#endif

#ifndef Py_BUILD_CORE
#  error "this header requires Py_BUILD_CORE define"
#endif

#include "opcode_ids.h"

#define MAX_REAL_OPCODE 254

#define IS_WITHIN_OPCODE_RANGE(opcode) \
        (((opcode) >= 0 && (opcode) <= MAX_REAL_OPCODE) || \
         IS_PSEUDO_INSTR(opcode))

#define IS_BLOCK_PUSH_OPCODE(opcode) \
        ((opcode) == SETUP_FINALLY || \
         (opcode) == SETUP_WITH || \
         (opcode) == SETUP_CLEANUP)

#define HAS_TARGET(opcode) \
        (OPCODE_HAS_JUMP(opcode) || IS_BLOCK_PUSH_OPCODE(opcode))

/* opcodes that must be last in the basicblock */
#define IS_TERMINATOR_OPCODE(opcode) \
        (OPCODE_HAS_JUMP(opcode) || IS_SCOPE_EXIT_OPCODE(opcode))

/* opcodes which are not emitted in codegen stage, only by the assembler */
#define IS_ASSEMBLER_OPCODE(opcode) \
        ((opcode) == JUMP_FORWARD || \
         (opcode) == JUMP_BACKWARD || \
         (opcode) == JUMP_BACKWARD_NO_INTERRUPT)

#define IS_BACKWARDS_JUMP_OPCODE(opcode) \
        ((opcode) == JUMP_BACKWARD || \
         (opcode) == JUMP_BACKWARD_NO_INTERRUPT)

#define IS_UNCONDITIONAL_JUMP_OPCODE(opcode) \
        ((opcode) == JUMP || \
         (opcode) == JUMP_NO_INTERRUPT || \
         (opcode) == JUMP_FORWARD || \
         (opcode) == JUMP_BACKWARD || \
         (opcode) == JUMP_BACKWARD_NO_INTERRUPT)

#define IS_SCOPE_EXIT_OPCODE(opcode) \
        ((opcode) == RETURN_VALUE || \
         (opcode) == RETURN_CONST || \
         (opcode) == RAISE_VARARGS || \
         (opcode) == RERAISE)


/* Flags used in the oparg for MAKE_FUNCTION */
#define MAKE_FUNCTION_DEFAULTS    0x01
#define MAKE_FUNCTION_KWDEFAULTS  0x02
#define MAKE_FUNCTION_ANNOTATIONS 0x04
#define MAKE_FUNCTION_CLOSURE     0x08

/* Values used in the oparg for RESUME */
#define RESUME_AT_FUNC_START 0
#define RESUME_AFTER_YIELD 1
#define RESUME_AFTER_YIELD_FROM 2
#define RESUME_AFTER_AWAIT 3

#define RESUME_OPARG_LOCATION_MASK 0x3
#define RESUME_OPARG_DEPTH1_MASK 0x4

#ifdef __cplusplus
}
#endif
#endif /* !Py_INTERNAL_OPCODE_UTILS_H */
PK       ! /s    (  python3.13/internal/pycore_tracemalloc.hnu [        #ifndef Py_INTERNAL_TRACEMALLOC_H
#define Py_INTERNAL_TRACEMALLOC_H
#ifdef __cplusplus
extern "C" {
#endif

#ifndef Py_BUILD_CORE
#  error "this header requires Py_BUILD_CORE define"
#endif

#include "pycore_hashtable.h"     // _Py_hashtable_t


/* Trace memory blocks allocated by PyMem_RawMalloc() */
#define TRACE_RAW_MALLOC


struct _PyTraceMalloc_Config {
    /* Module initialized?
       Variable protected by the GIL */
    enum {
        TRACEMALLOC_NOT_INITIALIZED,
        TRACEMALLOC_INITIALIZED,
        TRACEMALLOC_FINALIZED
    } initialized;

    /* Is tracemalloc tracing memory allocations?
       Variable protected by the GIL */
    int tracing;

    /* limit of the number of frames in a traceback, 1 by default.
       Variable protected by the GIL. */
    int max_nframe;
};


/* Pack the frame_t structure to reduce the memory footprint on 64-bit
   architectures: 12 bytes instead of 16. */
#if defined(_MSC_VER)
#pragma pack(push, 4)
#endif

struct
#ifdef __GNUC__
__attribute__((packed))
#endif
tracemalloc_frame {
    /* filename cannot be NULL: "<unknown>" is used if the Python frame
       filename is NULL */
    PyObject *filename;
    unsigned int lineno;
};
#ifdef _MSC_VER
#pragma pack(pop)
#endif

struct tracemalloc_traceback {
    Py_uhash_t hash;
    /* Number of frames stored */
    uint16_t nframe;
    /* Total number of frames the traceback had */
    uint16_t total_nframe;
    struct tracemalloc_frame frames[1];
};


struct _tracemalloc_runtime_state {
    struct _PyTraceMalloc_Config config;

    /* Protected by the GIL */
    struct {
        PyMemAllocatorEx mem;
        PyMemAllocatorEx raw;
        PyMemAllocatorEx obj;
    } allocators;

#if defined(TRACE_RAW_MALLOC)
    PyThread_type_lock tables_lock;
#endif
    /* Size in bytes of currently traced memory.
       Protected by TABLES_LOCK(). */
    size_t traced_memory;
    /* Peak size in bytes of traced memory.
       Protected by TABLES_LOCK(). */
    size_t peak_traced_memory;
    /* Hash table used as a set to intern filenames:
       PyObject* => PyObject*.
       Protected by the GIL */
    _Py_hashtable_t *filenames;
    /* Buffer to store a new traceback in traceback_new().
       Protected by the GIL. */
    struct tracemalloc_traceback *traceback;
    /* Hash table used as a set to intern tracebacks:
       traceback_t* => traceback_t*
       Protected by the GIL */
    _Py_hashtable_t *tracebacks;
    /* pointer (void*) => trace (trace_t*).
       Protected by TABLES_LOCK(). */
    _Py_hashtable_t *traces;
    /* domain (unsigned int) => traces (_Py_hashtable_t).
       Protected by TABLES_LOCK(). */
    _Py_hashtable_t *domains;

    struct tracemalloc_traceback empty_traceback;

    Py_tss_t reentrant_key;
};

#define _tracemalloc_runtime_state_INIT \
    { \
        .config = { \
            .initialized = TRACEMALLOC_NOT_INITIALIZED, \
            .tracing = 0, \
            .max_nframe = 1, \
        }, \
        .reentrant_key = Py_tss_NEEDS_INIT, \
    }


// Get the traceback where a memory block was allocated.
//
// Return a tuple of (filename: str, lineno: int) tuples.
//
// Return None if the tracemalloc module is disabled or if the memory block
// is not tracked by tracemalloc.
//
// Raise an exception and return NULL on error.
//
// Export for '_testinternalcapi' shared extension.
PyAPI_FUNC(PyObject*) _PyTraceMalloc_GetTraceback(
    unsigned int domain,
    uintptr_t ptr);

/* Return non-zero if tracemalloc is tracing */
extern int _PyTraceMalloc_IsTracing(void);

/* Clear the tracemalloc traces */
extern void _PyTraceMalloc_ClearTraces(void);

/* Clear the tracemalloc traces */
extern PyObject* _PyTraceMalloc_GetTraces(void);

/* Clear tracemalloc traceback for an object */
extern PyObject* _PyTraceMalloc_GetObjectTraceback(PyObject *obj);

/* Initialize tracemalloc */
extern PyStatus _PyTraceMalloc_Init(void);

/* Start tracemalloc */
extern int _PyTraceMalloc_Start(int max_nframe);

/* Stop tracemalloc */
extern void _PyTraceMalloc_Stop(void);

/* Get the tracemalloc traceback limit */
extern int _PyTraceMalloc_GetTracebackLimit(void);

/* Get the memory usage of tracemalloc in bytes */
extern size_t _PyTraceMalloc_GetMemory(void);

/* Get the current size and peak size of traced memory blocks as a 2-tuple */
extern PyObject* _PyTraceMalloc_GetTracedMemory(void);

/* Set the peak size of traced memory blocks to the current size */
extern void _PyTraceMalloc_ResetPeak(void);

#ifdef __cplusplus
}
#endif
#endif  // !Py_INTERNAL_TRACEMALLOC_H
PK       ! E]    3  python3.13/internal/pycore_runtime_init_generated.hnu [        #ifndef Py_INTERNAL_RUNTIME_INIT_GENERATED_H
#define Py_INTERNAL_RUNTIME_INIT_GENERATED_H
#ifdef __cplusplus
extern "C" {
#endif

#ifndef Py_BUILD_CORE
#  error "this header requires Py_BUILD_CORE define"
#endif

#include "pycore_long.h"          // _PyLong_DIGIT_INIT()


/* The following is auto-generated by Tools/build/generate_global_objects.py. */
#define _Py_small_ints_INIT { \
    _PyLong_DIGIT_INIT(-5), \
    _PyLong_DIGIT_INIT(-4), \
    _PyLong_DIGIT_INIT(-3), \
    _PyLong_DIGIT_INIT(-2), \
    _PyLong_DIGIT_INIT(-1), \
    _PyLong_DIGIT_INIT(0), \
    _PyLong_DIGIT_INIT(1), \
    _PyLong_DIGIT_INIT(2), \
    _PyLong_DIGIT_INIT(3), \
    _PyLong_DIGIT_INIT(4), \
    _PyLong_DIGIT_INIT(5), \
    _PyLong_DIGIT_INIT(6), \
    _PyLong_DIGIT_INIT(7), \
    _PyLong_DIGIT_INIT(8), \
    _PyLong_DIGIT_INIT(9), \
    _PyLong_DIGIT_INIT(10), \
    _PyLong_DIGIT_INIT(11), \
    _PyLong_DIGIT_INIT(12), \
    _PyLong_DIGIT_INIT(13), \
    _PyLong_DIGIT_INIT(14), \
    _PyLong_DIGIT_INIT(15), \
    _PyLong_DIGIT_INIT(16), \
    _PyLong_DIGIT_INIT(17), \
    _PyLong_DIGIT_INIT(18), \
    _PyLong_DIGIT_INIT(19), \
    _PyLong_DIGIT_INIT(20), \
    _PyLong_DIGIT_INIT(21), \
    _PyLong_DIGIT_INIT(22), \
    _PyLong_DIGIT_INIT(23), \
    _PyLong_DIGIT_INIT(24), \
    _PyLong_DIGIT_INIT(25), \
    _PyLong_DIGIT_INIT(26), \
    _PyLong_DIGIT_INIT(27), \
    _PyLong_DIGIT_INIT(28), \
    _PyLong_DIGIT_INIT(29), \
    _PyLong_DIGIT_INIT(30), \
    _PyLong_DIGIT_INIT(31), \
    _PyLong_DIGIT_INIT(32), \
    _PyLong_DIGIT_INIT(33), \
    _PyLong_DIGIT_INIT(34), \
    _PyLong_DIGIT_INIT(35), \
    _PyLong_DIGIT_INIT(36), \
    _PyLong_DIGIT_INIT(37), \
    _PyLong_DIGIT_INIT(38), \
    _PyLong_DIGIT_INIT(39), \
    _PyLong_DIGIT_INIT(40), \
    _PyLong_DIGIT_INIT(41), \
    _PyLong_DIGIT_INIT(42), \
    _PyLong_DIGIT_INIT(43), \
    _PyLong_DIGIT_INIT(44), \
    _PyLong_DIGIT_INIT(45), \
    _PyLong_DIGIT_INIT(46), \
    _PyLong_DIGIT_INIT(47), \
    _PyLong_DIGIT_INIT(48), \
    _PyLong_DIGIT_INIT(49), \
    _PyLong_DIGIT_INIT(50), \
    _PyLong_DIGIT_INIT(51), \
    _PyLong_DIGIT_INIT(52), \
    _PyLong_DIGIT_INIT(53), \
    _PyLong_DIGIT_INIT(54), \
    _PyLong_DIGIT_INIT(55), \
    _PyLong_DIGIT_INIT(56), \
    _PyLong_DIGIT_INIT(57), \
    _PyLong_DIGIT_INIT(58), \
    _PyLong_DIGIT_INIT(59), \
    _PyLong_DIGIT_INIT(60), \
    _PyLong_DIGIT_INIT(61), \
    _PyLong_DIGIT_INIT(62), \
    _PyLong_DIGIT_INIT(63), \
    _PyLong_DIGIT_INIT(64), \
    _PyLong_DIGIT_INIT(65), \
    _PyLong_DIGIT_INIT(66), \
    _PyLong_DIGIT_INIT(67), \
    _PyLong_DIGIT_INIT(68), \
    _PyLong_DIGIT_INIT(69), \
    _PyLong_DIGIT_INIT(70), \
    _PyLong_DIGIT_INIT(71), \
    _PyLong_DIGIT_INIT(72), \
    _PyLong_DIGIT_INIT(73), \
    _PyLong_DIGIT_INIT(74), \
    _PyLong_DIGIT_INIT(75), \
    _PyLong_DIGIT_INIT(76), \
    _PyLong_DIGIT_INIT(77), \
    _PyLong_DIGIT_INIT(78), \
    _PyLong_DIGIT_INIT(79), \
    _PyLong_DIGIT_INIT(80), \
    _PyLong_DIGIT_INIT(81), \
    _PyLong_DIGIT_INIT(82), \
    _PyLong_DIGIT_INIT(83), \
    _PyLong_DIGIT_INIT(84), \
    _PyLong_DIGIT_INIT(85), \
    _PyLong_DIGIT_INIT(86), \
    _PyLong_DIGIT_INIT(87), \
    _PyLong_DIGIT_INIT(88), \
    _PyLong_DIGIT_INIT(89), \
    _PyLong_DIGIT_INIT(90), \
    _PyLong_DIGIT_INIT(91), \
    _PyLong_DIGIT_INIT(92), \
    _PyLong_DIGIT_INIT(93), \
    _PyLong_DIGIT_INIT(94), \
    _PyLong_DIGIT_INIT(95), \
    _PyLong_DIGIT_INIT(96), \
    _PyLong_DIGIT_INIT(97), \
    _PyLong_DIGIT_INIT(98), \
    _PyLong_DIGIT_INIT(99), \
    _PyLong_DIGIT_INIT(100), \
    _PyLong_DIGIT_INIT(101), \
    _PyLong_DIGIT_INIT(102), \
    _PyLong_DIGIT_INIT(103), \
    _PyLong_DIGIT_INIT(104), \
    _PyLong_DIGIT_INIT(105), \
    _PyLong_DIGIT_INIT(106), \
    _PyLong_DIGIT_INIT(107), \
    _PyLong_DIGIT_INIT(108), \
    _PyLong_DIGIT_INIT(109), \
    _PyLong_DIGIT_INIT(110), \
    _PyLong_DIGIT_INIT(111), \
    _PyLong_DIGIT_INIT(112), \
    _PyLong_DIGIT_INIT(113), \
    _PyLong_DIGIT_INIT(114), \
    _PyLong_DIGIT_INIT(115), \
    _PyLong_DIGIT_INIT(116), \
    _PyLong_DIGIT_INIT(117), \
    _PyLong_DIGIT_INIT(118), \
    _PyLong_DIGIT_INIT(119), \
    _PyLong_DIGIT_INIT(120), \
    _PyLong_DIGIT_INIT(121), \
    _PyLong_DIGIT_INIT(122), \
    _PyLong_DIGIT_INIT(123), \
    _PyLong_DIGIT_INIT(124), \
    _PyLong_DIGIT_INIT(125), \
    _PyLong_DIGIT_INIT(126), \
    _PyLong_DIGIT_INIT(127), \
    _PyLong_DIGIT_INIT(128), \
    _PyLong_DIGIT_INIT(129), \
    _PyLong_DIGIT_INIT(130), \
    _PyLong_DIGIT_INIT(131), \
    _PyLong_DIGIT_INIT(132), \
    _PyLong_DIGIT_INIT(133), \
    _PyLong_DIGIT_INIT(134), \
    _PyLong_DIGIT_INIT(135), \
    _PyLong_DIGIT_INIT(136), \
    _PyLong_DIGIT_INIT(137), \
    _PyLong_DIGIT_INIT(138), \
    _PyLong_DIGIT_INIT(139), \
    _PyLong_DIGIT_INIT(140), \
    _PyLong_DIGIT_INIT(141), \
    _PyLong_DIGIT_INIT(142), \
    _PyLong_DIGIT_INIT(143), \
    _PyLong_DIGIT_INIT(144), \
    _PyLong_DIGIT_INIT(145), \
    _PyLong_DIGIT_INIT(146), \
    _PyLong_DIGIT_INIT(147), \
    _PyLong_DIGIT_INIT(148), \
    _PyLong_DIGIT_INIT(149), \
    _PyLong_DIGIT_INIT(150), \
    _PyLong_DIGIT_INIT(151), \
    _PyLong_DIGIT_INIT(152), \
    _PyLong_DIGIT_INIT(153), \
    _PyLong_DIGIT_INIT(154), \
    _PyLong_DIGIT_INIT(155), \
    _PyLong_DIGIT_INIT(156), \
    _PyLong_DIGIT_INIT(157), \
    _PyLong_DIGIT_INIT(158), \
    _PyLong_DIGIT_INIT(159), \
    _PyLong_DIGIT_INIT(160), \
    _PyLong_DIGIT_INIT(161), \
    _PyLong_DIGIT_INIT(162), \
    _PyLong_DIGIT_INIT(163), \
    _PyLong_DIGIT_INIT(164), \
    _PyLong_DIGIT_INIT(165), \
    _PyLong_DIGIT_INIT(166), \
    _PyLong_DIGIT_INIT(167), \
    _PyLong_DIGIT_INIT(168), \
    _PyLong_DIGIT_INIT(169), \
    _PyLong_DIGIT_INIT(170), \
    _PyLong_DIGIT_INIT(171), \
    _PyLong_DIGIT_INIT(172), \
    _PyLong_DIGIT_INIT(173), \
    _PyLong_DIGIT_INIT(174), \
    _PyLong_DIGIT_INIT(175), \
    _PyLong_DIGIT_INIT(176), \
    _PyLong_DIGIT_INIT(177), \
    _PyLong_DIGIT_INIT(178), \
    _PyLong_DIGIT_INIT(179), \
    _PyLong_DIGIT_INIT(180), \
    _PyLong_DIGIT_INIT(181), \
    _PyLong_DIGIT_INIT(182), \
    _PyLong_DIGIT_INIT(183), \
    _PyLong_DIGIT_INIT(184), \
    _PyLong_DIGIT_INIT(185), \
    _PyLong_DIGIT_INIT(186), \
    _PyLong_DIGIT_INIT(187), \
    _PyLong_DIGIT_INIT(188), \
    _PyLong_DIGIT_INIT(189), \
    _PyLong_DIGIT_INIT(190), \
    _PyLong_DIGIT_INIT(191), \
    _PyLong_DIGIT_INIT(192), \
    _PyLong_DIGIT_INIT(193), \
    _PyLong_DIGIT_INIT(194), \
    _PyLong_DIGIT_INIT(195), \
    _PyLong_DIGIT_INIT(196), \
    _PyLong_DIGIT_INIT(197), \
    _PyLong_DIGIT_INIT(198), \
    _PyLong_DIGIT_INIT(199), \
    _PyLong_DIGIT_INIT(200), \
    _PyLong_DIGIT_INIT(201), \
    _PyLong_DIGIT_INIT(202), \
    _PyLong_DIGIT_INIT(203), \
    _PyLong_DIGIT_INIT(204), \
    _PyLong_DIGIT_INIT(205), \
    _PyLong_DIGIT_INIT(206), \
    _PyLong_DIGIT_INIT(207), \
    _PyLong_DIGIT_INIT(208), \
    _PyLong_DIGIT_INIT(209), \
    _PyLong_DIGIT_INIT(210), \
    _PyLong_DIGIT_INIT(211), \
    _PyLong_DIGIT_INIT(212), \
    _PyLong_DIGIT_INIT(213), \
    _PyLong_DIGIT_INIT(214), \
    _PyLong_DIGIT_INIT(215), \
    _PyLong_DIGIT_INIT(216), \
    _PyLong_DIGIT_INIT(217), \
    _PyLong_DIGIT_INIT(218), \
    _PyLong_DIGIT_INIT(219), \
    _PyLong_DIGIT_INIT(220), \
    _PyLong_DIGIT_INIT(221), \
    _PyLong_DIGIT_INIT(222), \
    _PyLong_DIGIT_INIT(223), \
    _PyLong_DIGIT_INIT(224), \
    _PyLong_DIGIT_INIT(225), \
    _PyLong_DIGIT_INIT(226), \
    _PyLong_DIGIT_INIT(227), \
    _PyLong_DIGIT_INIT(228), \
    _PyLong_DIGIT_INIT(229), \
    _PyLong_DIGIT_INIT(230), \
    _PyLong_DIGIT_INIT(231), \
    _PyLong_DIGIT_INIT(232), \
    _PyLong_DIGIT_INIT(233), \
    _PyLong_DIGIT_INIT(234), \
    _PyLong_DIGIT_INIT(235), \
    _PyLong_DIGIT_INIT(236), \
    _PyLong_DIGIT_INIT(237), \
    _PyLong_DIGIT_INIT(238), \
    _PyLong_DIGIT_INIT(239), \
    _PyLong_DIGIT_INIT(240), \
    _PyLong_DIGIT_INIT(241), \
    _PyLong_DIGIT_INIT(242), \
    _PyLong_DIGIT_INIT(243), \
    _PyLong_DIGIT_INIT(244), \
    _PyLong_DIGIT_INIT(245), \
    _PyLong_DIGIT_INIT(246), \
    _PyLong_DIGIT_INIT(247), \
    _PyLong_DIGIT_INIT(248), \
    _PyLong_DIGIT_INIT(249), \
    _PyLong_DIGIT_INIT(250), \
    _PyLong_DIGIT_INIT(251), \
    _PyLong_DIGIT_INIT(252), \
    _PyLong_DIGIT_INIT(253), \
    _PyLong_DIGIT_INIT(254), \
    _PyLong_DIGIT_INIT(255), \
    _PyLong_DIGIT_INIT(256), \
}

#define _Py_bytes_characters_INIT { \
    _PyBytes_CHAR_INIT(0), \
    _PyBytes_CHAR_INIT(1), \
    _PyBytes_CHAR_INIT(2), \
    _PyBytes_CHAR_INIT(3), \
    _PyBytes_CHAR_INIT(4), \
    _PyBytes_CHAR_INIT(5), \
    _PyBytes_CHAR_INIT(6), \
    _PyBytes_CHAR_INIT(7), \
    _PyBytes_CHAR_INIT(8), \
    _PyBytes_CHAR_INIT(9), \
    _PyBytes_CHAR_INIT(10), \
    _PyBytes_CHAR_INIT(11), \
    _PyBytes_CHAR_INIT(12), \
    _PyBytes_CHAR_INIT(13), \
    _PyBytes_CHAR_INIT(14), \
    _PyBytes_CHAR_INIT(15), \
    _PyBytes_CHAR_INIT(16), \
    _PyBytes_CHAR_INIT(17), \
    _PyBytes_CHAR_INIT(18), \
    _PyBytes_CHAR_INIT(19), \
    _PyBytes_CHAR_INIT(20), \
    _PyBytes_CHAR_INIT(21), \
    _PyBytes_CHAR_INIT(22), \
    _PyBytes_CHAR_INIT(23), \
    _PyBytes_CHAR_INIT(24), \
    _PyBytes_CHAR_INIT(25), \
    _PyBytes_CHAR_INIT(26), \
    _PyBytes_CHAR_INIT(27), \
    _PyBytes_CHAR_INIT(28), \
    _PyBytes_CHAR_INIT(29), \
    _PyBytes_CHAR_INIT(30), \
    _PyBytes_CHAR_INIT(31), \
    _PyBytes_CHAR_INIT(32), \
    _PyBytes_CHAR_INIT(33), \
    _PyBytes_CHAR_INIT(34), \
    _PyBytes_CHAR_INIT(35), \
    _PyBytes_CHAR_INIT(36), \
    _PyBytes_CHAR_INIT(37), \
    _PyBytes_CHAR_INIT(38), \
    _PyBytes_CHAR_INIT(39), \
    _PyBytes_CHAR_INIT(40), \
    _PyBytes_CHAR_INIT(41), \
    _PyBytes_CHAR_INIT(42), \
    _PyBytes_CHAR_INIT(43), \
    _PyBytes_CHAR_INIT(44), \
    _PyBytes_CHAR_INIT(45), \
    _PyBytes_CHAR_INIT(46), \
    _PyBytes_CHAR_INIT(47), \
    _PyBytes_CHAR_INIT(48), \
    _PyBytes_CHAR_INIT(49), \
    _PyBytes_CHAR_INIT(50), \
    _PyBytes_CHAR_INIT(51), \
    _PyBytes_CHAR_INIT(52), \
    _PyBytes_CHAR_INIT(53), \
    _PyBytes_CHAR_INIT(54), \
    _PyBytes_CHAR_INIT(55), \
    _PyBytes_CHAR_INIT(56), \
    _PyBytes_CHAR_INIT(57), \
    _PyBytes_CHAR_INIT(58), \
    _PyBytes_CHAR_INIT(59), \
    _PyBytes_CHAR_INIT(60), \
    _PyBytes_CHAR_INIT(61), \
    _PyBytes_CHAR_INIT(62), \
    _PyBytes_CHAR_INIT(63), \
    _PyBytes_CHAR_INIT(64), \
    _PyBytes_CHAR_INIT(65), \
    _PyBytes_CHAR_INIT(66), \
    _PyBytes_CHAR_INIT(67), \
    _PyBytes_CHAR_INIT(68), \
    _PyBytes_CHAR_INIT(69), \
    _PyBytes_CHAR_INIT(70), \
    _PyBytes_CHAR_INIT(71), \
    _PyBytes_CHAR_INIT(72), \
    _PyBytes_CHAR_INIT(73), \
    _PyBytes_CHAR_INIT(74), \
    _PyBytes_CHAR_INIT(75), \
    _PyBytes_CHAR_INIT(76), \
    _PyBytes_CHAR_INIT(77), \
    _PyBytes_CHAR_INIT(78), \
    _PyBytes_CHAR_INIT(79), \
    _PyBytes_CHAR_INIT(80), \
    _PyBytes_CHAR_INIT(81), \
    _PyBytes_CHAR_INIT(82), \
    _PyBytes_CHAR_INIT(83), \
    _PyBytes_CHAR_INIT(84), \
    _PyBytes_CHAR_INIT(85), \
    _PyBytes_CHAR_INIT(86), \
    _PyBytes_CHAR_INIT(87), \
    _PyBytes_CHAR_INIT(88), \
    _PyBytes_CHAR_INIT(89), \
    _PyBytes_CHAR_INIT(90), \
    _PyBytes_CHAR_INIT(91), \
    _PyBytes_CHAR_INIT(92), \
    _PyBytes_CHAR_INIT(93), \
    _PyBytes_CHAR_INIT(94), \
    _PyBytes_CHAR_INIT(95), \
    _PyBytes_CHAR_INIT(96), \
    _PyBytes_CHAR_INIT(97), \
    _PyBytes_CHAR_INIT(98), \
    _PyBytes_CHAR_INIT(99), \
    _PyBytes_CHAR_INIT(100), \
    _PyBytes_CHAR_INIT(101), \
    _PyBytes_CHAR_INIT(102), \
    _PyBytes_CHAR_INIT(103), \
    _PyBytes_CHAR_INIT(104), \
    _PyBytes_CHAR_INIT(105), \
    _PyBytes_CHAR_INIT(106), \
    _PyBytes_CHAR_INIT(107), \
    _PyBytes_CHAR_INIT(108), \
    _PyBytes_CHAR_INIT(109), \
    _PyBytes_CHAR_INIT(110), \
    _PyBytes_CHAR_INIT(111), \
    _PyBytes_CHAR_INIT(112), \
    _PyBytes_CHAR_INIT(113), \
    _PyBytes_CHAR_INIT(114), \
    _PyBytes_CHAR_INIT(115), \
    _PyBytes_CHAR_INIT(116), \
    _PyBytes_CHAR_INIT(117), \
    _PyBytes_CHAR_INIT(118), \
    _PyBytes_CHAR_INIT(119), \
    _PyBytes_CHAR_INIT(120), \
    _PyBytes_CHAR_INIT(121), \
    _PyBytes_CHAR_INIT(122), \
    _PyBytes_CHAR_INIT(123), \
    _PyBytes_CHAR_INIT(124), \
    _PyBytes_CHAR_INIT(125), \
    _PyBytes_CHAR_INIT(126), \
    _PyBytes_CHAR_INIT(127), \
    _PyBytes_CHAR_INIT(128), \
    _PyBytes_CHAR_INIT(129), \
    _PyBytes_CHAR_INIT(130), \
    _PyBytes_CHAR_INIT(131), \
    _PyBytes_CHAR_INIT(132), \
    _PyBytes_CHAR_INIT(133), \
    _PyBytes_CHAR_INIT(134), \
    _PyBytes_CHAR_INIT(135), \
    _PyBytes_CHAR_INIT(136), \
    _PyBytes_CHAR_INIT(137), \
    _PyBytes_CHAR_INIT(138), \
    _PyBytes_CHAR_INIT(139), \
    _PyBytes_CHAR_INIT(140), \
    _PyBytes_CHAR_INIT(141), \
    _PyBytes_CHAR_INIT(142), \
    _PyBytes_CHAR_INIT(143), \
    _PyBytes_CHAR_INIT(144), \
    _PyBytes_CHAR_INIT(145), \
    _PyBytes_CHAR_INIT(146), \
    _PyBytes_CHAR_INIT(147), \
    _PyBytes_CHAR_INIT(148), \
    _PyBytes_CHAR_INIT(149), \
    _PyBytes_CHAR_INIT(150), \
    _PyBytes_CHAR_INIT(151), \
    _PyBytes_CHAR_INIT(152), \
    _PyBytes_CHAR_INIT(153), \
    _PyBytes_CHAR_INIT(154), \
    _PyBytes_CHAR_INIT(155), \
    _PyBytes_CHAR_INIT(156), \
    _PyBytes_CHAR_INIT(157), \
    _PyBytes_CHAR_INIT(158), \
    _PyBytes_CHAR_INIT(159), \
    _PyBytes_CHAR_INIT(160), \
    _PyBytes_CHAR_INIT(161), \
    _PyBytes_CHAR_INIT(162), \
    _PyBytes_CHAR_INIT(163), \
    _PyBytes_CHAR_INIT(164), \
    _PyBytes_CHAR_INIT(165), \
    _PyBytes_CHAR_INIT(166), \
    _PyBytes_CHAR_INIT(167), \
    _PyBytes_CHAR_INIT(168), \
    _PyBytes_CHAR_INIT(169), \
    _PyBytes_CHAR_INIT(170), \
    _PyBytes_CHAR_INIT(171), \
    _PyBytes_CHAR_INIT(172), \
    _PyBytes_CHAR_INIT(173), \
    _PyBytes_CHAR_INIT(174), \
    _PyBytes_CHAR_INIT(175), \
    _PyBytes_CHAR_INIT(176), \
    _PyBytes_CHAR_INIT(177), \
    _PyBytes_CHAR_INIT(178), \
    _PyBytes_CHAR_INIT(179), \
    _PyBytes_CHAR_INIT(180), \
    _PyBytes_CHAR_INIT(181), \
    _PyBytes_CHAR_INIT(182), \
    _PyBytes_CHAR_INIT(183), \
    _PyBytes_CHAR_INIT(184), \
    _PyBytes_CHAR_INIT(185), \
    _PyBytes_CHAR_INIT(186), \
    _PyBytes_CHAR_INIT(187), \
    _PyBytes_CHAR_INIT(188), \
    _PyBytes_CHAR_INIT(189), \
    _PyBytes_CHAR_INIT(190), \
    _PyBytes_CHAR_INIT(191), \
    _PyBytes_CHAR_INIT(192), \
    _PyBytes_CHAR_INIT(193), \
    _PyBytes_CHAR_INIT(194), \
    _PyBytes_CHAR_INIT(195), \
    _PyBytes_CHAR_INIT(196), \
    _PyBytes_CHAR_INIT(197), \
    _PyBytes_CHAR_INIT(198), \
    _PyBytes_CHAR_INIT(199), \
    _PyBytes_CHAR_INIT(200), \
    _PyBytes_CHAR_INIT(201), \
    _PyBytes_CHAR_INIT(202), \
    _PyBytes_CHAR_INIT(203), \
    _PyBytes_CHAR_INIT(204), \
    _PyBytes_CHAR_INIT(205), \
    _PyBytes_CHAR_INIT(206), \
    _PyBytes_CHAR_INIT(207), \
    _PyBytes_CHAR_INIT(208), \
    _PyBytes_CHAR_INIT(209), \
    _PyBytes_CHAR_INIT(210), \
    _PyBytes_CHAR_INIT(211), \
    _PyBytes_CHAR_INIT(212), \
    _PyBytes_CHAR_INIT(213), \
    _PyBytes_CHAR_INIT(214), \
    _PyBytes_CHAR_INIT(215), \
    _PyBytes_CHAR_INIT(216), \
    _PyBytes_CHAR_INIT(217), \
    _PyBytes_CHAR_INIT(218), \
    _PyBytes_CHAR_INIT(219), \
    _PyBytes_CHAR_INIT(220), \
    _PyBytes_CHAR_INIT(221), \
    _PyBytes_CHAR_INIT(222), \
    _PyBytes_CHAR_INIT(223), \
    _PyBytes_CHAR_INIT(224), \
    _PyBytes_CHAR_INIT(225), \
    _PyBytes_CHAR_INIT(226), \
    _PyBytes_CHAR_INIT(227), \
    _PyBytes_CHAR_INIT(228), \
    _PyBytes_CHAR_INIT(229), \
    _PyBytes_CHAR_INIT(230), \
    _PyBytes_CHAR_INIT(231), \
    _PyBytes_CHAR_INIT(232), \
    _PyBytes_CHAR_INIT(233), \
    _PyBytes_CHAR_INIT(234), \
    _PyBytes_CHAR_INIT(235), \
    _PyBytes_CHAR_INIT(236), \
    _PyBytes_CHAR_INIT(237), \
    _PyBytes_CHAR_INIT(238), \
    _PyBytes_CHAR_INIT(239), \
    _PyBytes_CHAR_INIT(240), \
    _PyBytes_CHAR_INIT(241), \
    _PyBytes_CHAR_INIT(242), \
    _PyBytes_CHAR_INIT(243), \
    _PyBytes_CHAR_INIT(244), \
    _PyBytes_CHAR_INIT(245), \
    _PyBytes_CHAR_INIT(246), \
    _PyBytes_CHAR_INIT(247), \
    _PyBytes_CHAR_INIT(248), \
    _PyBytes_CHAR_INIT(249), \
    _PyBytes_CHAR_INIT(250), \
    _PyBytes_CHAR_INIT(251), \
    _PyBytes_CHAR_INIT(252), \
    _PyBytes_CHAR_INIT(253), \
    _PyBytes_CHAR_INIT(254), \
    _PyBytes_CHAR_INIT(255), \
}

#define _Py_str_literals_INIT { \
    INIT_STR(anon_dictcomp, "<dictcomp>"), \
    INIT_STR(anon_genexpr, "<genexpr>"), \
    INIT_STR(anon_lambda, "<lambda>"), \
    INIT_STR(anon_listcomp, "<listcomp>"), \
    INIT_STR(anon_module, "<module>"), \
    INIT_STR(anon_null, "<NULL>"), \
    INIT_STR(anon_setcomp, "<setcomp>"), \
    INIT_STR(anon_string, "<string>"), \
    INIT_STR(anon_unknown, "<unknown>"), \
    INIT_STR(dbl_close_br, "}}"), \
    INIT_STR(dbl_open_br, "{{"), \
    INIT_STR(dbl_percent, "%%"), \
    INIT_STR(defaults, ".defaults"), \
    IN