@copyright  This  file  is   a  part  of  the   programming  language  Ruby.
 *             Permission  is hereby  granted,  to  either redistribute  and/or
 *             modify this file, provided that  the conditions mentioned in the
 *             file COPYING are met.  Consult the file for details.
 * @warning    Symbols   prefixed  with   either  `RBIMPL`   or  `rbimpl`   are
 *             implementation details.   Don't take  them as canon.  They could
 *             rapidly appear then vanish.  The name (path) of this header file
 *             is also an  implementation detail.  Do not expect  it to persist
 *             at the place it is now.  Developers are free to move it anywhere
 *             anytime at will.
 * @note       To  ruby-core:  remember  that   this  header  can  be  possibly
 *             recursively included  from extension  libraries written  in C++.
 *             Do not  expect for  instance `__VA_ARGS__` is  always available.
 *             We assume C99  for ruby itself but we don't  assume languages of
 *             extension libraries.  They could be written in C++98.
 * @brief      Function overloads to issue warnings around #ANYARGS.
 *
 * For instance ::rb_define_method  takes a pointer to  #ANYARGS -ed functions,
 * which in  fact varies 18  different prototypes.   We still need  to preserve
 * #ANYARGS for storages but why not check the consistencies if possible.  With
 * those complex macro overlays defined in  this header file, use of a function
 * pointer gets checked against the corresponding arity argument.
 *
 * ### Q&A ###
 *
 * - Q: Where did the magic number "18" came from in the description above?
 *
 * - A: Count the case branch of `vm_method.c:call_cfunc_invoker_func()`.  Note
 *      also that the 18  branches has lasted for at least  25 years.  See also
 *      commit 200e0ee2fd3c1c006c528874a88f684447215524.
 *
 * - Q: What is this `__weakref__` thing?
 *
 * - A: That is a kind of function overloading mechanism that GCC provides.  In
 *      this   case  for   instance  `rb_define_method_00`   is  an   alias  of
 *      ::rb_define_method, with a strong type.
 *
 * - Q: What is this `__transparent_union__` thing?
 *
 *   A: That  is  another  kind  of function  overloading  mechanism  that  GCC
 *      provides.   In this  case  the attributed  function  pointer is  either
 *      `VALUE(*)(int,VALUE*,VALUE)` or `VALUE(*)(int,const VALUE*,VALUE)`.
 *
 *      This is better than `void*` or #ANYARGS because we can reject all other
 *      possibilities than the two.
 *
 * - Q: What does this #rb_define_method macro mean?
 *
 * - A: It  selects  appropriate  alias  of  the  ::rb_define_method  function,
 *      depending on the last (arity) argument.
 *
 * - Q: Why the special case for ::rb_f_notimplement ?
 *
 * - A: Function   pointer  to   ::rb_f_notimplement   is   special  cased   in
 *      `vm_method.c:rb_add_method_cfunc()`.   That should  be  handled by  the
 *      `__builtin_choose_expr`   chain  inside   of  #rb_define_method   macro
 *      expansion.      In    order     to    do     so,    comparison     like
 *      `(func == rb_f_notimplement)`        is        inappropriate        for
 *      `__builtin_choose_expr`'s  expression  (which  must be  a  compile-time
 *      integer constant  but the address  of ::rb_f_notimplement is  not fixed
 *      until      the      linker).        Instead      we      are      using
 *      `__builtin_types_compatible_p`, and in doing  so we need to distinguish
 *      ::rb_f_notimplement from others, by type.
 */
#include "ruby/internal/attr/maybe_unused.h"
#include "ruby/internal/attr/nonnull.h"
#include "ruby/internal/attr/weakref.h"
#include "ruby/internal/cast.h"
#include "ruby/internal/config.h"
#include "ruby/internal/has/attribute.h"
#include "ruby/internal/intern/class.h"
#include "ruby/internal/intern/vm.h"
#include "ruby/internal/method.h"
#include "ruby/internal/value.h"
#include "ruby/backward/2/stdarg.h"

#if defined(__cplusplus)
# include "ruby/backward/cxxanyargs.hpp"

#elif defined(_WIN32) || defined(__CYGWIN__)
# /* Skip due to [Bug #16134] */
# define RBIMPL_CAST_FN_PTR 1

#elif ! RBIMPL_HAS_ATTRIBUTE(transparent_union)
# /* :TODO: improve here, please find a way to support. */
# define RBIMPL_CAST_FN_PTR 1

#elif ! defined(HAVE_VA_ARGS_MACRO)
# /* :TODO: improve here, please find a way to support. */
# define RBIMPL_CAST_FN_PTR 1

#else
# /** @cond INTERNAL_MACRO */
# if ! defined(HAVE_BUILTIN___BUILTIN_TYPES_COMPATIBLE_P)
#  define RBIMPL_CFUNC_IS_rb_f_notimplement(f) 0
# else
#  define RBIMPL_CFUNC_IS_rb_f_notimplement(f) \
    __builtin_types_compatible_p(             \
        __typeof__(f),                        \
        __typeof__(rb_f_notimplement))
# endif

# if ! defined(HAVE_BUILTIN___BUILTIN_CHOOSE_EXPR_CONSTANT_P)
#  define RBIMPL_ANYARGS_DISPATCH(expr, truthy, falsy) (falsy)
# else
#  define RBIMPL_ANYARGS_DISPATCH(expr, truthy, falsy) \
    __builtin_choose_expr(                            \
        __builtin_choose_expr(                        \
            __builtin_constant_p(expr),               \
            (expr), 0),                               \
        (truthy), (falsy))
# endif

# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_m2(n) RBIMPL_ANYARGS_DISPATCH((n) == -2, rb_define_singleton_method_m2, rb_define_singleton_method_m3)
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_m1(n) RBIMPL_ANYARGS_DISPATCH((n) == -1, rb_define_singleton_method_m1, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_m2(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_00(n) RBIMPL_ANYARGS_DISPATCH((n) ==  0, rb_define_singleton_method_00, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_m1(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_01(n) RBIMPL_ANYARGS_DISPATCH((n) ==  1, rb_define_singleton_method_01, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_00(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_02(n) RBIMPL_ANYARGS_DISPATCH((n) ==  2, rb_define_singleton_method_02, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_01(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_03(n) RBIMPL_ANYARGS_DISPATCH((n) ==  3, rb_define_singleton_method_03, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_02(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_04(n) RBIMPL_ANYARGS_DISPATCH((n) ==  4, rb_define_singleton_method_04, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_03(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_05(n) RBIMPL_ANYARGS_DISPATCH((n) ==  5, rb_define_singleton_method_05, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_04(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_06(n) RBIMPL_ANYARGS_DISPATCH((n) ==  6, rb_define_singleton_method_06, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_05(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_07(n) RBIMPL_ANYARGS_DISPATCH((n) ==  7, rb_define_singleton_method_07, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_06(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_08(n) RBIMPL_ANYARGS_DISPATCH((n) ==  8, rb_define_singleton_method_08, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_07(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_09(n) RBIMPL_ANYARGS_DISPATCH((n) ==  9, rb_define_singleton_method_09, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_08(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_10(n) RBIMPL_ANYARGS_DISPATCH((n) == 10, rb_define_singleton_method_10, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_09(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_11(n) RBIMPL_ANYARGS_DISPATCH((n) == 11, rb_define_singleton_method_11, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_10(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_12(n) RBIMPL_ANYARGS_DISPATCH((n) == 12, rb_define_singleton_method_12, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_11(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_13(n) RBIMPL_ANYARGS_DISPATCH((n) == 13, rb_define_singleton_method_13, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_12(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_14(n) RBIMPL_ANYARGS_DISPATCH((n) == 14, rb_define_singleton_method_14, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_13(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_15(n) RBIMPL_ANYARGS_DISPATCH((n) == 15, rb_define_singleton_method_15, RBIMPL_ANYARGS_DISPATCH_rb_define_singleton_method_14(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_protected_method_m2(n) RBIMPL_ANYARGS_DISPATCH((n) == -2, rb_define_protected_method_m2, rb_define_protected_method_m3)
# define RBIMPL_ANYARGS_DISPATCH_rb_define_protected_method_m1(n) RBIMPL_ANYARGS_DISPATCH((n) == -1, rb_define_protected_method_m1, RBIMPL_ANYARGS_DISPATCH_rb_define_protected_method_m2(n))
# define RBIMPL_ANYARGS_DISPATCH_rb_define_protected_method_00(n) RBIMPL_ANYARGS_DISPATCH((n) ==  0, rb_define_protected_method_00, RBIMPL_AN