So,  the format  string must  be
 * followed by three variable references, which  are to be assigned to captured
 * arguments.  For omitted arguments, variables are set to ::RUBY_Qnil.  `NULL`
 * can be put  in place of a variable reference,  which means the corresponding
 * captured argument(s) should be just dropped.
 *
 * The number of  given arguments, excluding an option hash  or iterator block,
 * is returned.
 *
 * @param[in]   argc          Length of `argv`.
 * @param[in]   argv          Pointer to the arguments to parse.
 * @param[in]   fmt           Format, in the language described above.
 * @param[out]  ...           Variables to fill in.
 * @exception   rb_eFatal     Malformed `fmt`.
 * @exception   rb_eArgError  Arity mismatch.
 * @return      Actually parsed number of given arguments.
 * @post        Each  values  passed to  `argv`  is  filled into  the  variadic
 *              arguments, according to the format.
 */
int rb_scan_args(int argc, const VALUE *argv, const char *fmt, ...);

RBIMPL_ATTR_NONNULL((3, 4))
/**
 * Identical to rb_scan_args(), except it also accepts `kw_splat`.
 *
 * @param[in]   kw_splat      How to understand the keyword arguments.
 *   - RB_SCAN_ARGS_PASS_CALLED_KEYWORDS: Same behaviour as rb_scan_args().
 *   - RB_SCAN_ARGS_KEYWORDS:             The final argument is a kwarg.
 *   - RB_SCAN_ARGS_LAST_HASH_KEYWORDS:   The final argument is a kwarg, iff it
 *                                        is a hash.
 * @param[in]   argc          Length of `argv`.
 * @param[in]   argv          Pointer to the arguments to parse.
 * @param[in]   fmt           Format, in the language described above.
 * @param[out]  ...           Variables to fill in.
 * @exception   rb_eFatal     Malformed `fmt`.
 * @exception   rb_eArgError  Arity mismatch.
 * @return      Actually parsed number of given arguments.
 * @post        Each  values  passed to  `argv`  is  filled into  the  variadic
 *              arguments, according to the format.
 */
int rb_scan_args_kw(int kw_splat, int argc, const VALUE *argv, const char *fmt, ...);

RBIMPL_ATTR_ERROR(("bad scan arg format"))
/**
 * @private
 *
 * This is  an implementation  detail of rb_scan_args().   People don't  use it
 * directly.
 */
void rb_scan_args_bad_format(const char*);

RBIMPL_ATTR_ERROR(("variable argument length doesn't match"))
/**
 * @private
 *
 * This is  an implementation  detail of rb_scan_args().   People don't  use it
 * directly.
 */
void rb_scan_args_length_mismatch(const char*,int);

RBIMPL_SYMBOL_EXPORT_END()

/** @cond INTERNAL_MACRO */

/* If we could use constexpr the following macros could be inline functions
 * ... but sadly we cannot. */

#define rb_scan_args_isdigit(c) (RBIMPL_CAST((unsigned char)((c)-'0'))<10)

#define rb_scan_args_count_end(fmt, ofs, vari) \
    ((fmt)[ofs] ? -1 : (vari))

#define rb_scan_args_count_block(fmt, ofs, vari) \
    ((fmt)[ofs]!='&' ? \
     rb_scan_args_count_end(fmt, ofs, vari) : \
     rb_scan_args_count_end(fmt, (ofs)+1, (vari)+1))

#define rb_scan_args_count_hash(fmt, ofs, vari) \
    ((fmt)[ofs]!=':' ? \
     rb_scan_args_count_block(fmt, ofs, vari) : \
     rb_scan_args_count_block(fmt, (ofs)+1, (vari)+1))

#define rb_scan_args_count_trail(fmt, ofs, vari) \
    (!rb_scan_args_isdigit((fmt)[ofs]) ? \
     rb_scan_args_count_hash(fmt, ofs, vari) : \
     rb_scan_args_count_hash(fmt, (ofs)+1, (vari)+((fmt)[ofs]-'0')))

#define rb_scan_args_count_var(fmt, ofs, vari) \
    ((fmt)[ofs]!='*' ? \
     rb_scan_args_count_trail(fmt, ofs, vari) : \
     rb_scan_args_count_trail(fmt, (ofs)+1, (vari)+1))

#define rb_scan_args_count_opt(fmt, ofs, vari) \
    (!rb_scan_args_isdigit((fmt)[ofs]) ? \
     rb_scan_args_count_var(fmt, ofs, vari) : \
     rb_scan_args_count_var(fmt, (ofs)+1, (vari)+(fmt)[ofs]-'0'))

#define rb_scan_args_count_lead(fmt, ofs, vari) \
    (!rb_scan_args_isdigit((fmt)[ofs]) ? \
     rb_scan_args_count_var(fmt, ofs, vari) : \
     rb_scan_args_count_opt(fmt, (ofs)+1, (vari)+(fmt)[ofs]-'0'))

#define rb_scan_args_count(fmt) rb_scan_args_count_lead(fmt, 0, 0)

#if RBIMPL_HAS_ATTRIBUTE(diagnose_if)
# /* Assertions done in the attribute. */
# define rb_scan_args_verify(fmt, varc) RBIMPL_ASSERT_NOTHING
#else
# /* At  one sight  it _seems_  the expressions  below could  be written  using
#  * static  assertions.  The  reality is  no, they  don't.  Because  fmt is  a
#  * string literal,  any operations  against fmt  cannot produce  the "integer
#  * constant  expression"s,  as  defined  in  ISO/IEC  9899:2018  section  6.6
#  * paragraph #6.  Static assertions need such integer constant expressions as
#  * defined in ISO/IEC 9899:2018 section 6.7.10 paragraph #3.
#  *
#  * GCC nonetheless constant-folds this into a no-op, though. */
# define rb_scan_args_verify(fmt, varc) \
    (sizeof(char[1-2*(rb_scan_args_count(fmt)<0)])!=1 ? \
     rb_scan_args_bad_format(fmt) : \
     sizeof(char[1-2*(rb_scan_args_count(fmt)!=(varc))])!=1 ? \
     rb_scan_args_length_mismatch(fmt, varc) : \
     RBIMPL_ASSERT_NOTHING)
#endif

static inline bool
rb_scan_args_keyword_p(int kw_flag, VALUE last)
{
    switch (kw_flag) {
      case RB_SCAN_ARGS_PASS_CALLED_KEYWORDS:
        return !! rb_keyword_given_p();
      case RB_SCAN_ARGS_KEYWORDS:
        return true;
      case RB_SCAN_ARGS_LAST_HASH_KEYWORDS:
        return RB_TYPE_P(last, T_HASH);
      default:
        return false;
    }
}

RBIMPL_ATTR_FORCEINLINE()
static bool
rb_scan_args_lead_p(const char *fmt)
{
    return rb_scan_args_isdigit(fmt[0]);
}

RBIMPL_ATTR_FORCEINLINE()
static int
rb_scan_args_n_lead(const char *fmt)
{
    return (rb_scan_args_lead_p(fmt) ? fmt[0]-'0' : 0);
}

RBIMPL_ATTR_FORCEINLINE()
static bool
rb_scan_args_opt_p(const char *fmt)
{
    return (rb_scan_args_lead_p(fmt) && rb_scan_args_isdigit(fmt[1]));
}

RBIMPL_ATTR_FORCEINLINE()
static int
rb_scan_args_n_opt(const char *fmt)
{
    return (rb_scan_args_opt_p(fmt) ? fmt[1]-'0' : 0);
}

RBIMPL_ATTR_FORCEINLINE()
static int
rb_scan_args_var_idx(const char *fmt)
{
    return (!rb_scan_args_lead_p(fmt) ? 0 : !rb_scan_args_isdigit(fmt[1]) ? 1 : 2);
}

RBIMPL_ATTR_FORCEINLINE()
static bool
rb_scan_args_f_var(const char *fmt)
{
    return (fmt[rb_scan_args_var_idx(fmt)]=='*');
}

RBIMPL_ATTR_FORCEINLINE()
static int
rb_scan_args_trail_idx(const char *fmt)
{
    const int idx = rb_scan_args_var_idx(fmt);
    return idx+(fmt[idx]=='*');
}

RBIMPL_ATTR_FORCEINLINE()
static int
rb_scan_args_n_trail(const char *fmt)
{
    const int idx = rb_scan_args_trail_idx(fmt);
    return (rb_scan_args_isdigit(fmt[idx]) ? fmt[idx]-'0' : 0);
}

RBIMPL_ATTR_FORCEINLINE()
static int
rb_scan_args_hash_idx(const char *fmt)
{
    const int idx = rb_scan_args_trail_idx(fmt);
    return idx+rb_scan_args_isdigit(fmt[idx]);
}

RBIMPL_ATTR_F