<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

require_once(__DIR__ . "/../../../../../../vendor/composer/vendor/autoload.php");
require_once(__DIR__ . "/../../Base.php");

use ILIAS\UI\Component as C;
use ILIAS\UI\Implementation as I;
use ILIAS\UI\Implementation\Component\SignalGenerator;

class ComponentDummy implements C\Component
{
    protected string $id;

    public function __construct($id = "")
    {
        $this->id = $id;
    }
    public function getCanonicalName(): string
    {
        return "Component Dummy";
    }
}

/**
 * Test on button implementation.
 */
class PanelTest extends ILIAS_UI_TestBase
{
    public function getUIFactory(): NoUIFactory
    {
        return new class () extends NoUIFactory {
            public function panelSecondary(): I\Component\Panel\Secondary\Factory
            {
                return new I\Component\Panel\Secondary\Factory();
            }
            public function dropdown(): I\Component\Dropdown\Factory
            {
                return new I\Component\Dropdown\Factory();
            }
            public function viewControl(): I\Component\ViewControl\Factory
            {
                return new I\Component\ViewControl\Factory(new SignalGenerator());
            }
            public function button(): I\Component\Button\Factory
            {
                return new I\Component\Button\Factory();
            }
            public function symbol(): C\Symbol\Factory
            {
                return new I\Component\Symbol\Factory(
                    new I\Component\Symbol\Icon\Factory(),
                    new I\Component\Symbol\Glyph\Factory(),
                    new I\Component\Symbol\Avatar\Factory()
                );
            }
        };
    }

    public function getPanelFactory(): I\Component\Panel\Factory
    {
        return new I\Component\Panel\Factory(
            $this->createMock(C\Panel\Listing\Factory::class)
        );
    }

    public function testImplementsFactoryInterface(): void
    {
        $f = $this->getPanelFactory();

        $this->assertInstanceOf("ILIAS\\UI\\Component\\Panel\\Factory", $f);
        $this->assertInstanceOf(
            "ILIAS\\UI\\Component\\Panel\\Standard",
            $f->standard("Title", array(new ComponentDummy()))
        );
        $this->assertInstanceOf(
            "ILIAS\\UI\\Component\\Panel\\Sub",
            $f->sub("Title", array(new ComponentDummy()))
        );
        $this->assertInstanceOf(
            "ILIAS\\UI\\Component\\Panel\\Report",
            $f->report("Title", $f->sub("Title", array(new ComponentDummy())))
        );
    }

    public function testStandardGetTitle(): void
    {
        $f = $this->getPanelFactory();
        $p = $f->standard("Title", array(new ComponentDummy()));

        $this->assertEquals("Title", $p->getTitle());
    }

    public function testStandardGetContent(): void
    {
        $f = $this->getPanelFactory();
        $c = new ComponentDummy();
        $p = $f->standard("Title", array($c));

        $this->assertEquals($p->getContent(), array($c));
    }

    public function testStandardWithActions(): void
    {
        $fp = $this->getPanelFactory();

        $p = $fp->standard("Title", array(new ComponentDummy()));

        $actions = new I\Component\Dropdown\Standard(array(
            new I\Component\Button\Shy("ILIAS", "https://www.ilias.de"),
            new I\Component\Button\Shy("GitHub", "https://www.github.com")
        ));

        $p = $p->withActions($actions);

        $this->assertEquals($p->getActions(), $actions);
    }

    public function testSubWithActions(): void
    {
        $fp = $this->getPanelFactory();

        $p = $fp->sub("Title", array(new ComponentDummy()));

        $actions = new I\Component\Dropdown\Standard(array(
            new I\Component\Button\Shy("ILIAS", "https://www.ilias.de"),
            new I\Component\Button\Shy("GitHub", "https://www.github.com")
        ));

        $p = $p->withActions($actions);

        $this->assertEquals($p->getActions(), $actions);
    }

    public function testSubWithCard(): void
    {
        $fp = $this->getPanelFactory();

        $p = $fp->sub("Title", array(new ComponentDummy()));

        $card = new I\Component\Card\Card("Card Title");

        $p = $p->withFurtherInformation($card);

        $this->assertEquals($p->getFurtherInformation(), $card);
    }

    public function testSubWithSecondaryPanel(): void
    {
        $fp = $this->getPanelFactory();

        $p = $fp->sub("Title", array(new ComponentDummy()));

        $legacy = new I\Component\Legacy\Legacy("Legacy content", new SignalGenerator());
        $secondary = new I\Component\Panel\Secondary\Legacy("Legacy panel title", $legacy);

        $p = $p->withFurtherInformation($secondary);

        $this->assertEquals($p->getFurtherInformation(), $secondary);
    }

    public function testReportWithActions(): void
    {
        $fp = $this->getPanelFactory();

        $p = $fp->report("Title", $fp->sub("Title", array(new ComponentDummy())));

        $actions = new I\Component\Dropdown\Standard(array(
            new I\Component\Button\Shy("ILIAS", "https://www.ilias.de"),
            new I\Component\Button\Shy("GitHub", "https://www.github.com")
        ));

        $p = $p->withActions($actions);

        $this->assertEquals($p->getActions(), $actions);
    }

    public function testReportGetTitle(): void
    {
        $f = $this->getPanelFactory();
        $sub = $f->sub("Title", array(new ComponentDummy()));
        $p = $f->report("Title", array($sub));

        $this->assertEquals("Title", $p->getTitle());
    }

    public function testReportGetContent(): void
    {
        $f = $this->getPanelFactory();
        $sub = $f->sub("Title", array(new ComponentDummy()));
        $p = $f->report("Title", [$sub]);

        $this->assertEquals($p->getContent(), array($sub));
    }

    public function testRenderStandard(): void
    {
        $f = $this->getPanelFactory();
        $r = $this->getDefaultRenderer();

        $actions = new I\Component\Dropdown\Standard(array(
            new I\Component\Button\Shy("ILIAS", "https://www.ilias.de"),
            new I\Component\Button\Shy("GitHub", "https://www.github.com")
        ));

        $p = $f->standard("Title", array())->withActions($actions);

        $html = $r->render($p);

        $expected_html = <<<EOT
<div class="panel panel-primary panel-flex">
    <div class="panel-heading ilHeader">
        <div class="panel-title"><h2>Title</h2></div>
        <div class="panel-controls">
            <div class="dropdown" id="id_3"><button class="btn btn-default dropdown-toggle" type="button" aria-label="actions" aria-haspopup="true" aria-expanded="false" aria-controls="id_3_menu"><span class="caret"></span></button>
                <ul id="id_3_menu" class="dropdown-menu">
                    <li><button class="btn btn-link" data-action="https://www.ilias.de" id="id_1">ILIAS</button></li>
                    <li><button class="btn btn-link" data-action="https://www.github.com" id="id_2">GitHub</button></li>
                </ul>
            </div>
        </div>
    </div>
    <div class="panel-body"></div>
</div>
EOT;
        $this->assertHTMLEquals($expected_html, $html);
    }

    public function testRenderSub(): void
    {
        $fp = $this->getPanelFactory();
        $r = $this->getDefaultRenderer();

        $actions = new I\Component\Dropdown\Standard(array(
            new I\Component\Button\Shy("ILIAS", "https://www.ilias.de"),
            new I\Component\Button\Shy("GitHub", "https://www.github.com")
        ));

        $p = $fp->sub("Title", array())->withActions($actions);
        $card = new I\Component\Card\Card("Card Title");

        $p = $p->withFurtherInformation($card);
        $html = $this->brutallyTrimHTML($r->render($p));

        $expected_html = <<<EOT
<div class="panel panel-sub panel-flex">
    <div class="panel-heading ilBlockHeader">
        <h3>Title</h3>
        <div class="panel-controls">
            <div class="dropdown" id="id_3"><button class="btn btn-default dropdown-toggle" type="button" aria-label="actions" aria-haspopup="true" aria-expanded="false" aria-controls="id_3_menu"><span class="caret"></span></button>
                <ul id="id_3_menu" class="dropdown-menu">
                    <li><button class="btn btn-link" data-action="https://www.ilias.de" id="id_1">ILIAS</button></li>
                    <li><button class="btn btn-link" data-action="https://www.github.com" id="id_2">GitHub</button></li>
                </ul>
            </div>
        </div>
    </div>
    <div class="panel-body">
        <div class="row">
            <div class="col-sm-8"></div>
            <div class="col-sm-4">
                <div class="il-card thumbnail">
                    <div class="card-no-highlight"></div>
                    <div class="caption card-title">Card Title</div>
                </div>
            </div>
        </div>
    </div>
</div>
EOT;

        $this->assertHTMLEquals($this->brutallyTrimHTML($expected_html), $html);
    }

    public function testRenderSubWithSecondaryPanel(): void
    {
        $fp = $this->getPanelFactory();
        $r = $this->getDefaultRenderer();

        $p = $fp->sub("Title", array());
        $legacy = new I\Component\Legacy\Legacy("Legacy content", new SignalGenerator());
        $secondary = new I\Component\Panel\Secondary\Legacy("Legacy panel title", $legacy);
        $p = $p->withFurtherInformation($secondary);
        $html = $r->render($p);

        $expected_html = <<<EOT
<div class="panel panel-sub panel-flex">
    <div class="panel-heading ilBlockHeader">
        <h3>Title</h3>
        <div class="panel-controls"></div>
    </div>
    <div class="panel-body">
        <div class="row">
            <div class="col-sm-8"></div>
            <div class="col-sm-4">
                <div class="panel panel-secondary panel-flex">
                    <div class="panel-heading ilHeader">
                        <div class="panel-title"><h2>Legacy panel title</h2></div>
                        <div class="panel-controls"></div>
                    </div>
                    <div class="panel-body">Legacy content</div>
                </div>
            </div>
        </div>
    </div>
</div>
EOT;

        $this->assertHTMLEquals(
            $this->brutallyTrimHTML($expected_html),
            $this->brutallyTrimHTML($html)
        );
    }

    public function testRenderReport(): void
    {
        $fp = $this->getPanelFactory();
        $r = $this->getDefaultRenderer();

        $actions = new I\Component\Dropdown\Standard(array(
            new I\Component\Button\Shy("ILIAS", "https://www.ilias.de"),
            new I\Component\Button\Shy("GitHub", "https://www.github.com")
        ));

        $sub = $fp->sub("Title", array());
        $card = new I\Component\Card\Card("Card Title");
        $sub = $sub->withFurtherInformation($card);
        $report = $fp->report("Title", $sub)->withActions($actions);

        $html = $this->brutallyTrimHTML($r->render($report));

        $expected_html = <<<EOT
<div class="panel panel-primary il-panel-report panel-flex">
    <div class="panel-heading ilHeader">
        <div class="panel-title"><h2>Title</h2></div>
        <div class="panel-controls">
            <div class="dropdown" id="id_3"><button class="btn btn-default dropdown-toggle" type="button" aria-label="actions" aria-haspopup="true" aria-expanded="false" aria-controls="id_3_menu"><span class="caret"></span></button>
                <ul id="id_3_menu" class="dropdown-menu">
                    <li><button class="btn btn-link" data-action="https://www.ilias.de" id="id_1">ILIAS</button></li>
                    <li><button class="btn btn-link" data-action="https://www.github.com" id="id_2">GitHub</button></li>
                </ul>
            </div>
        </div>
    </div>
    <div class="panel-body">
        <div class="panel panel-sub panel-flex">
            <div class="panel-heading ilBlockHeader">
                <h3>Title</h3>
                <div class="panel-controls"></div>
            </div>
            <div class="panel-body"><div class="row">
                <div class="col-sm-8"></div>
                    <div class="col-sm-4">
                        <div class="il-card thumbnail">
                            <div class="card-no-highlight"></div>
                            <div class="caption card-title">Card Title</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
EOT;

        $this->assertHTMLEquals($this->brutallyTrimHTML($expected_html), $html);
    }

    public function testStandardWithViewControls(): void
    {
        $sort_options = [
            'a' => 'A',
            'b' => 'B'
        ];
        $sortation = $this->getUIFactory()->viewControl()->sortation($sort_options, 'a');
        $f = $this->getPanelFactory();
        $p = $f->standard("Title", [])
            ->withViewControls([$sortation])
        ;

        $this->assertEquals($p->getViewControls(), [$sortation]);
    }

    public function testReportWithViewControls(): void
    {
        $sort_options = [
            'a' => 'A',
            'b' => 'B'
        ];
        $sortation = $this->getUIFactory()->viewControl()->sortation($sort_options, 'a');
        $f = $this->getPanelFactory();
        $p = $f->report("Title", [])
            ->withViewControls([$sortation])
        ;

        $this->assertEquals($p->getViewControls(), [$sortation]);
    }

    public function testRenderReportWithMode(): void
    {
        $modes = [
            'A' => 'a',
            'B' => 'b'
        ];
        $mode = $this->getUIFactory()->viewControl()->mode($modes, 'Presentation Mode');

        $f = $this->getPanelFactory();
        $r = $this->getDefaultRenderer();


        $p = $f->report("Title", [])
            ->withViewControls([$mode]);

        $html = $r->render($p);

        $expected_html = <<<EOT
<div class="panel panel-primary il-panel-report panel-flex">
    <div class="panel-heading ilHeader">
        <div class="panel-title"><h2>Title</h2></div>
        <div class="panel-viewcontrols l-bar__space-keeper">
            <div class="il-viewcontrol-mode l-bar__element" aria-label="Presentation Mode" role="group">
                <button class="btn btn-default engaged" aria-pressed="true" data-action="a" id="id_1">A</button>
                <button class="btn btn-default" aria-pressed="false" data-action="b" id="id_2">B</button>
            </div>
        </div>
        <div class="panel-controls"></div>
    </div>
    <div class="panel-body"></div>
</div>
EOT;
        $this->assertEquals(
            $this->brutallyTrimHTML($expected_html),
            $this->brutallyTrimHTML($html)
        );
    }

    public function testRenderWithSortation(): void
    {
        $sort_options = [
            'a' => 'A',
            'b' => 'B'
        ];

        $sortation = $this->getUIFactory()->viewControl()->sortation($sort_options, 'b');

        $f = $this->getPanelFactory();
        $r = $this->getDefaultRenderer();


        $p = $f->standard("Title", [])
            ->withViewControls([$sortation]);

        $html = $r->render($p);

        $expected_html = <<<EOT
<div class="panel panel-primary panel-flex">
    <div class="panel-heading ilHeader">
        <div class="panel-title"><h2>Title</h2></div>
        <div class="panel-viewcontrols l-bar__space-keeper">
            <div class="dropdown il-viewcontrol il-viewcontrol-sortation l-bar__element" id="id_1">
                <button class="btn btn-default dropdown-toggle" type="button" data-toggle="dropdown" aria-label="sortation" aria-haspopup="true" aria-expanded="false" aria-controls="id_1_ctrl">
                    <span class="label">vc_sort B</span>
                    <span class="caret"></span>
                </button>
                <ul id="id_1_ctrl" class="dropdown-menu">
                   <li><button class="btn btn-link" data-action="?sortation=a" id="id_2">A</button></li>
                   <li class="selected"><button class="btn btn-link" data-action="?sortation=b" id="id_3">B</button></li>
                </ul>
            </div>
        </div>
        <div class="panel-controls"></div>
    </div>
    <div class="panel-body"></div>
</div>
EOT;
        $this->assertEquals(
            $this->brutallyTrimHTML($expected_html),
            $this->brutallyTrimHTML($html)
        );
    }

    public function testRenderWithPagination(): void
    {
        $pagination = $this->getUIFactory()->viewControl()->pagination()
            ->withTargetURL('http://ilias.de', 'page')
            ->withTotalEntries(10)
            ->withPageSize(2)
            ->withCurrentPage(1);

        $f = $this->getPanelFactory();
        $r = $this->getDefaultRenderer();


        $p = $f->standard("Title", [])
            ->withViewControls([$pagination]);

        $html = $r->render($p);

        $expected_html = <<<EOT
<div class="panel panel-primary panel-flex">
    <div class="panel-heading ilHeader">
        <div class="panel-title"><h2>Title</h2></div>
        <div class="panel-viewcontrols l-bar__space-keeper">
            <div class="il-viewcontrol-pagination l-bar__element">
                <button class="btn btn-default" data-action="http://ilias.de?page=0" id="id_6">
                    <span class="glyph" aria-label="back" role="img"><span class="glyphicon glyphicon-chevron-left" aria-hidden="true"></span></span>
                </button>
                <button class="btn btn-link" data-action="http://ilias.de?page=0" id="id_1">1</button>
                <button class="btn btn-link engaged" aria-pressed="true" data-action="http://ilias.de?page=1" id="id_2">2</button>
                <button class="btn btn-link" data-action="http://ilias.de?page=2" id="id_3">3</button>
                <button class="btn btn-link" data-action="http://ilias.de?page=3" id="id_4">4</button>
                <button class="btn btn-link" data-action="http://ilias.de?page=4" id="id_5">5</button>
                <button class="btn btn-default" data-action="http://ilias.de?page=2" id="id_7">
                    <span class="glyph" aria-label="next" role="img"><span class="glyphicon glyphicon-chevron-right" aria-hidden="true"></span></span>
                </button>
            </div>
        </div>
        <div class="panel-controls"></div>
    </div>
    <div class="panel-body"></div>
</div>
EOT;
        $this->assertEquals(
            $this->brutallyTrimHTML($expected_html),
            $this->brutallyTrimHTML($html)
        );
    }
}
