<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

use ILIAS\News\StandardGUIRequest;

/**
 * Single news timeline item
 * @author Alexander Killing <killing@leifos.de>
 */
class ilNewsTimelineItemGUI implements ilTimelineItemInt
{
    protected \ILIAS\News\InternalGUIService $gui;
    protected \ILIAS\Notes\Service $notes;
    protected ilLanguage $lng;
    protected ilNewsItem $news_item;
    protected ilObjectDefinition $obj_def;
    protected ilObjUser $user;
    protected bool $user_edit_all;
    protected int $news_item_ref_id;
    protected int $ref_id;
    protected ilCtrl $ctrl;
    protected ilLikeGUI $like_gui;
    protected StandardGUIRequest $std_request;
    /**
     * @var array<int, \ILIAS\UI\Component\Image\Image>
     */
    protected array $item_image = [];
    /**
     * @var array<int, \ILIAS\UI\Component\Modal\Modal>
     */
    protected array $item_modal = [];

    protected function __construct(
        ilNewsItem $a_news_item,
        int $a_news_ref_id,
        ilLikeGUI $a_like_gui
    ) {
        global $DIC;

        $this->like_gui = $a_like_gui;
        $this->lng = $DIC->language();
        $this->ctrl = $DIC->ctrl();
        $this->setNewsItem($a_news_item);
        $this->user = $DIC->user();
        $this->obj_def = $DIC["objDefinition"];
        $this->news_item_ref_id = $a_news_ref_id;

        $this->std_request = $DIC->news()
            ->internal()
            ->gui()
            ->standardRequest();
        $this->ref_id = $this->std_request->getRefId();
        $this->gui = $DIC->news()
            ->internal()
            ->gui();
        $this->notes = $DIC->notes();
    }

    public static function getInstance(
        ilNewsItem $a_news_item,
        int $a_news_ref_id,
        ilLikeGUI $a_like_gui
    ): self {
        return new self($a_news_item, $a_news_ref_id, $a_like_gui);
    }

    public function setNewsItem(ilNewsItem $a_val): void
    {
        $this->news_item = $a_val;
    }

    public function getNewsItem(): ilNewsItem
    {
        return $this->news_item;
    }

    /**
     * Set user can edit other users postings
     */
    public function setUserEditAll(bool $a_val): void
    {
        $this->user_edit_all = $a_val;
    }

    /**
     * Get user can edit other users postings
     */
    public function getUserEditAll(): bool
    {
        return $this->user_edit_all;
    }

    public function getDateTime(): ilDateTime
    {
        $i = $this->getNewsItem();
        return new ilDateTime($i->getCreationDate(), IL_CAL_DATETIME);
    }

    public function render(): string
    {
        $i = $this->getNewsItem();
        $tpl = new ilTemplate("tpl.timeline_item.html", true, true, "components/ILIAS/News");
        $ui_factory = $this->gui->ui()->factory();
        $ui_renderer = $this->gui->ui()->renderer();

        $news_renderer = ilNewsRendererFactory::getRenderer($i->getContextObjType());
        $news_renderer->setLanguage($this->lng->getLangKey());
        $news_renderer->setNewsItem($i, $this->news_item_ref_id);

        $obj_id = $i->getContextObjId();

        // edited?
        if ($i->getCreationDate() !== $i->getUpdateDate()) {
            $tpl->setCurrentBlock("edited");
            $update_date = new ilDateTime($i->getUpdateDate(), IL_CAL_DATETIME);
            $tpl->setVariable("TXT_EDITED", $this->lng->txt("cont_news_edited"));
            if ($i->getUpdateUserId() > 0 && ($i->getUpdateUserId() !== $i->getUserId())) {
                $tpl->setVariable("TXT_USR_EDITED", ilUserUtil::getNamePresentation(
                    $i->getUpdateUserId(),
                    false,
                    true,
                    $this->ctrl->getLinkTargetByClass(ilNewsTimelineGUI::class)
                ) . " - ");
            }
            $tpl->setVariable("TIME_EDITED", ilDatePresentation::formatDate($update_date));
            $tpl->parseCurrentBlock();
        }

        // context object link
        if ($this->news_item_ref_id > 0 && $this->ref_id !== $this->news_item_ref_id) {
            $tpl->setCurrentBlock("object");
            $tpl->setVariable("OBJ_TITLE", ilObject::_lookupTitle($obj_id));
            $tpl->setVariable("OBJ_IMG", ilObject::_getIcon($obj_id));
            $tpl->setVariable("OBJ_HREF", $news_renderer->getObjectLink());
            $tpl->parseCurrentBlock();
        }

        // media
        if ($i->getMobId() > 0 && ilObject::_exists($i->getMobId())) {
            $media = $this->renderMedia($i);
            $tpl->setCurrentBlock("player");
            $tpl->setVariable("PLAYER", $media);
            $tpl->parseCurrentBlock();
        }

        $p = $this->gui->profile();
        $tpl->setVariable("USER_AVATAR", $this->gui->ui()->renderer()->render(
            $p->getAvatar($i->getUserId())
        ));
        $tpl->setVariable(
            "TITLE",
            ilNewsItem::determineNewsTitle($i->getContextObjType(), $i->getTitle(), $i->getContentIsLangVar())
        );

        // content
        $tpl->setVariable("CONTENT", $news_renderer->getTimelineContent());

        $tpl->setVariable("TXT_USR", $p->getNamePresentation(
            $i->getUserId(),
            true,
            $this->ctrl->getLinkTargetByClass(ilNewsTimelineGUI::class)
        ));

        $tpl->setVariable("TIME", ilDatePresentation::formatDate($this->getDateTime()));

        // actions
        $actions = [];

        if ($i->getPriority() === 1 && ($i->getUserId() === $this->user->getId() || $this->getUserEditAll())) {
            if (!$news_renderer->preventEditing()) {
                $actions[] = $ui_factory->button()->shy(
                    $this->lng->txt("edit"),
                    ""
                )->withOnLoadCode(static function ($id) use ($i) {
                    return "document.getElementById('$id').addEventListener('click', () => {il.News.edit(" . $i->getId() . ");});";
                });
                $actions[] = $ui_factory->button()->shy(
                    $this->lng->txt("delete"),
                    ""
                )->withOnLoadCode(static function ($id) use ($i) {
                    return "document.getElementById('$id').addEventListener('click', () => {il.News.delete(" . $i->getId() . ");});";
                });
            }
        }
        foreach ($news_renderer->getTimelineActions() as $action) {
            $actions[] = $action;
        }
        $dd = $ui_factory->dropdown()->standard($actions);
        $tpl->setVariable("ACTIONS", $ui_renderer->render($dd));

        return $tpl->get();
    }

    protected function renderMedia(ilNewsItem $i): string
    {
        $media_path = $this->getMediaPath($i);
        $mime = ilObjMediaObject::getMimeType($media_path);

        $ui_factory = $this->gui->ui()->factory();
        $ui_renderer = $this->gui->ui()->renderer();

        if (in_array($mime, ["image/jpeg", "image/svg+xml", "image/gif", "image/png"])) {
            if (isset($this->item_image[$i->getId()]) && isset($this->item_modal[$i->getId()])) {
                $image = $this->item_image[$i->getId()];
            } else {
                $title = basename($media_path);
                $image = $ui_factory->image()->responsive($media_path, $title);
                $modal_page = $ui_factory->modal()->lightboxImagePage($image, $title);
                $modal = $ui_factory->modal()->lightbox($modal_page);
                $image = $image->withAction($modal->getShowSignal());
                $this->item_image[$i->getId()] = $image;
                $this->item_modal[$i->getId()] = $modal;
            }
            $html = $ui_renderer->render($image);
        } elseif (in_array($mime, ["video/mp4", "video/youtube", "video/vimeo"])) {
            $video = $ui_factory->player()->video($media_path);
            $html = $ui_renderer->render($video);
        } elseif (in_array($mime, ["audio/mpeg"])) {
            $audio = $ui_factory->player()->audio($media_path);
            $html = $ui_renderer->render($audio);
        } elseif (in_array($mime, ["application/pdf"])) {
            $this->ctrl->setParameterByClass(ilNewsTimelineGUI::class, "news_id", $i->getId());
            $link = $ui_factory->link()->standard(
                basename($media_path),
                $this->ctrl->getLinkTargetByClass(ilNewsTimelineGUI::class, "downloadMob")
            );
            $html = $ui_renderer->render($link);
            $this->ctrl->setParameterByClass(ilNewsTimelineGUI::class, "news_id", null);
        } else {
            $html = "";
        }
        return $html;
    }

    protected function renderMediaModal(ilNewsItem $i): string
    {
        $ui_factory = $this->gui->ui()->factory();
        $ui_renderer = $this->gui->ui()->renderer();

        if (isset($this->item_image[$i->getId()]) && isset($this->item_modal[$i->getId()])) {
            $modal = $this->item_modal[$i->getId()];
            return $ui_renderer->render($modal);
        }

        $media_path = $this->getMediaPath($i);
        $mime = ilObjMediaObject::getMimeType($media_path);

        $modal_html = "";

        if (in_array($mime, ["image/jpeg", "image/svg+xml", "image/gif", "image/png"])) {
            $title = basename($media_path);
            $image = $ui_factory->image()->responsive($media_path, $title);
            $modal_page = $ui_factory->modal()->lightboxImagePage($image, $title);
            $modal = $ui_factory->modal()->lightbox($modal_page);
            $image = $image->withAction($modal->getShowSignal());
            $this->item_image[$i->getId()] = $image;
            $this->item_modal[$i->getId()] = $modal;
            $modal_html = $ui_renderer->render($modal);
        }
        return $modal_html;
    }

    public function renderFooter(): string
    {
        $i = $this->getNewsItem();

        // like
        $this->ctrl->setParameterByClass(ilNewsTimelineGUI::class, "news_id", $i->getId());
        $this->like_gui->setObject(
            $i->getContextObjId(),
            $i->getContextObjType(),
            $i->getContextSubObjId(),
            (string) $i->getContextSubObjType(),
            $i->getId()
        );
        $html = $this->ctrl->getHTML($this->like_gui);

        // comments
        $notes_obj_type = ($i->getContextSubObjType() == "")
            ? $i->getContextObjType()
            : $i->getContextSubObjType();
        $comments_gui = $this->notes->gui()->getCommentsGUI(
            $i->getContextObjId(),
            $i->getContextSubObjId(),
            $notes_obj_type,
            $i->getId()
        );
        $comments_gui->setDefaultCommand("getWidget");
        $comments_gui->setShowEmptyListMessage(false);
        $comments_gui->setShowHeader(false);
        $html .= $comments_gui->getWidget();
        //$html .= $this->ctrl->getHTML($comments_gui);

        $this->ctrl->setParameterByClass(ilNewsTimelineGUI::class, "news_id", $this->std_request->getNewsId());

        return $html . $this->renderMediaModal($i);
    }

    protected function getMediaPath(ilNewsItem $i): string
    {
        $media_path = "";
        if ($i->getMobId() > 0) {
            $mob = new ilObjMediaObject($i->getMobId());
            $media_path = $mob->getStandardSrc();
        }
        return $media_path;
    }
}
