<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

namespace ILIAS\MetaData\XML\Reader\Standard;

use PHPUnit\Framework\TestCase;
use ILIAS\MetaData\XML\Reader\NullReader;
use ILIAS\MetaData\XML\Version;
use ILIAS\MetaData\Elements\SetInterface;
use ILIAS\MetaData\Elements\NullSet;
use SimpleXMLElement;

class StandardTest extends TestCase
{
    protected function getReader(): Standard
    {
        $structurally_coupled = new class () extends NullReader {
            public function read(\SimpleXMLElement $xml, Version $version): SetInterface
            {
                return new class ($xml, $version) extends NullSet {
                    public string $exposed;

                    public function __construct(\SimpleXMLElement $xml, Version $version)
                    {
                        $this->exposed = 'generated by StructurallyCoupled in version ' .
                        $version->value . ' from xml ' . $xml->asXML();
                    }
                };
            }
        };

        $legacy = new class () extends NullReader {
            public function read(\SimpleXMLElement $xml, Version $version): SetInterface
            {
                return new class ($xml, $version) extends NullSet {
                    public string $exposed;

                    public function __construct(\SimpleXMLElement $xml, Version $version)
                    {
                        $this->exposed = 'generated by Legacy in version ' .
                            $version->value . ' from xml ' . $xml->asXML();
                    }
                };
            }
        };

        return new Standard($structurally_coupled, $legacy);
    }

    public function testReadWithVersion10_0(): void
    {
        $xml = new SimpleXMLElement('<some>xml</some>');
        $reader = $this->getReader();

        $set = $reader->read($xml, Version::V10_0);

        $this->assertSame(
            'generated by StructurallyCoupled in version 10.0 from xml ' . $xml->asXML(),
            $set->exposed
        );
    }

    public function testReadWithVersion4_1_0(): void
    {
        $xml = new SimpleXMLElement('<some>xml</some>');
        $reader = $this->getReader();

        $set = $reader->read($xml, Version::V4_1_0);

        $this->assertSame(
            'generated by Legacy in version 4.1.0 from xml ' . $xml->asXML(),
            $set->exposed
        );
    }
}
