<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

namespace ILIAS\Tests\Refinery\String;

use PHPUnit\Framework\TestCase;
use ILIAS\Refinery\String\MakeClickable;
use ILIAS\Refinery\ConstraintViolationException;

class MakeClickableTest extends TestCase
{
    public function testConstruct(): void
    {
        $this->assertInstanceOf(MakeClickable::class, new MakeClickable());
    }

    public function testTransformFailure(): void
    {
        $this->expectException(ConstraintViolationException::class);
        $clickable = new MakeClickable();

        $clickable->transform(3);
    }

    /**
     * @dataProvider provideInputInNewTab
     */
    public function testTransformSuccessInNewTab(string $expected, string $input): void
    {
        $clickable = new MakeClickable();
        $this->assertEquals($expected, $clickable->transform($input));
    }

    /**
     * @dataProvider provideInputWithoutAttributes
     */
    public function testTransformSuccess(string $expected, string $input): void
    {
        $clickable = new MakeClickable(false);
        $this->assertEquals($expected, $clickable->transform($input));
    }


    public static function provideInputInNewTab(): array
    {
        return self::createInput(' target="_blank" rel="noopener"');
    }

    public static function provideInputWithoutAttributes(): array
    {
        return self::createInput();
    }

    public static function createInput(string $attributes = ''): array
    {
        $templates = [
            'test empty string' => ['', ''],
            'test no links' => [
                'Aliquam erat volutpat.  Nunc eleifend leo vitae magna. <strong>Sed bibendum</strong>donec vitae dolor<p><b magna="vitae"> phasellus purus</b></p> Nunc aliquet, augue nec adipiscing interdum, lacus tellus malesuada massa, quis varius mi purus non odio.',
                'Aliquam erat volutpat.  Nunc eleifend leo vitae magna. <strong>Sed bibendum</strong>donec vitae dolor<p><b magna="vitae"> phasellus purus</b></p> Nunc aliquet, augue nec adipiscing interdum, lacus tellus malesuada massa, quis varius mi purus non odio.'
            ],
            'test only link' => [
                '<a%s href="https://www.ilias.de">https://www.ilias.de</a>',
                'https://www.ilias.de'
            ],
            'test simple link string' => [
                ' <a%s href="https://www.ilias.de">https://www.ilias.de</a> ',
                ' https://www.ilias.de '
            ],
            'test no nesting' => [
                'Phasellus at dui in ligula mollis ultricies <span><a%s href="https://www.ilias.de">https://www.ilias.de</a></span> <a href="baba">Nullam rutrum</a>',
                'Phasellus at dui in ligula mollis ultricies <span>https://www.ilias.de</span> <a href="baba">Nullam rutrum</a>'
            ],
            'match with links after match' => [
                ' Nullam rutrum <a%s href="https://www.ilias.de">https://www.ilias.de</a> <a href="baba">Integer placerat tristique nisl</a>',
                ' Nullam rutrum https://www.ilias.de <a href="baba">Integer placerat tristique nisl</a>'
            ],
            'test surrounded by diamonds' => [
                '><a%s href="https://www.ilias.de">https://www.ilias.de</a><',
                '>https://www.ilias.de<'
            ],
            'test surrounded by other links' => [
                '><a%s href="https://www.ilias.de">www.ilias.de</a><a href="www.ilias.de">www.ilias.de</a><a%s href="https://www.ilias.de">www.ilias.de</a>',
                '>www.ilias.de<a href="www.ilias.de">www.ilias.de</a>www.ilias.de'
            ],
            'test links next to each other' => [
                '<a%s href="https://www.ilias.de">www.ilias.de</a> <a%s href="https://www.ilias.de">www.ilias.de</a> <a%s href="https://www.ilias.de">www.ilias.de</a>',
                'www.ilias.de www.ilias.de www.ilias.de'
            ],
            'test multiple + nested' => [
                'Praesent fermentum tempor tellus <span>Aliquam erat volutpat<b><a href="www.ilias.de"><b>curabitur lacinia pulvinar nibh</b> www.ilias.de</a>nunc eleifend leo vitae magnawww.ilias.denunc rutrum turpis sed pede <a%s href="https://www.ilias.de">www.ilias.de</a><a href="www.ilias.de">www.ilias.de</a><a%s href="https://www.ilias.de">www.ilias.de</a>',
                'Praesent fermentum tempor tellus <span>Aliquam erat volutpat<b><a href="www.ilias.de"><b>curabitur lacinia pulvinar nibh</b> www.ilias.de</a>nunc eleifend leo vitae magnawww.ilias.denunc rutrum turpis sed pede www.ilias.de<a href="www.ilias.de">www.ilias.de</a>www.ilias.de'
            ],
            'test multiple + nested + multibyte' => [
                'Praesent fermentum tempor ♥ tellus <span>Aliquam erat volutpat<b><a href="www.ilias.de"><b>curabitur lacinia pulvinar nibh</b> www.ilias.de</a>nunc eleifend leo vitae magnawww.ilias.denunc rutrum turpis sed pede <a%s href="https://www.ilias.de">www.ilias.de</a> phasellus ♥ diam<a href="www.ilias.de">www.ilias.de</a><a%s href="https://www.ilias.de">www.ilias.de</a>',
                'Praesent fermentum tempor ♥ tellus <span>Aliquam erat volutpat<b><a href="www.ilias.de"><b>curabitur lacinia pulvinar nibh</b> www.ilias.de</a>nunc eleifend leo vitae magnawww.ilias.denunc rutrum turpis sed pede www.ilias.de phasellus ♥ diam<a href="www.ilias.de">www.ilias.de</a>www.ilias.de'
            ],
            'test with properties' => [
                '<a%s href="https://www.ilias.de">www.ilias.de</a> <a purus="pretium" href="www.ilias.de" tellus="bibendum">www.ilias.de</a>',
                'www.ilias.de <a purus="pretium" href="www.ilias.de" tellus="bibendum">www.ilias.de</a>'
            ],
            'test with tabs' => [
                'Mauris ac felis vel velit tristique imperdiet <a	href="www.ilias.de">www.ilias.de</a>',
                'Mauris ac felis vel velit tristique imperdiet <a	href="www.ilias.de">www.ilias.de</a>'
            ],
            'test example input with newlines' => [
                "Das ist eine URL: <a%s href=\"https://www.ilias.de\">https://www.ilias.de</a>\nDas ist sogar ein Link: <a href=\"https://www.ilias.de\">https://www.ilias.de</a>\nDas ist ein Link hinter einem Wort: <a href=\"https://www.ilias.de\">Link</a> und noch mehr Text.",
                "Das ist eine URL: https://www.ilias.de\nDas ist sogar ein Link: <a href=\"https://www.ilias.de\">https://www.ilias.de</a>\nDas ist ein Link hinter einem Wort: <a href=\"https://www.ilias.de\">Link</a> und noch mehr Text."
            ],
            'test without protocol to URL with protocol' => [
                '<a%s href="https://www.ilias.de">www.ilias.de</a>',
                'www.ilias.de'
            ],
            'test link with parameters' => [
                '<a%s href="http://ilias.de/ilias.php?ref_id=29&admin_mode=settings&cmd=view&cmdClass=ilobjdashboardsettingsgui&cmdNode=1c:lg&baseClass=ilAdministrationGUI">http://ilias.de/ilias.php?ref_id=29&admin_mode=settings&cmd=view&cmdClass=ilobjdashboardsettingsgui&cmdNode=1c:lg&baseClass=ilAdministrationGUI</a>',
                'http://ilias.de/ilias.php?ref_id=29&admin_mode=settings&cmd=view&cmdClass=ilobjdashboardsettingsgui&cmdNode=1c:lg&baseClass=ilAdministrationGUI'
            ],
            'test multibyte' => [
                'I♥ILIAS',
                'I♥ILIAS'
            ],
            'test links with multibytes in url' => [
                '<a%s href="https://www.ilias.de/?ilias=♥&other=parameter">www.ilias.de/?ilias=♥&other=parameter</a>',
                'www.ilias.de/?ilias=♥&other=parameter'
            ],
            'test multibyte at link end' => [
                '<a%s href="https://www.ilias.de/?ilias=♥">www.ilias.de/?ilias=♥</a>',
                'www.ilias.de/?ilias=♥'
            ],
            'test multiple multibytes' => [
                'Etiam laoreet quam sed arcu öä♥♥♥üåñç <a%s href="https://www.ilias.de">www.ilias.de</a> öä♥♥♥üå',
                'Etiam laoreet quam sed arcu öä♥♥♥üåñç www.ilias.de öä♥♥♥üå'
            ],
            'test links as attributes' => [
                'Etiam laoreet quam sed arcu <p data-target="https://ilias.de>Praesent augue</p>Vestibulum convallis, lorem a tempus semper, dui dui euismod elit, vitae placerat urna tortor vitae lacus',
                'Etiam laoreet quam sed arcu <p data-target="https://ilias.de>Praesent augue</p>Vestibulum convallis, lorem a tempus semper, dui dui euismod elit, vitae placerat urna tortor vitae lacus'
            ],
            'test svg attributes' => [
                'Etiam laoreet quam sed arcu <p xmlns:xlink="http://www.w3.org/1999/xlink" data-test="1">Praesent augue</p>Vestibulum convallis, lorem a tempus semper, dui dui euismod elit, vitae placerat urna tortor vitae lacus',
                'Etiam laoreet quam sed arcu <p xmlns:xlink="http://www.w3.org/1999/xlink" data-test="1">Praesent augue</p>Vestibulum convallis, lorem a tempus semper, dui dui euismod elit, vitae placerat urna tortor vitae lacus'
            ],
            'test svg elements' => [
                "<svg xmlns:xlink=\"http://www.w3.org/1999/xlink\"></svg>",
                "<svg xmlns:xlink=\"http://www.w3.org/1999/xlink\"></svg>"
            ],
            'test example mathjax svg element' => [
                "<svg xmlns:xlink=\"http://www.w3.org/1999/xlink\" width=\"17.471ex\" height=\"6.009ex\" style=\"vertical-align: -2.505ex;\" viewBox=\"0 -1508.9 7522 2587.3\" role=\"img\" focusable=\"false\" xmlns=\"http://www.w3.org/2000/svg\">\n<defs>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D453\" d=\"M248 437h37c63 0 58 77 101 141c31 47 74 90 150 90c50 0 79 -24 79 -59c0 -28 -21 -40 -41 -40c-28 0 -35 11 -35 25s9 29 9 37c0 5 -3 8 -15 8c-50 0 -89 -54 -110 -132l-19 -70h80l-12 -38h-76l-79 -306c-7 -27 -15 -65 -30 -102c-33 -83 -87 -178 -175 -178 c-48 0 -72 24 -72 54c0 24 14 49 43 49c27 0 37 -18 37 -33c0 -19 -16 -20 -16 -32c0 -6 6 -9 15 -9c65 0 95 158 110 215l87 342h-77Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-28\" d=\"M304 -161l-12 -16c-158 90 -244 259 -244 429c0 185 87 329 247 424l9 -16c-139 -119 -170 -212 -170 -405c0 -186 30 -299 170 -416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D465\" d=\"M305 288l45 69c26 40 61 84 111 84c32 0 49 -12 49 -33c0 -19 -13 -42 -39 -42c-22 0 -27 9 -44 9c-30 0 -41 -16 -55 -38l-55 -85c30 -62 53 -197 98 -197c16 0 40 14 61 57l14 -10c-12 -15 -57 -111 -115 -111c-41 0 -60 41 -73 77l-40 113l-63 -98 c-32 -49 -61 -92 -117 -92c-36 0 -52 25 -52 47c0 16 13 37 38 37c26 0 22 -20 47 -20c12 0 38 10 70 59l65 101l-33 92c-17 48 -43 89 -98 89c-13 0 -28 -2 -38 -6l-3 18l132 33c44 -24 63 -65 75 -98Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-29\" d=\"M29 660l12 16c153 -92 244 -259 244 -429c0 -185 -88 -327 -247 -424l-9 16c142 117 170 211 170 405c0 187 -25 302 -170 416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-3D\" d=\"M637 320h-589v66h589v-66zM637 120h-589v66h589v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBSIZE1-222B\" d=\"M280 -10l414 1680c48 195 126 330 268 330c52 0 73 -25 73 -55c0 -42 -32 -75 -71 -75c-19 0 -38 8 -38 30c0 27 21 37 23 51c0 7 -6 11 -15 11c-65 0 -94 -91 -126 -221l-419 -1704c-45 -181 -122 -306 -261 -306c-51 0 -72 25 -72 54c0 43 31 76 70 76 c19 0 38 -8 38 -30c0 -24 -21 -38 -23 -50c0 -8 6 -12 15 -12c65 0 91 88 124 221Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-2212\" d=\"M621 220h-557v66h557v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-221E\" d=\"M854 222c0 -97 -72 -222 -198 -222c-82 0 -139 53 -201 145c-45 -91 -113 -145 -209 -145c-86 0 -176 62 -176 200c0 184 133 230 191 230c73 0 123 -19 218 -147c46 83 92 147 193 147c90 0 182 -74 182 -208zM499 245v-3c72 -119 127 -183 195 -183 c56 0 100 55 100 119c0 91 -51 168 -139 168c-89 0 -127 -59 -156 -101zM433 181v3c-72 119 -135 187 -203 187c-56 0 -100 -55 -100 -119c0 -85 51 -168 139 -168c89 0 137 61 164 97Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D452\" d=\"M363 112l14 -13c-70 -86 -138 -110 -200 -110c-98 0 -137 84 -137 156c0 23 1 37 6 60c25 111 135 236 262 236c42 0 102 -14 102 -76c0 -127 -167 -176 -286 -182v-28c0 -64 52 -107 113 -107c42 0 90 18 126 64zM124 211h9c104 0 198 69 198 157c0 25 -19 43 -44 43 c-74 0 -134 -115 -163 -200Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D461\" d=\"M283 438l-10 -38h-70l-78 -320c-1 -3 -2 -8 -2 -11c0 -6 3 -12 16 -12c0 0 23 2 60 52l13 -11c-24 -31 -74 -107 -133 -107c-31 0 -39 25 -39 52c0 13 2 26 5 37l81 320h-77l5 29c75 14 136 51 177 138h13l-31 -129h70Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-32\" d=\"M474 137l-54 -137h-391v12l178 189c94 99 130 175 130 260c0 91 -54 141 -139 141c-72 0 -107 -32 -147 -130l-21 5c21 117 85 199 208 199c113 0 185 -77 185 -176c0 -79 -39 -154 -128 -248l-165 -176h234c42 0 63 11 96 67Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D451\" d=\"M527 668l-149 -598c-1 -3 -2 -9 -2 -12c0 -6 5 -9 15 -9c20 0 48 35 62 56l11 -12c-30 -45 -83 -105 -130 -105c-32 0 -40 23 -40 41c0 20 2 34 10 64h-5c-74 -93 -134 -105 -171 -105c-73 0 -88 74 -88 127c0 103 103 326 257 326c57 0 80 -26 81 -50h2l53 209 c1 4 2 8 2 12c0 13 -7 20 -33 20c-9 0 -20 -2 -27 -3l-4 15zM363 340c0 47 -15 71 -56 71c-99 0 -180 -200 -180 -296c0 -49 28 -66 56 -66c70 0 136 94 164 186c11 35 16 74 16 105Z\"></path>\n</defs>\n<g stroke=\"currentColor\" fill=\"currentColor\" stroke-width=\"0\" transform=\"matrix(1 0 0 -1 0 0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D453\" x=\"0\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-28\" x=\"615\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"949\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-29\" x=\"1499\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-3D\" x=\"2110\" y=\"0\"></use>\n<g transform=\"translate(3074,0)\">\n <use xlink:href=\"#E1-STIXWEBSIZE1-222B\" x=\"0\" y=\"-616\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"1655\" y=\"1573\"></use>\n<g transform=\"translate(585,-921)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-221E\" x=\"685\" y=\"0\"></use>\n</g>\n</g>\n<g transform=\"translate(5066,0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D452\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(445,412)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(484,0)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.574)\" xlink:href=\"#E1-STIXWEBMAIN-32\" x=\"386\" y=\"446\"></use>\n</g>\n</g>\n</g>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D451\" x=\"6676\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"7208\" y=\"0\"></use>\n</g>\n</svg>",
                "<svg xmlns:xlink=\"http://www.w3.org/1999/xlink\" width=\"17.471ex\" height=\"6.009ex\" style=\"vertical-align: -2.505ex;\" viewBox=\"0 -1508.9 7522 2587.3\" role=\"img\" focusable=\"false\" xmlns=\"http://www.w3.org/2000/svg\">\n<defs>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D453\" d=\"M248 437h37c63 0 58 77 101 141c31 47 74 90 150 90c50 0 79 -24 79 -59c0 -28 -21 -40 -41 -40c-28 0 -35 11 -35 25s9 29 9 37c0 5 -3 8 -15 8c-50 0 -89 -54 -110 -132l-19 -70h80l-12 -38h-76l-79 -306c-7 -27 -15 -65 -30 -102c-33 -83 -87 -178 -175 -178 c-48 0 -72 24 -72 54c0 24 14 49 43 49c27 0 37 -18 37 -33c0 -19 -16 -20 -16 -32c0 -6 6 -9 15 -9c65 0 95 158 110 215l87 342h-77Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-28\" d=\"M304 -161l-12 -16c-158 90 -244 259 -244 429c0 185 87 329 247 424l9 -16c-139 -119 -170 -212 -170 -405c0 -186 30 -299 170 -416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D465\" d=\"M305 288l45 69c26 40 61 84 111 84c32 0 49 -12 49 -33c0 -19 -13 -42 -39 -42c-22 0 -27 9 -44 9c-30 0 -41 -16 -55 -38l-55 -85c30 -62 53 -197 98 -197c16 0 40 14 61 57l14 -10c-12 -15 -57 -111 -115 -111c-41 0 -60 41 -73 77l-40 113l-63 -98 c-32 -49 -61 -92 -117 -92c-36 0 -52 25 -52 47c0 16 13 37 38 37c26 0 22 -20 47 -20c12 0 38 10 70 59l65 101l-33 92c-17 48 -43 89 -98 89c-13 0 -28 -2 -38 -6l-3 18l132 33c44 -24 63 -65 75 -98Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-29\" d=\"M29 660l12 16c153 -92 244 -259 244 -429c0 -185 -88 -327 -247 -424l-9 16c142 117 170 211 170 405c0 187 -25 302 -170 416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-3D\" d=\"M637 320h-589v66h589v-66zM637 120h-589v66h589v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBSIZE1-222B\" d=\"M280 -10l414 1680c48 195 126 330 268 330c52 0 73 -25 73 -55c0 -42 -32 -75 -71 -75c-19 0 -38 8 -38 30c0 27 21 37 23 51c0 7 -6 11 -15 11c-65 0 -94 -91 -126 -221l-419 -1704c-45 -181 -122 -306 -261 -306c-51 0 -72 25 -72 54c0 43 31 76 70 76 c19 0 38 -8 38 -30c0 -24 -21 -38 -23 -50c0 -8 6 -12 15 -12c65 0 91 88 124 221Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-2212\" d=\"M621 220h-557v66h557v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-221E\" d=\"M854 222c0 -97 -72 -222 -198 -222c-82 0 -139 53 -201 145c-45 -91 -113 -145 -209 -145c-86 0 -176 62 -176 200c0 184 133 230 191 230c73 0 123 -19 218 -147c46 83 92 147 193 147c90 0 182 -74 182 -208zM499 245v-3c72 -119 127 -183 195 -183 c56 0 100 55 100 119c0 91 -51 168 -139 168c-89 0 -127 -59 -156 -101zM433 181v3c-72 119 -135 187 -203 187c-56 0 -100 -55 -100 -119c0 -85 51 -168 139 -168c89 0 137 61 164 97Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D452\" d=\"M363 112l14 -13c-70 -86 -138 -110 -200 -110c-98 0 -137 84 -137 156c0 23 1 37 6 60c25 111 135 236 262 236c42 0 102 -14 102 -76c0 -127 -167 -176 -286 -182v-28c0 -64 52 -107 113 -107c42 0 90 18 126 64zM124 211h9c104 0 198 69 198 157c0 25 -19 43 -44 43 c-74 0 -134 -115 -163 -200Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D461\" d=\"M283 438l-10 -38h-70l-78 -320c-1 -3 -2 -8 -2 -11c0 -6 3 -12 16 -12c0 0 23 2 60 52l13 -11c-24 -31 -74 -107 -133 -107c-31 0 -39 25 -39 52c0 13 2 26 5 37l81 320h-77l5 29c75 14 136 51 177 138h13l-31 -129h70Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-32\" d=\"M474 137l-54 -137h-391v12l178 189c94 99 130 175 130 260c0 91 -54 141 -139 141c-72 0 -107 -32 -147 -130l-21 5c21 117 85 199 208 199c113 0 185 -77 185 -176c0 -79 -39 -154 -128 -248l-165 -176h234c42 0 63 11 96 67Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D451\" d=\"M527 668l-149 -598c-1 -3 -2 -9 -2 -12c0 -6 5 -9 15 -9c20 0 48 35 62 56l11 -12c-30 -45 -83 -105 -130 -105c-32 0 -40 23 -40 41c0 20 2 34 10 64h-5c-74 -93 -134 -105 -171 -105c-73 0 -88 74 -88 127c0 103 103 326 257 326c57 0 80 -26 81 -50h2l53 209 c1 4 2 8 2 12c0 13 -7 20 -33 20c-9 0 -20 -2 -27 -3l-4 15zM363 340c0 47 -15 71 -56 71c-99 0 -180 -200 -180 -296c0 -49 28 -66 56 -66c70 0 136 94 164 186c11 35 16 74 16 105Z\"></path>\n</defs>\n<g stroke=\"currentColor\" fill=\"currentColor\" stroke-width=\"0\" transform=\"matrix(1 0 0 -1 0 0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D453\" x=\"0\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-28\" x=\"615\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"949\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-29\" x=\"1499\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-3D\" x=\"2110\" y=\"0\"></use>\n<g transform=\"translate(3074,0)\">\n <use xlink:href=\"#E1-STIXWEBSIZE1-222B\" x=\"0\" y=\"-616\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"1655\" y=\"1573\"></use>\n<g transform=\"translate(585,-921)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-221E\" x=\"685\" y=\"0\"></use>\n</g>\n</g>\n<g transform=\"translate(5066,0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D452\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(445,412)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(484,0)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.574)\" xlink:href=\"#E1-STIXWEBMAIN-32\" x=\"386\" y=\"446\"></use>\n</g>\n</g>\n</g>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D451\" x=\"6676\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"7208\" y=\"0\"></use>\n</g>\n</svg>"
            ],
            'test example mathjaxcomplete html output' => [
                "<span style=\"color:#202020;font-family:Barlow, '-apple-system', BlinkMacSystemFont, 'system-ui', sans-serif;font-size:14px;font-style:normal;font-weight:normal;letter-spacing:normal;text-align:left;text-indent:0px;text-transform:none;white-space:normal;word-spacing:0px;background-color:#eaf1f7;text-decoration:none;float:none;\"><svg xmlns:xlink=\"http://www.w3.org/1999/xlink\" width=\"17.471ex\" height=\"6.009ex\" style=\"vertical-align: -2.505ex;\" viewBox=\"0 -1508.9 7522 2587.3\" role=\"img\" focusable=\"false\" xmlns=\"http://www.w3.org/2000/svg\">\n<defs>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D453\" d=\"M248 437h37c63 0 58 77 101 141c31 47 74 90 150 90c50 0 79 -24 79 -59c0 -28 -21 -40 -41 -40c-28 0 -35 11 -35 25s9 29 9 37c0 5 -3 8 -15 8c-50 0 -89 -54 -110 -132l-19 -70h80l-12 -38h-76l-79 -306c-7 -27 -15 -65 -30 -102c-33 -83 -87 -178 -175 -178 c-48 0 -72 24 -72 54c0 24 14 49 43 49c27 0 37 -18 37 -33c0 -19 -16 -20 -16 -32c0 -6 6 -9 15 -9c65 0 95 158 110 215l87 342h-77Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-28\" d=\"M304 -161l-12 -16c-158 90 -244 259 -244 429c0 185 87 329 247 424l9 -16c-139 -119 -170 -212 -170 -405c0 -186 30 -299 170 -416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D465\" d=\"M305 288l45 69c26 40 61 84 111 84c32 0 49 -12 49 -33c0 -19 -13 -42 -39 -42c-22 0 -27 9 -44 9c-30 0 -41 -16 -55 -38l-55 -85c30 -62 53 -197 98 -197c16 0 40 14 61 57l14 -10c-12 -15 -57 -111 -115 -111c-41 0 -60 41 -73 77l-40 113l-63 -98 c-32 -49 -61 -92 -117 -92c-36 0 -52 25 -52 47c0 16 13 37 38 37c26 0 22 -20 47 -20c12 0 38 10 70 59l65 101l-33 92c-17 48 -43 89 -98 89c-13 0 -28 -2 -38 -6l-3 18l132 33c44 -24 63 -65 75 -98Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-29\" d=\"M29 660l12 16c153 -92 244 -259 244 -429c0 -185 -88 -327 -247 -424l-9 16c142 117 170 211 170 405c0 187 -25 302 -170 416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-3D\" d=\"M637 320h-589v66h589v-66zM637 120h-589v66h589v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBSIZE1-222B\" d=\"M280 -10l414 1680c48 195 126 330 268 330c52 0 73 -25 73 -55c0 -42 -32 -75 -71 -75c-19 0 -38 8 -38 30c0 27 21 37 23 51c0 7 -6 11 -15 11c-65 0 -94 -91 -126 -221l-419 -1704c-45 -181 -122 -306 -261 -306c-51 0 -72 25 -72 54c0 43 31 76 70 76 c19 0 38 -8 38 -30c0 -24 -21 -38 -23 -50c0 -8 6 -12 15 -12c65 0 91 88 124 221Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-2212\" d=\"M621 220h-557v66h557v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-221E\" d=\"M854 222c0 -97 -72 -222 -198 -222c-82 0 -139 53 -201 145c-45 -91 -113 -145 -209 -145c-86 0 -176 62 -176 200c0 184 133 230 191 230c73 0 123 -19 218 -147c46 83 92 147 193 147c90 0 182 -74 182 -208zM499 245v-3c72 -119 127 -183 195 -183 c56 0 100 55 100 119c0 91 -51 168 -139 168c-89 0 -127 -59 -156 -101zM433 181v3c-72 119 -135 187 -203 187c-56 0 -100 -55 -100 -119c0 -85 51 -168 139 -168c89 0 137 61 164 97Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D452\" d=\"M363 112l14 -13c-70 -86 -138 -110 -200 -110c-98 0 -137 84 -137 156c0 23 1 37 6 60c25 111 135 236 262 236c42 0 102 -14 102 -76c0 -127 -167 -176 -286 -182v-28c0 -64 52 -107 113 -107c42 0 90 18 126 64zM124 211h9c104 0 198 69 198 157c0 25 -19 43 -44 43 c-74 0 -134 -115 -163 -200Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D461\" d=\"M283 438l-10 -38h-70l-78 -320c-1 -3 -2 -8 -2 -11c0 -6 3 -12 16 -12c0 0 23 2 60 52l13 -11c-24 -31 -74 -107 -133 -107c-31 0 -39 25 -39 52c0 13 2 26 5 37l81 320h-77l5 29c75 14 136 51 177 138h13l-31 -129h70Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-32\" d=\"M474 137l-54 -137h-391v12l178 189c94 99 130 175 130 260c0 91 -54 141 -139 141c-72 0 -107 -32 -147 -130l-21 5c21 117 85 199 208 199c113 0 185 -77 185 -176c0 -79 -39 -154 -128 -248l-165 -176h234c42 0 63 11 96 67Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D451\" d=\"M527 668l-149 -598c-1 -3 -2 -9 -2 -12c0 -6 5 -9 15 -9c20 0 48 35 62 56l11 -12c-30 -45 -83 -105 -130 -105c-32 0 -40 23 -40 41c0 20 2 34 10 64h-5c-74 -93 -134 -105 -171 -105c-73 0 -88 74 -88 127c0 103 103 326 257 326c57 0 80 -26 81 -50h2l53 209 c1 4 2 8 2 12c0 13 -7 20 -33 20c-9 0 -20 -2 -27 -3l-4 15zM363 340c0 47 -15 71 -56 71c-99 0 -180 -200 -180 -296c0 -49 28 -66 56 -66c70 0 136 94 164 186c11 35 16 74 16 105Z\"></path>\n</defs>\n<g stroke=\"currentColor\" fill=\"currentColor\" stroke-width=\"0\" transform=\"matrix(1 0 0 -1 0 0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D453\" x=\"0\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-28\" x=\"615\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"949\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-29\" x=\"1499\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-3D\" x=\"2110\" y=\"0\"></use>\n<g transform=\"translate(3074,0)\">\n <use xlink:href=\"#E1-STIXWEBSIZE1-222B\" x=\"0\" y=\"-616\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"1655\" y=\"1573\"></use>\n<g transform=\"translate(585,-921)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-221E\" x=\"685\" y=\"0\"></use>\n</g>\n</g>\n<g transform=\"translate(5066,0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D452\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(445,412)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(484,0)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.574)\" xlink:href=\"#E1-STIXWEBMAIN-32\" x=\"386\" y=\"446\"></use>\n</g>\n</g>\n</g>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D451\" x=\"6676\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"7208\" y=\"0\"></use>\n</g>\n</svg></span>",
                "<span style=\"color:#202020;font-family:Barlow, '-apple-system', BlinkMacSystemFont, 'system-ui', sans-serif;font-size:14px;font-style:normal;font-weight:normal;letter-spacing:normal;text-align:left;text-indent:0px;text-transform:none;white-space:normal;word-spacing:0px;background-color:#eaf1f7;text-decoration:none;float:none;\"><svg xmlns:xlink=\"http://www.w3.org/1999/xlink\" width=\"17.471ex\" height=\"6.009ex\" style=\"vertical-align: -2.505ex;\" viewBox=\"0 -1508.9 7522 2587.3\" role=\"img\" focusable=\"false\" xmlns=\"http://www.w3.org/2000/svg\">\n<defs>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D453\" d=\"M248 437h37c63 0 58 77 101 141c31 47 74 90 150 90c50 0 79 -24 79 -59c0 -28 -21 -40 -41 -40c-28 0 -35 11 -35 25s9 29 9 37c0 5 -3 8 -15 8c-50 0 -89 -54 -110 -132l-19 -70h80l-12 -38h-76l-79 -306c-7 -27 -15 -65 -30 -102c-33 -83 -87 -178 -175 -178 c-48 0 -72 24 -72 54c0 24 14 49 43 49c27 0 37 -18 37 -33c0 -19 -16 -20 -16 -32c0 -6 6 -9 15 -9c65 0 95 158 110 215l87 342h-77Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-28\" d=\"M304 -161l-12 -16c-158 90 -244 259 -244 429c0 185 87 329 247 424l9 -16c-139 -119 -170 -212 -170 -405c0 -186 30 -299 170 -416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D465\" d=\"M305 288l45 69c26 40 61 84 111 84c32 0 49 -12 49 -33c0 -19 -13 -42 -39 -42c-22 0 -27 9 -44 9c-30 0 -41 -16 -55 -38l-55 -85c30 -62 53 -197 98 -197c16 0 40 14 61 57l14 -10c-12 -15 -57 -111 -115 -111c-41 0 -60 41 -73 77l-40 113l-63 -98 c-32 -49 -61 -92 -117 -92c-36 0 -52 25 -52 47c0 16 13 37 38 37c26 0 22 -20 47 -20c12 0 38 10 70 59l65 101l-33 92c-17 48 -43 89 -98 89c-13 0 -28 -2 -38 -6l-3 18l132 33c44 -24 63 -65 75 -98Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-29\" d=\"M29 660l12 16c153 -92 244 -259 244 -429c0 -185 -88 -327 -247 -424l-9 16c142 117 170 211 170 405c0 187 -25 302 -170 416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-3D\" d=\"M637 320h-589v66h589v-66zM637 120h-589v66h589v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBSIZE1-222B\" d=\"M280 -10l414 1680c48 195 126 330 268 330c52 0 73 -25 73 -55c0 -42 -32 -75 -71 -75c-19 0 -38 8 -38 30c0 27 21 37 23 51c0 7 -6 11 -15 11c-65 0 -94 -91 -126 -221l-419 -1704c-45 -181 -122 -306 -261 -306c-51 0 -72 25 -72 54c0 43 31 76 70 76 c19 0 38 -8 38 -30c0 -24 -21 -38 -23 -50c0 -8 6 -12 15 -12c65 0 91 88 124 221Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-2212\" d=\"M621 220h-557v66h557v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-221E\" d=\"M854 222c0 -97 -72 -222 -198 -222c-82 0 -139 53 -201 145c-45 -91 -113 -145 -209 -145c-86 0 -176 62 -176 200c0 184 133 230 191 230c73 0 123 -19 218 -147c46 83 92 147 193 147c90 0 182 -74 182 -208zM499 245v-3c72 -119 127 -183 195 -183 c56 0 100 55 100 119c0 91 -51 168 -139 168c-89 0 -127 -59 -156 -101zM433 181v3c-72 119 -135 187 -203 187c-56 0 -100 -55 -100 -119c0 -85 51 -168 139 -168c89 0 137 61 164 97Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D452\" d=\"M363 112l14 -13c-70 -86 -138 -110 -200 -110c-98 0 -137 84 -137 156c0 23 1 37 6 60c25 111 135 236 262 236c42 0 102 -14 102 -76c0 -127 -167 -176 -286 -182v-28c0 -64 52 -107 113 -107c42 0 90 18 126 64zM124 211h9c104 0 198 69 198 157c0 25 -19 43 -44 43 c-74 0 -134 -115 -163 -200Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D461\" d=\"M283 438l-10 -38h-70l-78 -320c-1 -3 -2 -8 -2 -11c0 -6 3 -12 16 -12c0 0 23 2 60 52l13 -11c-24 -31 -74 -107 -133 -107c-31 0 -39 25 -39 52c0 13 2 26 5 37l81 320h-77l5 29c75 14 136 51 177 138h13l-31 -129h70Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-32\" d=\"M474 137l-54 -137h-391v12l178 189c94 99 130 175 130 260c0 91 -54 141 -139 141c-72 0 -107 -32 -147 -130l-21 5c21 117 85 199 208 199c113 0 185 -77 185 -176c0 -79 -39 -154 -128 -248l-165 -176h234c42 0 63 11 96 67Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D451\" d=\"M527 668l-149 -598c-1 -3 -2 -9 -2 -12c0 -6 5 -9 15 -9c20 0 48 35 62 56l11 -12c-30 -45 -83 -105 -130 -105c-32 0 -40 23 -40 41c0 20 2 34 10 64h-5c-74 -93 -134 -105 -171 -105c-73 0 -88 74 -88 127c0 103 103 326 257 326c57 0 80 -26 81 -50h2l53 209 c1 4 2 8 2 12c0 13 -7 20 -33 20c-9 0 -20 -2 -27 -3l-4 15zM363 340c0 47 -15 71 -56 71c-99 0 -180 -200 -180 -296c0 -49 28 -66 56 -66c70 0 136 94 164 186c11 35 16 74 16 105Z\"></path>\n</defs>\n<g stroke=\"currentColor\" fill=\"currentColor\" stroke-width=\"0\" transform=\"matrix(1 0 0 -1 0 0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D453\" x=\"0\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-28\" x=\"615\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"949\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-29\" x=\"1499\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-3D\" x=\"2110\" y=\"0\"></use>\n<g transform=\"translate(3074,0)\">\n <use xlink:href=\"#E1-STIXWEBSIZE1-222B\" x=\"0\" y=\"-616\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"1655\" y=\"1573\"></use>\n<g transform=\"translate(585,-921)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-221E\" x=\"685\" y=\"0\"></use>\n</g>\n</g>\n<g transform=\"translate(5066,0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D452\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(445,412)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(484,0)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.574)\" xlink:href=\"#E1-STIXWEBMAIN-32\" x=\"386\" y=\"446\"></use>\n</g>\n</g>\n</g>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D451\" x=\"6676\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"7208\" y=\"0\"></use>\n</g>\n</svg></span>"
            ],
            'test complete mathjax html output with link at the end' => [
                "<span style=\"color:#202020;font-family:Barlow, '-apple-system', BlinkMacSystemFont, 'system-ui', sans-serif;font-size:14px;font-style:normal;font-weight:normal;letter-spacing:normal;text-align:left;text-indent:0px;text-transform:none;white-space:normal;word-spacing:0px;background-color:#eaf1f7;text-decoration:none;float:none;\"><svg xmlns:xlink=\"http://www.w3.org/1999/xlink\" width=\"17.471ex\" height=\"6.009ex\" style=\"vertical-align: -2.505ex;\" viewBox=\"0 -1508.9 7522 2587.3\" role=\"img\" focusable=\"false\" xmlns=\"http://www.w3.org/2000/svg\">\n<defs>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D453\" d=\"M248 437h37c63 0 58 77 101 141c31 47 74 90 150 90c50 0 79 -24 79 -59c0 -28 -21 -40 -41 -40c-28 0 -35 11 -35 25s9 29 9 37c0 5 -3 8 -15 8c-50 0 -89 -54 -110 -132l-19 -70h80l-12 -38h-76l-79 -306c-7 -27 -15 -65 -30 -102c-33 -83 -87 -178 -175 -178 c-48 0 -72 24 -72 54c0 24 14 49 43 49c27 0 37 -18 37 -33c0 -19 -16 -20 -16 -32c0 -6 6 -9 15 -9c65 0 95 158 110 215l87 342h-77Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-28\" d=\"M304 -161l-12 -16c-158 90 -244 259 -244 429c0 185 87 329 247 424l9 -16c-139 -119 -170 -212 -170 -405c0 -186 30 -299 170 -416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D465\" d=\"M305 288l45 69c26 40 61 84 111 84c32 0 49 -12 49 -33c0 -19 -13 -42 -39 -42c-22 0 -27 9 -44 9c-30 0 -41 -16 -55 -38l-55 -85c30 -62 53 -197 98 -197c16 0 40 14 61 57l14 -10c-12 -15 -57 -111 -115 -111c-41 0 -60 41 -73 77l-40 113l-63 -98 c-32 -49 -61 -92 -117 -92c-36 0 -52 25 -52 47c0 16 13 37 38 37c26 0 22 -20 47 -20c12 0 38 10 70 59l65 101l-33 92c-17 48 -43 89 -98 89c-13 0 -28 -2 -38 -6l-3 18l132 33c44 -24 63 -65 75 -98Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-29\" d=\"M29 660l12 16c153 -92 244 -259 244 -429c0 -185 -88 -327 -247 -424l-9 16c142 117 170 211 170 405c0 187 -25 302 -170 416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-3D\" d=\"M637 320h-589v66h589v-66zM637 120h-589v66h589v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBSIZE1-222B\" d=\"M280 -10l414 1680c48 195 126 330 268 330c52 0 73 -25 73 -55c0 -42 -32 -75 -71 -75c-19 0 -38 8 -38 30c0 27 21 37 23 51c0 7 -6 11 -15 11c-65 0 -94 -91 -126 -221l-419 -1704c-45 -181 -122 -306 -261 -306c-51 0 -72 25 -72 54c0 43 31 76 70 76 c19 0 38 -8 38 -30c0 -24 -21 -38 -23 -50c0 -8 6 -12 15 -12c65 0 91 88 124 221Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-2212\" d=\"M621 220h-557v66h557v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-221E\" d=\"M854 222c0 -97 -72 -222 -198 -222c-82 0 -139 53 -201 145c-45 -91 -113 -145 -209 -145c-86 0 -176 62 -176 200c0 184 133 230 191 230c73 0 123 -19 218 -147c46 83 92 147 193 147c90 0 182 -74 182 -208zM499 245v-3c72 -119 127 -183 195 -183 c56 0 100 55 100 119c0 91 -51 168 -139 168c-89 0 -127 -59 -156 -101zM433 181v3c-72 119 -135 187 -203 187c-56 0 -100 -55 -100 -119c0 -85 51 -168 139 -168c89 0 137 61 164 97Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D452\" d=\"M363 112l14 -13c-70 -86 -138 -110 -200 -110c-98 0 -137 84 -137 156c0 23 1 37 6 60c25 111 135 236 262 236c42 0 102 -14 102 -76c0 -127 -167 -176 -286 -182v-28c0 -64 52 -107 113 -107c42 0 90 18 126 64zM124 211h9c104 0 198 69 198 157c0 25 -19 43 -44 43 c-74 0 -134 -115 -163 -200Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D461\" d=\"M283 438l-10 -38h-70l-78 -320c-1 -3 -2 -8 -2 -11c0 -6 3 -12 16 -12c0 0 23 2 60 52l13 -11c-24 -31 -74 -107 -133 -107c-31 0 -39 25 -39 52c0 13 2 26 5 37l81 320h-77l5 29c75 14 136 51 177 138h13l-31 -129h70Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-32\" d=\"M474 137l-54 -137h-391v12l178 189c94 99 130 175 130 260c0 91 -54 141 -139 141c-72 0 -107 -32 -147 -130l-21 5c21 117 85 199 208 199c113 0 185 -77 185 -176c0 -79 -39 -154 -128 -248l-165 -176h234c42 0 63 11 96 67Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D451\" d=\"M527 668l-149 -598c-1 -3 -2 -9 -2 -12c0 -6 5 -9 15 -9c20 0 48 35 62 56l11 -12c-30 -45 -83 -105 -130 -105c-32 0 -40 23 -40 41c0 20 2 34 10 64h-5c-74 -93 -134 -105 -171 -105c-73 0 -88 74 -88 127c0 103 103 326 257 326c57 0 80 -26 81 -50h2l53 209 c1 4 2 8 2 12c0 13 -7 20 -33 20c-9 0 -20 -2 -27 -3l-4 15zM363 340c0 47 -15 71 -56 71c-99 0 -180 -200 -180 -296c0 -49 28 -66 56 -66c70 0 136 94 164 186c11 35 16 74 16 105Z\"></path>\n</defs>\n<g stroke=\"currentColor\" fill=\"currentColor\" stroke-width=\"0\" transform=\"matrix(1 0 0 -1 0 0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D453\" x=\"0\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-28\" x=\"615\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"949\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-29\" x=\"1499\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-3D\" x=\"2110\" y=\"0\"></use>\n<g transform=\"translate(3074,0)\">\n <use xlink:href=\"#E1-STIXWEBSIZE1-222B\" x=\"0\" y=\"-616\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"1655\" y=\"1573\"></use>\n<g transform=\"translate(585,-921)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-221E\" x=\"685\" y=\"0\"></use>\n</g>\n</g>\n<g transform=\"translate(5066,0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D452\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(445,412)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(484,0)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.574)\" xlink:href=\"#E1-STIXWEBMAIN-32\" x=\"386\" y=\"446\"></use>\n</g>\n</g>\n</g>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D451\" x=\"6676\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"7208\" y=\"0\"></use>\n</g>\n</svg></span><a%s href=\"https://www.ilias.de\">www.ilias.de</a>",
                "<span style=\"color:#202020;font-family:Barlow, '-apple-system', BlinkMacSystemFont, 'system-ui', sans-serif;font-size:14px;font-style:normal;font-weight:normal;letter-spacing:normal;text-align:left;text-indent:0px;text-transform:none;white-space:normal;word-spacing:0px;background-color:#eaf1f7;text-decoration:none;float:none;\"><svg xmlns:xlink=\"http://www.w3.org/1999/xlink\" width=\"17.471ex\" height=\"6.009ex\" style=\"vertical-align: -2.505ex;\" viewBox=\"0 -1508.9 7522 2587.3\" role=\"img\" focusable=\"false\" xmlns=\"http://www.w3.org/2000/svg\">\n<defs>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D453\" d=\"M248 437h37c63 0 58 77 101 141c31 47 74 90 150 90c50 0 79 -24 79 -59c0 -28 -21 -40 -41 -40c-28 0 -35 11 -35 25s9 29 9 37c0 5 -3 8 -15 8c-50 0 -89 -54 -110 -132l-19 -70h80l-12 -38h-76l-79 -306c-7 -27 -15 -65 -30 -102c-33 -83 -87 -178 -175 -178 c-48 0 -72 24 -72 54c0 24 14 49 43 49c27 0 37 -18 37 -33c0 -19 -16 -20 -16 -32c0 -6 6 -9 15 -9c65 0 95 158 110 215l87 342h-77Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-28\" d=\"M304 -161l-12 -16c-158 90 -244 259 -244 429c0 185 87 329 247 424l9 -16c-139 -119 -170 -212 -170 -405c0 -186 30 -299 170 -416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D465\" d=\"M305 288l45 69c26 40 61 84 111 84c32 0 49 -12 49 -33c0 -19 -13 -42 -39 -42c-22 0 -27 9 -44 9c-30 0 -41 -16 -55 -38l-55 -85c30 -62 53 -197 98 -197c16 0 40 14 61 57l14 -10c-12 -15 -57 -111 -115 -111c-41 0 -60 41 -73 77l-40 113l-63 -98 c-32 -49 -61 -92 -117 -92c-36 0 -52 25 -52 47c0 16 13 37 38 37c26 0 22 -20 47 -20c12 0 38 10 70 59l65 101l-33 92c-17 48 -43 89 -98 89c-13 0 -28 -2 -38 -6l-3 18l132 33c44 -24 63 -65 75 -98Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-29\" d=\"M29 660l12 16c153 -92 244 -259 244 -429c0 -185 -88 -327 -247 -424l-9 16c142 117 170 211 170 405c0 187 -25 302 -170 416Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-3D\" d=\"M637 320h-589v66h589v-66zM637 120h-589v66h589v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBSIZE1-222B\" d=\"M280 -10l414 1680c48 195 126 330 268 330c52 0 73 -25 73 -55c0 -42 -32 -75 -71 -75c-19 0 -38 8 -38 30c0 27 21 37 23 51c0 7 -6 11 -15 11c-65 0 -94 -91 -126 -221l-419 -1704c-45 -181 -122 -306 -261 -306c-51 0 -72 25 -72 54c0 43 31 76 70 76 c19 0 38 -8 38 -30c0 -24 -21 -38 -23 -50c0 -8 6 -12 15 -12c65 0 91 88 124 221Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-2212\" d=\"M621 220h-557v66h557v-66Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-221E\" d=\"M854 222c0 -97 -72 -222 -198 -222c-82 0 -139 53 -201 145c-45 -91 -113 -145 -209 -145c-86 0 -176 62 -176 200c0 184 133 230 191 230c73 0 123 -19 218 -147c46 83 92 147 193 147c90 0 182 -74 182 -208zM499 245v-3c72 -119 127 -183 195 -183 c56 0 100 55 100 119c0 91 -51 168 -139 168c-89 0 -127 -59 -156 -101zM433 181v3c-72 119 -135 187 -203 187c-56 0 -100 -55 -100 -119c0 -85 51 -168 139 -168c89 0 137 61 164 97Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D452\" d=\"M363 112l14 -13c-70 -86 -138 -110 -200 -110c-98 0 -137 84 -137 156c0 23 1 37 6 60c25 111 135 236 262 236c42 0 102 -14 102 -76c0 -127 -167 -176 -286 -182v-28c0 -64 52 -107 113 -107c42 0 90 18 126 64zM124 211h9c104 0 198 69 198 157c0 25 -19 43 -44 43 c-74 0 -134 -115 -163 -200Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D461\" d=\"M283 438l-10 -38h-70l-78 -320c-1 -3 -2 -8 -2 -11c0 -6 3 -12 16 -12c0 0 23 2 60 52l13 -11c-24 -31 -74 -107 -133 -107c-31 0 -39 25 -39 52c0 13 2 26 5 37l81 320h-77l5 29c75 14 136 51 177 138h13l-31 -129h70Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBMAIN-32\" d=\"M474 137l-54 -137h-391v12l178 189c94 99 130 175 130 260c0 91 -54 141 -139 141c-72 0 -107 -32 -147 -130l-21 5c21 117 85 199 208 199c113 0 185 -77 185 -176c0 -79 -39 -154 -128 -248l-165 -176h234c42 0 63 11 96 67Z\"></path>\n<path stroke-width=\"1\" id=\"E1-STIXWEBNORMALI-1D451\" d=\"M527 668l-149 -598c-1 -3 -2 -9 -2 -12c0 -6 5 -9 15 -9c20 0 48 35 62 56l11 -12c-30 -45 -83 -105 -130 -105c-32 0 -40 23 -40 41c0 20 2 34 10 64h-5c-74 -93 -134 -105 -171 -105c-73 0 -88 74 -88 127c0 103 103 326 257 326c57 0 80 -26 81 -50h2l53 209 c1 4 2 8 2 12c0 13 -7 20 -33 20c-9 0 -20 -2 -27 -3l-4 15zM363 340c0 47 -15 71 -56 71c-99 0 -180 -200 -180 -296c0 -49 28 -66 56 -66c70 0 136 94 164 186c11 35 16 74 16 105Z\"></path>\n</defs>\n<g stroke=\"currentColor\" fill=\"currentColor\" stroke-width=\"0\" transform=\"matrix(1 0 0 -1 0 0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D453\" x=\"0\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-28\" x=\"615\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"949\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-29\" x=\"1499\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBMAIN-3D\" x=\"2110\" y=\"0\"></use>\n<g transform=\"translate(3074,0)\">\n <use xlink:href=\"#E1-STIXWEBSIZE1-222B\" x=\"0\" y=\"-616\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D465\" x=\"1655\" y=\"1573\"></use>\n<g transform=\"translate(585,-921)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-221E\" x=\"685\" y=\"0\"></use>\n</g>\n</g>\n<g transform=\"translate(5066,0)\">\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D452\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(445,412)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBMAIN-2212\" x=\"0\" y=\"0\"></use>\n<g transform=\"translate(484,0)\">\n <use transform=\"scale(0.707)\" xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"0\" y=\"0\"></use>\n <use transform=\"scale(0.574)\" xlink:href=\"#E1-STIXWEBMAIN-32\" x=\"386\" y=\"446\"></use>\n</g>\n</g>\n</g>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D451\" x=\"6676\" y=\"0\"></use>\n <use xlink:href=\"#E1-STIXWEBNORMALI-1D461\" x=\"7208\" y=\"0\"></use>\n</g>\n</svg></span>www.ilias.de"
            ],
        ];

        return array_map(static function (array $arguments) use ($attributes): array {
            return [str_replace('%s', $attributes, $arguments[0]), $arguments[1]];
        }, $templates);
    }
}
