<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

/**
 * User Interface for Place Holder Management
 *
 * @author Hendrik Holtmann <holtmann@me.com>
 * @ilCtrl_Calls ilPCPlaceHolderGUI: ilPCMediaObjectGUI
 */
class ilPCPlaceHolderGUI extends ilPageContentGUI
{
    public const TYPE_TEXT = "Text";
    public const TYPE_QUESTION = "Question";
    public const TYPE_MEDIA = "Media";
    public const TYPE_VERIFICATION = "Verification";
    protected ilPropertyFormGUI $form_gui;

    public function __construct(
        ilPageObject $a_pg_obj,
        ?ilPageContent $a_content_obj,
        string $a_hier_id,
        string $a_pc_id = ""
    ) {
        global $DIC;

        $this->lng = $DIC->language();
        $this->pg_obj = $a_pg_obj;
        $this->content_obj = $a_content_obj;
        $this->hier_id = $a_hier_id;
        $this->pc_id = $a_pc_id;

        parent::__construct($a_pg_obj, $a_content_obj, $a_hier_id, $a_pc_id);
    }

    public function executeCommand(): void
    {
        // get next class that processes or forwards current command
        $next_class = $this->ctrl->getNextClass($this);
        // get current command
        $cmd = $this->ctrl->getCmd();

        switch ($next_class) {
            case 'ilpcmediaobjectgui':  //special handling
                $media_gui = new ilPCMediaObjectGUI($this->pg_obj, $this->content_obj, $this->hier_id, $this->pc_id);
                $ret = $this->ctrl->forwardCommand($media_gui);
                break;

            default:
                $this->$cmd();
                break;
        }
    }

    protected function insert(): void
    {
        $this->propertyGUI("create", self::TYPE_TEXT, "100px", "insert");
    }

    protected function create(): void
    {
        $plach_height = $this->request->getString("plach_height");
        if ($plach_height == "" ||
            !preg_match("/[0-9]+/", $plach_height)) {
            $this->insert();
            return;
        }

        $this->content_obj = new ilPCPlaceHolder($this->getPage());
        $this->content_obj->create($this->pg_obj, $this->hier_id, $this->pc_id);
        $this->content_obj->setHeight($plach_height . "px");
        $this->content_obj->setContentClass(
            $this->request->getString("plach_type")
        );
        $this->updated = $this->pg_obj->update();
        if ($this->updated === true) {
            $this->ctrl->returnToParent($this, "jump" . $this->hier_id);
        } else {
            $this->insert();
        }
    }

    public function edit(): void
    {
        if ($this->getPageConfig()->getEnablePCType("PlaceHolder")) {
            $this->edit_object();
        } else {
            $this->forward_edit();
        }
    }

    public function setStyleId(int $a_styleid): void
    {
        $this->styleid = $a_styleid;
    }

    public function getStyleId(): int
    {
        return $this->styleid;
    }

    protected function edit_object(): void
    {
        $this->propertyGUI(
            "saveProperties",
            $this->content_obj->getContentClass(),
            $this->content_obj->getHeight(),
            "save"
        );
    }

    protected function forward_edit(): void
    {
        switch ($this->content_obj->getContentClass()) {
            case self::TYPE_MEDIA:
                $this->ctrl->saveParameterByClass(
                    ilPCMediaObjectGUI::class,
                    ["pc_id", "hier_id", "obj_id", "active_node"]
                );
                $this->ctrl->redirectByClass(
                    [ilPCMediaObjectGUI::class],
                    "insert"
                );
                break;

            case self::TYPE_QUESTION:
                $this->ctrl->saveParameterByClass(
                    ilPCQuestionGUI::class,
                    ["pc_id", "hier_id", "obj_id", "active_node"]
                );
                $this->ctrl->redirectByClass(
                    [ilPCQuestionGUI::class],
                    "insert"
                );
                break;

            default:
                break;
        }
    }


    /**
     * save placeholder properties in db and return to page edit screen
     */
    protected function saveProperties(): void
    {
        $plach_height = $this->request->getString("plach_height");
        if ($plach_height == "" ||
            !preg_match("/[0-9]+/", $plach_height)) {
            $this->edit_object();
            return;
        }

        $this->content_obj->setContentClass($this->request->getString("plach_type"));
        $this->content_obj->setHeight($plach_height . "px");

        $this->updated = $this->pg_obj->update();
        if ($this->updated === true) {
            $this->ctrl->returnToParent($this, "jump" . $this->hier_id);
        } else {
            $this->pg_obj->addHierIDs();
            $this->edit();
        }
    }

    /**
     * Property Form
     */
    public function initCreationForm(): ilPropertyFormGUI
    {
        $lng = $this->lng;

        $this->form_gui = new ilPropertyFormGUI();
        $this->form_gui->setFormAction($this->ctrl->getFormAction($this));
        $this->form_gui->setTitle($lng->txt("cont_ed_plachprop"));

        $ttype_input = new ilRadioGroupInputGUI($lng->txt("type"), "plach_type");
        $type_captions = $this->getTypeCaptions();
        foreach ($this->getAvailableTypes("") as $type) {
            $ttype_input->addOption(new ilRadioOption($type_captions[$type], $type));
        }
        $ttype_input->setValue("Text");
        $ttype_input->setRequired(true);
        $this->form_gui->addItem($ttype_input);

        /*
        $theight_input = new ilTextInputGUI($lng->txt("height"), "plach_height");
        $theight_input->setSize(4);
        $theight_input->setMaxLength(3);
        $theight_input->setTitle($lng->txt("height") . " (px)");
        $theight_input->setRequired(true);
        $this->form_gui->addItem($theight_input);

        $theight_input->setValue(preg_replace("/px/", "", $a_height));*/
        if ($this->content_obj) {
            $ttype_input->setValue($this->content_obj->getContentClass());
        }

        $this->form_gui->addCommandButton("insert", $lng->txt("insert"));
        $this->form_gui->addCommandButton("cancelCreate", $lng->txt("cancel"));
        return $this->form_gui;
    }

    public function cancel(): void
    {
        $this->ctrl->returnToParent($this, "jump" . $this->hier_id);
    }

    protected function getAvailableTypes(
        string $a_selected_type = ""
    ): array {
        // custom config?
        if (method_exists($this->getPage()->getPageConfig(), "getAvailablePlaceholderTypes")) {
            $types = $this->getPage()->getPageConfig()->getAvailablePlaceholderTypes();
        } else {
            $types = array(self::TYPE_TEXT, self::TYPE_MEDIA, self::TYPE_QUESTION);
        }
        return $types;
    }

    protected function getTypeCaptions(): array
    {
        $lng = $this->lng;

        return array(
                self::TYPE_TEXT => $lng->txt("cont_ed_plachtext"),
                self::TYPE_MEDIA => $lng->txt("cont_ed_plachmedia"),
                self::TYPE_QUESTION => $lng->txt("cont_ed_plachquestion"),
                self::TYPE_VERIFICATION => $lng->txt("cont_ed_plachverification")
            );
    }
}
