<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

namespace ILIAS\UI\Component\ViewControl;

use ILIAS\UI\Component\Button\Button;
use ILIAS\UI\Component\Component;
use ILIAS\UI\Component\Button\Month;

/**
 * This is how the factory for UI elements looks.
 */
interface Factory
{
    /**
     * ---
     * description:
     *   purpose: >
     *      Mode View Controls enable the switching between different aspects of some data. The different modes are mutually
     *      exclusive and can therefore not be activated at once.
     *   composition: >
     *      Mode View Controls are composed of Buttons switching between active/engaged and inactive states.
     *   effect: >
     *      Clicking on an inactive Button turns this button active/engaged and all other inactive.
     *      Clicking on an active/engaged button has no effect.
     * rules:
     *   usage:
     *      1: Exactly one Button MUST always be active/engaged.
     *   accessibility:
     *      1: The HTML container enclosing the buttons of the Mode View Control MUST cary the role-attribute "group".
     *      2: The HTML container enclosing the buttons of the Mode View Control MUST set an aria-label describing the element. Eg. "Mode View Control"
     * ---
     * @param    array $labelled_actions Set of labelled actions (string|string)[]. The label of the action is used as key, the action itself as value.
     *          The first of the actions will be activated by default.
     * @param string $aria_label Defines the functionality.
     * @return \ILIAS\UI\Component\ViewControl\Mode
     */
    public function mode(array $labelled_actions, string $aria_label): Mode;

    /**
     * ---
     * description:
     *   purpose: >
     *      Section View Controls enable the switching between different sections of some data. Examples are subsequent
     *      days/weeks/months in a calendar or entries in a blog.
     *   composition: >
     *      Section View Controls are composed of three Buttons. The Button on the left caries a Back Glyph, the Button
     *      in the middle is either a Default- or Split Button labeling the data displayed below and the Button on the right carries
     *      a next Glyph.
     *   effect: >
     *      Clicking on the Buttons left or right changes the selection of the displayed data by a fixed interval. Clicking
     *      the Button in the middle opens the sections hinted by the label of the button (e.g. "Today").
     * ---
     * @param   \ILIAS\UI\Component\Button\Button $previous_action Button to be placed in the left.
     * @param   \ILIAS\UI\Component\Button\Button|\ILIAS\UI\Component\Button\Month $button Button to be placed in the middle (Month Button or Default Button).
     * @param   \ILIAS\UI\Component\Button\Button $next_action Button to be placed in the right.
     * @return \ILIAS\UI\Component\ViewControl\Section
     */
    public function section(Button $previous_action, Component $button, Button $next_action): Section;

    /**
     * ---
     * description:
     *   purpose: >
     *      The Sortation View Control enables users to change the order in which
     *      some data is presented.
     *      This control applies to all sorts of _structured_ data, like tables and lists.
     *   composition: >
     *      Sortation uses a Dropdown to display a collection of shy-buttons.
     *   effect: >
     *      A click on an option will change the ordering of the associated data-list
     *      by calling a page with a parameter according to the selected option or triggering a signal.
     *      The label displayed in the dropdown will be set to the selected sorting.
     *      Also, the according entry in the dropdown will be marked as selected by an icon.
     *
     * rules:
     *   usage:
     *      1: A Sortation MUST NOT be used standalone.
     *      2: Sortations MUST BE visually close to the list or table their operation will have effect upon.
     *      3: There SHOULD NOT be more than one Sortation per view.
     *      4: >
     *        The current ordering SHOULD be indicated by the Sortation's label.
     *        When space is limited, the label MAY be omitted.
     *      5: >
     *        The labels of the options SHOULD be formulated as short as possible;
     *        the currently selected label will be prefixed by somthing like 'sorted by:'.
     *
     *   accessibility:
     *      1: Sortation MUST be operable via keyboard only.
     *
     * background:
     *   https://mantis.ilias.de/view.php?id=26634
     *
     * ---
     * @param array<string,string> $options a dictionary with value=>title
     * @param string $selected a value from $options
     * @return \ILIAS\UI\Component\ViewControl\Sortation
     */
    public function sortation(array $options, string $selected): Sortation;

    /**
     * ---
     * description:
     *   purpose: >
     *      Pagination allows structured data being displayed in chunks by
     *      limiting the number of entries shown. It provides the user with
     *      controls to leaf through the chunks of entries.
     *   composition: >
     *      Pagination is a collection of shy-buttons to access distinct chunks
     *      of data, framed by next/back glyphs.
     *      When used with the "DropdownAt" option, a dropdown is rendered if
     *      the number of chunks exceeds the option's value.
     *   effect: >
     *      A click on a chunk-option will change the offset of the displayed data-list,
     *      thus displaying the respective chunk of entries.
     *      The active option is rendered as an unavailable shy-button.
     *      Clicking the next/back-glyphs, the previous (respectively: the next)
     *      chunk of entries is being displayed.
     *      If a previous/next chunk is not available, the glyph is
     *      rendered unavailable.
     *      If the pagination is used with a maximum of chunk-options to be shown,
     *      both first and last options are always displayed.
     *
     * rules:
     *   usage:
     *      1: A Pagination MUST only be used for structured data, like tables and lists.
     *      2: A Pagination MUST NOT be used standalone.
     *      3: Paginations MUST be visually close to the list or table their operation will have effect upon. They MAY be placed directly above and/or below the list.
     *      4: You MUST use the default label if dealing with tables.
     *      5: >
     *        You MAY use a different label, if the default one is not working for
     *        the use case. But indicating the total number of items (X of Y)
     *        MUST be kept anyway.
     *   accessibility:
     *      1: Pagination MUST be operable via keyboard only.
     *
     * ---
     * @return \ILIAS\UI\Component\ViewControl\Pagination
     */
    public function pagination(): Pagination;
}
