/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Observable, root, Window } from "../component/index.js";
import { FunctionUtil } from "../util/index.js";
import { BrowserStore } from "../util/BrowserStorage.js";
import { t } from "../Translate";
/**
 * @category Data
 */
export var CommitErrorType;
(function (CommitErrorType) {
    CommitErrorType[CommitErrorType["forbidden"] = 0] = "forbidden";
    CommitErrorType[CommitErrorType["overQuota"] = 1] = "overQuota";
    CommitErrorType[CommitErrorType["tooLarge"] = 2] = "tooLarge";
    CommitErrorType[CommitErrorType["rateLimit"] = 3] = "rateLimit";
    CommitErrorType[CommitErrorType["notFound"] = 4] = "notFound";
    CommitErrorType[CommitErrorType["invalidPatch"] = 5] = "invalidPatch";
    CommitErrorType[CommitErrorType["willDestroy"] = 6] = "willDestroy";
    CommitErrorType[CommitErrorType["invalidProperties"] = 7] = "invalidProperties";
    CommitErrorType[CommitErrorType["singleton"] = 8] = "singleton";
    CommitErrorType[CommitErrorType["requestTooLarge"] = 9] = "requestTooLarge";
    CommitErrorType[CommitErrorType["stateMismatch"] = 10] = "stateMismatch";
})(CommitErrorType || (CommitErrorType = {}));
/**
 * Abstract DataSource class
 *
 * A DataSource collection is a single source of truth for all types of data.
 * When the DataSource changes it fires an event. All components and stores listen to the
 * 'change' event to update themselves. This approach reduces the amount of code that has
 * to be written and maintained.
 *
 * Use a {@see DataSourceStore} in components to list data from datasources.
 * The {@see Form} component can also load from a datasource.
 *
 * @category Data
 */
export class AbstractDataSource extends Observable {
    /**
     * Get the local server state ID of the store
     * @protected
     */
    async getState() {
        if (!this._state && this.persist) {
            this._state = await this.browserStore.getItem("__state__");
        }
        return this._state;
    }
    /**
     * Set's the local server state ID
     *
     * Setting it to undefined will reset the store.
     *
     * @param state
     * @protected
     */
    async setState(state) {
        this._state = state;
        if (!this.persist) {
            return;
        }
        return this.browserStore.setItem("__state__", state);
    }
    clearCache() {
        this.data = {};
        return this.browserStore.clear();
    }
    /**
     * Get the browser storage object to save state to the browser
     * @private
     */
    get browserStore() {
        if (!this._browserStore) {
            this._browserStore = new BrowserStore("ds-" + this.id);
        }
        return this._browserStore;
    }
    constructor(id) {
        super();
        this.id = id;
        /**
         * Store data in the browser storage so it will persist across sessions
         */
        this.persist = true;
        /**
         * Extra parameters to send to the Foo/set
         */
        this.commitBaseParams = {};
        /**
         * Extra /set parameters that will reset after commit
         */
        this.setParams = {};
        this.data = {};
        this.creates = {};
        this.updates = {};
        this.destroys = {};
        this.getIds = {};
        this._createId = 0;
        this.delayedCommit = FunctionUtil.buffer(0, () => {
            void this.commit();
        });
        this.delayedGet = FunctionUtil.buffer(0, () => {
            void this.doGet();
        });
    }
    /**
     * Get entities from the store
     *
     * It will return a list of entities ordered by the requested ID's
     *
     * @param ids
     */
    async get(ids) {
        const promises = [], order = {};
        if (ids == undefined) {
            ids = (await this.query()).ids;
        }
        //first see if we have it in our data property
        ids.forEach((id, index) => {
            //keep order for sorting the result
            order[id] = index++;
            promises.push(this.single(id));
        });
        // Call class method to fetch additional
        let entities = await Promise.all(promises);
        const response = {
            list: [],
            notFound: [],
            state: await this.getState()
        };
        entities.forEach((e, index) => {
            if (e === undefined) {
                response.notFound.push(ids[index]);
            }
            else {
                response.list.push(e);
            }
        });
        response.list = response.list.sort(function (a, b) {
            return order[a.id] - order[b.id];
        });
        return response;
    }
    async add(data) {
        this.data[data.id] = data;
        if (!this.persist) {
            return Promise.resolve(data);
        }
        await this.browserStore.setItem(data.id, data);
        return data;
    }
    async remove(id) {
        console.debug("Removing " + this.id + ": " + id);
        delete this.data[id];
        if (!this.persist) {
            return Promise.resolve(id);
        }
        await this.browserStore.removeItem(id);
        return id;
    }
    /**
     * Get a single entity.
     *
     * Multiple calls will be buffered and returned together on the next event loop. This way multiple calls can
     * be joined together in a single HTTP request to the server.
     *
     * @param id
     */
    async single(id) {
        const p = new Promise((resolve, reject) => {
            if (!this.getIds[id]) {
                this.getIds[id] = {
                    resolves: [resolve],
                    rejects: [reject]
                };
            }
            else {
                this.getIds[id].resolves.push(resolve);
                this.getIds[id].rejects.push(reject);
            }
        });
        this.delayedGet();
        return p;
    }
    returnGet(id) {
        let r;
        if (!this.getIds[id]) {
            return;
        }
        while (r = this.getIds[id].resolves.shift()) {
            // this.getIds[id].rejects.shift();
            r.call(this, structuredClone(this.data[id]));
        }
        delete this.getIds[id];
    }
    /**
     * Does the actual getting of entities. First checks if present in this onbject, otherwise it will be requested
     * from the remote source.
     *
     * @protected
     */
    async doGet() {
        const unknownIds = [];
        for (let id in this.getIds) {
            if (this.data[id]) {
                this.returnGet(id);
            }
            else if (this.persist) {
                const data = await this.browserStore.getItem(id);
                if (data) {
                    this.data[id] = data;
                    this.returnGet(id);
                }
                else {
                    unknownIds.push(id);
                }
            }
            else {
                unknownIds.push(id);
            }
        }
        if (!unknownIds.length) {
            // Can we return without a server call? State won't be checked.
            // In the detail view we call an additional validateState() function to do this to
            // save a lot of empty calls.
            return;
        }
        this.internalGet(unknownIds)
            .then(response => this.checkState(response.state, response))
            .then(response => {
            var _a;
            response.list.forEach((e) => {
                this.add(e);
                this.returnGet(e.id);
            });
            (_a = response.notFound) === null || _a === void 0 ? void 0 : _a.forEach((id) => {
                let r;
                while (r = this.getIds[id].resolves.shift()) {
                    r.call(this, undefined);
                }
                delete this.getIds[id];
            });
        }).catch((e) => {
            //reject all
            unknownIds.forEach((id) => {
                if (this.getIds[id]) {
                    let r;
                    while (r = this.getIds[id].rejects.shift()) {
                        r.call(this, e);
                    }
                    delete this.getIds[id];
                }
            });
        });
    }
    /**
     * Create entity
     *
     * Multiple calls will be joined together in a single call on the next event loop
     *
     * @param data
     * @param createId The create ID to use when committing this entity to the server
     */
    create(data, createId) {
        if (createId === undefined) {
            createId = this.createID();
        }
        const p = new Promise((resolve, reject) => {
            this.creates[createId] = {
                data: data,
                resolve: resolve,
                reject: reject
            };
        }).finally(() => {
            delete this.creates[createId];
        });
        this.delayedCommit();
        return p;
    }
    /**
     * Reset the data source.
     *
     * Clears all data and will resync
     */
    async reset() {
        return this.setState(undefined);
    }
    /**
     * Update an entity
     *
     * Multiple calls will be joined together in a single call on the next event loop
     *
     * @param id
     * @param data
     */
    update(id, data) {
        const p = new Promise((resolve, reject) => {
            this.updates[id] = {
                data: data,
                resolve: resolve,
                reject: reject
            };
        }).finally(() => {
            delete this.updates[id];
        });
        this.delayedCommit();
        return p;
    }
    createID() {
        return "_new_" + (++this._createId);
    }
    /**
     * Destroy an entity
     *
     * Multiple calls will be joined together in a single call on the next event loop
     *
     * @param id
     */
    destroy(id) {
        const p = new Promise((resolve, reject) => {
            this.destroys[id] = {
                resolve: resolve,
                reject: reject
            };
        }).finally(() => {
            delete this.destroys[id];
        });
        this.delayedCommit();
        return p;
    }
    /**
     * Ask for confirmation and delete entities by ID
     *
     * @example
     * ```
     * const tbl = this.projectTable!,
     * 	ids = tbl.rowSelection!.selected.map(index => tbl.store.get(index)!.id);
     *
     * const result = await jmapds("Project3")
     * 	.confirmDestroy(ids);
     *
     * if(result != false) {
     * 	btn.parent!.hide();
     * }
     * ```
     * @param ids The ID's to delete
     */
    async confirmDestroy(ids) {
        const count = ids.length;
        if (!count) {
            return false;
        }
        let msg;
        if (count == 1) {
            msg = t("Are you sure you want to delete the selected item?");
        }
        else {
            msg = t("Are you sure you want to delete {count} items?").replace('{count}', count);
        }
        const confirmed = await Window.confirm(msg);
        if (!confirmed) {
            return false;
        }
        root.mask(300);
        return Promise.all(ids.map(id => {
            return this.destroy(id);
        })).finally(() => {
            root.unmask();
        });
    }
    /**
     * Fetch updates from remote
     */
    async updateFromServer() {
        let hasMoreChanges = true, hasAChange = false;
        const allChanges = {
            created: [],
            updated: [],
            destroyed: [],
            oldState: "",
            newState: "",
        }, promises = [];
        try {
            while (hasMoreChanges) {
                const state = await this.getState();
                if (state === undefined) {
                    // no state so nothing to update
                    return;
                }
                if (!allChanges.oldState) {
                    allChanges.oldState = state;
                }
                const changes = await this.internalRemoteChanges(state);
                if (changes.created) {
                    for (let id of changes.created) {
                        promises.push(this.remove(id));
                        allChanges.created.push(id + "");
                        hasAChange = true;
                    }
                }
                if (changes.updated) {
                    for (let id of changes.updated) {
                        promises.push(this.remove(id));
                        allChanges.updated.push(id + "");
                        hasAChange = true;
                    }
                }
                if (changes.destroyed) {
                    for (let id of changes.destroyed) {
                        promises.push(this.remove(id));
                        allChanges.destroyed.push(id + "");
                        hasAChange = true;
                    }
                }
                //Set the new server state
                await Promise.all(promises);
                await this.setState(changes.newState);
                allChanges.newState = changes.newState;
                hasMoreChanges = !!changes.hasMoreChanges;
            }
        }
        catch (e) {
            console.error(this.id + " Error while updating from server. Resetting data source.");
            console.error(e);
            await this.reset();
        }
        if (hasAChange) {
            this.fire("change", this, allChanges);
        }
    }
    /**
     * Commit pending changes to remote
     */
    async commit() {
        const params = Object.assign({
            create: {},
            update: {},
            destroy: [],
            ifInState: await this.getState(),
        }, this.commitBaseParams, this.setParams);
        this.setParams = {}; // unset after /set is sent
        for (let id in this.creates) {
            params.create[id] = this.creates[id].data;
        }
        for (let id in this.updates) {
            params.update[id] = this.updates[id].data;
        }
        for (let id in this.destroys) {
            params.destroy.push(id);
        }
        this.internalCommit(params).then(async (response) => {
            if (response.created) {
                for (let clientId in response.created) {
                    //merge client data with server defaults.
                    let data = Object.assign(params.create ? (params.create[clientId] || {}) : {}, response.created[clientId] || {});
                    this.add(data).then(() => this.creates[clientId].resolve(data));
                }
            }
            if (response.notCreated) {
                for (let clientId in response.notCreated) {
                    //merge client data with server defaults.
                    this.creates[clientId].reject(response.notCreated[clientId]);
                }
            }
            if (response.updated) {
                for (let serverId in response.updated) {
                    //server updated something we don't have
                    if (!this.data[serverId]) {
                        continue;
                    }
                    //merge existing data, with updates from client and server
                    let data = params.update && params.update[serverId] ? Object.assign(this.data[serverId], params.update[serverId]) : this.data[serverId];
                    data = Object.assign(data, response.updated[serverId] || {});
                    this.add(data).then((data) => this.updates[serverId].resolve(data));
                }
            }
            if (response.notUpdated) {
                for (let serverId in response.notUpdated) {
                    //merge client data with server defaults.
                    this.updates[serverId].reject(response.notUpdated[serverId]);
                }
            }
            if (response.destroyed) {
                for (let i = 0, l = response.destroyed.length; i < l; i++) {
                    this.remove(response.destroyed[i]).then((id) => this.destroys[id].resolve(id));
                }
            }
            if (response.notDestroyed) {
                for (let serverId in response.notDestroyed) {
                    this.destroys[serverId].reject(response.notDestroyed[serverId]);
                }
            }
            await this.setState(response.newState);
            this.fire("change", this, {
                created: response.created ? Object.keys(response.created) : [],
                updated: response.updated ? Object.keys(response.updated) : [],
                destroyed: response.destroyed || [],
                oldState: response.oldState,
                newState: response.newState
            });
        })
            .catch(e => {
            for (let clientId in this.creates) {
                this.creates[clientId].reject(e);
            }
            for (let clientId in this.updates) {
                this.updates[clientId].reject(e);
            }
            for (let clientId in this.destroys) {
                this.destroys[clientId].reject(e);
            }
        })
            .finally(() => {
            this.creates = {};
            this.updates = {};
            this.destroys = {};
        });
    }
    /**
     * Query the server for a list of entity ID's
     *
     * It takes filters and sort parameters.
     *
     * @link https://jmap.io/spec-core.html#query
     */
    async query(params = {}) {
        let r = await this.internalQuery(params);
        return this.checkState(r.queryState, r);
    }
    /**
     * Check's if we are up-to-date with the server and fetches updates if needed.
     *
     * If no state is returned by the data source this function will ignore states and the data source should then
     * always refresh data.
     *
     * @param serverState
     * @param retVal
     * @private
     */
    async checkState(serverState, retVal) {
        let state = await this.getState();
        if (!state && serverState) {
            // // We are empty!
            // if(this.persist) {
            // 	console.warn("Emptying store as there's no server state")
            // 	this.data = {};
            // 	await this.browserStore.clear();
            // }
            await this.setState(serverState);
            state = serverState;
        }
        // Check if our data is up-to-date
        if (serverState != state) {
            return this.updateFromServer().then(() => retVal);
        }
        else {
            return Promise.resolve(retVal);
        }
    }
}
//# sourceMappingURL=AbstractDataSource.js.map