/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { createComponent } from "../Component.js";
import { t } from "../../Translate";
import { InputField } from "./InputField";
/**
 * NumberField component
 */
export class NumberField extends InputField {
    constructor() {
        super();
        this.baseCls = 'goui-form-field number';
        this.type = "number";
    }
    validate() {
        super.validate();
        const v = this.value;
        if (v && isNaN(v)) {
            this.setInvalid("Incorrect number format");
        }
        if (this.max !== undefined && v && v > this.max) {
            this.setInvalid(t("Number is too big"));
        }
        if (this.min !== undefined && (!v || v < this.min)) {
            this.setInvalid(t("Number is too small"));
        }
    }
    set value(v) {
        if (isNaN(v)) {
            throw new Error("Invalid number");
        }
        else if (!this.isEmptyNumber(v)) {
            super.value = +v.toFixed(this.decimals);
        }
    }
    isEmptyNumber(v) {
        return (v === undefined || v === null || v === "");
    }
    get value() {
        const v = super.value;
        return (this.isEmptyNumber(v) || isNaN(v)) ? undefined : +(+v).toFixed(this.decimals);
    }
    /**
     * The step attribute is a number that specifies the granularity that the value must adhere to or the keyword any.
     *
     * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/step
     * @param step
     */
    set step(step) {
        if (!step) {
            this.input.attr('step', undefined);
        }
        else {
            this.input.attr('step', step);
        }
    }
    get step() {
        return parseFloat(this.input.attr('step'));
    }
    /**
     * Set the number of decimals. It uses the step attribute to accomplish this.
     *
     * @param decimals
     */
    set decimals(decimals) {
        if (!decimals) {
            this.input.attr('step', undefined);
        }
        else {
            this.input.attr('step', '0.'.padEnd(decimals + 1, "0") + "1");
        }
    }
    get decimals() {
        const step = this.input.attr('step');
        if (!step) {
            return undefined;
        }
        return step.length - 2;
    }
    /**
     * The minimum number allowed
     *
     * @param min
     */
    set min(min) {
        this.input.attr('min', min);
    }
    get min() {
        return parseInt(this.input.attr('min'));
    }
    /**
     * The maximum number allowed
     *
     * @param max
     */
    set max(max) {
        this.input.attr('max', max);
    }
    get max() {
        return parseInt(this.input.attr('max'));
    }
}
export const numberfield = (config) => createComponent(new NumberField(), config);
//# sourceMappingURL=NumberField.js.map