/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component, createComponent } from "./Component.js";
import { Menu } from "./menu/Menu.js";
import { router } from "../Router.js";
/**
 * Button component
 *
 * @example
 *
 * ```typescript
 * btn({
 *   icon: "star",
 *   text: "Test 1"
 *   handler: (e) => alert("Hi!")
 * })
 * ```
 *
 */
export class Button extends Component {
    constructor() {
        super("button");
        this.baseCls = "goui-button";
        /**
         * Turn on if you want this button to be clickable fast.
         * We disable this by default because some users tend to double click on all buttons and we don't
         * want to double submit.
         */
        this.allowFastClick = false;
        this.type = "button";
    }
    /**
     * Find the first menu in the tree of submenu's
     */
    findTopMenu() {
        if (!(this.parent instanceof Menu)) {
            return undefined;
        }
        if (!(this.parent.parent instanceof Button)) {
            return this.parent;
        }
        else {
            const next = this.parent.parent.findTopMenu();
            if (next) {
                return next;
            }
            else {
                return this.parent;
            }
        }
    }
    /**
     * Button type. "button" or "submit", defaults to "button".
     */
    set type(type) {
        this.el.type = type;
    }
    get type() {
        return this.el.type;
    }
    internalRender() {
        const el = super.internalRender();
        if (this.route != undefined) {
            if (this.handler) {
                throw "You can't set both handler and route on a button";
            }
            this.handler = () => {
                router.goto(this.route);
            };
        }
        // The first menu of a button will expand on click, sub menus will show on hover and are hidden with css.
        // Before I made this without JS with the :focus-within selector but that didn't work in Safari because it
        // doesn't focus buttons on click.
        // First menu is rendered directly in body so it's positioned absolute on the page and there's no need for overflow
        // visible in windows. Sub menu's are rendered inside the parent menu button.
        if (this.menu) {
            this.menu.hide();
            if (!(this.parent instanceof Menu)) {
                // When a menu is opened. other top level will open on mouse enter
                this.el.addEventListener("mouseenter", this.onMenuMouseEnter.bind(this));
                this.el.addEventListener("click", this.onMenuButtonClick.bind(this));
            }
            else {
                // Setting renderTo to undefined will make it render to it's parent
                // which is this button
                this.menu.renderTo = undefined;
            }
        }
        el.addEventListener("click", (e) => {
            // check detail for being the first click. We don't want double clicks to call the handler twice.
            // the detail property contains the click count. When spacebar is used it will be 0
            // Michael had problems with e.detail < 2 but we don't remember why. Discuss when we run into this.
            if (this.handler && e.button == 0 && (this.allowFastClick || e.detail < 2)) {
                this.handler.call(this, this, e);
                // close dropdown menu if handler is set
                const topMenu = this.findTopMenu();
                if (topMenu && topMenu.isDropdown) {
                    topMenu.close();
                }
            }
            this.fire("click", this, e);
        });
        return el;
    }
    onMenuButtonClick(ev) {
        if (this._menu.hidden) {
            this.showMenu();
        }
        else {
            this._menu.hide();
        }
    }
    onMenuMouseEnter(ev) {
        if (Menu.openedMenu && Menu.openedMenu != this._menu && Menu.openedMenu.parent.parent === this._menu.parent.parent) {
            this.showMenu();
        }
    }
    /**
     * Add menu to this button
     */
    set menu(menu) {
        if (menu) {
            menu.parent = this;
            menu.removeOnClose = false;
            menu.isDropdown = true;
            this.el.classList.add("has-menu");
        }
        this._menu = menu;
    }
    get menu() {
        return this._menu;
    }
    showMenu() {
        if (!this._menu) {
            return;
        }
        if (!this._menu.hidden) {
            return;
        }
        // noinspection PointlessBooleanExpressionJS
        if (this.fire("beforeshowmenu", this, this._menu) === false) {
            return;
        }
        if (!this._menu.alignTo) {
            this._menu.alignTo = this.el;
        }
        this._menu.show();
        this.fire("showmenu", this, this._menu);
    }
    internalRemove() {
        if (this.menu) {
            this.menu.remove();
        }
        super.internalRemove();
    }
    /**
     * Button text
     *
     * Set's the button text and adds a "text" css class
     *
     * This overrides the "html" property. Use html if you want something custom.
     */
    set text(text) {
        if (text) {
            this._text = text;
            this.el.classList.add("with-text");
        }
        else {
            this.el.classList.remove("with-text");
        }
        this.textEl.innerText = text + "";
    }
    get text() {
        return this._text + "";
    }
    /**
     * Set's the button icon and adds a "icon" css class
     */
    set icon(icon) {
        this._icon = icon;
        if (this._icon != undefined) {
            this.el.classList.add("with-icon");
        }
        else {
            this.el.classList.remove("with-icon");
        }
        this.iconEl.innerText = icon + "";
    }
    get icon() {
        return this._icon;
    }
    get iconEl() {
        if (!this._iconEl) {
            this._iconEl = document.createElement("i");
            this._iconEl.classList.add("icon");
            if (this._textEl) {
                this.el.insertBefore(this._iconEl, this._textEl);
            }
            else {
                this.el.appendChild(this._iconEl);
            }
        }
        return this._iconEl;
    }
    get textEl() {
        if (!this._textEl) {
            this._textEl = document.createElement("span");
            this._textEl.classList.add("text");
            this.el.appendChild(this._textEl);
        }
        return this._textEl;
    }
}
/**
 * Shorthand function to create {@see Button}
 *
 * @param config
 */
export const btn = (config) => createComponent(new Button(), config);
//# sourceMappingURL=Button.js.map