/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Format, FunctionUtil, Observable } from "@intermesh/goui";
import { fetchEventSource } from "@fortaine/fetch-event-source";
import { jmapds } from "./JmapDataSource.js";
export class Client extends Observable {
    constructor() {
        super();
        this._lastCallCounter = 0;
        this._requests = [];
        this._requestData = {};
        this.debugParam = ""; // "XDEBUG_SESSION=1"
        this.uri = "";
        this.CSRFToken = "";
        /**
         * Either a cookie + CSRFToken are used when the API is on the same site. If it's not then an access token can be used
         *
         * @private
         */
        this.accessToken = "";
        this.SSEEventsRegistered = false;
        this.delayedJmap = FunctionUtil.buffer(0, () => {
            this.doJmap();
        });
    }
    set session(session) {
        // Remove some extjs stuff that's not required
        delete session.debug;
        delete session.accounts;
        delete session.state;
        if (session.accessToken) {
            this.accessToken = session.accessToken;
            sessionStorage.setItem("accessToken", this.accessToken);
            // don't put this in the session to prevent token theft
            delete session.accessToken;
        }
        this._session = session;
        if (session.CSRFToken) {
            this.CSRFToken = session.CSRFToken;
        }
        // this.fire("authenticated", this, session);
    }
    /**
     * this should be firing on set session() but in GO we first have to load custom fields and modules before this fires.
     */
    fireAuth() {
        this.fire("authenticated", this, this._session);
    }
    get session() {
        if (this._session) {
            return Promise.resolve(this._session);
        }
        if (!this.accessToken) {
            this.accessToken = sessionStorage.getItem("accessToken") || "";
        }
        return this.request().then(response => {
            return response.json();
        }).then(session => {
            this.session = session;
            return this._session;
        });
    }
    /**
     * The ID of the last JMAP method call
     */
    get lastCallId() {
        return this._lastCallId;
    }
    async isLoggedIn() {
        if (this.user) {
            return this.user;
        }
        else {
            try {
                const user = await this.getUser();
                return user || false;
            }
            catch (e) {
                return false;
            }
        }
    }
    async request(data) {
        const response = await fetch(this.uri + "jmap.php" + (this.debugParam ? '?' + this.debugParam : ''), {
            method: data ? "POST" : "GET",
            mode: "cors",
            credentials: "include", // for cookie auth
            headers: this.buildHeaders(),
            body: data ? JSON.stringify(data) : undefined
        });
        if (response.status != 200) {
            throw response.statusText;
        }
        return response;
    }
    async logout() {
        await fetch(this.uri + "auth.php" + (this.debugParam ? '?' + this.debugParam : ''), {
            method: "DELETE",
            mode: "cors",
            credentials: "include",
            headers: this.buildHeaders()
        });
        this.CSRFToken = "";
        this.accessToken = "";
        sessionStorage.removeItem("accessToken");
        this.fire("logout", this);
    }
    getBlobURL(blobId) {
        if (!Client.blobCache[blobId]) {
            let type;
            Client.blobCache[blobId] = fetch(client.downloadUrl(blobId), {
                method: 'GET',
                credentials: "include",
                headers: this.buildHeaders()
            })
                .then(r => {
                type = r.headers.get("Content-Type") || undefined;
                return r.arrayBuffer();
            })
                .then(ab => URL.createObjectURL(new Blob([ab], { type: type })));
        }
        return Client.blobCache[blobId];
    }
    async downloadBlobId(blobId, filename) {
        // Create a URL for the blob
        const url = await this.getBlobURL(blobId);
        // Create an anchor element to "point" to it
        const anchor = document.createElement('a');
        anchor.href = url;
        anchor.download = filename;
        // Simulate a click on our anchor element
        anchor.click();
        console.log("Downloading: " + url);
        // Discard the object data
        URL.revokeObjectURL(url);
    }
    auth(data) {
        return fetch(this.uri + "auth.php" + (this.debugParam ? '?' + this.debugParam : ''), {
            method: "POST",
            mode: "cors",
            credentials: "include",
            headers: this.buildHeaders(),
            body: JSON.stringify(data)
        });
    }
    /**
     * Get the logged-in user.
     */
    async getUser() {
        if (!this.user) {
            try {
                const session = await this.session;
                if (!session) {
                    return undefined;
                }
                const ds = jmapds("User");
                this.user = await ds.single(session.userId);
                if (this.user) {
                    Format.dateFormat = this.user.dateFormat;
                    Format.timeFormat = this.user.timeFormat;
                    Format.timezone = this.user.timezone;
                    Format.currency = this.user.currency;
                    Format.thousandsSeparator = this.user.thousandsSeparator;
                    Format.decimalSeparator = this.user.decimalSeparator;
                    return this.user;
                }
                else {
                    return undefined;
                }
            }
            catch (reason) {
                this.user = undefined;
                return Promise.reject(reason);
            }
        }
        return this.user;
    }
    downloadUrl(blobId) {
        return this.uri + "download.php?blob=" + encodeURIComponent(blobId);
    }
    pageUrl(path) {
        return `${this.uri}page.php/${path}`;
    }
    getDefaultHeaders() {
        const headers = {
            'Content-Type': 'application/json'
        };
        if (this.accessToken) {
            headers.Authorization = "Bearer " + this.accessToken;
        }
        if (this.CSRFToken) {
            headers['X-CSRF-Token'] = this.CSRFToken;
        }
        return headers;
    }
    buildHeaders(headers = {}) {
        return Object.assign(this.getDefaultHeaders(), headers);
    }
    /**
     * Upload a file to the API
     *
     * @todo Progress. Not possible ATM with fetch() so we probably need XMLHttpRequest()
     * @param file
     */
    upload(file) {
        return fetch(this.uri + "upload.php" + (this.debugParam ? '?' + this.debugParam : ''), {
            method: 'POST',
            credentials: "include",
            headers: this.buildHeaders({
                'X-File-Name': "UTF-8''" + encodeURIComponent(file.name),
                'Content-Type': file.type,
                'X-File-LastModified': Math.round(file['lastModified'] / 1000).toString()
            }),
            body: file
        }).then((response) => {
            if (response.status > 201) {
                throw response.statusText;
            }
            return response;
        }).then(response => response.json())
            .then(response => Object.assign(response, { file: file }));
    }
    /**
     * Upload multiple files to the API
     *
     * @example
     * ```
     * btn({
     * 	type: "button",
     * 	text: t("Attach files"),
     * 	icon: "attach_file",
     * 	handler: async () => {
     *
     * 		const files = await browser.pickLocalFiles(true);
     * 		this.mask();
     * 		const blobs = await client.uploadMultiple(files);
     * 		this.unmask();
     * 	  console.warn(blobs);
     *
     * 	}
     * })
     * ```
     * @param files
     */
    uploadMultiple(files) {
        const p = [];
        for (let f of files) {
            p.push(this.upload(f));
        }
        return Promise.all(p);
    }
    /**
     * Execute JMAP method
     *
     * Multiple calls will be joined together in a single call on the next event loop
     *
     * @param method
     * @param params
     */
    jmap(method, params = {}, callId = undefined) {
        if (callId === undefined) {
            callId = "call-" + (++this._lastCallCounter);
        }
        this._lastCallId = callId;
        const promise = new Promise((resolve, reject) => {
            this._requestData[callId] = {
                reject: reject,
                resolve: resolve,
                params: params,
                method: method
            };
        });
        this._requests.push([method, params, callId]);
        this.delayedJmap();
        return promise;
    }
    /**
     * Performs the requests queued in the jmap() method
     *
     * @private
     */
    doJmap() {
        this.request(this._requests)
            .then((response) => {
            return response.json();
        })
            .then((responseData) => {
            responseData.forEach((response) => {
                const callId = response[2];
                if (!this._requestData[callId]) {
                    //aborted
                    console.debug("Aborted");
                    return true;
                }
                const success = response[0] !== "error";
                if (success) {
                    this._requestData[callId].resolve(response[1]);
                }
                else {
                    this._requestData[callId].reject(response[1]);
                }
                delete this._requestData[callId];
            });
        });
        this._requests = [];
    }
    /**
     * When SSE is disabled we'll poll the server for changes every 2 minutes.
     * This also keeps the token alive. Which expires in 30M.
     */
    updateAllDataSources(entities) {
        entities.forEach(function (entity) {
            const ds = jmapds(entity);
            ds.getState().then((state) => {
                if (state)
                    ds.updateFromServer();
            });
        });
    }
    startPolling(entities) {
        this.updateAllDataSources(entities);
        this.pollInterval = setInterval(() => {
            this.updateAllDataSources(entities);
        }, 60000);
    }
    stopSSE() {
        if (this.SSEABortController) {
            this.SSEABortController.abort();
        }
        if (this.pollInterval) {
            clearInterval(this.pollInterval);
            this.pollInterval = undefined;
        }
    }
    /**
     * Initializes Server Sent Events via EventSource. This function is called in MainLayout.onAuthenticated()
     *
     * Note: disable this if you want to use xdebug because it will crash if you use SSE.
     *
     * @returns {Boolean}
     */
    async startSSE(entities) {
        try {
            this.SSELastEntities = entities;
            if (!this.SSEEventsRegistered) {
                this.registerSSEEvents();
            }
            if (!window.navigator.onLine) {
                console.log("SSE not stated because we're offline");
                return false;
            }
            const session = await this.session;
            if (!session.eventSourceUrl) {
                console.debug("Server Sent Events (EventSource) is disabled on the server.");
                this.startPolling(entities);
                return false;
            }
            console.debug("Starting SSE");
            const url = this.uri + 'sse.php?types=' + entities.join(',') + (this.debugParam ? '&' + this.debugParam : '');
            const headers = this.buildHeaders();
            delete headers['Content-Type'];
            // Event source will stop when document is hidden. Other tab is selected. When coming back check all sources for
            // updates
            document.addEventListener('visibilitychange', () => {
                if (!document.hidden) {
                    this.updateAllDataSources(entities);
                }
            });
            this.SSEABortController = new AbortController();
            // let retry = 0;
            void fetchEventSource(url, {
                headers: headers,
                signal: this.SSEABortController.signal,
                onmessage: (msg) => {
                    try {
                        // console.warn(msg);
                        const data = JSON.parse(msg.data);
                        for (let entity in data) {
                            let ds = jmapds(entity);
                            ds.getState().then(state => {
                                // console.warn(entity, state, data[entity]);
                                if (!state || state == data[entity]) {
                                    //don't fetch updates if there's no state yet because it never was used in that case.
                                    return;
                                }
                                ds.updateFromServer();
                            });
                        }
                    }
                    catch (e) {
                        console.warn(e);
                    }
                },
                onclose: () => {
                    // if the server closes the connection then retry.
                    this.startSSE(entities);
                }
            });
        }
        catch (e) {
            console.error("Failed to start Server Sent Events. Perhaps the API URL in the system settings is invalid?", e);
        }
    }
    registerSSEEvents() {
        this.SSEEventsRegistered = true;
        window.addEventListener('offline', () => {
            console.log("Closing SSE because we're offline");
            this.stopSSE();
        });
        window.addEventListener('online', () => {
            console.log("Starting SSE because we're online");
            this.startSSE(this.SSELastEntities);
        });
    }
}
Client.blobCache = {};
export const client = new Client();
//# sourceMappingURL=Client.js.map