/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { comp, Component, createComponent } from "@intermesh/goui";
import { client } from "./Client.js";
/**
 * Image component
 *
 * Uses the fetch API to fetch the image with an Authorization header and creates an objectURL using URL.createObjectURL()
 */
export class Image extends Component {
    constructor() {
        super("img");
        this.blobId = "";
    }
    internalRender() {
        const el = super.internalRender();
        el.src = Image.defaultSrc;
        if (this.blobId) {
            this.setBlobId(this.blobId);
        }
        return el;
    }
    setBlobId(blobId) {
        this.blobId = blobId;
        client.getBlobURL(this.blobId).then(src => {
            this.el.src = src;
        })
            .catch(console.error);
    }
    static replace(html) {
        const c = comp({
            html: html
        });
        Image.replaceImages(c.el);
        return c;
    }
    /**
     * Replaces all img tags with a blob ID source from group-office with an objectURL
     *
     * @param el
     * @return Promise that resolves when all images are fully loaded
     */
    static replaceImages(el) {
        const promises = [];
        el.querySelectorAll("img").forEach((img) => {
            let blobId = img.dataset.blobId;
            if (!blobId) {
                const regex = new RegExp('blob=([^">\'&\?].*)');
                const matches = regex.exec(img.src);
                if (matches && matches[1]) {
                    blobId = matches[1];
                }
            }
            if (blobId) {
                img.src = Image.defaultSrc;
                promises.push(client.getBlobURL(blobId).then(src => {
                    img.src = src;
                }).then(() => {
                    //wait till image is fully loaded
                    return new Promise(resolve => { img.onload = img.onerror = resolve; });
                }));
            }
        });
        return Promise.all(promises);
    }
}
Image.cache = {};
Image.defaultSrc = 'data:image/gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAAABAAEAAAICTAEAOw==';
/**
 * Shorthand function to create table
 *
 * @param config
 */
export const img = (config) => createComponent(new Image, config);
//# sourceMappingURL=Image.js.map