<?php


namespace GO\Projects2\Report;

use GO;
use GO\Base\Db\FindParams;
use GO\Base\Db\FindCriteria;
use GO\Base\Util\Number;
use GO\Base\Util\Date\DateTime;
use GO\Projects2\Model\Expense;
use GO\Projects2\Model\TimeEntry;

class TimeFinanceReportProject extends AbstractReport
{
	public function name(){
		return GO::t('Time & finance for project', 'projects2');
	}

	public function supportsBatchReport(){
		return false;
	}

	public function supportsSelectedProject(){
		return true;
	}

	public function fileExtension(){
		return 'csv';
	}

	public function supportsDateRange(){
		return true;
	}

	public function supportsStatusFilter(){
		return false;
	}

	/**
	 * @inheritDoc
	 */
	public function render($return = false)
	{
		// Step 0: Prepare the CSV file
		$this->setFilename($this->getFileName());

		$csvFile = \GO\Base\Fs\CsvFile::tempFile($this->filename, $this->fileExtension());
		$csvFile->putRecord(['', go()->t('Time'), '', '','' /*, go()->t('Expenses', 'legacy', 'projects2'), '', '',''*/]);
		$csvFile->putRecord([
			go()->t('Project', 'legacy', 'projects2'),

			go()->t('Spent', 'legacy', 'projects2') . " " . go()->t('Period', 'legacy', 'projects2'),
			go()->t('Spent', 'legacy', 'projects2') . " " . strip_tags(go()->t('Total', 'legacy', 'projects2')),
			go()->t('Budget', 'legacy', 'projects2') . " " . strip_tags(go()->t('Total', 'legacy', 'projects2')),
			go()->t('Balance', 'legacy', 'projects2'),

//			go()->t('Spent', 'legacy', 'projects2') . " " . go()->t('Period', 'legacy', 'projects2'),
//			go()->t('Spent', 'legacy', 'projects2') . " " . strip_tags(go()->t('Total', 'legacy', 'projects2')),
//			go()->t('Budget', 'legacy', 'projects2') . " " . strip_tags(go()->t('Total', 'legacy', 'projects2')),
//			go()->t('Balance', 'legacy', 'projects2'),
		]);

		// Step 1: Retrieve relevant project IDs
		$conn = GO::getDbConnection();
		$q = 'SELECT `t`.`id`,`t`.`name`,`t`.`path`, `r`.`total_budgeted_units` 
				FROM `pr2_projects` `t` 
					LEFT JOIN (
					SELECT `project_id`, SUM(`budgeted_units`) AS `total_budgeted_units` FROM `pr2_resources` GROUP BY `project_id`
					) `r` ON `t`.`id` = `r`.`project_id`
				WHERE `t`.`id` = ' .$this->project->id;


		// Prepared statement for budgets. We'll get there
		$bpq = GO::getDbConnection()->prepare(
			'SELECT SUM(nett) AS total_budget FROM pr2_expense_budgets WHERE project_id = :project_id;'
		);


		// Step 2: for each project ID:
		$row = $conn->query($q)->fetch();

		// 2a. Retrieve all expenses;
		$st2 = Expense::model()->find(FindParams::newInstance()
			->ignoreAcl()
			->select('t.nett,t.date')
			->criteria(FindCriteria::newInstance()->addCondition('project_id', $this->project->id))
			->order('date'));

		$total_expenses = 0;
		$total_period = 0;
		foreach ($st2 as $expEntry) {
			// 2b. Sum all expenses
			$total_expenses += $expEntry->nett;
			if ($expEntry->date >= $this->startDate && $expEntry->date < $this->endDate) {
				// 2c. Filter out current period
				$total_period += $expEntry->nett;
			}
		}

		// 2d. Retrieve all expense_budget, sum netto / nett+tax
		$bpq->execute(['project_id' => $this->project->id]);
		$bprow = $bpq->fetch(\PDO::FETCH_ASSOC);
		$totalBudget = !empty($bprow) ? $bprow['total_budget'] : 0;

		// Step 2e: Retrieve worked hours, Step 2f: Filter out the hours for the current period
		$st2 = TimeEntry::model()->find(FindParams::newInstance()
			->ignoreAcl()
			->select('t.units, t.date, t.standard_task_id, a.name, t.user_id, u.displayName')
			->join('business_activity', FindCriteria::newInstance()->addRawCondition('a.id = t.standard_task_id'), 'a', 'LEFT')
			->join('core_user', FindCriteria::newInstance()->addRawCondition('u.id = t.user_id'), 'u', 'LEFT')
			->criteria(FindCriteria::newInstance()->addCondition('project_id', $this->project->id))
			->order('t.date'));

		$numWorkedHours = 0;
		$numPeriodicalHours = 0;
		$activities = [];
		$employees = [];
		foreach ($st2 as $entry) {
			if($entry->standard_task_id === null) {
				$entry->standard_task_id = 0;
			}
			if(!isset($activities[$entry->standard_task_id])) {
				$activities[$entry->standard_task_id] = [$entry->standard_task_id == 0 ? '-' : $entry->name,0,0];
			}
			if(!isset($employees[$entry->user_id])) {
				$employees[$entry->user_id] = [$entry->displayName,0,0];
			}
			$employees[$entry->user_id][2] += $entry->units;
			$activities[$entry->standard_task_id][2] += $entry->units;

			$numWorkedHours += $entry->units;
			if($entry->date >= $this->startDate && $entry->date < $this->endDate) {
				$numPeriodicalHours += $entry->units;
				$employees[$entry->user_id][1] += $entry->units;
				$activities[$entry->standard_task_id][1] += $entry->units;
			}
		}

		$csvFile->putRecord([
			go()->t('Total') . ' '.$row['name'],
			Number::localize($numPeriodicalHours),
			Number::localize($numWorkedHours),
			Number::localize($row['total_budgeted_units']), // Step 2g: Total budgeted hours for the current project, cunningly retrieved from the initial query
			Number::localize($row['total_budgeted_units'] - $numWorkedHours),
//			Number::localize($total_period),
//			Number::localize($total_expenses),
//			Number::localize($totalBudget),
//			$totalBudget - $total_expenses
		]);



		$csvFile->putRecord(['','','','','']);
		$csvFile->putRecord([go()->t('Per activity'),'','','','']);
		foreach($activities as $activity) {
			$csvFile->putRecord([$activity[0], Number::localize($activity[1]),Number::localize($activity[2])]);
		}

		$csvFile->putRecord(['','','','','']);
		$csvFile->putRecord([go()->t('Per employee'),'','','','']);
		foreach($employees as $employee) {
			$csvFile->putRecord([$employee[0], Number::localize($employee[1]),Number::localize($employee[2])]);
		}

		if ($return) {
			return $csvFile->getContents();
		} else {
			\GO\Base\Util\Http::outputDownloadHeaders($csvFile, false);
			$csvFile->output();
		}
	}

	private function getFileName()
	{
		$oDT = DateTime::fromUnixTime($this->startDate);
		$strName = $this->name() . ' ';
		$strName .= $oDT->formatDate() . ' ';
		$oDT = DateTime::fromUnixtime($this->endDate)->sub(new \DateInterval('P1D'));
		$strName .= GO::t("Till", "projects2") . ' ' . $oDT->formatDate();
		return $strName;
	}

}