/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { comp, Component, createComponent } from "../Component.js";
import { ObjectUtil } from "../../util/ObjectUtil.js";
import { menu } from "../menu/Menu.js";
import { checkbox } from "../form/CheckboxField.js";
import { Notifier } from "../../Notifier.js";
import { draggable } from "../DraggableComponent.js";
import { List } from "../List.js";
import { t } from "../../Translate";
/**
 * Table component
 *
 * @example
 * ```
 * const records:StoreRecord[] = [];
 *
 * 	for(let i = 1; i <= 100; i++) {
 * 		records.push({number: i, description: "Test " + i, createdAt: (new DateTime()).format("c")});
 * 	}
 *
 * 	const table = table({
 * 		store: const store = jmapstore({
 * 			entity: "TaskList",
 * 			properties: ['id', 'name', 'support'],
 * 			queryParams: {
 * 				limit: 20,
 * 				filter: {
 * 					forSupport: true,
 * 					role: "support", //support tasklists
 * 				}
 * 			},
 * 			sort: [{property: "name", isAscending: true}]
 * 		}),
 * 		cls: "fit",
 * 		columns: [
 * 			{
 * 				header: "Index",
 * 				id: "id",
 * 				renderer: (value, record, td, table) => {
 * 					return table.getStore().findRecordIndex(r => record.number == r.number).toString();
 * 				},
 * 				resizable: true,
 * 				width: 60,
 * 				sortable: false
 * 			},
 * 			{
 * 				header: "Number",
 * 				id: "number",
 * 				sortable: true,
 * 				resizable: true,
 * 				width: 200
 * 			},
 * 			{
 * 				header: "Description",
 * 				id: "description",
 * 				sortable: true,
 * 				resizable: true,
 * 				width: 300
 * 			},
 * 			datecolumn({
 * 				header: "Created At",
 * 				id: "createdAt",
 * 				sortable: true
 * 			})
 * 			]
 * 	});
 *  ```
 */
export class Table extends List {
    /**
     *
     * @param store Store to provide data
     * @param columns The table columns
     */
    constructor(store, columns) {
        super(store, (record, row, me, storeIndex) => {
            for (let c of this.columns) {
                c.parent = this;
                if (c.hidden) {
                    continue;
                }
                const td = document.createElement("td");
                if (c.align) {
                    td.style.textAlign = c.align;
                }
                if (c.cls) {
                    td.classList.add(...c.cls.split(" "));
                }
                let value = c.property ? ObjectUtil.path(record, c.property) : undefined;
                if (c.renderer) {
                    const r = c.renderer(value, record, td, this, storeIndex);
                    if (typeof r === "string") {
                        td.innerHTML = r;
                    }
                    else if (r instanceof Component) {
                        r.parent = this;
                        r.render(td);
                    }
                    else {
                        r.then((s) => {
                            if (s instanceof Component) {
                                s.parent = this;
                                s.render(td);
                            }
                            else {
                                td.innerHTML = s;
                            }
                        });
                    }
                }
                else {
                    td.innerText = value ? value : "";
                }
                row.append(td);
            }
        }, "table");
        this.columns = columns;
        /**
         * Make the table fits its container in width by setting min-width: 100%
         * Defaults to true
         */
        this.fitParent = false;
        /**
         * Show headers
         */
        this.headers = true;
        this.emptyStateTag = 'caption';
        /**
         * Group renderer function
         */
        this.groupByRenderer = groupBy => groupBy !== null && groupBy !== void 0 ? groupBy : t("None");
        this.minCellWidth = 30;
        this.baseCls = "goui-table";
        this.itemTag = 'tr';
        this.colsAreFixed = false;
    }
    getRowElements() {
        return Array.from(this.el.getElementsByClassName("data"));
    }
    internalRemove() {
        if (this.columnMenu) {
            this.columnMenu.remove();
        }
        return super.internalRemove();
    }
    restoreState(state) {
        if (state.sort) {
            this.store.sort = state.sort;
        }
        if (state.columns) {
            for (let id in state.columns) {
                let col = this.findColumnById(id);
                if (col) {
                    Object.assign(col, state.columns[id]);
                }
            }
        }
    }
    /**
     * Find column by "property" property.
     *
     * It's the property path of the data linked to the column
     *
     * @param id
     */
    findColumnById(id) {
        return this.columns.find((col) => {
            return col.id == id;
        });
    }
    buildState() {
        const cols = {};
        this.columns.forEach((c) => {
            cols[c.id] = {
                width: c.width,
                hidden: c.hidden
            };
        });
        return {
            sort: this.store.sort,
            columns: cols
        };
    }
    renderBody() {
        if (this.fitParent) {
            this.el.style.minWidth = "100%";
        }
        if (this.headers) {
            this.renderHeaders();
        }
        else {
            this.renderColGroup();
        }
        super.renderBody();
    }
    rerender() {
        const el = this.el;
        this.groupEl = undefined;
        el.innerHTML = "";
        this.renderBody();
    }
    showColumnMenu(ev) {
        ev.preventDefault();
        if (!this.columnMenu) {
            this.columnMenu = menu({
                isDropdown: true,
                removeOnClose: false
            });
            this.columns.forEach((c) => {
                if (c.header && c.hidable) {
                    this.columnMenu.items.add(checkbox({
                        label: c.header,
                        name: c.id,
                        value: !c.hidden,
                        listeners: {
                            change: (field) => {
                                c.hidden = !field.value;
                                this.saveState();
                                this.rerender();
                            }
                        }
                    }));
                }
            });
        }
        this.columnMenu.showAt(ev);
    }
    createColumnSplitter(h, header, colIndex) {
        if (h.resizable) {
            const splitter = draggable({
                tagName: "hr",
                setPosition: false,
                parent: this
            });
            splitter.on("dragstart", (cmp, dragData) => {
                if (!this.colsAreFixed) {
                    this.fixColumnWidths();
                }
                dragData.data.startWidth = header.offsetWidth;
                splitter.dragConstrainTo(this.headersRow, {
                    left: this.calcTableWidth(colIndex) + this.minCellWidth,
                    right: -10000
                });
            });
            splitter.on("drag", (cmp, dragData) => {
                const w = dragData.data.startWidth + dragData.x - dragData.startX;
                header.style.width = w + "px";
                h.width = w;
                this.el.style.width = this.calcTableWidth() + "px";
            });
            splitter.on("drop", () => {
                this.saveState();
            });
            return splitter;
        }
        else {
            return comp({ tagName: "hr" });
        }
    }
    renderColGroup() {
        const colGroup = document.createElement("colgroup");
        let index = -1;
        for (let h of this.columns) {
            index++;
            if (h.hidden) {
                continue;
            }
            const col = document.createElement("col");
            if (h.width) {
                col.style.width = h.width + "px";
            }
            if (h.align) {
                col.style.textAlign = h.align;
            }
            if (h.cls) {
                col.classList.add(...h.cls.split(" "));
            }
            colGroup.appendChild(col);
        }
        this.el.appendChild(colGroup);
        return colGroup;
    }
    renderHeaders() {
        const thead = document.createElement('thead');
        this.headersRow = document.createElement("tr");
        this.headersRow.addEventListener('contextmenu', ev => {
            this.showColumnMenu(ev);
        });
        let index = -1;
        for (let h of this.columns) {
            index++;
            if (h.hidden) {
                continue;
            }
            const header = document.createElement("th");
            if (h.width) {
                header.style.width = h.width + "px";
            }
            if (h.align) {
                header.style.textAlign = h.align;
            }
            if (h.cls) {
                header.classList.add(...h.cls.split(" "));
            }
            if (h.headerRenderer) {
                const r = h.headerRenderer(h, header, this);
                if (typeof r === "string") {
                    header.innerHTML = r;
                }
                else if (r instanceof Component) {
                    r.render(header);
                }
            }
            else {
                header.innerHTML = h.header || "";
            }
            h.headerEl = header;
            if (h.resizable) {
                const splitter = this.createColumnSplitter(h, header, index);
                splitter.render(header);
            }
            else {
                comp({ tagName: "hr" }).render(header);
            }
            if (h.sortable && h.property) {
                header.addEventListener("click", () => {
                    this.onSort(h.property, header);
                });
                const sort = this.store.sort;
                if (sort.length) {
                    sort.forEach((comparator) => {
                        if (h.property == comparator.property) {
                            header.classList.add("sorted");
                            header.classList.add(comparator.isAscending || comparator.isAscending === undefined ? "asc" : "desc");
                        }
                    });
                }
            }
            this.headersRow.appendChild(header);
        }
        thead.appendChild(this.headersRow);
        this.el.appendChild(thead);
        return this.headersRow;
    }
    onSort(dataIndex, header) {
        this.fire("sort", this, dataIndex);
        const s = this.store.sort;
        let sortIndex = s.length - 1 || 0;
        if (s[sortIndex]) {
            if (s[sortIndex].property == dataIndex) {
                s[sortIndex].isAscending = !s[sortIndex].isAscending;
            }
            else {
                s[sortIndex].property = dataIndex;
            }
        }
        else {
            s[sortIndex] = {
                isAscending: true,
                property: dataIndex
            };
        }
        this.headersRow.childNodes.forEach((node) => {
            let th = node;
            if (th == header) {
                th.classList.add("sorted");
                th.classList.remove(s[sortIndex].isAscending ? "desc" : "asc");
                th.classList.add(s[sortIndex].isAscending ? "asc" : "desc");
            }
            else {
                th.classList.remove("sorted");
                th.classList.remove("asc");
                th.classList.remove("desc");
            }
        });
        this.store.reload().catch((reason) => {
            Notifier.error(reason);
        });
        this.saveState();
    }
    /**
     * When resizing columns we must calculate absolute pixel widths
     *
     * @private
     */
    fixColumnWidths() {
        this.columns.forEach(col => {
            if (!col.hidden) {
                col.width = col.headerEl.offsetWidth;
                col.headerEl.style.width = col.width + "px";
            }
        });
        this.el.style.minWidth = "";
        this.el.style.width = this.calcTableWidth() + "px";
    }
    /**
     * Returns the sum of column widths
     *
     * @param untilColumnIndex Calc width until this column
     */
    calcTableWidth(untilColumnIndex = -1) {
        return this.columns.reduce((previousValue, nextValue, currentIndex) => {
            if (nextValue.hidden || (untilColumnIndex > -1 && currentIndex >= untilColumnIndex)) {
                return previousValue;
            }
            return previousValue + nextValue.width;
        }, 0);
    }
    renderGroup(record) {
        if (!this.groupBy) {
            if (!this.groupEl) {
                this.groupEl = document.createElement('tbody');
                this.el.append(this.groupEl);
            }
            return this.groupEl;
        }
        const groupBy = ObjectUtil.path(record, this.groupBy);
        // console.warn(this.groupEl,groupBy, this.lastGroup)
        if (!this.groupEl || groupBy != this.lastGroup) {
            const tr = document.createElement("tr");
            tr.classList.add("group");
            const th = document.createElement("th");
            th.colSpan = this.columns.length;
            const r = this.groupByRenderer(groupBy, record, th, this);
            if (typeof r === "string") {
                th.innerHTML = r;
            }
            else if (r instanceof Component) {
                r.render(th);
            }
            else if (r instanceof Promise) {
                r.then((s) => {
                    if (s instanceof Component) {
                        s.render(th);
                    }
                    else {
                        th.innerHTML = s;
                    }
                });
            }
            else {
                console.warn("Renderer returned invalid value: ", r);
            }
            tr.append(th);
            this.groupEl = document.createElement('tbody');
            this.groupEl.append(tr);
            this.el.append(this.groupEl);
            this.lastGroup = groupBy;
        }
        return this.groupEl;
    }
    onRecordRemove(collection, item, index) {
        var _a;
        let groupEl;
        if (this.groupBy) {
            const rows = this.getRowElements();
            groupEl = (_a = rows[index]) === null || _a === void 0 ? void 0 : _a.parentElement;
        }
        super.onRecordRemove(collection, item, index);
        //console.warn(groupEl, groupEl?.children.length);
        //cleanup group if only group header is left
        if (groupEl && groupEl.children.length == 1) {
            if (groupEl == this.groupEl) {
                this.groupEl = undefined;
            }
            groupEl.remove();
        }
    }
    findDropRow(e) {
        return e.target.closest("TR");
    }
}
/**
 * Shorthand function to create {@see Table}
 *
 * @param config
 */
export const table = (config) => createComponent(new Table(config.store, config.columns), config);
//# sourceMappingURL=Table.js.map