import { Field } from "./Field";
import { createComponent } from "../Component";
import { DateInterval, E, FunctionUtil } from "../../util";
import { t } from "../../Translate";
/**
 * Duration field
 *
 * Represents a period of time
 */
export class DurationField extends Field {
    /**
     *
     * @param outputFormat Format it will as value
     *
     * {@link DurationField.value}
     *
     * It can be any string format supported by {@link DateInterval.format}
     */
    constructor(outputFormat = "h:I") {
        super();
        this.outputFormat = outputFormat;
        this.baseCls = "goui-form-field duration no-floating-label";
        /**
         * Minimum allowed duration to be entered
         */
        this.min = undefined;
        /**
         * Maximum allowed duration to be entered
         */
        this.max = undefined;
    }
    validate() {
        super.validate();
        const v = this.getValueAsDateInterval();
        if (this.max !== undefined && v.compare(this.max) == 1) {
            this.setInvalid(t("The maximum duration is {duration}").replace("{duration}", this.max.format("h:I")));
        }
        if (this.min !== undefined && v.compare(this.min) == -1) {
            this.setInvalid(t("The minimum duration is {duration}").replace("{duration}", this.min.format("h:I")));
        }
    }
    /**
     * Get the value DateInterval object
     */
    getValueAsDateInterval() {
        const di = new DateInterval();
        di.hours = parseInt(this.hoursInput.value);
        di.minutes = parseInt(this.minutesInput.value);
        return di;
    }
    createControl() {
        const ctrl = E("div").cls("goui hbox");
        const onBlur = function () {
            if (!this.value) {
                return;
            }
            this.value = this.value.padStart(2, "0");
            return true;
        };
        const onFocus = function (ev) {
            ev.preventDefault();
            this.focus();
            this.setSelectionRange(0, this.value.length);
        };
        const onKeyDown = (ev) => {
            switch (ev.key) {
                case "Tab":
                case "Enter":
                case "Backspace":
                case "Delete":
                    return true;
                case ':':
                    this.minutesInput.focus();
                    ev.preventDefault();
                    break;
                default:
                    if (!/^[0-9]$/i.test(ev.key)) {
                        //only allow numbers
                        ev.preventDefault();
                    }
            }
        };
        this.hoursInput = document.createElement("input");
        this.hoursInput.classList.add("text");
        this.hoursInput.classList.add("hour");
        this.hoursInput.inputMode = "numeric";
        this.hoursInput.type = "text";
        this.hoursInput.pattern = "[0-9]+";
        this.hoursInput.onblur = onBlur;
        this.hoursInput.onfocus = onFocus;
        this.hoursInput.onmousedown = onFocus;
        this.hoursInput.oninput = FunctionUtil.buffer(500, function (e) {
            onBlur.call(this);
            onFocus.call(this, e);
        });
        this.hoursInput.placeholder = "--";
        this.hoursInput.autocomplete = "off";
        this.hoursInput.onkeydown = onKeyDown;
        this.minutesInput = document.createElement("input");
        this.minutesInput.classList.add("text");
        this.minutesInput.classList.add("minute");
        this.minutesInput.inputMode = "numeric";
        this.minutesInput.type = "text";
        this.minutesInput.pattern = "[0-9]+";
        this.minutesInput.maxLength = 2;
        this.minutesInput.oninput = FunctionUtil.buffer(500, function (e) {
            if (parseInt(this.value) > 59) {
                this.value = "59";
            }
            onBlur.call(this);
            onFocus.call(this, e);
        });
        const hoursInput = this.hoursInput;
        this.minutesInput.onmousedown = onFocus;
        this.minutesInput.onblur = function () {
            onBlur.call(this);
            if (!this.value && hoursInput.value) {
                this.value = "00";
            }
        };
        this.minutesInput.onfocus = onFocus;
        this.minutesInput.onkeydown = onKeyDown;
        this.minutesInput.placeholder = "--";
        this.minutesInput.autocomplete = "off";
        ctrl.append(this.hoursInput, ":", this.minutesInput);
        return ctrl;
    }
    internalSetValue(v) {
        if (v && this.hoursInput && this.minutesInput) {
            const dateInterval = DateInterval.createFromFormat(v, this.outputFormat);
            if (dateInterval) {
                this.hoursInput.value = dateInterval.format("h");
                this.minutesInput.value = dateInterval.format("I");
            }
            else {
                this.minutesInput.value = "00";
            }
        }
    }
    internalGetValue() {
        if (!this.hoursInput.value) {
            return undefined;
        }
        const dateInterval = new DateInterval();
        dateInterval.hours = parseInt(this.hoursInput.value);
        dateInterval.minutes = parseInt(this.minutesInput.value);
        return dateInterval.format(this.outputFormat);
    }
}
export const durationfield = (config) => { var _a; return createComponent(new DurationField((_a = config === null || config === void 0 ? void 0 : config.outputFormat) !== null && _a !== void 0 ? _a : "h:I"), config); };
//# sourceMappingURL=DurationField.js.map