/**
 * Copyright Intermesh
 *
 * This file is part of Group-Office. You should have received a copy of the
 * Group-Office license along with Group-Office. See the file /LICENSE.TXT
 *
 * If you have questions write an e-mail to info@intermesh.nl
 *
 * @version $Id: TicketsGrid.js 23483 2018-03-07 10:57:52Z mdhart $
 * @copyright Copyright Intermesh
 * @author Michiel Schmidt <michiel@intermesh.nl>
 * @author Merijn Schering <mschering@intermesh.nl>
 */

GO.tickets.TicketsGrid = function(config){
	
	if(!config)
	{
		config = {};
	}
	
	config.layout='fit';
	config.autoScroll=true;
	config.split=true;
	config.paging=true;
	config.noDelete = !GO.settings.modules.tickets.write_permission;

	var fields ={
		fields:['id','ticket_number','priority','priority_name','agent_id','agent_name','due_date', 'contact_name','subject','ctime','mtime','status','status_id','type','unseen','email','phone','company','level','late','type_id','last_response', 'user_id'].concat(go.customfields.CustomFields.getFieldDefinitions("Ticket")),
		columns:[
		{
			header: '&nbsp;',
			id:'icon',
			dataIndex: 'unseen',
			hideInExport:true,
			width:dp(56),
			renderer: this.renderIcon
		},{
			header: t("Number", "tickets"), 
			dataIndex: 'ticket_number',
			width: dp(140),
			hidden:true
		},{
			header: t("Subject"), 
			dataIndex: 'subject',
			renderer: function(v) {
				return Ext.util.Format.htmlDecode(v);
			},
			width:dp(280)
		},{
			header: t("Date"), 
			dataIndex: 'last_response',
				xtype: "datecolumn",
			// renderer: GO.util.format.smartDate,
			// width: dp(112)
		},{
			header: t("Status", "tickets"), 
			dataIndex: 'status',
			renderer: function(v) {
				return Ext.util.Format.htmlDecode(v);
			},
			width:80
		},{
			header: t("Type", "tickets"),
			dataIndex: 'type',
			hidden: true,
			renderer: function(v) {
				return Ext.util.Format.htmlDecode(v);
			},
			width: dp(140),
			groupable: true
		},{
			id:'agent_name',
			header: t("Responsible", "tickets"),
			dataIndex: 'agent_name',
			width:150,
			renderer: function(v) {
				return Ext.util.Format.htmlDecode(v);
			},
			sortable:true
		},{
			header: t("Contact", "tickets"),
			dataIndex: 'contact_name',
			renderer: function(v) {
				return Ext.util.Format.htmlDecode(v);
			},

			width:150
		},{
			header: t("Company"),
			dataIndex: 'company',
			renderer: function(v) {
				return Ext.util.Format.htmlDecode(v);
			},
			width:150,
			hidden:true
		},{
			header: t("E-mail"),
			dataIndex: 'email',
			hidden:true,
			width: dp(140)
		},{
			header: t("Due at", "tickets"),
			dataIndex: 'due_date',
			hidden:true,
			width:110
		},{
			header: t("Phone"),
			dataIndex: 'phone',
			hidden:true,
			width: dp(140)
		},{
			header: t("Created at"), 
			dataIndex: 'ctime',
			hidden:true,
			width: dp(140)
		},{
			header: t("Modified at"), 
			dataIndex: 'mtime',
			hidden:true,
			width: dp(140)
		}
		].concat(go.customfields.CustomFields.getColumns("Ticket"))
	};

	config.store = new GO.data.JsonStore({
		url: GO.url('tickets/ticket/store'),
		baseParams:{},
		root: 'results',
		id: 'id',
		totalProperty:'total',
		fields: fields.fields,
		remoteSort: true,
		sortInfo: {
			field: "last_response",
			direction: "DESC"
		}
	});


	config.bbar = new Ext.PagingToolbar({
		cls: 'go-paging-tb',
		store: config.store,
		pageSize: parseInt(GO.settings['max_rows_list']),
		//displayInfo: true,
		//displayMsg: t("Total: {2}"),
		emptyMsg: t("No items to display")
	})


	config.store.on('load', function()
	{
		if(config.store.reader.jsonData.feedback)
		{
			alert(config.store.reader.jsonData.feedback);
		}
	},this)

	var columnModel =  new Ext.grid.ColumnModel({
		defaults:{
			sortable:true,
			groupable:false
		},
		columns:fields.columns
	});
	config.cm=columnModel;

	config.view=new go.grid.GridView({
		totalDisplay: true,
		emptyText: t("No items to display"),
		getRowClass:function(row, index, rowParams) {                                        
			var cls = [];
			if(row.data.late){
				cls.push('ti-late');
			}else if (row.data.unseen == true) {                                
				cls.push('ti-unseen');
			}else{
				cls.push('ti-seen');
			}
			return cls.join(' ');
		}		
	});
			
	config.sm=new Ext.grid.RowSelectionModel();
	config.loadMask=true;

	GO.tickets.TicketsGrid.superclass.constructor.call(this, config);

	this.on('rowcontextmenu', this.onContextClick, this);

};

Ext.extend(GO.tickets.TicketsGrid, go.grid.GridPanel,{

	deleteSelected: GO.grid.GridPanel.prototype.deleteSelected,

	mtime: 0,
	total: 0,
	
	setVars : function(mtime, total)
	{
		this.mtime = (mtime) ? mtime : 0;
		this.total = (total) ? total : 0;
	},
	flagTickets : function(tickets, unseen, flag)
	{
		if(tickets.length){
			flag = flag || 'unseen';
			this.store.baseParams['flag_tickets'] = Ext.encode(tickets);
			this.store.baseParams[flag] = unseen;
			this.store.reload();

			this.selModel.clearSelections();

			delete this.store.baseParams.flag_tickets;
			delete this.store.baseParams[flag];
		}
	},
	onContextClick : function(grid, index, e)
	{
		if(!this.menu)
		{
			this.menu = new Ext.menu.Menu({
				id:'grid-ctx',
				items: [{
					text:t("Mark as read", "tickets"),
					iconCls: 'ic-visibility',
					scope:this,
					handler: function(){
						var records = grid.selModel.getSelections();
						var selectedRows = [];
						for(var i=0;i<records.length;i++){
							if(records[i].get('unseen')=='1'){// && records[i].get('agent_id')==GO.settings.user_id){
								selectedRows.push(records[i].id);
								GO.tickets.totalUnseen--;
							}
						}
						this.flagTickets(selectedRows, 0);
					}
				},{
					text:t("Mark as unread", "tickets"),
					scope:this,
					iconCls: 'ic-visibility-off',
					handler: function(){

						var records = grid.selModel.getSelections();
						var selectedRows = [];
						for(var i=0;i<records.length;i++){
							if(records[i].get('unseen')=='0'){// && records[i].get('agent_id')==GO.settings.user_id){
								selectedRows.push(records[i].id);
								GO.tickets.totalUnseen++;
							}
						}

						this.flagTickets(selectedRows, 1);
					}
				},
					'-',

					{
						itemId:'close',
						text:t("Close ticket", "tickets"),
						iconCls: 'ic-lock',
						scope:this,
						handler: function(){
							var records = grid.selModel.getSelections();
							var selectedRows = [];
							for(var i=0;i<records.length;i++){
								if(records[i].get('status_id') != -1){ // 0 = open, -1 = closed
									selectedRows.push(records[i].id);
								}
							}
							this.flagTickets(selectedRows, -1, 'closed');
						}
					},{
						itemId:'reopen',
						text:t("Reopen ticket", "tickets"),
						scope:this,
						iconCls: 'ic-lock-open',
						handler: function(){

							var records = grid.selModel.getSelections();
							var selectedRows = [];
							for(var i=0;i<records.length;i++){
								if(records[i].get('status_id') == -1){
									selectedRows.push(records[i].id);
								}
							}

							this.flagTickets(selectedRows, 0, 'closed');
						}
					},

					'-',

					this.setAgentMenuItem = new Ext.menu.Item({
					text: t("Change responsible agent", "tickets"),
					iconCls: 'ic-supervisor-account',
					menu: {
						items: []
					},
					multiple:true,
					scope: this,
					listeners: {
						'activate' : function(menuItem) {
							var records = grid.selModel.getSelections();
							var selectedRowIds = [];
							for (var i=0;i<records.length;i++) {
								selectedRowIds.push(records[i].id);
							}
							this.selectedRowIds = selectedRowIds;
							GO.request({
								url: "tickets/ticket/possibleAgents",
								params: {
									ticketIds: Ext.encode(this.selectedRowIds)
								},
								success: function(options, response, result)
								{			
									this.setAgentMenuItem.menu.removeAll();
									for (var i=0; i<result.results.length; i++) {
										var name = result.results[i].name;
										var id = result.results[i].id;
										var subMenuItem = new Ext.menu.Item({
											text: name,
											agentId: id,
											listeners: {
												scope: this,
												'click' : function(item,event) {
//													console.log('TODO: change to agent ID '+item.agentId);
													GO.request({
														url: 'tickets/ticket/setAgent',
														params: {
															agentId: item.agentId,
															ticketIds: Ext.encode(this.selectedRowIds)
														},
														success: function(options, response, result) {
															this.store.load();
														},
														scope: this
													})
												},
												scope: this
											},
											scope: this
										});
										this.setAgentMenuItem.menu.add(subMenuItem);
									}
								},
								scope: this
							});
						},
						scope: this
					}
				})
				]
			});
   		if(GO.settings.modules.tickets.write_permission){
				this.menu.add({
					text: t("Batch edit"),
					iconCls: 'ic-edit',
					scope:this,
					handler: function()
					{
						this.showBatchEditDialog(grid);
					}
				});
			}

			this.menu.on('hide', this.onContextHide, this);
		}
        
		e.stopEvent();

		this.ctxRecord = this.store.getAt(index);
		this.menu.items.each(function(item){
				item.setDisabled(this.ctxRecord.data.level<GO.permissionLevels.manage);
		},this);

		this.menu.showAt(e.getXY());
        
	}, 
	renderIcon : function(value, cell, record) {

		var icons = [];
		var unseen = '';

		if(value=='1') {
			var late = record.data.late ? ' red':'';
			var unseen = '<div class="ml-unseen-dot'+late+'"></div>';
		} 
		if(record.data.replied){
			icons.push('reply');
		}
		if(record.data['hasAttachments']) {
			icons.push('attachment');
		}
		if(record.data.priority=="2") {
			icons.push('priority_high');
		}
		if(record.data.priority=="0") {
			icons.push('low_priority');
		}
		if(Ext.isEmpty(icons)) {
			return unseen;
		}
		return unseen + '<i class="icon">'+icons.join('</i><i class="icon">')+'</i>';
		
	},
	onContextHide : function()
	{
		if(this.selectedRows)
		{
			this.selectedRows = null;
		}
	},
	showBatchEditDialog : function(grid) {
		var ids = [];
		var selected = grid.selModel.getSelections();
		for (var i = 0; i < selected.length; i++) {
			if (!GO.util.empty(selected[i].data.id))
				ids.push(selected[i].data.id);
		}
		
		// Delete the acl_id baseParam in case that it's set
		delete GO.tickets.agentsStore.baseParams.acl_id;
		
		if(!GO.tickets.writableTypesStore.loaded)
			GO.tickets.writableTypesStore.load();
		

		GO.base.model.showBatchEditModelDialog("GO\\Tickets\\Model\\Ticket", ids,	'id', {
			status_id		:	GO.tickets.SelectStatus,
			agent_id		:	GO.tickets.SelectAgent,
			priority		: GO.form.SelectPriority,
			type_id			:	GO.tickets.SelectType,
			unseen			: Ext.ux.form.XCheckbox
		},
		['id,ticket_verifier,ticket_number,muser_id,order,last_response_time,cc_addresses,files_folder_id,order_id,group_id']
		).on('submit', function () {
			grid.getStore().reload();
		});
	}
    
});
