import {commenteditor} from "./CommentEditor.js";
import {
	browser,
	btn,
	combobox,
	comp,
	DateTime,
	fieldset,
	Form,
	form,
	router,
	t,
	tbar,
	textfield,
	Window
} from "@intermesh/goui";

import {client, jmapds} from "@intermesh/groupoffice-core";


export class SupportCreateWindow extends Window {
	constructor() {
		super();

		Object.assign(this, {
				title: t("Add"),
				width: 800,
				height: 500,
				modal: true,
				resizable: true,
				maximizable: true
			}
		);

		const taskListCombo = combobox({
			label:  t("Tasklist", "community", "tasks"),
			name:  "tasklistId",
			flex: 1,
			dataSource: jmapds("SupportList"),
			tableConfig: {
				groupBy: "group.name"
			},
			storeConfig: {
				relations: {
					group: {
						path: "groupingId",
						dataSource: jmapds("TaskListGrouping")
					}
				},
				sort: [{property: "group", isAscending: true},{property: "name", isAscending: true}],
				filters: {
					permission: {
						permissionLevel: 5
					}
				}
			},
		});

		const commentEditor = commenteditor();

		this.items.add(
			form({
				flex: 1,
				cls: "vbox",
				handler: () => {
					this.send();
				}
			},

			fieldset({
					cls: "flow scroll",
					flex: 1
				},

				comp({cls: "hbox gap"},

					textfield({
						label: t("Subject"),
						name: "subject",
						required: true,
						flex: 2
					}),
					taskListCombo
					),

				commentEditor
			),

			tbar({},
				btn({
					type: "button",
					text: t("Attach files"),
					icon: "attach_file",
					handler: async () => {

						const files = await browser.pickLocalFiles(true);
						this.mask();
						const blobs = await client.uploadMultiple(files);
						this.unmask();


						const atts = commentEditor.attachments.value;
						for (let blob of blobs) {
							atts.push({
								name: blob.name,
								blobId: blob.id
							});
						}
						commentEditor.attachments.value = atts;

					}
				}),

				'->',

				btn({
					type: "submit",
					text: t("Send")
				})
			)))

			this.on("focus", () => this.items.get(0)!.focus());
	}

	private async send  () {
		this.mask();
		const v = this.findChildByType(Form)!.value;

		const now = new DateTime(), tomorrow = (new DateTime).setMonthDay(now.getMonthDay() + 1);

		const taskDS = jmapds("SupportTicket"),
			commentDS = jmapds("Comment");

		taskDS.commitBaseParams = {
			forSupport: true
		};

		try {
			const task = await taskDS.create({
				title: v.subject,
				start: now.format("c"),
				due: tomorrow.format("c"),
				tasklistId: v.tasklistId
			});

			delete v.subject;
			delete v.tasklistId;

			const comment = await commentDS.create(
				Object.assign({
					entity: "SupportTicket",
					entityId: task.id
				}, v)
			);
			this.unmask();
			void router.goto("supportclient/" + task.id);
			this.close();
		} catch(e:any) {
			this.unmask();
			if(e.type && e.type == 'stateMismatch') {
				Promise.all([
					taskDS.updateFromServer(),
					commentDS.updateFromServer()]
				).then(() => this.send());
			} else {
				console.error(e);
				Window.alert(t("Sorry, an error occured. Please try again."));
			}
		}


	};

}

