/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component } from "./Component.js";
import { Toolbar } from "./Toolbar.js";
import { DraggableComponent, DraggableComponentEventMap } from "./DraggableComponent.js";
import { Config, Listener, ObservableListenerOpts } from "./Observable.js";
/**
 * @inheritDoc
 */
export interface WindowEventMap<Type> extends DraggableComponentEventMap<Type> {
    /**
     * Fires when the window is closed
     *
     * @param window
     */
    close: (window: Type) => void;
    /**
     * Fires before closing window
     * return false to cancel close
     *
     * @param window
     */
    beforeclose: (window: Type) => void;
    /**
     * Fires when the window is maximized
     *
     * @param window
     */
    maximize: (window: Type) => void;
    /**
     * Fires when the window is restored after being maximized
     *
     * @param window
     */
    unmaximize: (window: Type) => void;
}
export interface Window extends DraggableComponent {
    on<K extends keyof WindowEventMap<this>, L extends Listener>(eventName: K, listener: Partial<WindowEventMap<this>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof WindowEventMap<this>>(eventName: K, listener: Partial<WindowEventMap<this>>[K]): boolean;
    fire<K extends keyof WindowEventMap<this>>(eventName: K, ...args: Parameters<WindowEventMap<any>[K]>): boolean;
}
/**
 * Window component
 *
 * @example
 *
 * ```
 * const win = Window.create({
 * 	title: "Hello World",
 * 	items: [Component.create({tagName: "h1", cls: "pad", html: "Just saying hi!"})]
 * });
 *
 * win.open();
 * ```
 */
export declare class Window extends DraggableComponent {
    constructor();
    protected baseCls: string;
    /**
     * Maximize the window
     */
    maximized: boolean;
    /**
     * Enable tool to maximize window
     */
    maximizable: boolean;
    /**
     * Enable tool to close window
     */
    closable: boolean;
    /**
     * Enable tool to collapse window
     */
    collapsible: boolean;
    /**
     * Make the window modal so the user can only interact with this window.
     */
    modal: boolean;
    private titleCmp;
    private header;
    private modalOverlay;
    private resizeObserver?;
    /**
     * Return focus to element focussed before opening it when closing the window
     * @private
     */
    private focussedBeforeOpen?;
    private initMaximizeTool;
    private _title;
    set title(title: string);
    get title(): string;
    protected getDragHandle(): HTMLElement;
    getHeader(): Toolbar;
    set collapsed(collapsed: boolean);
    get collapsed(): boolean;
    protected internalRender(): HTMLElement;
    private observerResize;
    protected buildState(): Record<string, any>;
    protected restoreState(s: Record<string, any>): void;
    private constrainViewport;
    /**
     * Open the window by rendering it into the DOM body element
     * Use show()
     * @deprecated
     */
    open(): boolean;
    protected internalSetHidden(hidden: boolean): void;
    private shrinkToFit;
    private disableBodyScroll;
    private enableBodyScroll;
    /**
     * @inheritDoc
     */
    remove(): boolean;
    /**
     * Close the window by removing it
     */
    close(): void;
    /**
     * Center the window in the screen
     */
    center(): this;
    /**
     * Returns true if the window is maximized
     */
    isMaximized(): boolean;
    /**
     * Grow window to the maximum of the viewport
     */
    maximize(): this;
    /**
     * Make the window smaller than the viewport
     */
    unmaximize(): this;
    /**
     * Display an error message
     *
     * @param msg - The error message to be displayed.
     * @return Promise<void> - A promise that resolves when the alert window is closed
     */
    static error(msg: string | any): Promise<void>;
    /**
     * Show modal alert window
     *
     * @param text - The alert message or an object with a 'message' property
     * @param [title="Alert"] - The title of the alert window
     * @return Promise<void> - A promise that resolves when the alert window is closed
     */
    static alert(text: any, title?: string): Promise<void>;
    /**
     * Prompt the user for a text input value.
     *
     * @param text - The message to display to the user.
     * @param inputLabel - The label for the input field.
     * @param [defaultValue=""] - The default value for the input field.
     * @param [title="Please enter"] - The title for the prompt window.
     * @returns {Promise<string | undefined>} - A promise that resolves with the input value or undefined if the user cancelled.
     */
    static prompt(text: string, inputLabel: string, defaultValue?: string, title?: string): Promise<string | undefined>;
    /**
     * Asks the user for confirmation.
     * @param {string} text - The text to display in the confirmation dialog.
     * @param {string} [title=t("Please confirm")] - The title of the confirmation dialog.
     * @return {Promise<boolean>} - A promise that resolves to `true` if the user confirms, or `false` if the user cancels.
     */
    static confirm(text: string, title?: string): Promise<boolean>;
}
type WindowConfig = Omit<Config<Window, WindowEventMap<Window>>, "close" | "maximize" | "center" | "dragConstrainTo" | "constrainTo" | "calcConstrainBox">;
/**
 * Shorthand function to create {@see Window}
 *
 * @param config
 * @param items
 */
export declare const win: (config?: WindowConfig, ...items: Component[]) => Window;
export {};
//# sourceMappingURL=Window.d.ts.map