<?php
/*
 * Fusio - Self-Hosted API Management for Builders.
 * For the current version and information visit <https://www.fusio-project.org/>
 *
 * Copyright (c) Christoph Kappestein <christoph.kappestein@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Fusio\Impl\Installation;

use Fusio\Adapter;
use Fusio\Engine\Inflection\ClassName;
use Fusio\Impl\Action\Scheme;
use Fusio\Impl\Authorization;
use Fusio\Impl\Authorization\TokenGenerator;
use Fusio\Impl\Backend;
use Fusio\Impl\Connection\System as ConnectionSystem;
use Fusio\Impl\Consumer;
use Fusio\Impl\System;
use Fusio\Impl\Table;
use Fusio\Marketplace;
use Fusio\Model;
use Psr\Container\ContainerInterface;
use PSX\Api\Model\Passthru;
use PSX\Api\OperationInterface;
use PSX\Schema\ContentType;
use PSX\Schema\Type\Factory\PropertyTypeFactory;

/**
 * NewInstallation
 *
 * @author  Christoph Kappestein <christoph.kappestein@gmail.com>
 * @license http://www.apache.org/licenses/LICENSE-2.0
 * @link    https://www.fusio-project.org
 */
class NewInstallation
{
    public static function getData(?string $tenantId = null): DataBag
    {
        $password = \password_hash(TokenGenerator::generateUserPassword(), PASSWORD_DEFAULT);

        $bag = new DataBag();
        $bag->addCategory('default', tenantId: $tenantId);
        $bag->addCategory('backend', tenantId: $tenantId);
        $bag->addCategory('consumer', tenantId: $tenantId);
        $bag->addCategory('system', tenantId: $tenantId);
        $bag->addCategory('authorization', tenantId: $tenantId);
        $bag->addRole('default', 'Administrator', tenantId: $tenantId);
        $bag->addRole('default', 'Backend', tenantId: $tenantId);
        $bag->addRole('default', 'Consumer', tenantId: $tenantId);
        $bag->addUser('Administrator', 'Administrator', 'admin@localhost.com', $password, tenantId: $tenantId);
        $bag->addScope('backend', 'backend', 'Global access to the backend API', tenantId: $tenantId);
        $bag->addScope('consumer', 'consumer', 'Global access to the consumer API', tenantId: $tenantId);
        $bag->addScope('authorization', 'authorization', 'Authorization API endpoint', tenantId: $tenantId);
        $bag->addScope('authorization', 'openid', 'OpenID scope', tenantId: $tenantId);
        $bag->addScope('default', 'default', 'Default scope', tenantId: $tenantId);
        $bag->addConfig('app_approval', Table\Config::FORM_BOOLEAN, 0, 'If true the status of a new app is PENDING so that an administrator has to manually activate the app', tenantId: $tenantId);
        $bag->addConfig('consumer_max_apps', Table\Config::FORM_NUMBER, 16, 'The max amount of apps a consumer can generate', tenantId: $tenantId);
        $bag->addConfig('consumer_max_tokens', Table\Config::FORM_NUMBER, 16, 'The max amount of tokens a consumer can generate', tenantId: $tenantId);
        $bag->addConfig('consumer_max_webhooks', Table\Config::FORM_NUMBER, 8, 'The max amount of webhooks a consumer can register', tenantId: $tenantId);
        $bag->addConfig('authorization_url', Table\Config::FORM_STRING, '', 'Url where the user can authorize for the OAuth2 flow', tenantId: $tenantId);
        $bag->addConfig('info_title', Table\Config::FORM_STRING, 'Fusio', 'The title of the application', tenantId: $tenantId);
        $bag->addConfig('info_description', Table\Config::FORM_STRING, 'Self-Hosted API Management for Builders.', 'A short description of the application. CommonMark syntax MAY be used for rich text representation', tenantId: $tenantId);
        $bag->addConfig('info_tos', Table\Config::FORM_STRING, '', 'A URL to the Terms of Service for the API. MUST be in the format of a URL', tenantId: $tenantId);
        $bag->addConfig('info_contact_name', Table\Config::FORM_STRING, '', 'The identifying name of the contact person/organization', tenantId: $tenantId);
        $bag->addConfig('info_contact_url', Table\Config::FORM_STRING, '', 'The URL pointing to the contact information. MUST be in the format of a URL', tenantId: $tenantId);
        $bag->addConfig('info_contact_email', Table\Config::FORM_STRING, '', 'The email address of the contact person/organization. MUST be in the format of an email address', tenantId: $tenantId);
        $bag->addConfig('info_license_name', Table\Config::FORM_STRING, '', 'The license name used for the API', tenantId: $tenantId);
        $bag->addConfig('info_license_url', Table\Config::FORM_STRING, '', 'A URL to the license used for the API. MUST be in the format of a URL', tenantId: $tenantId);
        $bag->addConfig('mail_register_subject', Table\Config::FORM_STRING, 'Fusio registration', 'Subject of the activation mail', tenantId: $tenantId);
        $bag->addConfig('mail_register_body', Table\Config::FORM_TEXT, 'Hello {name},' . "\n\n" . 'you have successful registered at Fusio.' . "\n" . 'To activate you account please visit the following link:' . "\n" . '{apps_url}/developer/register/activate/{token}', 'Body of the activation mail', tenantId: $tenantId);
        $bag->addConfig('mail_pw_reset_subject', Table\Config::FORM_STRING, 'Fusio password reset', 'Subject of the password reset mail', tenantId: $tenantId);
        $bag->addConfig('mail_pw_reset_body', Table\Config::FORM_TEXT, 'Hello {name},' . "\n\n" . 'you have requested to reset your password.' . "\n" . 'To set a new password please visit the following link:' . "\n" . '{apps_url}/developer/password/confirm/{token}' . "\n\n" . 'Please ignore this email if you have not requested a password reset.', 'Body of the password reset mail', tenantId: $tenantId);
        $bag->addConfig('mail_points_subject', Table\Config::FORM_STRING, 'Fusio points threshold reached', 'Subject of the points threshold mail', tenantId: $tenantId);
        $bag->addConfig('mail_points_body', Table\Config::FORM_TEXT, 'Hello {name},' . "\n\n" . 'your account has reached the configured threshold of {points} points.' . "\n" . 'If your account reaches 0 points your are not longer able to invoke specific endpoints.' . "\n" . 'To prevent this please go to the developer portal to purchase new points:' . "\n" . '{apps_url}/developer', 'Body of the points threshold mail', tenantId: $tenantId);
        $bag->addConfig('recaptcha_key', Table\Config::FORM_STRING, '', 'ReCaptcha Key', tenantId: $tenantId);
        $bag->addConfig('recaptcha_secret', Table\Config::FORM_STRING, '', 'ReCaptcha Secret', tenantId: $tenantId);
        $bag->addConfig('payment_stripe_secret', Table\Config::FORM_STRING, '', 'The stripe webhook secret which is needed to verify a webhook request', tenantId: $tenantId);
        $bag->addConfig('payment_stripe_portal_configuration', Table\Config::FORM_STRING, '', 'The stripe portal configuration id', tenantId: $tenantId);
        $bag->addConfig('payment_currency', Table\Config::FORM_STRING, '', 'The three-character ISO-4217 currency code which is used to process payments', tenantId: $tenantId);
        $bag->addConfig('role_default', Table\Config::FORM_STRING, 'Consumer', 'Default role which a user gets assigned on registration', tenantId: $tenantId);
        $bag->addConfig('points_default', Table\Config::FORM_NUMBER, 0, 'The default amount of points which a user receives if he registers', tenantId: $tenantId);
        $bag->addConfig('points_threshold', Table\Config::FORM_NUMBER, 0, 'If a user goes below this points threshold we send an information to the user', tenantId: $tenantId);
        $bag->addConfig('system_mailer', Table\Config::FORM_STRING, '', 'Optional the name of an SMTP connection which is used as mailer, by default the system uses the connection configured through the APP_MAILER environment variable', tenantId: $tenantId);
        $bag->addConfig('system_dispatcher', Table\Config::FORM_STRING, '', 'Optional the name of an HTTP or Message-Queue connection which is used to dispatch events. By default the system uses simply cron and an internal table to dispatch such events, for better performance you can provide a Message-Queue connection and Fusio will only dispatch the event to the queue, then your worker must execute the actual webhook HTTP request', tenantId: $tenantId);
        $bag->addConfig('user_pw_length', Table\Config::FORM_NUMBER, 8, 'Minimal required password length', tenantId: $tenantId);
        $bag->addConfig('user_approval', Table\Config::FORM_BOOLEAN, 1, 'Whether the user needs to activate the account through an email', tenantId: $tenantId);
        $bag->addConfig('marketplace_client_id', Table\Config::FORM_STRING, '', 'Marketplace Client-Id, this is either your username or app key of the Fusio marketplace (marketplace.fusio-project.org)', tenantId: $tenantId);
        $bag->addConfig('marketplace_client_secret', Table\Config::FORM_STRING, '', 'Marketplace Client-Secret, this is either your password or app secret of the Fusio marketplace (marketplace.fusio-project.org)', tenantId: $tenantId);
        $bag->addConfig('sdkgen_client_id', Table\Config::FORM_STRING, '', 'SDKgen Client-Id, this is either your username or app key of the SDKgen app (sdkgen.app)', tenantId: $tenantId);
        $bag->addConfig('sdkgen_client_secret', Table\Config::FORM_STRING, '', 'SDKgen Client-Secret, this is either your password or app secret of the SDKgen app (sdkgen.app)', tenantId: $tenantId);
        if ($tenantId === null) {
            // we add the system connection only at the root tenant
            $bag->addConnection('System', ClassName::serialize(ConnectionSystem::class), tenantId: $tenantId);
        }
        $bag->addRate('Default', 0, 3600, 'PT1H', tenantId: $tenantId);
        $bag->addRate('Default-Anonymous', 4, 900, 'PT1H', tenantId: $tenantId);
        $bag->addRateAllocation('Default', tenantId: $tenantId);
        $bag->addRateAllocation('Default-Anonymous', null, null, null, null, false, tenantId: $tenantId);
        $bag->addCronjob('backend', 'Renew_Token', '0 * * * *', Scheme::wrap(Backend\Action\Connection\RenewToken::class), tenantId: $tenantId);
        $bag->addRoleScope('Administrator', 'authorization', tenantId: $tenantId);
        $bag->addRoleScope('Administrator', 'backend', tenantId: $tenantId);
        $bag->addRoleScope('Administrator', 'consumer', tenantId: $tenantId);
        $bag->addRoleScope('Administrator', 'default', tenantId: $tenantId);
        $bag->addRoleScope('Backend', 'authorization', tenantId: $tenantId);
        $bag->addRoleScope('Backend', 'backend', tenantId: $tenantId);
        $bag->addRoleScope('Backend', 'default', tenantId: $tenantId);
        $bag->addRoleScope('Consumer', 'authorization', tenantId: $tenantId);
        $bag->addRoleScope('Consumer', 'consumer', tenantId: $tenantId);
        $bag->addRoleScope('Consumer', 'default', tenantId: $tenantId);
        $bag->addSchema('default', 'Passthru', Passthru::class, tenantId: $tenantId);
        $bag->addSchema('default', 'Message', Model\Common\Message::class, tenantId: $tenantId);
        $bag->addUserScope('Administrator', 'backend', tenantId: $tenantId);
        $bag->addUserScope('Administrator', 'consumer', tenantId: $tenantId);
        $bag->addUserScope('Administrator', 'authorization', tenantId: $tenantId);
        $bag->addUserScope('Administrator', 'default', tenantId: $tenantId);
        $bag->addPage('Overview', 'overview', self::readFile('overview.html'), Table\Page::STATUS_INVISIBLE, tenantId: $tenantId);
        $bag->addPage('Getting started', 'getting-started', self::readFile('getting-started.html'), tenantId: $tenantId);
        $bag->addPage('API', 'api', self::readFile('api.html'), tenantId: $tenantId);
        $bag->addPage('Authorization', 'authorization', self::readFile('authorization.html'), tenantId: $tenantId);
        $bag->addPage('Support', 'support', self::readFile('support.html'), tenantId: $tenantId);
        $bag->addPage('SDK', 'sdk', self::readFile('sdk.html'), tenantId: $tenantId);

        foreach (self::getOperations() as $category => $operations) {
            $bag->addOperations($tenantId, $category, $operations);
        }

        return $bag;
    }

    private static function getOperations(): array
    {
        return [
            'default' => [
                'meta.getAbout' => new Operation(
                    action: System\Action\Meta\GetAbout::class,
                    httpMethod: 'GET',
                    httpPath: '/',
                    httpCode: 200,
                    outgoing: Model\System\About::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns meta information and links about the current installed Fusio version',
                ),
            ],
            'backend' => [
                'account.get' => new Operation(
                    action: Backend\Action\Account\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/account',
                    httpCode: 200,
                    outgoing: Model\Backend\User::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns user data of the authenticated user',
                ),
                'account.update' => new Operation(
                    action: Backend\Action\Account\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/account',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\UserUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates user data of the authenticated user',
                ),
                'account.changePassword' => new Operation(
                    action: Backend\Action\Account\ChangePassword::class,
                    httpMethod: 'PUT',
                    httpPath: '/account/change_password',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\AccountChangePassword::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Changes the password of the authenticated user',
                ),
                'action.getAll' => new Operation(
                    action: Backend\Action\Action\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/action',
                    httpCode: 200,
                    outgoing: Model\Backend\ActionCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of actions',
                ),
                'action.create' => new Operation(
                    action: Backend\Action\Action\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/action',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\ActionCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.action.create',
                    description: 'Creates a new action',
                ),
                'action.getClasses' => new Operation(
                    action: Backend\Action\Action\GetIndex::class,
                    httpMethod: 'GET',
                    httpPath: '/action/list',
                    httpCode: 200,
                    outgoing: Model\Backend\ActionIndex::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all available action classes',
                ),
                'action.getForm' => new Operation(
                    action: Backend\Action\Action\GetForm::class,
                    httpMethod: 'GET',
                    httpPath: '/action/form',
                    httpCode: 200,
                    outgoing: Model\Common\FormContainer::class,
                    parameters: ['class' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns the action config form',
                ),
                'action.execute' => new Operation(
                    action: Backend\Action\Action\Execute::class,
                    httpMethod: 'POST',
                    httpPath: '/action/execute/:action_id',
                    httpCode: 200,
                    outgoing: Model\Backend\ActionExecuteResponse::class,
                    incoming: Model\Backend\ActionExecuteRequest::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Executes a specific action',
                ),
                'action.get' => new Operation(
                    action: Backend\Action\Action\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/action/$action_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Action::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific action',
                ),
                'action.update' => new Operation(
                    action: Backend\Action\Action\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/action/$action_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\ActionUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.action.update',
                    description: 'Updates an existing action',
                ),
                'action.delete' => new Operation(
                    action: Backend\Action\Action\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/action/$action_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.action.delete',
                    description: 'Deletes an existing action',
                ),
                'app.getAll' => new Operation(
                    action: Backend\Action\App\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/app',
                    httpCode: 200,
                    outgoing: Model\Backend\AppCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of apps',
                ),
                'app.create' => new Operation(
                    action: Backend\Action\App\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/app',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\AppCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.app.create',
                    description: 'Creates a new app',
                ),
                'app.get' => new Operation(
                    action: Backend\Action\App\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/app/$app_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\App::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific app',
                ),
                'app.update' => new Operation(
                    action: Backend\Action\App\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/app/$app_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\AppUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.app.update',
                    description: 'Updates an existing app',
                ),
                'app.delete' => new Operation(
                    action: Backend\Action\App\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/app/$app_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.app.delete',
                    description: 'Deletes an existing app',
                ),
                'app.deleteToken' => new Operation(
                    action: Backend\Action\App\DeleteToken::class,
                    httpMethod: 'DELETE',
                    httpPath: '/app/$app_id<[0-9]+>/token/:token_id',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Deletes an existing token from an app',
                ),
                'audit.getAll' => new Operation(
                    action: Backend\Action\Audit\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/audit',
                    httpCode: 200,
                    outgoing: Model\Backend\AuditCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'event' => PropertyTypeFactory::getString(), 'ip' => PropertyTypeFactory::getString(), 'message' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of audits',
                ),
                'audit.get' => new Operation(
                    action: Backend\Action\Audit\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/audit/$audit_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Backend\Audit::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific audit',
                ),
                'category.getAll' => new Operation(
                    action: Backend\Action\Category\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/category',
                    httpCode: 200,
                    outgoing: Model\Backend\CategoryCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of categories',
                ),
                'category.create' => new Operation(
                    action: Backend\Action\Category\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/category',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\CategoryCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.category.create',
                    description: 'Creates a new category',
                ),
                'category.get' => new Operation(
                    action: Backend\Action\Category\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/category/$category_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Category::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific category',
                ),
                'category.update' => new Operation(
                    action: Backend\Action\Category\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/category/$category_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\CategoryUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.category.update',
                    description: 'Updates an existing category',
                ),
                'category.delete' => new Operation(
                    action: Backend\Action\Category\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/category/$category_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.category.delete',
                    description: 'Deletes an existing category',
                ),
                'config.getAll' => new Operation(
                    action: Backend\Action\Config\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/config',
                    httpCode: 200,
                    outgoing: Model\Backend\ConfigCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of configuration values',
                ),
                'config.get' => new Operation(
                    action: Backend\Action\Config\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/config/$config_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Config::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific config',
                ),
                'config.update' => new Operation(
                    action: Backend\Action\Config\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/config/$config_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\ConfigUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.config.update',
                    description: 'Updates an existing config value',
                ),
                'connection.getAll' => new Operation(
                    action: Backend\Action\Connection\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/connection',
                    httpCode: 200,
                    outgoing: Model\Backend\ConnectionCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'class' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of connections',
                ),
                'connection.create' => new Operation(
                    action: Backend\Action\Connection\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/connection',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\ConnectionCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.connection.create',
                    description: 'Creates a new connection',
                ),
                'connection.getClasses' => new Operation(
                    action: Backend\Action\Connection\GetIndex::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/list',
                    httpCode: 200,
                    outgoing: Model\Backend\ConnectionIndex::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all available connection classes',
                ),
                'connection.getForm' => new Operation(
                    action: Backend\Action\Connection\GetForm::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/form',
                    httpCode: 200,
                    outgoing: Model\Common\FormContainer::class,
                    parameters: ['class' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns the connection config form',
                ),
                'connection.get' => new Operation(
                    action: Backend\Action\Connection\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/$connection_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Connection::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific connection',
                ),
                'connection.update' => new Operation(
                    action: Backend\Action\Connection\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/connection/$connection_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\ConnectionUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.connection.update',
                    description: 'Updates an existing connection',
                ),
                'connection.delete' => new Operation(
                    action: Backend\Action\Connection\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/connection/$connection_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.connection.delete',
                    description: 'Deletes an existing connection',
                ),
                'connection.getRedirect' => new Operation(
                    action: Backend\Action\Connection\GetRedirect::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:connection_id/redirect',
                    httpCode: 200,
                    outgoing: Model\Backend\ConnectionRedirectResponse::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a redirect url to start the OAuth2 authorization flow for the given connection',
                ),
                'connection.database.getTables' => new Operation(
                    action: Backend\Action\Connection\Database\Table\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:connection_id/database',
                    httpCode: 200,
                    outgoing: Model\Backend\DatabaseTableCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all available tables on a database',
                ),
                'connection.database.getTable' => new Operation(
                    action: Backend\Action\Connection\Database\Table\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:connection_id/database/:table_name',
                    httpCode: 200,
                    outgoing: Model\Backend\DatabaseTable::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns the schema of a specific table on a database',
                ),
                'connection.database.createTable' => new Operation(
                    action: Backend\Action\Connection\Database\Table\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/connection/:connection_id/database',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\DatabaseTable::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Creates a new table on a database',
                ),
                'connection.database.updateTable' => new Operation(
                    action: Backend\Action\Connection\Database\Table\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/connection/:connection_id/database/:table_name',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\DatabaseTable::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates an existing table on a database',
                ),
                'connection.database.deleteTable' => new Operation(
                    action: Backend\Action\Connection\Database\Table\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/connection/:connection_id/database/:table_name',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Deletes an existing table on a database',
                ),
                'connection.database.getRows' => new Operation(
                    action: Backend\Action\Connection\Database\Row\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:connection_id/database/:table_name/rows',
                    httpCode: 200,
                    outgoing: Model\Backend\DatabaseRowCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'filterBy' => PropertyTypeFactory::getString(), 'filterOp' => PropertyTypeFactory::getString(), 'filterValue' => PropertyTypeFactory::getString(), 'sortBy' => PropertyTypeFactory::getString(), 'sortOrder' => PropertyTypeFactory::getString(), 'columns' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns paginated rows at a table on a database',
                ),
                'connection.database.getRow' => new Operation(
                    action: Backend\Action\Connection\Database\Row\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:connection_id/database/:table_name/rows/:id',
                    httpCode: 200,
                    outgoing: Model\Backend\DatabaseRow::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific row at a table on a database',
                ),
                'connection.database.createRow' => new Operation(
                    action: Backend\Action\Connection\Database\Row\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/connection/:connection_id/database/:table_name/rows',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\DatabaseRow::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Creates a new row at a table on a database',
                ),
                'connection.database.updateRow' => new Operation(
                    action: Backend\Action\Connection\Database\Row\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/connection/:connection_id/database/:table_name/rows/:id',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\DatabaseRow::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates an existing row at a table on a database',
                ),
                'connection.database.deleteRow' => new Operation(
                    action: Backend\Action\Connection\Database\Row\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/connection/:connection_id/database/:table_name/rows/:id',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Deletes an existing row at a table on a database',
                ),
                'connection.filesystem.getAll' => new Operation(
                    action: Backend\Action\Connection\Filesystem\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:connection_id/filesystem',
                    httpCode: 200,
                    outgoing: Model\Backend\FileCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all available files on the filesystem connection',
                ),
                'connection.filesystem.get' => new Operation(
                    action: Backend\Action\Connection\Filesystem\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:connection_id/filesystem/:file_id',
                    httpCode: 200,
                    outgoing: ContentType::BINARY,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns the content of the provided file id on the filesystem connection',
                ),
                'connection.filesystem.create' => new Operation(
                    action: Backend\Action\Connection\Filesystem\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/connection/:connection_id/filesystem',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: ContentType::MULTIPART,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Uploads one or more files on the filesystem connection',
                ),
                'connection.filesystem.update' => new Operation(
                    action: Backend\Action\Connection\Filesystem\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/connection/:connection_id/filesystem/:file_id',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: ContentType::MULTIPART,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates an existing file on the filesystem connection',
                ),
                'connection.filesystem.delete' => new Operation(
                    action: Backend\Action\Connection\Filesystem\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/connection/:connection_id/filesystem/:file_id',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Deletes an existing file on the filesystem connection',
                ),
                'connection.http.execute' => new Operation(
                    action: Backend\Action\Connection\Http\Execute::class,
                    httpMethod: 'POST',
                    httpPath: '/connection/:connection_id/http',
                    httpCode: 200,
                    outgoing: Model\Backend\HttpResponse::class,
                    incoming: Model\Backend\HttpRequest::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Sends an arbitrary HTTP request to the connection',
                ),
                'connection.sdk.get' => new Operation(
                    action: Backend\Action\Connection\Sdk\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:connection_id/sdk',
                    httpCode: 200,
                    outgoing: Passthru::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns the SDK specification',
                ),
                'cronjob.getAll' => new Operation(
                    action: Backend\Action\Cronjob\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/cronjob',
                    httpCode: 200,
                    outgoing: Model\Backend\CronjobCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of cronjobs',
                ),
                'cronjob.create' => new Operation(
                    action: Backend\Action\Cronjob\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/cronjob',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\CronjobCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.cronjob.create',
                    description: 'Creates a new cronjob',
                ),
                'cronjob.get' => new Operation(
                    action: Backend\Action\Cronjob\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/cronjob/$cronjob_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Cronjob::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific cronjob',
                ),
                'cronjob.update' => new Operation(
                    action: Backend\Action\Cronjob\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/cronjob/$cronjob_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\CronjobUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.cronjob.update',
                    description: 'Updates an existing cronjob',
                ),
                'cronjob.delete' => new Operation(
                    action: Backend\Action\Cronjob\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/cronjob/$cronjob_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.cronjob.delete',
                    description: 'Deletes an existing cronjob',
                ),
                'dashboard.getAll' => new Operation(
                    action: Backend\Action\Dashboard\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/dashboard',
                    httpCode: 200,
                    outgoing: Model\Backend\Dashboard::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all available dashboard widgets',
                ),
                'event.getAll' => new Operation(
                    action: Backend\Action\Event\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/event',
                    httpCode: 200,
                    outgoing: Model\Backend\EventCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of events',
                ),
                'event.create' => new Operation(
                    action: Backend\Action\Event\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/event',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\EventCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.event.create',
                    description: 'Creates a new event',
                ),
                'event.get' => new Operation(
                    action: Backend\Action\Event\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/event/$event_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Event::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific event',
                ),
                'event.update' => new Operation(
                    action: Backend\Action\Event\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/event/$event_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\EventUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.event.update',
                    description: 'Updates an existing event',
                ),
                'event.delete' => new Operation(
                    action: Backend\Action\Event\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/event/$event_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.event.delete',
                    description: 'Deletes an existing event',
                ),
                'backup.export' => new Operation(
                    action: Backend\Action\Backup\Export::class,
                    httpMethod: 'POST',
                    httpPath: '/backup/export',
                    httpCode: 200,
                    outgoing: Model\Backend\BackupExport::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Generates an backup of the current system',
                ),
                'backup.import' => new Operation(
                    action: Backend\Action\Backup\Import::class,
                    httpMethod: 'POST',
                    httpPath: '/backup/import',
                    httpCode: 200,
                    outgoing: Model\Backend\BackupImportResult::class,
                    incoming: Model\Backend\BackupImport::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Imports an backup to the current system',
                ),
                'firewall.getAll' => new Operation(
                    action: Backend\Action\Firewall\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/firewall',
                    httpCode: 200,
                    outgoing: Model\Backend\FirewallCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of firewall rules',
                ),
                'firewall.create' => new Operation(
                    action: Backend\Action\Firewall\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/firewall',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\FirewallCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.firewall.create',
                    description: 'Creates a new firewall rule',
                ),
                'firewall.get' => new Operation(
                    action: Backend\Action\Firewall\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/firewall/$firewall_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Firewall::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific firewall rule',
                ),
                'firewall.update' => new Operation(
                    action: Backend\Action\Firewall\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/firewall/$firewall_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\FirewallUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.firewall.update',
                    description: 'Updates an existing firewall rule',
                ),
                'firewall.delete' => new Operation(
                    action: Backend\Action\Firewall\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/firewall/$firewall_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.firewall.delete',
                    description: 'Deletes an existing firewall rule',
                ),
                'form.getAll' => new Operation(
                    action: Backend\Action\Form\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/form',
                    httpCode: 200,
                    outgoing: Model\Backend\FormCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of forms',
                ),
                'form.create' => new Operation(
                    action: Backend\Action\Form\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/form',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\FormCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.form.create',
                    description: 'Creates a new form',
                ),
                'form.get' => new Operation(
                    action: Backend\Action\Form\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/form/$form_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Form::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific form',
                ),
                'form.update' => new Operation(
                    action: Backend\Action\Form\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/form/$form_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\FormUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.form.update',
                    description: 'Updates an existing form',
                ),
                'form.delete' => new Operation(
                    action: Backend\Action\Form\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/form/$form_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.form.delete',
                    description: 'Deletes an existing form',
                ),
                'generator.getClasses' => new Operation(
                    action: Backend\Action\Generator\GetIndex::class,
                    httpMethod: 'GET',
                    httpPath: '/generator',
                    httpCode: 200,
                    outgoing: Model\Backend\GeneratorIndexProviders::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all available generator classes',
                ),
                'generator.getForm' => new Operation(
                    action: Backend\Action\Generator\GetForm::class,
                    httpMethod: 'GET',
                    httpPath: '/generator/:provider',
                    httpCode: 200,
                    outgoing: Model\Common\FormContainer::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns the generator config form',
                ),
                'generator.executeProvider' => new Operation(
                    action: Backend\Action\Generator\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/generator/:provider',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\GeneratorProvider::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Executes a generator with the provided config',
                ),
                'generator.getChangelog' => new Operation(
                    action: Backend\Action\Generator\Changelog::class,
                    httpMethod: 'PUT',
                    httpPath: '/generator/:provider',
                    httpCode: 200,
                    outgoing: Model\Backend\GeneratorProviderChangelog::class,
                    incoming: Model\Backend\GeneratorProviderConfig::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Generates a changelog of all potential changes if you execute this generator with the provided config',
                ),
                'identity.getAll' => new Operation(
                    action: Backend\Action\Identity\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/identity',
                    httpCode: 200,
                    outgoing: Model\Backend\IdentityCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of identities',
                ),
                'identity.create' => new Operation(
                    action: Backend\Action\Identity\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/identity',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\IdentityCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.identity.create',
                    description: 'Creates a new identity',
                ),
                'identity.getClasses' => new Operation(
                    action: Backend\Action\Identity\GetIndex::class,
                    httpMethod: 'GET',
                    httpPath: '/identity/list',
                    httpCode: 200,
                    outgoing: Model\Backend\IdentityIndex::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all available identity classes',
                ),
                'identity.getForm' => new Operation(
                    action: Backend\Action\Identity\GetForm::class,
                    httpMethod: 'GET',
                    httpPath: '/identity/form',
                    httpCode: 200,
                    outgoing: Model\Common\FormContainer::class,
                    parameters: ['class' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns the identity config form',
                ),
                'identity.get' => new Operation(
                    action: Backend\Action\Identity\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/identity/$identity_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Identity::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific identity',
                ),
                'identity.update' => new Operation(
                    action: Backend\Action\Identity\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/identity/$identity_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\IdentityUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.identity.update',
                    description: 'Updates an existing identity',
                ),
                'identity.delete' => new Operation(
                    action: Backend\Action\Identity\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/identity/$identity_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.identity.delete',
                    description: 'Deletes an existing identity',
                ),
                'log.getAllErrors' => new Operation(
                    action: Backend\Action\Log\Error\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/log/error',
                    httpCode: 200,
                    outgoing: Model\Backend\LogErrorCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of log errors',
                ),
                'log.getError' => new Operation(
                    action: Backend\Action\Log\Error\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/log/error/$error_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Backend\LogError::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific error',
                ),
                'log.getAll' => new Operation(
                    action: Backend\Action\Log\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/log',
                    httpCode: 200,
                    outgoing: Model\Backend\LogCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of logs',
                ),
                'log.get' => new Operation(
                    action: Backend\Action\Log\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/log/$log_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Backend\Log::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific log',
                ),
                'marketplace.action.getAll' => new Operation(
                    action: Backend\Action\Marketplace\Action\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/marketplace/action',
                    httpCode: 200,
                    outgoing: Marketplace\MarketplaceActionCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'query' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of marketplace actions',
                ),
                'marketplace.action.install' => new Operation(
                    action: Backend\Action\Marketplace\Action\Install::class,
                    httpMethod: 'POST',
                    httpPath: '/marketplace/action',
                    httpCode: 201,
                    outgoing: Marketplace\MarketplaceMessage::class,
                    incoming: Marketplace\MarketplaceInstall::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Installs an action from the marketplace',
                ),
                'marketplace.action.get' => new Operation(
                    action: Backend\Action\Marketplace\Action\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/marketplace/action/:user/:name',
                    httpCode: 200,
                    outgoing: Marketplace\MarketplaceAction::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific marketplace action',
                ),
                'marketplace.action.upgrade' => new Operation(
                    action: Backend\Action\Marketplace\Action\Upgrade::class,
                    httpMethod: 'PUT',
                    httpPath: '/marketplace/action/:user/:name',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Upgrades an action from the marketplace',
                ),
                'marketplace.app.getAll' => new Operation(
                    action: Backend\Action\Marketplace\App\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/marketplace/app',
                    httpCode: 200,
                    outgoing: Marketplace\MarketplaceAppCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'query' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of marketplace apps',
                ),
                'marketplace.app.install' => new Operation(
                    action: Backend\Action\Marketplace\App\Install::class,
                    httpMethod: 'POST',
                    httpPath: '/marketplace/app',
                    httpCode: 201,
                    outgoing: Marketplace\MarketplaceMessage::class,
                    incoming: Marketplace\MarketplaceInstall::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Installs an app from the marketplace',
                ),
                'marketplace.app.get' => new Operation(
                    action: Backend\Action\Marketplace\App\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/marketplace/app/:user/:name',
                    httpCode: 200,
                    outgoing: Marketplace\MarketplaceApp::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific marketplace app',
                ),
                'marketplace.app.upgrade' => new Operation(
                    action: Backend\Action\Marketplace\App\Upgrade::class,
                    httpMethod: 'PUT',
                    httpPath: '/marketplace/app/:user/:name',
                    httpCode: 200,
                    outgoing: Marketplace\MarketplaceMessage::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Upgrades an app from the marketplace',
                ),
                'page.getAll' => new Operation(
                    action: Backend\Action\Page\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/page',
                    httpCode: 200,
                    outgoing: Model\Backend\PageCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of pages',
                ),
                'page.create' => new Operation(
                    action: Backend\Action\Page\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/page',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\PageCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.page.create',
                    description: 'Creates a new page',
                ),
                'page.get' => new Operation(
                    action: Backend\Action\Page\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/page/$page_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Page::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific page',
                ),
                'page.update' => new Operation(
                    action: Backend\Action\Page\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/page/$page_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\PageUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.page.update',
                    description: 'Updates an existing page',
                ),
                'page.delete' => new Operation(
                    action: Backend\Action\Page\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/page/$page_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.page.delete',
                    description: 'Deletes an existing page',
                ),
                'plan.getAll' => new Operation(
                    action: Backend\Action\Plan\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/plan',
                    httpCode: 200,
                    outgoing: Model\Backend\PlanCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of plans',
                ),
                'plan.create' => new Operation(
                    action: Backend\Action\Plan\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/plan',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\PlanCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.plan.create',
                    description: 'Creates a new plan',
                ),
                'plan.get' => new Operation(
                    action: Backend\Action\Plan\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/plan/$plan_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Plan::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific plan',
                ),
                'plan.update' => new Operation(
                    action: Backend\Action\Plan\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/plan/$plan_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\PlanUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.plan.update',
                    description: 'Updates an existing plan',
                ),
                'plan.delete' => new Operation(
                    action: Backend\Action\Plan\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/plan/$plan_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.plan.delete',
                    description: 'Deletes an existing plan',
                ),
                'rate.getAll' => new Operation(
                    action: Backend\Action\Rate\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/rate',
                    httpCode: 200,
                    outgoing: Model\Backend\RateCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of rate limitations',
                ),
                'rate.create' => new Operation(
                    action: Backend\Action\Rate\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/rate',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\RateCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.rate.create',
                    description: 'Creates a new rate limitation',
                ),
                'rate.get' => new Operation(
                    action: Backend\Action\Rate\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/rate/$rate_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Rate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific rate',
                ),
                'rate.update' => new Operation(
                    action: Backend\Action\Rate\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/rate/$rate_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\RateUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.rate.update',
                    description: 'Updates an existing rate',
                ),
                'rate.delete' => new Operation(
                    action: Backend\Action\Rate\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/rate/$rate_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.rate.delete',
                    description: 'Deletes an existing rate',
                ),
                'role.getAll' => new Operation(
                    action: Backend\Action\Role\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/role',
                    httpCode: 200,
                    outgoing: Model\Backend\RoleCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of roles',
                ),
                'role.create' => new Operation(
                    action: Backend\Action\Role\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/role',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\RoleCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.role.create',
                    description: 'Creates a new role',
                ),
                'role.get' => new Operation(
                    action: Backend\Action\Role\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/role/$role_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Role::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific role',
                ),
                'role.update' => new Operation(
                    action: Backend\Action\Role\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/role/$role_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\RoleUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.role.update',
                    description: 'Updates an existing role',
                ),
                'role.delete' => new Operation(
                    action: Backend\Action\Role\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/role/$role_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.role.delete',
                    description: 'Deletes an existing role',
                ),
                'operation.getAll' => new Operation(
                    action: Backend\Action\Operation\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/operation',
                    httpCode: 200,
                    outgoing: Model\Backend\OperationCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of operations',
                ),
                'operation.create' => new Operation(
                    action: Backend\Action\Operation\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/operation',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\OperationCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.operation.create',
                    description: 'Creates a new operation',
                ),
                'operation.get' => new Operation(
                    action: Backend\Action\Operation\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/operation/$operation_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Operation::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific operation',
                ),
                'operation.update' => new Operation(
                    action: Backend\Action\Operation\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/operation/$operation_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\OperationUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.operation.update',
                    description: 'Updates an existing operation',
                ),
                'operation.delete' => new Operation(
                    action: Backend\Action\Operation\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/operation/$operation_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.operation.delete',
                    description: 'Deletes an existing operation',
                ),
                'schema.getAll' => new Operation(
                    action: Backend\Action\Schema\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/schema',
                    httpCode: 200,
                    outgoing: Model\Backend\SchemaCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of schemas',
                ),
                'schema.create' => new Operation(
                    action: Backend\Action\Schema\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/schema',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\SchemaCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.schema.create',
                    description: 'Creates a new schema',
                ),
                'schema.getPreview' => new Operation(
                    action: Backend\Action\Schema\GetPreview::class,
                    httpMethod: 'POST',
                    httpPath: '/schema/preview/:schema_id',
                    httpCode: 200,
                    outgoing: Model\Backend\SchemaPreviewResponse::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a HTML preview of the provided schema',
                ),
                'schema.updateForm' => new Operation(
                    action: Backend\Action\Schema\Form::class,
                    httpMethod: 'PUT',
                    httpPath: '/schema/form/$schema_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\SchemaForm::class,
                    throws: [999 => Model\Common\Message::class],
                    stability: OperationInterface::STABILITY_LEGACY,
                    description: 'Updates an existing schema form',
                ),
                'schema.get' => new Operation(
                    action: Backend\Action\Schema\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/schema/$schema_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Schema::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific schema',
                ),
                'schema.update' => new Operation(
                    action: Backend\Action\Schema\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/schema/$schema_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\SchemaUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.schema.update',
                    description: 'Updates an existing schema',
                ),
                'schema.delete' => new Operation(
                    action: Backend\Action\Schema\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/schema/$schema_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.schema.delete',
                    description: 'Deletes an existing schema',
                ),
                'scope.getAll' => new Operation(
                    action: Backend\Action\Scope\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/scope',
                    httpCode: 200,
                    outgoing: Model\Backend\ScopeCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of scopes',
                ),
                'scope.create' => new Operation(
                    action: Backend\Action\Scope\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/scope',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\ScopeCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.scope.create',
                    description: 'Creates a new scope',
                ),
                'scope.getCategories' => new Operation(
                    action: Backend\Action\Scope\GetCategories::class,
                    httpMethod: 'GET',
                    httpPath: '/scope/categories',
                    httpCode: 200,
                    outgoing: Model\Backend\ScopeCategories::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all available scopes grouped by category',
                ),
                'scope.get' => new Operation(
                    action: Backend\Action\Scope\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/scope/$scope_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Scope::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific scope',
                ),
                'scope.update' => new Operation(
                    action: Backend\Action\Scope\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/scope/$scope_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\ScopeUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.scope.update',
                    description: 'Updates an existing scope',
                ),
                'scope.delete' => new Operation(
                    action: Backend\Action\Scope\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/scope/$scope_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.scope.delete',
                    description: 'Deletes an existing scope',
                ),
                'sdk.getAll' => new Operation(
                    action: Backend\Action\Sdk\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/sdk',
                    httpCode: 200,
                    outgoing: Model\Backend\SdkResponse::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of SDKs',
                ),
                'sdk.generate' => new Operation(
                    action: Backend\Action\Sdk\Generate::class,
                    httpMethod: 'POST',
                    httpPath: '/sdk',
                    httpCode: 200,
                    outgoing: Model\Backend\SdkMessage::class,
                    incoming: Model\Backend\SdkGenerate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Generates a specific SDK',
                ),
                'statistic.getActivitiesPerUser' => new Operation(
                    action: Backend\Action\Statistic\GetActivitiesPerUser::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/activities_per_user',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the activities per user',
                ),
                'statistic.getCountRequests' => new Operation(
                    action: Backend\Action\Statistic\GetCountRequests::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/count_requests',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticCount::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the request count',
                ),
                'statistic.getErrorsPerOperation' => new Operation(
                    action: Backend\Action\Statistic\GetErrorsPerOperation::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/errors_per_operation',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the errors per operation',
                ),
                'statistic.getIncomingRequests' => new Operation(
                    action: Backend\Action\Statistic\GetIncomingRequests::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/incoming_requests',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the incoming requests',
                ),
                'statistic.getIncomingTransactions' => new Operation(
                    action: Backend\Action\Statistic\GetIncomingTransactions::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/incoming_transactions',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the incoming transactions',
                ),
                'statistic.getIssuedTokens' => new Operation(
                    action: Backend\Action\Statistic\GetIssuedTokens::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/issued_tokens',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the issues tokens',
                ),
                'statistic.getMostUsedActivities' => new Operation(
                    action: Backend\Action\Statistic\GetMostUsedActivities::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/most_used_activities',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the most used activities',
                ),
                'statistic.getMostUsedApps' => new Operation(
                    action: Backend\Action\Statistic\GetMostUsedApps::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/most_used_apps',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the most used apps',
                ),
                'statistic.getMostUsedOperations' => new Operation(
                    action: Backend\Action\Statistic\GetMostUsedOperations::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/most_used_operations',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the most used operations',
                ),
                'statistic.getRequestsPerIP' => new Operation(
                    action: Backend\Action\Statistic\GetRequestsPerIP::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/requests_per_ip',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the requests per ip',
                ),
                'statistic.getRequestsPerOperation' => new Operation(
                    action: Backend\Action\Statistic\GetRequestsPerOperation::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/requests_per_operation',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the requests per operation',
                ),
                'statistic.getRequestsPerUser' => new Operation(
                    action: Backend\Action\Statistic\GetRequestsPerUser::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/requests_per_user',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the requests per user',
                ),
                'statistic.getTestCoverage' => new Operation(
                    action: Backend\Action\Statistic\GetTestCoverage::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/test_coverage',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the test coverage',
                ),
                'statistic.getTimeAverage' => new Operation(
                    action: Backend\Action\Statistic\GetTimeAverage::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/time_average',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the time average',
                ),
                'statistic.getTimePerOperation' => new Operation(
                    action: Backend\Action\Statistic\GetTimePerOperation::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/time_per_operation',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the time per operation',
                ),
                'statistic.getUsedPoints' => new Operation(
                    action: Backend\Action\Statistic\GetUsedPoints::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/used_points',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the used points',
                ),
                'statistic.getUserRegistrations' => new Operation(
                    action: Backend\Action\Statistic\GetUserRegistrations::class,
                    httpMethod: 'GET',
                    httpPath: '/statistic/user_registrations',
                    httpCode: 200,
                    outgoing: Model\Backend\StatisticChart::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'operationId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'ip' => PropertyTypeFactory::getString(), 'userAgent' => PropertyTypeFactory::getString(), 'method' => PropertyTypeFactory::getString(), 'path' => PropertyTypeFactory::getString(), 'header' => PropertyTypeFactory::getString(), 'body' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a statistic containing the user registrations',
                ),
                'tenant.setup' => new Operation(
                    action: Backend\Action\Tenant\Setup::class,
                    httpMethod: 'PUT',
                    httpPath: '/tenant/:tenant_id',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Setup a new tenant',
                ),
                'tenant.remove' => new Operation(
                    action: Backend\Action\Tenant\Remove::class,
                    httpMethod: 'DELETE',
                    httpPath: '/tenant/:tenant_id',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Removes an existing tenant',
                ),
                'test.getAll' => new Operation(
                    action: Backend\Action\Test\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/test',
                    httpCode: 200,
                    outgoing: Model\Backend\TestCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of tests',
                ),
                'test.get' => new Operation(
                    action: Backend\Action\Test\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/test/$test_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Backend\Test::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific test',
                ),
                'test.refresh' => new Operation(
                    action: Backend\Action\Test\Refresh::class,
                    httpMethod: 'PUT',
                    httpPath: '/test',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Refresh all tests',
                ),
                'test.run' => new Operation(
                    action: Backend\Action\Test\Run::class,
                    httpMethod: 'POST',
                    httpPath: '/test',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Run all tests',
                ),
                'test.update' => new Operation(
                    action: Backend\Action\Test\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/test/$test_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\Test::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates an existing test',
                ),
                'token.getAll' => new Operation(
                    action: Backend\Action\Token\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/token',
                    httpCode: 200,
                    outgoing: Model\Backend\TokenCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'appId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'status' => PropertyTypeFactory::getInteger(), 'scope' => PropertyTypeFactory::getString(), 'ip' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of tokens',
                ),
                'token.get' => new Operation(
                    action: Backend\Action\Token\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/token/$token_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Backend\Token::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific token',
                ),
                'transaction.getAll' => new Operation(
                    action: Backend\Action\Transaction\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/transaction',
                    httpCode: 200,
                    outgoing: Model\Backend\TransactionCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString(), 'from' => PropertyTypeFactory::getDateTime(), 'to' => PropertyTypeFactory::getDateTime(), 'planId' => PropertyTypeFactory::getInteger(), 'userId' => PropertyTypeFactory::getInteger(), 'appId' => PropertyTypeFactory::getInteger(), 'status' => PropertyTypeFactory::getString(), 'provider' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of transactions',
                ),
                'transaction.get' => new Operation(
                    action: Backend\Action\Transaction\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/transaction/$transaction_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Backend\Transaction::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific transaction',
                ),
                'trash.getTypes' => new Operation(
                    action: Backend\Action\Trash\GetTypes::class,
                    httpMethod: 'GET',
                    httpPath: '/trash',
                    httpCode: 200,
                    outgoing: Model\Backend\TrashTypes::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all trash types',
                ),
                'trash.getAllByType' => new Operation(
                    action: Backend\Action\Trash\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/trash/:type',
                    httpCode: 200,
                    outgoing: Model\Backend\TrashDataCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all deleted records by trash type',
                ),
                'trash.restore' => new Operation(
                    action: Backend\Action\Trash\Restore::class,
                    httpMethod: 'POST',
                    httpPath: '/trash/:type',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\TrashRestore::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Restores a previously deleted record',
                ),
                'trigger.getAll' => new Operation(
                    action: Backend\Action\Trigger\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/trigger',
                    httpCode: 200,
                    outgoing: Model\Backend\TriggerCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of triggers',
                ),
                'trigger.create' => new Operation(
                    action: Backend\Action\Trigger\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/trigger',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\TriggerCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.trigger.create',
                    description: 'Creates a new trigger',
                ),
                'trigger.get' => new Operation(
                    action: Backend\Action\Trigger\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/trigger/$trigger_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Trigger::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific trigger',
                ),
                'trigger.update' => new Operation(
                    action: Backend\Action\Trigger\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/trigger/$trigger_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\TriggerUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.trigger.update',
                    description: 'Updates an existing trigger',
                ),
                'trigger.delete' => new Operation(
                    action: Backend\Action\Trigger\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/trigger/$trigger_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.trigger.delete',
                    description: 'Deletes an existing trigger',
                ),
                'user.getAll' => new Operation(
                    action: Backend\Action\User\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/user',
                    httpCode: 200,
                    outgoing: Model\Backend\UserCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of users',
                ),
                'user.create' => new Operation(
                    action: Backend\Action\User\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/user',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\UserCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.user.create',
                    description: 'Creates a new user',
                ),
                'user.get' => new Operation(
                    action: Backend\Action\User\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/user/$user_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\User::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific user',
                ),
                'user.update' => new Operation(
                    action: Backend\Action\User\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/user/$user_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\UserUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.user.update',
                    description: 'Updates an existing user',
                ),
                'user.delete' => new Operation(
                    action: Backend\Action\User\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/user/$user_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.user.delete',
                    description: 'Deletes an existing user',
                ),
                'user.resend' => new Operation(
                    action: Backend\Action\User\Resend::class,
                    httpMethod: 'POST',
                    httpPath: '/user/$user_id<[0-9]+|^~>/resend',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Passthru::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.user.update',
                    description: 'Resend the activation mail to the provided user',
                ),
                'webhook.getAll' => new Operation(
                    action: Backend\Action\Webhook\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/webhook',
                    httpCode: 200,
                    outgoing: Model\Backend\WebhookCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of webhooks',
                ),
                'webhook.create' => new Operation(
                    action: Backend\Action\Webhook\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/webhook',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\WebhookCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.webhook.create',
                    description: 'Creates a new webhook',
                ),
                'webhook.get' => new Operation(
                    action: Backend\Action\Webhook\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/webhook/$webhook_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Backend\Webhook::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific webhook',
                ),
                'webhook.update' => new Operation(
                    action: Backend\Action\Webhook\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/webhook/$webhook_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\WebhookUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.webhook.update',
                    description: 'Updates an existing webhook',
                ),
                'webhook.delete' => new Operation(
                    action: Backend\Action\Webhook\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/webhook/$webhook_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    eventName: 'fusio.webhook.delete',
                    description: 'Deletes an existing webhook',
                ),
            ],
            'consumer' => [
                'app.getAll' => new Operation(
                    action: Consumer\Action\App\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/app',
                    httpCode: 200,
                    outgoing: Model\Consumer\AppCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of apps which are assigned to the authenticated user',
                ),
                'app.create' => new Operation(
                    action: Consumer\Action\App\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/app',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\AppCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Creates a new app for the authenticated user',
                ),
                'app.get' => new Operation(
                    action: Consumer\Action\App\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/app/$app_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Consumer\App::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific app for the authenticated user',
                ),
                'app.update' => new Operation(
                    action: Consumer\Action\App\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/app/$app_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\AppUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates an existing app for the authenticated user',
                ),
                'app.delete' => new Operation(
                    action: Consumer\Action\App\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/app/$app_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Deletes an existing app for the authenticated user',
                ),
                'event.getAll' => new Operation(
                    action: Consumer\Action\Event\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/event',
                    httpCode: 200,
                    outgoing: Model\Consumer\EventCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of apps which are assigned to the authenticated user',
                ),
                'event.get' => new Operation(
                    action: Consumer\Action\Event\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/event/$event_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Consumer\Event::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific event for the authenticated user',
                ),
                'grant.getAll' => new Operation(
                    action: Consumer\Action\Grant\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/grant',
                    httpCode: 200,
                    outgoing: Model\Consumer\GrantCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of grants which are assigned to the authenticated user',
                ),
                'grant.delete' => new Operation(
                    action: Consumer\Action\Grant\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/grant/$grant_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Deletes an existing grant for an app which was created by the authenticated user',
                ),
                'log.getAll' => new Operation(
                    action: Consumer\Action\Log\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/log',
                    httpCode: 200,
                    outgoing: Model\Consumer\LogCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of logs which are assigned to the authenticated user',
                ),
                'log.get' => new Operation(
                    action: Consumer\Action\Log\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/log/$log_id<[0-9]+>',
                    httpCode: 200,
                    outgoing: Model\Consumer\Log::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific log for the authenticated user',
                ),
                'page.getAll' => new Operation(
                    action: Consumer\Action\Page\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/page',
                    httpCode: 200,
                    outgoing: Model\Consumer\PageCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns a paginated list of pages which are relevant to the authenticated user',
                ),
                'page.get' => new Operation(
                    action: Consumer\Action\Page\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/page/:page_id',
                    httpCode: 200,
                    outgoing: Model\Consumer\Page::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns a specific page for the authenticated user',
                ),
                'form.getAll' => new Operation(
                    action: Consumer\Action\Form\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/form',
                    httpCode: 200,
                    outgoing: Model\Consumer\FormCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns a paginated list of forms which are relevant to the authenticated user',
                ),
                'form.get' => new Operation(
                    action: Consumer\Action\Form\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/form/:form_id',
                    httpCode: 200,
                    outgoing: Model\Consumer\Form::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns a specific form for the authenticated user',
                ),
                'payment.portal' => new Operation(
                    action: Consumer\Action\Payment\Portal::class,
                    httpMethod: 'POST',
                    httpPath: '/payment/:provider/portal',
                    httpCode: 200,
                    outgoing: Model\Consumer\PaymentPortalResponse::class,
                    incoming: Model\Consumer\PaymentPortalRequest::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Generates a payment portal link for the authenticated user',
                ),
                'payment.checkout' => new Operation(
                    action: Consumer\Action\Payment\Checkout::class,
                    httpMethod: 'POST',
                    httpPath: '/payment/:provider/checkout',
                    httpCode: 200,
                    outgoing: Model\Consumer\PaymentCheckoutResponse::class,
                    incoming: Model\Consumer\PaymentCheckoutRequest::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Start the checkout process for a specific plan',
                ),
                'plan.getAll' => new Operation(
                    action: Consumer\Action\Plan\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/plan',
                    httpCode: 200,
                    outgoing: Model\Consumer\PlanCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of plans which are relevant to the authenticated user',
                ),
                'plan.get' => new Operation(
                    action: Consumer\Action\Plan\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/plan/$plan_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Consumer\Plan::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific plan for the authenticated user',
                ),
                'scope.getAll' => new Operation(
                    action: Consumer\Action\Scope\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/scope',
                    httpCode: 200,
                    outgoing: Model\Consumer\ScopeCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of scopes which are assigned to the authenticated user',
                ),
                'scope.getCategories' => new Operation(
                    action: Consumer\Action\Scope\GetCategories::class,
                    httpMethod: 'GET',
                    httpPath: '/scope/categories',
                    httpCode: 200,
                    outgoing: Model\Consumer\ScopeCategories::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns all scopes by category',
                ),
                'token.getAll' => new Operation(
                    action: Consumer\Action\Token\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/token',
                    httpCode: 200,
                    outgoing: Model\Consumer\TokenCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of tokens which are assigned to the authenticated user',
                ),
                'token.create' => new Operation(
                    action: Consumer\Action\Token\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/token',
                    httpCode: 201,
                    outgoing: Model\Consumer\TokenAccessToken::class,
                    incoming: Model\Consumer\TokenCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Creates a new token for the authenticated user',
                ),
                'token.get' => new Operation(
                    action: Consumer\Action\Token\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/token/$token_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Consumer\Token::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific token for the authenticated user',
                ),
                'token.update' => new Operation(
                    action: Consumer\Action\Token\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/token/$token_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Consumer\TokenAccessToken::class,
                    incoming: Model\Consumer\TokenUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates an existing token for the authenticated user',
                ),
                'token.delete' => new Operation(
                    action: Consumer\Action\Token\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/token/$token_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Deletes an existing token for the authenticated user',
                ),
                'webhook.getAll' => new Operation(
                    action: Consumer\Action\Webhook\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/webhook',
                    httpCode: 200,
                    outgoing: Model\Consumer\WebhookCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of webhooks which are assigned to the authenticated user',
                ),
                'webhook.create' => new Operation(
                    action: Consumer\Action\Webhook\Create::class,
                    httpMethod: 'POST',
                    httpPath: '/webhook',
                    httpCode: 201,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\WebhookCreate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Creates a new webhook for the authenticated user',
                ),
                'webhook.get' => new Operation(
                    action: Consumer\Action\Webhook\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/webhook/$webhook_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Consumer\Webhook::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific webhook for the authenticated user',
                ),
                'webhook.update' => new Operation(
                    action: Consumer\Action\Webhook\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/webhook/$webhook_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\WebhookUpdate::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates an existing webhook for the authenticated user',
                ),
                'webhook.delete' => new Operation(
                    action: Consumer\Action\Webhook\Delete::class,
                    httpMethod: 'DELETE',
                    httpPath: '/webhook/$webhook_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Deletes an existing webhook for the authenticated user',
                ),
                'transaction.getAll' => new Operation(
                    action: Consumer\Action\Transaction\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/transaction',
                    httpCode: 200,
                    outgoing: Model\Consumer\TransactionCollection::class,
                    parameters: ['startIndex' => PropertyTypeFactory::getInteger(), 'count' => PropertyTypeFactory::getInteger(), 'search' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a paginated list of transactions which are assigned to the authenticated user',
                ),
                'transaction.get' => new Operation(
                    action: Consumer\Action\Transaction\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/transaction/$transaction_id<[0-9]+|^~>',
                    httpCode: 200,
                    outgoing: Model\Consumer\Transaction::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a specific transaction for the authenticated user',
                ),
                'account.get' => new Operation(
                    action: Consumer\Action\User\Get::class,
                    httpMethod: 'GET',
                    httpPath: '/account',
                    httpCode: 200,
                    outgoing: Model\Consumer\UserAccount::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns a user data for the authenticated user',
                ),
                'account.update' => new Operation(
                    action: Consumer\Action\User\Update::class,
                    httpMethod: 'PUT',
                    httpPath: '/account',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\UserAccount::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Updates user data for the authenticated user',
                ),
                'account.changePassword' => new Operation(
                    action: Consumer\Action\User\ChangePassword::class,
                    httpMethod: 'PUT',
                    httpPath: '/account/change_password',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Backend\AccountChangePassword::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Change the password for the authenticated user',
                ),
                'account.activate' => new Operation(
                    action: Consumer\Action\User\Activate::class,
                    httpMethod: 'POST',
                    httpPath: '/activate',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\UserActivate::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Activates an previously registered account through a token which was provided to the user via email',
                ),
                'account.getApp' => new Operation(
                    action: Consumer\Action\User\GetApp::class,
                    httpMethod: 'GET',
                    httpPath: '/authorize',
                    httpCode: 200,
                    outgoing: Model\Consumer\AuthorizeMeta::class,
                    parameters: ['client_id' => PropertyTypeFactory::getString(), 'scope' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns information about a specific app to start the OAuth2 authorization code flow',
                ),
                'account.authorize' => new Operation(
                    action: Consumer\Action\User\Authorize::class,
                    httpMethod: 'POST',
                    httpPath: '/authorize',
                    httpCode: 200,
                    outgoing: Model\Consumer\AuthorizeResponse::class,
                    incoming: Model\Consumer\AuthorizeRequest::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Authorizes the access of a specific app for the authenticated user',
                ),
                'account.login' => new Operation(
                    action: Consumer\Action\User\Login::class,
                    httpMethod: 'POST',
                    httpPath: '/login',
                    httpCode: 200,
                    outgoing: Model\Consumer\UserJWT::class,
                    incoming: Model\Consumer\UserLogin::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'User login by providing a username and password',
                ),
                'account.refresh' => new Operation(
                    action: Consumer\Action\User\Refresh::class,
                    httpMethod: 'PUT',
                    httpPath: '/login',
                    httpCode: 200,
                    outgoing: Model\Consumer\UserJWT::class,
                    incoming: Model\Consumer\UserRefresh::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Refresh a previously obtained access token',
                ),
                'account.register' => new Operation(
                    action: Consumer\Action\User\Register::class,
                    httpMethod: 'POST',
                    httpPath: '/register',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\UserRegister::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Register a new user account',
                ),
                'account.requestPasswordReset' => new Operation(
                    action: Consumer\Action\User\ResetPassword\Request::class,
                    httpMethod: 'POST',
                    httpPath: '/password_reset',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\UserEmail::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Start the password reset flow',
                ),
                'account.executePasswordReset' => new Operation(
                    action: Consumer\Action\User\ResetPassword\Execute::class,
                    httpMethod: 'PUT',
                    httpPath: '/password_reset',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    incoming: Model\Consumer\UserPasswordReset::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Change the password after the password reset flow was started',
                ),
                'identity.getAll' => new Operation(
                    action: Consumer\Action\Identity\GetAll::class,
                    httpMethod: 'GET',
                    httpPath: '/identity',
                    httpCode: 200,
                    outgoing: Model\Consumer\IdentityCollection::class,
                    parameters: ['appId' => PropertyTypeFactory::getInteger()->setDeprecated(true), 'appKey' => PropertyTypeFactory::getString()],
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns a paginated list of identities which are relevant to the authenticated user',
                ),
                'identity.exchange' => new Operation(
                    action: Consumer\Action\Identity\Exchange::class,
                    httpMethod: 'GET',
                    httpPath: '/identity/:identity/exchange',
                    httpCode: 200,
                    outgoing: Passthru::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Identity callback endpoint to exchange an access token',
                ),
                'identity.redirect' => new Operation(
                    action: Consumer\Action\Identity\Redirect::class,
                    httpMethod: 'GET',
                    httpPath: '/identity/:identity/redirect',
                    httpCode: 200,
                    outgoing: Passthru::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Redirect the user to the configured identity provider',
                ),
            ],
            'system' => [
                'connection.callback' => new Operation(
                    action: System\Action\Connection\Callback::class,
                    httpMethod: 'GET',
                    httpPath: '/connection/:name/callback',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Connection OAuth2 callback to authorize a connection',
                ),
                'meta.getAbout' => new Operation(
                    action: System\Action\Meta\GetAbout::class,
                    httpMethod: 'GET',
                    httpPath: '/about',
                    httpCode: 200,
                    outgoing: Model\System\About::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns meta information and links about the current installed Fusio version',
                ),
                'meta.getDebug' => new Operation(
                    action: System\Action\Meta\GetDebug::class,
                    httpMethod: 'POST',
                    httpPath: '/debug',
                    httpCode: 200,
                    outgoing: Passthru::class,
                    incoming: Passthru::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Debug endpoint which returns the provided data',
                ),
                'meta.getHealth' => new Operation(
                    action: System\Action\Meta\GetHealth::class,
                    httpMethod: 'GET',
                    httpPath: '/health',
                    httpCode: 200,
                    outgoing: Model\System\HealthCheck::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Health check endpoint which returns information about the health status of the system',
                ),
                'meta.getRoutes' => new Operation(
                    action: System\Action\Meta\GetRoutes::class,
                    httpMethod: 'GET',
                    httpPath: '/route',
                    httpCode: 200,
                    outgoing: Model\System\Route::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns all available routes',
                ),
                'meta.getSchema' => new Operation(
                    action: System\Action\Meta\GetSchema::class,
                    httpMethod: 'GET',
                    httpPath: '/schema/:name',
                    httpCode: 200,
                    outgoing: Model\System\Schema::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Returns details of a specific schema',
                ),
                'payment.webhook' => new Operation(
                    action: System\Action\Payment\Webhook::class,
                    httpMethod: 'POST',
                    httpPath: '/payment/:provider/webhook',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    public: true,
                    description: 'Payment webhook endpoint after successful purchase of a plan',
                ),
            ],
            'authorization' => [
                'revoke' => new Operation(
                    action: Authorization\Action\Revoke::class,
                    httpMethod: 'POST',
                    httpPath: '/revoke',
                    httpCode: 200,
                    outgoing: Model\Common\Message::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Revoke the access token of the current authenticated user',
                ),
                'getWhoami' => new Operation(
                    action: Authorization\Action\GetWhoami::class,
                    httpMethod: 'GET',
                    httpPath: '/whoami',
                    httpCode: 200,
                    outgoing: Model\Backend\User::class,
                    throws: [999 => Model\Common\Message::class],
                    description: 'Returns user data of the current authenticated user',
                ),
            ],
        ];
    }

    /**
     * Reads files in new line neutral way that means we always use \n
     */
    private static function readFile(string $file): string
    {
        $lines = file(__DIR__ . '/resources/' . $file);
        if ($lines === false) {
            throw new \RuntimeException('Could not read file: ' . $file);
        }

        $lines = array_map('rtrim', $lines);
        return implode("\n", $lines);
    }

    private static function getContainer(): ContainerInterface
    {
        global $container;
        if ($container instanceof ContainerInterface) {
            return $container;
        }

        throw new \RuntimeException('Could not detect global container');
    }
}
