<?php
/*
 * Nom du projet : Flatboard
 * URL du projet : https://flatboard.org
 * Auteur : Frédéric Kaplon et contributeurs
 * Tout le code de Flatboard est publié sous la licence MIT.
 */

// Définition du type de contenu pour la sortie
$out['self'] = 'feed';
require_once __DIR__ . '/header.php';

// Configuration des en-têtes HTTP
header('Content-Type: application/atom+xml; charset=utf-8');
header('Cache-Control: public, max-age=300'); // Cache de 5 minutes

// Constantes pour améliorer la lisibilité
const FEED_NAMESPACE = 'http://www.w3.org/2005/Atom';
const XML_ENCODING = 'UTF-8';
const DEFAULT_ENTRIES_LIMIT = 20;

/**
 * Classe utilitaire pour la génération de flux RSS/Atom
 */
class AtomFeedGenerator
{
    private array $config;
    private array $out;
    private array $lang;
    /** @var array Cache pour les topics lus (évite les lectures multiples) */
    private array $topicCache = [];

    public function __construct(array $config, array $out, array $lang)
    {
        $this->config = $config;
        $this->out = $out;
        $this->lang = $lang;
    }

    /**
     * Échappe et formate du contenu pour XML
     */
    private function escapeXml(?string $content): string
    {
        if ($content === null) {
            return '';
        }
        return htmlspecialchars($content, ENT_XML1 | ENT_QUOTES | ENT_SUBSTITUTE, XML_ENCODING);
    }

    /**
     * Valide et nettoie une URL
     */
    private function sanitizeUrl(string $url): string
    {
        // Supprime les caractères potentiellement dangereux
        $url = filter_var($url, FILTER_SANITIZE_URL);
        return $url !== false ? $url : '';
    }

    /**
     * Génère une entrée Atom
     */
    private function generateAtomEntry(string $id, string $title, string $url, string $updated, string $summary): string
    {
        $escapedId = $this->escapeXml($id);
        $escapedTitle = $this->escapeXml($title);
        $escapedUrl = $this->sanitizeUrl($url);
        $escapedSummary = $this->escapeXml($summary);

        // Validation des données requises
        if (empty($escapedId) || empty($escapedTitle)) {
            error_log("Feed Generation Warning: Missing required data for entry - ID: $id, Title: $title");
            return '';
        }

        // Validation du format de date
        if (!$this->isValidAtomDate($updated)) {
            $updated = date('c'); // Fallback vers la date actuelle
        }

        return <<<XML
    <entry>
        <id>{$escapedId}</id>
        <title>{$escapedTitle}</title>
        <updated>{$updated}</updated>
        <link href="{$escapedUrl}"/>
        <summary type="html">{$escapedSummary}</summary>
    </entry>
XML;
    }

    /**
     * Valide le format de date Atom (ISO 8601)
     */
    private function isValidAtomDate(string $date): bool
    {
        $datetime = DateTime::createFromFormat(DateTime::ATOM, $date);
        return $datetime !== false && $datetime->format(DateTime::ATOM) === $date;
    }

    /**
     * Génère l'en-tête du flux Atom
     */
    private function generateFeedHeader(): string
    {
        try {
            $title = $this->escapeXml($this->config['name'] ?? 'Flatboard');
            $baseUrl = $this->escapeXml($this->out['baseURL'] ?? '');
            $updated = date('c');

            // Validation de l'URL de base
            if (empty($baseUrl)) {
                throw new Exception("Base URL is not configured properly");
            }

            return <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<feed xmlns="http://www.w3.org/2005/Atom">
    <title>{$title}</title>
    <link href="{$baseUrl}"/>
    <updated>{$updated}</updated>
    <author><name>{$title}</name></author>
    <id>{$baseUrl}</id>
XML;
        } catch (Exception $e) {
            error_log("Feed Header Generation Error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Traite et génère les entrées pour les sujets
     */
    private function processTopicEntries(array $topics, string $type = 'topic', int $limit = DEFAULT_ENTRIES_LIMIT): string
    {
        if (empty($topics)) {
            return $this->generateEmptyEntry("No {$type}s found");
        }

        // Réinitialise le cache pour chaque traitement
        $this->topicCache = [];

        // Tri et limitation des résultats
        arsort($topics, SORT_NATURAL | SORT_FLAG_CASE);
        $topics = array_slice($topics, 0, $limit, true);

        $entries = [];
        
        foreach ($topics as $topicId) {
            $entry = $this->generateTopicEntry($topicId, $type);
            if (!empty($entry)) {
                $entries[] = $entry;
            }
        }

        if (empty($entries)) {
            return $this->generateEmptyEntry("No valid {$type}s found");
        }

        return implode("\n", $entries);
    }

    /**
     * Génère une entrée pour un sujet spécifique
     */
    private function generateTopicEntry(string $topicId, string $type): ?string
    {
        // Validation de l'ID
        if (empty($topicId) || !is_string($topicId)) {
            error_log("Invalid topic ID: " . var_export($topicId, true));
            return null;
        }

        $topicEntry = flatDB::readEntry($type, $topicId);
        if (!$topicEntry || !is_array($topicEntry)) {
            error_log("Topic entry not found or invalid for {$type}/{$topicId}");
            return null;
        }

        $entryData = $this->buildEntryData($topicId, $topicEntry, $type);
        if (!$entryData) {
            return null;
        }

        return $this->generateAtomEntry(
            $entryData['id'],
            $entryData['title'],
            $entryData['url'],
            $entryData['updated'],
            $entryData['summary']
        );
    }

    /**
     * Construit les données d'une entrée
     */
    private function buildEntryData(string $topicId, array $topicEntry, string $type): ?array
    {
        // Vérification des données requises
        if (!isset($this->out['baseURL'])) {
            error_log("Base URL not configured");
            return null;
        }

        // Génération sécurisée du summary
        $summary = '';
        if (isset($topicEntry['content']) && class_exists('Parser')) {
            try {
                $summary = Parser::summary(Parser::content($topicEntry['content'], true));
            } catch (Exception $e) {
                error_log("Parser error for {$type} {$topicId}: " . $e->getMessage());
                $content = $topicEntry['content'] ?? '';
                $summary = $this->escapeXml(mb_substr($content, 0, 200, 'UTF-8')) . '...';
            }
        }

        $baseData = [
            'id' => $this->out['baseURL'] . 'view.php' . DS . $type . DS . $topicId,
            'updated' => $this->generateValidDate($topicId),
            'summary' => $summary
        ];

        if ($type === 'reply') {
            return $this->buildReplyEntryData($topicId, $topicEntry, $baseData);
        }

        // Données pour les topics standard
        $trip = $topicEntry['trip'] ?? 'Anonymous';
        $title = $topicEntry['title'] ?? 'Untitled';

        return array_merge($baseData, [
            'title' => $trip . ': ' . $title,
            'url' => 'view.php' . DS . $type . DS . $topicId
        ]);
    }

    /**
     * Génère une date valide au format ISO 8601 (Atom) à partir d'un ID
     */
    private function generateValidDate(string $topicId): string
    {
        // Utiliser Util::toDate si disponible (gère mieux le format)
        if (method_exists('Util', 'toDate')) {
            try {
                // Util::toDate avec 'c' retourne le format ISO 8601
                $dateStr = Util::toDate($topicId, 'c', false);
                // Vérifier que c'est un format ISO 8601 valide
                $dateTime = DateTime::createFromFormat(DateTime::ATOM, $dateStr);
                if ($dateTime !== false) {
                    return $dateStr;
                }
                // Si ce n'est pas ISO 8601, convertir
                $dateTime = DateTime::createFromFormat('Y/m/d H:i', $dateStr);
                if ($dateTime !== false) {
                    return $dateTime->format('c');
                }
            } catch (Exception $e) {
                error_log("Util::toDate error for ID {$topicId}: " . $e->getMessage());
            }
        }
        
        // Fallback: parse l'ID comme date (format: YYYY-MM-DDHHMMSS)
        if (strlen($topicId) >= 16) {
            $dateStr = substr($topicId, 0, 16);
            // Format: YYYY-MM-DDHHMMSS -> YYYY-MM-DD HH:MM:SS
            if (preg_match('/^(\d{4})-(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})$/', $dateStr, $matches)) {
                try {
                    $formattedDate = "{$matches[1]}-{$matches[2]}-{$matches[3]} {$matches[4]}:{$matches[5]}:{$matches[6]}";
                    $dateTime = new DateTime($formattedDate, new DateTimeZone('Europe/Paris'));
                    return $dateTime->format('c');
                } catch (Exception $e) {
                    error_log("Date parsing error for ID {$topicId}: " . $e->getMessage());
                }
            }
        }
        
        // Dernière solution: date actuelle
        return date('c');
    }

    /**
     * Récupère un topic depuis le cache ou la base de données
     */
    private function getCachedTopic(string $topicId): ?array
    {
        if (!isset($this->topicCache[$topicId])) {
            $topic = flatDB::readEntry('topic', $topicId);
            if ($topic && is_array($topic)) {
                $this->topicCache[$topicId] = $topic;
            } else {
                return null;
            }
        }
        return $this->topicCache[$topicId];
    }

    /**
     * Construit les données spécifiques pour une réponse
     */
    private function buildReplyEntryData(string $topicId, array $topicEntry, array $baseData): ?array
    {
        if (!isset($topicEntry['topic'])) {
            error_log("Reply entry missing topic reference for {$topicId}");
            return null;
        }

        $forumTopic = $this->getCachedTopic($topicEntry['topic']);
        if (!$forumTopic) {
            error_log("Forum topic not found for reply {$topicId}, topic: " . $topicEntry['topic']);
            return null;
        }

        // Calcul sécurisé de la page
        $pageNumber = 1;
        if (isset($forumTopic['reply']) && is_array($forumTopic['reply'])) {
            try {
                $pageNumber = Util::onPage($topicId, $forumTopic['reply']);
            } catch (Exception $e) {
                error_log("Error calculating page number for reply {$topicId}: " . $e->getMessage());
            }
        }

        $url = 'view.php' . DS . 'topic' . DS . $topicEntry['topic'] . 
               DS . 'p' . DS . $pageNumber . '#' . $topicId;

        $trip = $topicEntry['trip'] ?? 'Anonymous';
        $title = $forumTopic['title'] ?? 'Untitled';

        return array_merge($baseData, [
            'title' => $trip . ': ' . $title,
            'url' => $url
        ]);
    }

    /**
     * Génère une entrée vide avec un message
     */
    private function generateEmptyEntry(string $message): string
    {
        return $this->generateAtomEntry(
            $this->out['baseURL'] ?? '',
            $message,
            '',
            date('c'),
            $message
        );
    }

    /**
     * Génère le flux pour tous les sujets
     */
    public function generateTopicsFeed(): string
    {
        $this->out['subtitle'] = $this->lang['topic'] ?? 'Topics';
        $this->out['type'] = 'topic';
        
        $topics = flatDB::listEntry('topic');
        if ($topics === false || empty($topics)) {
            return $this->generateEmptyEntry('No topics found');
        }
        
        return $this->processTopicEntries($topics, 'topic');
    }

    /**
     * Génère le flux pour toutes les réponses
     */
    public function generateRepliesFeed(): string
    {
        $this->out['subtitle'] = $this->lang['reply'] ?? 'Replies';
        $this->out['type'] = 'reply';
        
        $replies = flatDB::listEntry('reply');
        if ($replies === false || empty($replies)) {
            return $this->generateEmptyEntry('No replies found');
        }
        
        return $this->processTopicEntries($replies, 'reply');
    }

    /**
     * Génère le flux pour un forum spécifique
     */
    public function generateForumFeed(string $forumId): string
    {
        // Validation de l'ID du forum
        if (!flatDB::isValidEntry('forum', $forumId)) {
            error_log("Invalid forum ID: " . $forumId);
            return $this->generateEmptyEntry('Forum not found');
        }

        $forumEntry = flatDB::readEntry('forum', $forumId);
        if (!$forumEntry || !is_array($forumEntry)) {
            return $this->generateEmptyEntry('Forum not found');
        }

        $this->out['subtitle'] = $this->escapeXml($forumEntry['name'] ?? 'Unknown Forum');
        $this->out['type'] = 'forum';
        
        $topics = $forumEntry['topic'] ?? [];
        if (!is_array($topics)) {
            error_log("Forum topics data is not an array for forum {$forumId}");
            return $this->generateEmptyEntry('Forum has no topics');
        }
        
        return $this->processTopicEntries($topics, 'topic');
    }

    /**
     * Génère le flux pour un sujet spécifique
     */
    public function generateTopicFeed(string $topicId): string
    {
        // Validation de l'ID du sujet
        if (!flatDB::isValidEntry('topic', $topicId)) {
            error_log("Invalid topic ID: " . $topicId);
            return $this->generateEmptyEntry('Topic not found');
        }

        $topicEntry = flatDB::readEntry('topic', $topicId);
        if (!$topicEntry || !is_array($topicEntry)) {
            return $this->generateEmptyEntry('Topic not found');
        }

        $this->out['subtitle'] = $this->escapeXml($topicEntry['title'] ?? 'Untitled');
        $this->out['type'] = 'thread';

        $url = 'view.php' . DS . 'topic' . DS . $topicId;
        
        // Génération sécurisée du summary
        $summary = '';
        if (isset($topicEntry['content']) && class_exists('Parser')) {
            try {
                $summary = Parser::summary(Parser::content($topicEntry['content'], true));
            } catch (Exception $e) {
                error_log("Parser error for topic {$topicId}: " . $e->getMessage());
                $content = $topicEntry['content'] ?? '';
                $summary = $this->escapeXml(mb_substr($content, 0, 200, 'UTF-8')) . '...';
            }
        }

        return $this->generateAtomEntry(
            $this->out['baseURL'] . $url,
            ($topicEntry['trip'] ?? 'Anonymous') . ': ' . ($topicEntry['title'] ?? 'Untitled'),
            $url,
            $this->generateValidDate($topicId),
            $summary
        );
    }

    /**
     * Génère le flux complet
     */
    public function generateFeed(): string
    {
        $content = '';

        // Détermine le type de flux à générer
        if (Util::isGET('topic')) {
            $content = $this->generateTopicsFeed();
        } elseif (Util::isGET('reply')) {
            $content = $this->generateRepliesFeed();
        } elseif (Util::isGETValidEntry('forum', 'forum')) {
            $content = $this->generateForumFeed($_GET['forum']);
        } elseif (Util::isGETValidEntry('topic', 'topic')) {
            $content = $this->generateTopicFeed($_GET['topic']);
        } else {
            // Retourne un flux d'erreur valide au lieu de rediriger
            error_log("Feed Error: Invalid or missing parameters. GET parameters: " . json_encode($_GET));
            return $this->generateErrorFeed("Invalid or missing feed parameters");
        }

        $header = $this->generateFeedHeader();
        return $header . "\n" . $content . "\n</feed>";
    }

    /**
     * Génère un flux d'erreur minimal mais valide XML
     */
    private function generateErrorFeed(string $errorMessage): string
    {
        $title = $this->escapeXml($this->config['name'] ?? 'Flatboard');
        $baseUrl = $this->escapeXml($this->out['baseURL'] ?? 'http://localhost/');
        $updated = date('c');
        $escapedError = $this->escapeXml($errorMessage);

        return <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<feed xmlns="http://www.w3.org/2005/Atom">
    <title>{$title} - Error</title>
    <link href="{$baseUrl}"/>
    <updated>{$updated}</updated>
    <author><name>{$title}</name></author>
    <id>{$baseUrl}</id>
    <entry>
        <id>{$baseUrl}error</id>
        <title>Feed Error</title>
        <updated>{$updated}</updated>
        <link href="{$baseUrl}"/>
        <summary type="html">Feed temporarily unavailable: {$escapedError}</summary>
    </entry>
</feed>
XML;
    }
}

// Utilisation de la classe optimisée avec gestion d'erreur améliorée
try {
    // Vérification des prérequis
    if (!isset($config) || !is_array($config)) {
        throw new Exception("Configuration not properly loaded");
    }
    
    if (!isset($out) || !is_array($out)) {
        throw new Exception("Output configuration not properly loaded");
    }
    
    if (!isset($lang) || !is_array($lang)) {
        throw new Exception("Language configuration not properly loaded");
    }

    // Vérification de la classe flatDB
    if (!class_exists('flatDB')) {
        throw new Exception("flatDB class not found - check if Flatdb.lib.php is properly loaded");
    }

    // Vérification de la classe Util
    if (!class_exists('Util')) {
        throw new Exception("Util class not found - check if Utils.lib.php is properly loaded");
    }

    $feedGenerator = new AtomFeedGenerator($config, $out, $lang);
    echo $feedGenerator->generateFeed();
    
} catch (Error $e) {
    // Erreurs PHP fatales
    error_log("PHP Fatal Error in RSS Feed: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    
    // En-têtes d'erreur appropriés
    if (!headers_sent()) {
        header('Content-Type: application/atom+xml; charset=utf-8');
        header('HTTP/1.1 500 Internal Server Error');
    }
    
    // Flux d'erreur minimal
    echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    echo '<feed xmlns="http://www.w3.org/2005/Atom">' . "\n";
    echo '<title>Feed Error</title>' . "\n";
    echo '<updated>' . date('c') . '</updated>' . "\n";
    echo '<entry><title>Error</title><updated>' . date('c') . '</updated><summary>Feed temporarily unavailable due to a system error.</summary></entry>' . "\n";
    echo '</feed>';
    
} catch (Exception $e) {
    // Exceptions normales - retourne un flux d'erreur valide au lieu de rediriger
    error_log("RSS Feed Exception: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    
    if (!headers_sent()) {
        header('Content-Type: application/atom+xml; charset=utf-8');
        header('HTTP/1.1 500 Internal Server Error');
    }
    
    // Génère un flux d'erreur valide
    $title = isset($config['name']) ? htmlspecialchars($config['name'], ENT_XML1 | ENT_QUOTES, 'UTF-8') : 'Flatboard';
    $baseUrl = isset($out['baseURL']) ? htmlspecialchars($out['baseURL'], ENT_XML1 | ENT_QUOTES, 'UTF-8') : 'http://localhost/';
    $updated = date('c');
    $escapedError = htmlspecialchars($e->getMessage(), ENT_XML1 | ENT_QUOTES, 'UTF-8');
    
    echo <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<feed xmlns="http://www.w3.org/2005/Atom">
    <title>{$title} - Error</title>
    <link href="{$baseUrl}"/>
    <updated>{$updated}</updated>
    <author><name>{$title}</name></author>
    <id>{$baseUrl}</id>
    <entry>
        <id>{$baseUrl}error</id>
        <title>Feed Error</title>
        <updated>{$updated}</updated>
        <link href="{$baseUrl}"/>
        <summary type="html">Feed temporarily unavailable: {$escapedError}</summary>
    </entry>
</feed>
XML;
}
?>