<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
 */

class Paginate
{
    /**
     * Constructeur protégé car il s'agit d'une classe statique.
     *
     * @access protected
     */
    protected function __construct()
    {
        // Rien ici
    }

    /**
     * Générer les liens de pagination.
     *
     * @param int $p Numéro de la page actuelle
     * @param int $total Nombre total de pages
     * @param string $loc Lien de base pour la pagination
     * @return string|false Les liens de pagination ou false en cas d'échec
     */
    public static function pageLink($p, $total, $loc)
    {
        // Déterminer le premier et le dernier lien à afficher
        $start = max(1, $p - 4);
        $end = min($total, $p + 4);
        
        $out = ' 		
            <nav aria-label="pagination">
              <ul class="pagination justify-content-center mt-3">' .
                ($p > 1 
                    ? '<li class="page-item page-prev"><a class="page-link" href="' . $loc . DS . 'p' . DS . ($p - 1) . '">&larr;</a></li>' 
                    : '<li class="page-item disabled"><a class="page-link" href="' . $loc . DS . 'p' . DS . ($p - 1) . '" tabindex="-1">&larr;</a></li>') .
                ($start === 1 ? '' : '<li class="page-item"><a class="page-link" href="' . $loc . DS . 'p' . DS . ($start - 1) . '">…</a></li>');

        for ($i = $start; $i <= $end; $i++) {
            if ($p === $i) {
                $out .= '<li class="page-item active"><span class="page-link">' . $i . '</span></li>';
            } else {
                $out .= '<li class="page-item"><a class="page-link" href="' . $loc . DS . 'p' . DS . $i . '">' . $i . '</a></li>';
            }
        }

        $out .= ($end === $total ? '' : '<li class="page-item"><a class="page-link" href="' . $loc . DS . 'p' . DS . ($end + 1) . '">…</a></li>') .
                ($p < $total 
                    ? '<li id="next" class="page-item"><a class="page-link" href="' . $loc . DS . 'p' . DS . ($p + 1) . '">&rarr;</a></li>' 
                    : '<li class="page-item disabled"><a class="page-link" href="' . $loc . DS . 'p' . DS . ($p + 1) . '" tabindex="-1">&rarr;</a></li>');

        $out .= '</ul>
          </nav>';

        // Retourner tous les liens de pagination  
        return $out;
    }

    /**
     * Compter le nombre de pages nécessaires.
     *
     * @param array $items Liste des éléments
     * @param int $nb Nombre d'éléments par page
     * @return int Nombre total de pages
     */
    public static function countPage($items, $nb)
    {
        $itemNum = count($items);
        return $itemNum === 0 ? 1 : (int) ceil($itemNum / $nb);
    }

    /**
     * Obtenir les éléments à afficher pour une page donnée.
     *
     * @param array $items Liste des éléments
     * @param int $p Numéro de la page actuelle
     * @param int $nb Nombre d'éléments par page
     * @return array Les éléments à afficher
     */
    public static function viewPage($items, $p, $nb)
    {
        return array_slice($items, $nb * ($p - 1), $nb);
    }

    /**
     * Obtenir le numéro de la page à afficher.
     *
     * @param int $total Nombre total de pages
     * @return int Numéro de la page valide
     */
    public static function pid($total)
    {
        if (!Util::isGET('p')) {
            return 1; // Retourne la première page si 'p' n'est pas défini
        }

        $p = (int) $_GET['p'];
        if ($p >= 1 && $p <= $total) {
            return $p; // Retourne le numéro de page valide
        } else {
            return 1; // Retourne la première page si le numéro est invalide
        }
    }
}