<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
 *
 * BBCode to HTML converter
 * Version: 3.8 - Optimized
 */

class BBCode
{
    protected $bbcode_table = array();
    
    // Cache statique partagé pour les quotes (évite les lectures répétées de flatDB)
    protected static $quoteCache = [];
    
    // Cache pour les IDs générés (évite les collisions)
    protected static $generatedIds = [];

    /**
     * Constructor - Initialize BBCode patterns
     */
    public function __construct()
    {
        $this->initializeBBCodePatterns();
    }

    /**
     * Initialize all BBCode patterns
     */
    protected function initializeBBCodePatterns()
    {
        // Text formatting
        $this->bbcode_table["/\[b\](.*?)\[\/b\]/is"] = function ($match) {
            return "<strong>{$match[1]}</strong>";
        };

        $this->bbcode_table["/\[i\](.*?)\[\/i\]/is"] = function ($match) {
            return "<em>{$match[1]}</em>";
        };

        $this->bbcode_table["/\[u\](.*?)\[\/u\]/is"] = function ($match) {
            return '<span style="text-decoration:underline;">' . $match[1] . '</span>';
        };

        $this->bbcode_table["/\[s\](.*?)\[\/s\]/is"] = function ($match) {
            return "<del>{$match[1]}</del>";
        };

        // Font size
        $this->bbcode_table["/\[size=(\d+)\](.*?)\[\/size\]/is"] = function ($match) {
            $size = min(max((int)$match[1], 8), 72); // Limiter entre 8px et 72px
            return "<span style=\"font-size:{$size}px\">{$match[2]}</span>";
        };

        // Font color
        $this->bbcode_table["/\[color=([#a-zA-Z0-9]+)\](.*?)\[\/color\]/is"] = function ($match) {
            $color = $this->sanitizeColor($match[1]);
            return "<span style=\"color:{$color}\">{$match[2]}</span>";
        };

        // Code blocks
        $this->bbcode_table["/\[code\](.*?)\[\/code\]/is"] = function ($match) {
            global $cur;
            if ($cur === 'home') {
                return '🗐…';
            }
            $code = str_replace('<br />', '', $match[1]);
            return '<pre class="code" data-lang="CODE"><code>' . htmlspecialchars($code, ENT_QUOTES, 'UTF-8') . '</code></pre>';
        };

        // Blockquote
        $this->bbcode_table["/\[blockquote\](.*?)\[\/blockquote\]/is"] = function ($match) {
            return "<blockquote><p>{$match[1]}</p></blockquote>";
        };

        // Quote with reply ID
        $this->bbcode_table["/\[quote\](\d{4}-\d{2}-\d{8}[a-z\d]{5})\[\/quote\]/is"] = function ($match) {
            return $this->processQuote($match[1]);
        };

        // Alignment
        $this->bbcode_table["/\[center\](.*?)\[\/center\]/is"] = function ($match) {
            return '<div style="text-align:center;">' . $match[1] . '</div>';
        };

        $this->bbcode_table["/\[right\](.*?)\[\/right\]/is"] = function ($match) {
            return '<div style="text-align:right;">' . $match[1] . '</div>';
        };

        // Hide content
        $this->registerHidePatterns();

        // Links and emails
        $this->registerLinkPatterns();

        // Images
        $this->registerImagePatterns();

        // Lists
        $this->registerListPatterns();

        // Media (videos)
        $this->registerMediaPatterns();
    }

    /**
     * Register hide patterns
     */
    protected function registerHidePatterns()
    {
        // Basic hide
        $this->bbcode_table["/\[hide=(.*?)\](.*?)\[\/hide\]/is"] = function ($match) {
            global $cur;
            if ($cur === 'home') return '…';
            
            $hideId = $this->generateUniqueId('hide');
            $buttonText = htmlspecialchars($match[1], ENT_QUOTES, 'UTF-8');
            
            return $this->renderToggleContent($hideId, $buttonText, $match[2]);
        };

        // Hide for logged users
        $this->bbcode_table["/\[hideuser=(.*?)\](.*?)\[\/hideuser\]/is"] = function ($match) {
            global $sessionTrip, $cur, $lang;
            
            if ($cur === 'home') return '…';
            
            if ($sessionTrip || User::isWorker()) {
                $hideId = $this->generateUniqueId('hideuser');
                $buttonText = htmlspecialchars($match[1], ENT_QUOTES, 'UTF-8');
                return $this->renderToggleContent($hideId, $buttonText, $match[2]);
            }
            
            return '<div class="alert alert-danger" role="alert"><i class="fa fa-exclamation-triangle" aria-hidden="true"></i> ' . $lang['visible_for_logged'] . '</div>';
        };

        // Hide for staff
        $this->bbcode_table["/\[hideworker=(.*?)\](.*?)\[\/hideworker\]/is"] = function ($match) {
            global $cur, $lang;
            
            if ($cur === 'home') return '…';
            
            if (User::isWorker()) {
                $hideId = $this->generateUniqueId('hideworker');
                $buttonText = htmlspecialchars($match[1], ENT_QUOTES, 'UTF-8');
                return $this->renderToggleContent($hideId, $buttonText, $match[2]);
            }
            
            return '<div class="alert alert-danger" role="alert"><i class="fa fa-exclamation-triangle" aria-hidden="true"></i> ' . $lang['visible_for_staff'] . '</div>';
        };

        // Hide for specific user
        $this->bbcode_table["/\[hidetrip=(.*?)\](.*?)\[\/hidetrip\]/is"] = function ($match) {
            global $cur, $sessionTrip, $lang;
            
            if ($cur === 'home') return '…';
            
            $tripCrypt = HTMLForm::trip($sessionTrip, null);
            
            if (User::isWorker() || $tripCrypt === $match[1]) {
                $hideId = $this->generateUniqueId('hidetrip');
                return $this->renderToggleContent($hideId, $lang['hide_show_more'], $match[2]);
            }
            
            return '<div class="alert alert-danger" role="alert"><i class="fa fa-exclamation-triangle" aria-hidden="true"></i> ' . $lang['visible_for_specific_user'] . '</div>';
        };
    }

    /**
     * Register link patterns
     */
    protected function registerLinkPatterns()
    {
        // Email
        $this->bbcode_table["/\[email\](.*?)\[\/email\]/is"] = function ($match) {
            $email = htmlspecialchars($match[1], ENT_QUOTES, 'UTF-8');
            return "<a href=\"mailto:{$email}\">{$email}</a>";
        };

        $this->bbcode_table["/\[email=(.*?)\](.*?)\[\/email\]/is"] = function ($match) {
            $email = htmlspecialchars($match[1], ENT_QUOTES, 'UTF-8');
            $text = htmlspecialchars($match[2], ENT_QUOTES, 'UTF-8');
            return "<a href=\"mailto:{$email}\">{$text}</a>";
        };

        // URL
        $this->bbcode_table["/\[url\](.*?)\[\/url\]/is"] = function ($match) {
            $url = $this->sanitizeUrl($match[1]);
            return "<a href=\"{$url}\" rel=\"noopener noreferrer\" target=\"_blank\">{$url}</a>";
        };

        $this->bbcode_table["/\[url=(.*?)\](.*?)\[\/url\]/is"] = function ($match) {
            $url = $this->sanitizeUrl($match[1]);
            $text = htmlspecialchars($match[2], ENT_QUOTES, 'UTF-8');
            return "<a href=\"{$url}\" rel=\"noopener noreferrer\" target=\"_blank\">{$text}</a>";
        };
    }

    /**
     * Register image patterns
     */
    protected function registerImagePatterns()
    {
        // Basic image
        $this->bbcode_table["/\[img\](.*?)\[\/img\]/is"] = function ($match) {
            global $cur;
            if ($cur === 'home') return '🖼…';
            
            $url = $this->sanitizeUrl($match[1]);
            return '<img src="' . $url . '" class="img-fluid img-thumbnail" alt="image" loading="lazy">';
        };

        // Image with dimensions
        $this->bbcode_table["/\[img width=(\d+),height=(\d+)\](.*?)\[\/img\]/is"] = function ($match) {
            global $cur;
            if ($cur === 'home') return '🖼…';
            
            $width = min((int)$match[1], 2000);
            $height = min((int)$match[2], 2000);
            $url = $this->sanitizeUrl($match[3]);
            
            return '<img style="width:' . $width . 'px;height:' . $height . 'px" src="' . $url . '" class="img-fluid img-thumbnail" alt="image" loading="lazy">';
        };
    }

    /**
     * Register list patterns
     */
    protected function registerListPatterns()
    {
        // Unordered list
        $this->bbcode_table["/\[list\](.*?)\[\/list\]/is"] = function ($match) {
            $items = preg_replace_callback("/\[\*\]([^\[\*\]]*)/is", function ($submatch) {
                return "<li>" . trim($submatch[1]) . "</li>";
            }, $match[1]);
            
            return "<ul>" . preg_replace("/[\n\r]/", "", $items) . "</ul>";
        };

        // Ordered list
        $this->bbcode_table["/\[list=(1|a)\](.*?)\[\/list\]/is"] = function ($match) {
            $listType = $match[1] === 'a' 
                ? '<ol style="list-style-type: lower-alpha">' 
                : '<ol>';
            
            $items = preg_replace_callback("/\[\*\]([^\[\*\]]*)/is", function ($submatch) {
                return "<li>" . trim($submatch[1]) . "</li>";
            }, $match[2]);
            
            return $listType . preg_replace("/[\n\r]/", "", $items) . "</ol>";
        };
    }

    /**
     * Register media patterns (videos)
     */
    protected function registerMediaPatterns()
    {
        // YouTube
        $this->bbcode_table["/\[youtube\](.*?)\[\/youtube\]/s"] = function ($match) {
            global $cur;
            if ($cur === 'home') return '🎬… ';
            
            $videoId = $this->sanitizeYoutubeId($match[1]);
            return '<div class="embed-responsive embed-responsive-16by9"><iframe class="embed-responsive-item" src="https://www.youtube.com/embed/' . $videoId . '" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe></div>';
        };

        // Dailymotion
        $this->bbcode_table["/\[dailymotion\](.*?)\[\/dailymotion\]/s"] = function ($match) {
            global $cur;
            if ($cur === 'home') return '🎬… ';
            
            $videoId = $this->sanitizeDailymotionId($match[1]);
            return '<div class="embed-responsive embed-responsive-16by9"><iframe class="embed-responsive-item" src="https://www.dailymotion.com/embed/video/' . $videoId . '" allowfullscreen frameborder="0"></iframe></div>';
        };

        // Generic video
        $this->bbcode_table["/\[video width=(\d+),height=(\d+)\](.*?)\[\/video\]/s"] = function ($match) {
            global $cur;
            if ($cur === 'home') return '🎬… ';
            
            $width = min((int)$match[1], 1920);
            $height = min((int)$match[2], 1080);
            $url = $this->sanitizeUrl($match[3]);
            
            return '<video width="' . $width . '" height="' . $height . '" controls preload="metadata"><source src="' . $url . '" type="video/mp4">Your browser does not support the video tag.</video>';
        };
    }

    /**
     * Process quote with reply ID
     */
    protected function processQuote($replyId)
    {
        if (!flatDB::isValidEntry('reply', $replyId)) {
            return '<a class="badge badge-pill badge-info">[?]</a>';
        }

        global $lang;

        // Utiliser le cache statique si disponible
        if (!isset(self::$quoteCache[$replyId])) {
            $replyEntry = flatDB::readEntry('reply', $replyId);
            if ($replyEntry && isset($replyEntry['topic'])) {
                $topicEntry = flatDB::readEntry('topic', $replyEntry['topic']);
                self::$quoteCache[$replyId] = [
                    'replyEntry' => $replyEntry,
                    'topicEntry' => $topicEntry
                ];
            } else {
                self::$quoteCache[$replyId] = null;
            }
        }

        if (self::$quoteCache[$replyId] === null) {
            return '<a class="badge badge-pill badge-info">[?]</a>';
        }

        $replyEntry = self::$quoteCache[$replyId]['replyEntry'];
        $topicEntry = self::$quoteCache[$replyId]['topicEntry'];
        
        $page = Util::onPage($replyId, $topicEntry['reply']);
        $trip = htmlspecialchars($replyEntry['trip'], ENT_QUOTES, 'UTF-8');
        
        return '<a class="badge badge-pill badge-info" href="view.php/topic/' . $replyEntry['topic'] . '/p/' . $page . '#' . $replyId . '" data-toggle="tooltip" data-placement="top" title="' . $lang['quote_by'] . ' ' . $trip . '"><i class="fa fa-quote-left"></i></a>&nbsp;';
    }

    /**
     * Render toggle content (for hide features)
     */
    protected function renderToggleContent($hideId, $buttonText, $content)
    {
        return '<button class="btn btn-outline-secondary btn-sm" type="button" data-toggle="collapse" data-target="#' . $hideId . '" aria-expanded="false" aria-controls="' . $hideId . '">' . 
            $buttonText . 
            '</button><div class="collapse" id="' . $hideId . '">' . 
            $content . 
            '</div>';
    }

    /**
     * Convert BBCode to HTML
     */
    public function toHTML($str, $escapeHTML = false, $nr2br = false)
    {
        if (empty($str)) return '';
        
        if ($escapeHTML) {
            $str = htmlspecialchars($str, ENT_QUOTES, 'UTF-8');
        }
        
        foreach ($this->bbcode_table as $pattern => $callback) {
            $str = preg_replace_callback($pattern, $callback, $str);
        }
        
        if ($nr2br) {
            $str = nl2br($str, false);
        }
        
        return $str;
    }

    /**
     * Generate unique ID
     */
    protected function generateUniqueId($prefix = 'bbcode', $length = 8)
    {
        do {
            $id = $prefix . '_' . $this->genUid($length);
        } while (isset(self::$generatedIds[$id]));
        
        self::$generatedIds[$id] = true;
        return $id;
    }

    /**
     * Generate random UID
     */
    public function genUid($length = 4)
    {
        return substr(str_shuffle("0123456789abcdefghijklmnopqrstuvwxyz"), 0, $length);
    }

    /**
     * Sanitize color value
     */
    protected function sanitizeColor($color)
    {
        // Allow hex colors and named colors
        if (preg_match('/^#[0-9A-Fa-f]{3,6}$/', $color)) {
            return $color;
        }
        
        $allowedColors = ['red', 'blue', 'green', 'yellow', 'orange', 'purple', 'black', 'white', 'gray'];
        if (in_array(strtolower($color), $allowedColors)) {
            return strtolower($color);
        }
        
        return '#000000';
    }

    /**
     * Sanitize URL
     */
    protected function sanitizeUrl($url)
    {
        $url = trim($url);
        
        // Block javascript: and data: URLs for security
        if (preg_match('/^(javascript|data|vbscript):/i', $url)) {
            return '#';
        }
        
        return htmlspecialchars($url, ENT_QUOTES, 'UTF-8');
    }

    /**
     * Sanitize YouTube video ID
     */
    protected function sanitizeYoutubeId($input)
    {
        // Extract video ID from various YouTube URL formats
        if (preg_match('/^[a-zA-Z0-9_-]{11}$/', $input)) {
            return $input;
        }
        
        // Try to extract from URL
        if (preg_match('/(?:youtube\.com\/watch\?v=|youtu\.be\/)([a-zA-Z0-9_-]{11})/', $input, $matches)) {
            return $matches[1];
        }
        
        return '';
    }

    /**
     * Sanitize Dailymotion video ID
     */
    protected function sanitizeDailymotionId($input)
    {
        // Extract video ID from various Dailymotion URL formats
        if (preg_match('/^[a-zA-Z0-9]+$/', $input)) {
            return $input;
        }
        
        // Try to extract from URL
        if (preg_match('/dailymotion\.com\/video\/([a-zA-Z0-9]+)/', $input, $matches)) {
            return $matches[1];
        }
        
        return '';
    }

    /**
     * Clear static caches (useful for testing or memory management)
     */
    public static function clearCaches()
    {
        self::$quoteCache = [];
        self::$generatedIds = [];
    }
}